/*
 * Decompiled with CFR 0.152.
 */
package net.sf.picard.illumina;

import java.io.BufferedWriter;
import java.io.File;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import net.sf.picard.cmdline.CommandLineProgram;
import net.sf.picard.cmdline.Option;
import net.sf.picard.cmdline.Usage;
import net.sf.picard.illumina.parser.ClusterData;
import net.sf.picard.illumina.parser.IlluminaDataProvider;
import net.sf.picard.illumina.parser.IlluminaDataProviderFactory;
import net.sf.picard.illumina.parser.IlluminaDataType;
import net.sf.picard.illumina.parser.ReadDescriptor;
import net.sf.picard.illumina.parser.ReadStructure;
import net.sf.picard.illumina.parser.ReadType;
import net.sf.picard.io.IoUtil;
import net.sf.picard.metrics.MetricBase;
import net.sf.picard.metrics.MetricsFile;
import net.sf.picard.util.IlluminaUtil;
import net.sf.picard.util.Log;
import net.sf.picard.util.TabbedTextFileWithHeaderParser;
import net.sf.samtools.util.SequenceUtil;
import net.sf.samtools.util.StringUtil;

public class ExtractIlluminaBarcodes
extends CommandLineProgram {
    @Usage
    public String USAGE = this.getStandardUsagePreamble() + "Determine the barcode for each read in an Illumina lane.\n" + "For each tile, a file is written to the basecalls directory of the form s_<lane>_<tile>_barcode.txt. " + "An output file contains a line for each read in the tile, aligned with the regular basecall output. \n" + "The output file contains the following tab-separated columns: \n" + "    * read subsequence at barcode position\n" + "    * Y or N indicating if there was a barcode match\n" + "    * matched barcode sequence\n" + "Note that the order of specification of barcodes can cause arbitrary differences in output for poorly matching barcodes.\n\n";
    @Option(doc="The Illumina basecalls directory. ", shortName="B")
    public File BASECALLS_DIR;
    @Option(doc="Where to write _barcode.txt files.  By default, these are written to BASECALLS_DIR.", optional=true)
    public File OUTPUT_DIR;
    @Option(doc="Lane number. ", shortName="L")
    public Integer LANE;
    @Option(doc="A description of the logical structure of clusters in an Illumina Run, i.e. a description of the structure IlluminaBasecallsToSam assumes the  data to be in. It should consist of integer/character pairs describing the number of cycles and the type of those cycles (B for Barcode, T for Template, and S for skip).  E.g. If the input data consists of 80 base clusters and we provide a read structure of \"36T8B8S28T\" then, before being converted to SAM records those bases will be split into 4 reads where read one consists of 36 cycles of template, read two consists of 8 cycles of barcode, read three will be an 8 base read of skipped cycles and read four is another 28 cycle template read.  The read consisting of skipped cycles would NOT be included in output SAM/BAM file read groups.", shortName="RS")
    public String READ_STRUCTURE;
    @Option(doc="Barcode sequence.  These must be unique, and all the same length.  This cannot be used with reads that have more than one barcode; use BARCODE_FILE in that case. ", mutex={"BARCODE_FILE"})
    public List<String> BARCODE = new ArrayList<String>();
    @Option(doc="Tab-delimited file of barcode sequences, barcode name and, optionally, library name.  Barcodes must be unique and all the same length.  Column headers must be 'barcode_sequence_1', 'barcode_sequence_2' (optional), 'barcode_name', and 'library_name'.", mutex={"BARCODE"})
    public File BARCODE_FILE;
    @Option(doc="Per-barcode and per-lane metrics written to this file.", shortName="M")
    public File METRICS_FILE;
    @Option(doc="Maximum mismatches for a barcode to be considered a match.")
    public int MAX_MISMATCHES = 1;
    @Option(doc="Minimum difference between number of mismatches in the best and second best barcodes for a barcode to be considered a match.")
    public int MIN_MISMATCH_DELTA = 1;
    @Option(doc="Maximum allowable number of no-calls in a barcode read before it is considered unmatchable.")
    public int MAX_NO_CALLS = 2;
    @Option(shortName="Q", doc="Minimum base quality. Any barcode bases falling below this quality will be considered a mismatch even in the bases match.")
    public int MINIMUM_BASE_QUALITY = 0;
    @Option(shortName="GZIP", doc="Compress output s_l_t_barcode.txt files using gzip and append a .gz extension to the filenames.")
    public boolean COMPRESS_OUTPUTS = false;
    @Option(doc="Run this many PerTileBarcodeExtractors in parallel.  If NUM_PROCESSORS = 0, number of cores is automatically set to the number of cores available on the machine. If NUM_PROCESSORS < 0 then the number of cores used will be the number available on the machine less NUM_PROCESSORS.")
    public int NUM_PROCESSORS = 1;
    private final Log log = Log.getInstance(ExtractIlluminaBarcodes.class);
    private ReadStructure readStructure;
    private ReadStructure outputReadStructure;
    private IlluminaDataProviderFactory factory;
    private final Map<String, BarcodeMetric> barcodeToMetrics = new LinkedHashMap<String, BarcodeMetric>();
    private BarcodeMetric noMatchMetric = null;
    private final NumberFormat tileNumberFormatter = NumberFormat.getNumberInstance();
    private static final String BARCODE_SEQUENCE_COLUMN = "barcode_sequence";
    private static final String BARCODE_SEQUENCE_1_COLUMN = "barcode_sequence_1";
    private static final String BARCODE_NAME_COLUMN = "barcode_name";
    private static final String LIBRARY_NAME_COLUMN = "library_name";

    public ExtractIlluminaBarcodes() {
        this.tileNumberFormatter.setMinimumIntegerDigits(4);
        this.tileNumberFormatter.setGroupingUsed(false);
    }

    @Override
    protected int doWork() {
        IoUtil.assertDirectoryIsWritable(this.BASECALLS_DIR);
        IoUtil.assertFileIsWritable(this.METRICS_FILE);
        if (this.OUTPUT_DIR == null) {
            this.OUTPUT_DIR = this.BASECALLS_DIR;
        }
        IoUtil.assertDirectoryIsWritable(this.OUTPUT_DIR);
        String[] noMatchBarcode = new String[this.readStructure.barcodes.length()];
        int index = 0;
        for (ReadDescriptor d : this.readStructure.descriptors) {
            if (d.type != ReadType.Barcode) continue;
            noMatchBarcode[index++] = StringUtil.repeatCharNTimes((char)'N', (int)d.length);
        }
        this.noMatchMetric = new BarcodeMetric(null, null, IlluminaUtil.barcodeSeqsToString(noMatchBarcode), noMatchBarcode);
        int numProcessors = this.NUM_PROCESSORS == 0 ? Runtime.getRuntime().availableProcessors() : (this.NUM_PROCESSORS < 0 ? Runtime.getRuntime().availableProcessors() + this.NUM_PROCESSORS : this.NUM_PROCESSORS);
        this.log.info("Processing with " + numProcessors + " PerTileBarcodeExtractor(s).");
        ExecutorService pool = Executors.newFixedThreadPool(numProcessors);
        ArrayList<PerTileBarcodeExtractor> extractors = new ArrayList<PerTileBarcodeExtractor>(this.factory.getAvailableTiles().size());
        for (int tile : this.factory.getAvailableTiles()) {
            PerTileBarcodeExtractor extractor = new PerTileBarcodeExtractor(tile, this.getBarcodeFile(tile));
            pool.submit(extractor);
            extractors.add(extractor);
        }
        pool.shutdown();
        try {
            if (!pool.awaitTermination(6L, TimeUnit.HOURS)) {
                pool.shutdownNow();
                if (!pool.awaitTermination(60L, TimeUnit.SECONDS)) {
                    this.log.error("Pool did not terminate");
                }
                return 1;
            }
        }
        catch (InterruptedException ie) {
            pool.shutdownNow();
            return 2;
        }
        this.log.info("Processed " + extractors.size() + " tiles.");
        for (PerTileBarcodeExtractor extractor : extractors) {
            for (String key : this.barcodeToMetrics.keySet()) {
                this.barcodeToMetrics.get(key).merge(extractor.getMetrics().get(key));
            }
            this.noMatchMetric.merge(extractor.getNoMatchMetric());
            if (extractor.getException() == null) continue;
            this.log.error("Abandoning metrics calculation because one or more PerTileBarcodeExtractors failed.");
            return 4;
        }
        int totalReads = this.noMatchMetric.READS;
        int totalPfReads = this.noMatchMetric.PF_READS;
        int totalPfReadsAssigned = 0;
        for (BarcodeMetric barcodeMetric : this.barcodeToMetrics.values()) {
            totalReads += barcodeMetric.READS;
            totalPfReads += barcodeMetric.PF_READS;
            totalPfReadsAssigned += barcodeMetric.PF_READS;
        }
        if (totalReads > 0) {
            this.noMatchMetric.PCT_MATCHES = (double)this.noMatchMetric.READS / (double)totalReads;
            double bestPctOfAllBarcodeMatches = 0.0;
            for (BarcodeMetric barcodeMetric : this.barcodeToMetrics.values()) {
                barcodeMetric.PCT_MATCHES = (double)barcodeMetric.READS / (double)totalReads;
                if (!(barcodeMetric.PCT_MATCHES > bestPctOfAllBarcodeMatches)) continue;
                bestPctOfAllBarcodeMatches = barcodeMetric.PCT_MATCHES;
            }
            if (bestPctOfAllBarcodeMatches > 0.0) {
                this.noMatchMetric.RATIO_THIS_BARCODE_TO_BEST_BARCODE_PCT = this.noMatchMetric.PCT_MATCHES / bestPctOfAllBarcodeMatches;
                for (BarcodeMetric barcodeMetric : this.barcodeToMetrics.values()) {
                    barcodeMetric.RATIO_THIS_BARCODE_TO_BEST_BARCODE_PCT = barcodeMetric.PCT_MATCHES / bestPctOfAllBarcodeMatches;
                }
            }
        }
        if (totalPfReads > 0) {
            this.noMatchMetric.PF_PCT_MATCHES = (double)this.noMatchMetric.PF_READS / (double)totalPfReads;
            double bestPctOfAllBarcodeMatches = 0.0;
            for (BarcodeMetric barcodeMetric : this.barcodeToMetrics.values()) {
                barcodeMetric.PF_PCT_MATCHES = (double)barcodeMetric.PF_READS / (double)totalPfReads;
                if (!(barcodeMetric.PF_PCT_MATCHES > bestPctOfAllBarcodeMatches)) continue;
                bestPctOfAllBarcodeMatches = barcodeMetric.PF_PCT_MATCHES;
            }
            if (bestPctOfAllBarcodeMatches > 0.0) {
                this.noMatchMetric.PF_RATIO_THIS_BARCODE_TO_BEST_BARCODE_PCT = this.noMatchMetric.PF_PCT_MATCHES / bestPctOfAllBarcodeMatches;
                for (BarcodeMetric barcodeMetric : this.barcodeToMetrics.values()) {
                    barcodeMetric.PF_RATIO_THIS_BARCODE_TO_BEST_BARCODE_PCT = barcodeMetric.PF_PCT_MATCHES / bestPctOfAllBarcodeMatches;
                }
            }
        }
        if (totalPfReadsAssigned > 0) {
            double mean = (double)totalPfReadsAssigned / (double)this.barcodeToMetrics.values().size();
            for (BarcodeMetric m : this.barcodeToMetrics.values()) {
                m.PF_NORMALIZED_MATCHES = (double)m.PF_READS / mean;
            }
        }
        MetricsFile metrics = this.getMetricsFile();
        for (BarcodeMetric barcodeMetric : this.barcodeToMetrics.values()) {
            metrics.addMetric(barcodeMetric);
        }
        metrics.addMetric(this.noMatchMetric);
        metrics.write(this.METRICS_FILE);
        return 0;
    }

    private File getBarcodeFile(int tile) {
        return new File(this.OUTPUT_DIR, "s_" + this.LANE + "_" + this.tileNumberFormatter.format(tile) + "_barcode.txt" + (this.COMPRESS_OUTPUTS ? ".gz" : ""));
    }

    @Override
    protected String[] customCommandLineValidation() {
        IlluminaDataType[] illuminaDataTypeArray;
        ArrayList<String> messages = new ArrayList<String>();
        this.readStructure = new ReadStructure(this.READ_STRUCTURE.replaceAll("T", "S"));
        if (this.MINIMUM_BASE_QUALITY > 0) {
            IlluminaDataType[] illuminaDataTypeArray2 = new IlluminaDataType[3];
            illuminaDataTypeArray2[0] = IlluminaDataType.BaseCalls;
            illuminaDataTypeArray2[1] = IlluminaDataType.PF;
            illuminaDataTypeArray = illuminaDataTypeArray2;
            illuminaDataTypeArray2[2] = IlluminaDataType.QualityScores;
        } else {
            IlluminaDataType[] illuminaDataTypeArray3 = new IlluminaDataType[2];
            illuminaDataTypeArray3[0] = IlluminaDataType.BaseCalls;
            illuminaDataTypeArray = illuminaDataTypeArray3;
            illuminaDataTypeArray3[1] = IlluminaDataType.PF;
        }
        IlluminaDataType[] datatypes = illuminaDataTypeArray;
        this.factory = new IlluminaDataProviderFactory(this.BASECALLS_DIR, this.LANE, this.readStructure, datatypes);
        this.outputReadStructure = this.factory.getOutputReadStructure();
        if (this.BARCODE_FILE != null) {
            this.parseBarcodeFile(messages);
        } else {
            HashSet<String> barcodes = new HashSet<String>();
            for (String barcode : this.BARCODE) {
                if (barcodes.contains(barcode)) {
                    messages.add("Barcode " + barcode + " specified more than once.");
                }
                barcodes.add(barcode);
                BarcodeMetric metric = new BarcodeMetric(null, null, barcode, new String[]{barcode});
                this.barcodeToMetrics.put(barcode, metric);
            }
        }
        if (this.barcodeToMetrics.keySet().size() == 0) {
            messages.add("No barcodes have been specified.");
        }
        if (messages.size() == 0) {
            return null;
        }
        return messages.toArray(new String[messages.size()]);
    }

    public static void main(String[] argv) {
        System.exit(new ExtractIlluminaBarcodes().instanceMain(argv));
    }

    private void parseBarcodeFile(ArrayList<String> messages) {
        String sequenceColumn;
        TabbedTextFileWithHeaderParser barcodesParser = new TabbedTextFileWithHeaderParser(this.BARCODE_FILE);
        String string = barcodesParser.hasColumn(BARCODE_SEQUENCE_COLUMN) ? BARCODE_SEQUENCE_COLUMN : (sequenceColumn = barcodesParser.hasColumn(BARCODE_SEQUENCE_1_COLUMN) ? BARCODE_SEQUENCE_1_COLUMN : null);
        if (sequenceColumn == null) {
            messages.add(this.BARCODE_FILE + " does not have " + BARCODE_SEQUENCE_COLUMN + " or " + BARCODE_SEQUENCE_1_COLUMN + " column header");
            return;
        }
        boolean hasBarcodeName = barcodesParser.hasColumn(BARCODE_NAME_COLUMN);
        boolean hasLibraryName = barcodesParser.hasColumn(LIBRARY_NAME_COLUMN);
        int numBarcodes = this.readStructure.barcodes.length();
        HashSet<String> barcodes = new HashSet<String>();
        for (TabbedTextFileWithHeaderParser.Row row : barcodesParser) {
            Object[] bcStrings = new String[numBarcodes];
            int barcodeNum = 1;
            for (ReadDescriptor rd : this.readStructure.descriptors) {
                if (rd.type != ReadType.Barcode) continue;
                String header = barcodeNum == 1 ? sequenceColumn : "barcode_sequence_" + String.valueOf(barcodeNum);
                bcStrings[barcodeNum - 1] = row.getField(header);
                ++barcodeNum;
            }
            String bcStr = IlluminaUtil.barcodeSeqsToString((String[])bcStrings);
            if (barcodes.contains(bcStr)) {
                messages.add("Barcode " + bcStr + " specified more than once in " + this.BARCODE_FILE);
            }
            barcodes.add(bcStr);
            String barcodeName = hasBarcodeName ? row.getField(BARCODE_NAME_COLUMN) : "";
            String libraryName = hasLibraryName ? row.getField(LIBRARY_NAME_COLUMN) : "";
            BarcodeMetric metric = new BarcodeMetric(barcodeName, libraryName, bcStr, (String[])bcStrings);
            this.barcodeToMetrics.put(StringUtil.join((String)"", (Object[])bcStrings), metric);
        }
        barcodesParser.close();
    }

    private class PerTileBarcodeExtractor
    implements Runnable {
        private final int tile;
        private final File barcodeFile;
        private final Map<String, BarcodeMetric> metrics;
        private final BarcodeMetric noMatch;
        private Exception exception = null;
        private final boolean usingQualityScores;

        public PerTileBarcodeExtractor(int tile, File barcodeFile) {
            this.usingQualityScores = ExtractIlluminaBarcodes.this.MINIMUM_BASE_QUALITY > 0;
            this.tile = tile;
            this.barcodeFile = barcodeFile;
            this.metrics = new LinkedHashMap<String, BarcodeMetric>(ExtractIlluminaBarcodes.this.barcodeToMetrics.size());
            for (String key : ExtractIlluminaBarcodes.this.barcodeToMetrics.keySet()) {
                this.metrics.put(key, BarcodeMetric.copy((BarcodeMetric)ExtractIlluminaBarcodes.this.barcodeToMetrics.get(key)));
            }
            this.noMatch = BarcodeMetric.copy(ExtractIlluminaBarcodes.this.noMatchMetric);
        }

        public synchronized Map<String, BarcodeMetric> getMetrics() {
            return this.metrics;
        }

        public synchronized BarcodeMetric getNoMatchMetric() {
            return this.noMatch;
        }

        public synchronized Exception getException() {
            return this.exception;
        }

        @Override
        public synchronized void run() {
            ExtractIlluminaBarcodes.this.log.info("Extracting barcodes for tile " + this.tile);
            IlluminaDataProvider provider = ExtractIlluminaBarcodes.this.factory.makeDataProvider(Arrays.asList(this.tile));
            int[] barcodeIndices = ((ExtractIlluminaBarcodes)ExtractIlluminaBarcodes.this).outputReadStructure.barcodes.getIndices();
            BufferedWriter writer = IoUtil.openFileForBufferedWriting(this.barcodeFile);
            try {
                Object qualityScores;
                byte[][] barcodeSubsequences = new byte[barcodeIndices.length][];
                Object object = qualityScores = this.usingQualityScores ? (Object)new byte[barcodeIndices.length][] : (byte[][])null;
                while (provider.hasNext()) {
                    ClusterData cluster = provider.next();
                    for (int i = 0; i < barcodeIndices.length; ++i) {
                        barcodeSubsequences[i] = cluster.getRead(barcodeIndices[i]).getBases();
                        if (!this.usingQualityScores) continue;
                        qualityScores[i] = cluster.getRead(barcodeIndices[i]).getQualities();
                    }
                    boolean passingFilter = cluster.isPf();
                    BarcodeMatch match = this.findBestBarcodeAndUpdateMetrics(barcodeSubsequences, (byte[][])qualityScores, passingFilter, this.metrics, ExtractIlluminaBarcodes.this.noMatchMetric);
                    String yOrN = match.matched ? "Y" : "N";
                    for (byte[] bc : barcodeSubsequences) {
                        writer.write(StringUtil.bytesToString((byte[])bc));
                    }
                    writer.write("\t" + yOrN + "\t" + match.barcode + "\t" + String.valueOf(match.mismatches) + "\t" + String.valueOf(match.mismatchesToSecondBest));
                    writer.newLine();
                }
                writer.close();
            }
            catch (Exception e) {
                ExtractIlluminaBarcodes.this.log.error(e, "Error processing tile ", this.tile);
                this.exception = e;
            }
        }

        private BarcodeMatch findBestBarcodeAndUpdateMetrics(byte[][] readSubsequences, byte[][] qualityScores, boolean passingFilter, Map<String, BarcodeMetric> metrics, BarcodeMetric noMatchBarcodeMetric) {
            BarcodeMetric bestBarcodeMetric = null;
            int totalBarcodeReadBases = 0;
            int numNoCalls = 0;
            for (byte[] bc : readSubsequences) {
                totalBarcodeReadBases += bc.length;
                for (byte b : bc) {
                    if (!SequenceUtil.isNoCall((byte)b)) continue;
                    ++numNoCalls;
                }
            }
            int numMismatchesInBestBarcode = totalBarcodeReadBases + 1;
            int numMismatchesInSecondBestBarcode = totalBarcodeReadBases + 1;
            for (BarcodeMetric barcodeMetric : metrics.values()) {
                int numMismatches = this.countMismatches(barcodeMetric.barcodeBytes, readSubsequences, qualityScores);
                if (numMismatches < numMismatchesInBestBarcode) {
                    if (bestBarcodeMetric != null) {
                        numMismatchesInSecondBestBarcode = numMismatchesInBestBarcode;
                    }
                    numMismatchesInBestBarcode = numMismatches;
                    bestBarcodeMetric = barcodeMetric;
                    continue;
                }
                if (numMismatches >= numMismatchesInSecondBestBarcode) continue;
                numMismatchesInSecondBestBarcode = numMismatches;
            }
            boolean matched = bestBarcodeMetric != null && numNoCalls <= ExtractIlluminaBarcodes.this.MAX_NO_CALLS && numMismatchesInBestBarcode <= ExtractIlluminaBarcodes.this.MAX_MISMATCHES && numMismatchesInSecondBestBarcode - numMismatchesInBestBarcode >= ExtractIlluminaBarcodes.this.MIN_MISMATCH_DELTA;
            BarcodeMatch match = new BarcodeMatch();
            if (numNoCalls + numMismatchesInBestBarcode < totalBarcodeReadBases) {
                match.mismatches = numMismatchesInBestBarcode;
                match.mismatchesToSecondBest = numMismatchesInSecondBestBarcode;
                match.barcode = bestBarcodeMetric.BARCODE.toLowerCase().replaceAll("-", "");
            } else {
                match.mismatches = totalBarcodeReadBases;
                match.barcode = "";
            }
            if (matched) {
                ++bestBarcodeMetric.READS;
                if (passingFilter) {
                    ++bestBarcodeMetric.PF_READS;
                }
                if (numMismatchesInBestBarcode == 0) {
                    ++bestBarcodeMetric.PERFECT_MATCHES;
                    if (passingFilter) {
                        ++bestBarcodeMetric.PF_PERFECT_MATCHES;
                    }
                } else if (numMismatchesInBestBarcode == 1) {
                    ++bestBarcodeMetric.ONE_MISMATCH_MATCHES;
                    if (passingFilter) {
                        ++bestBarcodeMetric.PF_ONE_MISMATCH_MATCHES;
                    }
                }
                match.matched = true;
                match.barcode = bestBarcodeMetric.BARCODE.replaceAll("-", "");
            } else {
                ++noMatchBarcodeMetric.READS;
                if (passingFilter) {
                    ++noMatchBarcodeMetric.PF_READS;
                }
            }
            return match;
        }

        private int countMismatches(byte[][] barcodeBytes, byte[][] readSubsequence, byte[][] qualities) {
            int numMismatches = 0;
            for (int j = 0; j < barcodeBytes.length; ++j) {
                int basesToCheck = Math.min(barcodeBytes[j].length, readSubsequence[j].length);
                for (int i = 0; i < basesToCheck; ++i) {
                    if (SequenceUtil.isNoCall((byte)readSubsequence[j][i])) continue;
                    if (!SequenceUtil.basesEqual((byte)barcodeBytes[j][i], (byte)readSubsequence[j][i])) {
                        ++numMismatches;
                        continue;
                    }
                    if (qualities == null || qualities[j][i] >= ExtractIlluminaBarcodes.this.MINIMUM_BASE_QUALITY) continue;
                    ++numMismatches;
                }
            }
            return numMismatches;
        }

        class BarcodeMatch {
            boolean matched;
            String barcode;
            int mismatches;
            int mismatchesToSecondBest;

            BarcodeMatch() {
            }
        }
    }

    public static class BarcodeMetric
    extends MetricBase {
        public String BARCODE;
        public String BARCODE_NAME = "";
        public String LIBRARY_NAME = "";
        public int READS = 0;
        public int PF_READS = 0;
        public int PERFECT_MATCHES = 0;
        public int PF_PERFECT_MATCHES = 0;
        public int ONE_MISMATCH_MATCHES = 0;
        public int PF_ONE_MISMATCH_MATCHES = 0;
        public double PCT_MATCHES = 0.0;
        public double RATIO_THIS_BARCODE_TO_BEST_BARCODE_PCT = 0.0;
        public double PF_PCT_MATCHES = 0.0;
        public double PF_RATIO_THIS_BARCODE_TO_BEST_BARCODE_PCT = 0.0;
        public double PF_NORMALIZED_MATCHES;
        protected byte[][] barcodeBytes;

        public BarcodeMetric(String barcodeName, String libraryName, String barcodeDisplay, String[] barcodeSeqs) {
            this.BARCODE = barcodeDisplay;
            this.BARCODE_NAME = barcodeName;
            this.LIBRARY_NAME = libraryName;
            this.barcodeBytes = new byte[barcodeSeqs.length][];
            for (int i = 0; i < barcodeSeqs.length; ++i) {
                this.barcodeBytes[i] = StringUtil.stringToBytes((String)barcodeSeqs[i]);
            }
        }

        public BarcodeMetric() {
            this.barcodeBytes = null;
        }

        public static BarcodeMetric copy(BarcodeMetric metric) {
            BarcodeMetric result = new BarcodeMetric();
            result.BARCODE = metric.BARCODE;
            result.BARCODE_NAME = metric.BARCODE_NAME;
            result.LIBRARY_NAME = metric.LIBRARY_NAME;
            result.barcodeBytes = metric.barcodeBytes;
            return result;
        }

        public void merge(BarcodeMetric metric) {
            this.READS += metric.READS;
            this.PF_READS += metric.PF_READS;
            this.PERFECT_MATCHES += metric.PERFECT_MATCHES;
            this.PF_PERFECT_MATCHES += metric.PF_PERFECT_MATCHES;
            this.ONE_MISMATCH_MATCHES += metric.ONE_MISMATCH_MATCHES;
            this.PF_ONE_MISMATCH_MATCHES += metric.PF_ONE_MISMATCH_MATCHES;
        }
    }
}

