# TOOL mothur-classifyseqs.R: "Classify sequences to taxonomic units with Mothur" (Classify 16S rRNA sequences to taxonomic units using the Wang method with Silva reference set and taxonomy. This functionality is based on the Mothur package. In addition to classification, this tool generates a count table and a phenodata file, which you can use to assign samples to different experimental groups for further statistical analyses. As inputs you need a fasta file of aligned sequences, and a groups file generated by Mothur.)
# INPUT a.fasta: "Aligned reads in FASTA file" TYPE FASTA
# INPUT all.grp: "Groups file" TYPE GENERIC
# OUTPUT reads-taxonomy-assignment.txt
# OUTPUT classification-summary.txt
# OUTPUT counttable.tsv: counttable.tsv
# OUTPUT META phenodata.tsv: phenodata.tsv
# PARAMETER OPTIONAL cutlevel: "Cutting level for taxonomic names" TYPE INTEGER FROM 0 TO 9 DEFAULT 0 (Cutting level for taxonomic names for the count table. 0 means retain full names, e.g. Bacteria;Actinobacteria;Actinobacteria;Coriobacteridae;Coriobacteriales;Coriobacterineae;Coriobacteriaceae;Slackia;unclassified.)

# EK 18.06.2013
# JTT 28.8.2013 count table and phenodata added

# binary
binary <- c(file.path(chipster.tools.path, "mothur", "mothur"))
data.path <- c(file.path(chipster.tools.path, "mothur-data"))
template.path <- c(file.path(data.path, "silva.bacteria.fasta"))
taxonomy.path <- c(file.path(data.path, "silva.bacteria.silva.tax"))

# batch file
write(paste("classify.seqs(fasta=a.fasta, iters=1000, template=", template.path, ", taxonomy=", taxonomy.path, ")", sep=""), "batch.mth", append=F)

# command
command <- paste(binary, "batch.mth", "> log.txt 2>&1")

# run
system(command)

# Postprocess output
system("mv a.silva.wang.taxonomy reads-taxonomy-assignment.txt")
system("mv a.silva.wang.tax.summary classification-summary.txt")

# Count table and phenodata preparations:

# Reads the data and tabulates it
grp<-read.table("all.grp", header=F, sep="\t")
tax<-read.table("reads-taxonomy-assignment.txt", header=F, sep="\t")
dat<-merge(grp, tax, by.x="V1", by.y="V1")
dat2<-dat
dat2$V2.y<-gsub(".[[:digit:]]{1,}.?[[:digit:]]?)", "", as.character(dat2$V2.y))

# Cutting the taxonomic names
if(cutlevel==0) {
	dat2$newnames<-dat2$V2.y
} else {
	sp<-strsplit(dat2$V2.y, ";")
	sp2<-rep(NA, nrow(dat2))
	for(i in 1:nrow(dat2)) {
		sp2[i]<-paste(sp[[i]][1:cutlevel], collapse=";")
	}
	dat2$newnames<-sp2
}

# Creating the count table
tab<-table(dat2$V2.x, dat2$newnames)
tab2<-as.data.frame.matrix(tab)
chiptype<-c("metagenomics")

# Writing the table to disk
write.table(tab, "counttable.tsv", col.names=T, row.names=T, sep="\t", quote=FALSE)
write.table(data.frame(sample=rownames(tab2), chiptype=chiptype, group=rep("", length(rownames(tab2)))), "phenodata.tsv", col.names=T, row.names=F, sep="\t", quote=F)


