/* ======================================================================= 
 * A visualisation library extension for JFreeChart. Please see JFreeChart
 * for further information.
 * =======================================================================
 * Copyright (C) 2006  University of Helsinki, Department of Computer Science
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * -----------------------------
 * Contact:  ohtu@cs.helsinki.fi
 * -----------------------------
 *
 */


package org.jfree.chart.plot;

import java.awt.AlphaComposite;
import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Composite;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Paint;
import java.awt.Polygon;
import java.awt.Shape;
import java.awt.Stroke;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ResourceBundle;
import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.jfree.chart.ChartMouseEvent;
import org.jfree.chart.ChartMouseListener;
import org.jfree.chart.entity.EntityCollection;
import org.jfree.chart.entity.SOMItemEntity;
import org.jfree.chart.event.PlotChangeEvent;
import org.jfree.chart.labels.SOMToolTipGenerator;
import org.jfree.data.general.DatasetUtilities;
import org.jfree.data.som.SOMDataItem;
import org.jfree.data.som.SOMDataset;
import org.jfree.io.SerialUtilities;
import org.jfree.text.G2TextMeasurer;
import org.jfree.text.TextBlock;
import org.jfree.text.TextBlockAnchor;
import org.jfree.text.TextUtilities;
import org.jfree.util.ObjectUtilities;


/**
 * A plot that displays data in the form of a SOM-map. The plot uses
 * data from a {@link SOMDataset} -object.
 *
 * @author viski project.
 */
public class SOMPlot extends Plot implements ChartMouseListener, ChangeListener, Cloneable, Serializable {

    /** The dataset. */
    private SOMDataset dataset;

    /** The tooltip generator. */
    private SOMToolTipGenerator toolTipGenerator;
    
    /** The cell color hue adjustment value. */
    private int colorHueAdjustment;
    
    /** The List of cells selected on-screen. */
    private List selectedCells;
    
    /** The cell color hue adjuster. */
    private JSlider colorHueSlider;
    
    /** The distance selector for 'select neighbors' */
    private JSlider distanceSlider;
    
    /** The cell information text is drawn with this font. */
    private Font descriptionFont;

    /** The default font used in the chart. */
    private static final Font DEFAULT_DESCRIPTION_FONT = new Font("SansSerif", Font.PLAIN, 8);

    /** The default size of the cell description. */
    private static final float DESCRIPTION_WIDTH_RATIO = 0.9f;

    /** The number of empty pixels between cells.
     *  Use an odd number to achieve cell symmetry.
     */
    private static final int HORIZONTAL_GAP = 3;
    
    /** The number of empty pixels between cells */
    private static final int VERTICAL_GAP = 3;
    
    /** The resourceBundle for the localization. */
    protected static ResourceBundle localizationResources 
        = ResourceBundle.getBundle("org.jfree.chart.plot.LocalizationBundle");
    
    /**
     * Creates a new plot that will draw a SOM-map for the given dataset. Sets
     * the toolTipGenerator to null by default.
     *
     * @param dataset  the dataset.
     * @throws NullPointerException
     */
    public SOMPlot(SOMDataset dataset) throws NullPointerException {
        if (DatasetUtilities.isEmptyOrNull(dataset))
            throw new NullPointerException("Dataset given to SOMPlot was null or empty.");
        
        this.dataset = dataset;
        this.toolTipGenerator = null;
        this.colorHueAdjustment = 0;
        this.selectedCells = new LinkedList();
        this.distanceSlider = new JSlider(0,444,0);
        this.colorHueSlider = new JSlider(0,359,this.colorHueAdjustment);
        this.descriptionFont = DEFAULT_DESCRIPTION_FONT;
    }
    
    /**
     * Returns the dataset of a plot.
     *
     * @return  The dataset.
     */
    public SOMDataset getDataset() {
        return this.dataset;
    }

    /**
     * Sets the dataset of a plot.
     *
     * @param dataset   the dataset.
     * @throws NullPointerException  if dataset is null.
     */
    public void setDataset(SOMDataset dataset) throws NullPointerException {
        if (dataset == null)
            throw new NullPointerException("Dataset given to SOMPlot was null");
        this.dataset = dataset;
    }

    /** 
     * Returns the tooltip generator of a plot.
     *
     * @return  The tooltip generator.
     */
    public SOMToolTipGenerator getToolTipGenerator() {
        return toolTipGenerator;
    }

    /**
     * Sets the tooltip generator of a plot.
     * Parameter value can be null.
     *
     * @param toolTipGenerator  the tooltip generator (null permitted.)
     */
    public void setToolTipGenerator(SOMToolTipGenerator toolTipGenerator) {
        // No null check.
        // Null value means that tooltips will not be displayed.
        this.toolTipGenerator = toolTipGenerator;
    }

    /**
     * Returns the name of the plot.
     * Thr name is used as the name of the SOM specific panel that can be 
     * opened from the context menu.
     *
     * @return  The name of the panel.
     */
    public String getPlotType() {
        return this.localizationResources.getString("SOM_Plot");
    }

    /**
     * Returns the font used for cell descriptions.
     *
     * @return Description font.
     */
    public Font getDescriptionFont() {
        return this.descriptionFont;
    }
    
    /**
     * Sets the font used for cell descriptions.
     *
     * @param descriptionFont  the font.
     * @throws NullPointerException  if descriptionFont is null.
     */
    public void setDescriptionFont(Font descriptionFont) throws NullPointerException {
        if (descriptionFont == null) {
            throw new NullPointerException("Description font given to SOMPlot was null");
        }
        this.descriptionFont = descriptionFont;
    }

    public JSlider getColorHueSlider() {
        return colorHueSlider;
    }

    public JSlider getDistanceSlider() {
        return distanceSlider;
    }

    /**
     * Draws the plot on a Java2D graphics device (such as the screen or 
     * a printer).
     *
     * @param g2  the graphics device.
     * @param area  the area within which the plot should be drawn.
     * @param anchor  the anchor point (<code>null</code> permitted).
     * @param parentState  the state from the parent plot, if there is one
     * (<code>null</code> permitted.)
     * @param info  collects info about the drawing (<code>null</code> permitted).
     * @throws NullPointerException  if g2 or area is null.
     */ 
    public void draw(java.awt.Graphics2D g2, java.awt.geom.Rectangle2D area, 
            java.awt.geom.Point2D anchor, PlotState parentState, 
            PlotRenderingInfo info) {
        
        // adjust for insets...
        this.getInsets().trim(area);

        if (info != null) {
            info.setPlotArea(area);
            info.setDataArea(area);
        }

        drawBackground(g2, area);
        drawOutline(g2, area);

        Shape savedClip = g2.getClip();
        g2.clip(area);

        Composite originalComposite = g2.getComposite();
        g2.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 
                getForegroundAlpha()));

        drawSOM(g2, area, info);
        
        g2.setClip(savedClip);
        g2.setComposite(originalComposite);

        drawOutline(g2, area);
    }

    /**
     * Does the actual drawing for the SOMPlot.draw()-method
     * and initializes the entities associated with a cell.
     *
     * @param g2  the graphics device.
     * @param area  the area within which the plot should be drawn.
     * @param info  collects info about the drawing.
     *
     * @throws NullPointerException  if any of the parameters is null.
     */
    private void drawSOM(Graphics2D g2, java.awt.geom.Rectangle2D area, 
            PlotRenderingInfo info) {
        
        double edgeLengthByWidth = ((area.getWidth() - this.dataset.getColumnCount() * (this.HORIZONTAL_GAP+1)) / 
                ((1+2*this.dataset.getColumnCount())*Math.cos(Math.PI/6))
                );
        double edgeLengthByHeight = ((area.getHeight() - this.dataset.getRowCount() * (this.VERTICAL_GAP+1))/ 
                (this.dataset.getRowCount() + (this.dataset.getRowCount() + 1) * Math.sin(Math.PI/6)
                ));
        double edgeLength = Math.min(edgeLengthByWidth, edgeLengthByHeight);
        
        Polygon hexagon = createHexagon(edgeLength);
        hexagon.translate((int)area.getMinX()+1, (int)area.getMinY()+1);
        
        int deltaX = hexagon.xpoints[1]-hexagon.xpoints[5]+1+this.HORIZONTAL_GAP;
        int deltaY = hexagon.ypoints[2]-hexagon.ypoints[0]+1+this.VERTICAL_GAP;
        for (int y=0; y < this.dataset.getRowCount(); ++y) {
            for (int x=0; x < this.dataset.getColumnCount(); ++x) {
                SOMDataItem item = this.dataset.getValue(x, y);
                if (item != null) {
                    if (item.isSelected()) {
                        g2.setPaint(item.getColor());
                        g2.fillPolygon(hexagon);
                        g2.setPaint(contrastingColor(item.getColor()));
                        
                        //Shape savedClip = g2.getClip();
                        //g2.clip(hexagon);
                        Stroke stroke = g2.getStroke();
                        g2.setStroke(new BasicStroke(1.5f));
                        g2.drawPolygon(hexagon);
                        g2.setStroke(stroke);
                        //g2.setClip(savedClip);
                    } else {
                        g2.setPaint(item.getColor());
                        g2.fillPolygon(hexagon);
                    }
                    
                    drawDescriptions(g2, 
                                     hexagon,
                                     item);
                    createEntity(x, y, new Polygon(hexagon.xpoints,
                                                   hexagon.ypoints,
                                                   hexagon.npoints), info);
                    hexagon.translate(deltaX , 0);
                }
            }
            if (y % 2 == 1) {
                hexagon.translate((int)(area.getMinX()-hexagon.xpoints[5]+1), deltaY);
                //                                                        ^
                // the +1 moves the lefth most edge into the visible/drawable area
            } 
            else {
                hexagon.translate((int)(area.getMinX()-hexagon.xpoints[5]+1), deltaY);
                //                                                        ^
                // the +1 moves the lefth most edge into the visible/drawable area
                hexagon.translate(hexagon.xpoints[0]-hexagon.xpoints[5]+
                        (((this.HORIZONTAL_GAP>>1)+1)), 0);
                // (this.HORIZONTAL_GAP/2)+1 centers the top vertex in
                // the gap between the two hexagons above it.
            }
        }
    }
    
   /**
    * Draws the textual descriptions associated with a SOM-cell.
    *
    * @param g2  the graphics device.
    * @param area  the area to draw the descriptions into.
    * @param item  the SOMDataItem describing a cell.
    *
    * @throws NullPointerException  if any of the parameters is null.
    */
   private void drawDescriptions(Graphics2D g2, Polygon area, SOMDataItem item) {
        Shape savedClip = g2.getClip();
        g2.clip(area);
        StringBuffer sb = new StringBuffer();
        for (int i=0; i < item.getDescriptions().length; ++i) {
            sb.append(item.getDescriptions()[i]);
            if (i < item.getDescriptions().length - 1)
                sb.append(" ");
        }
        String message = sb.toString();
        
        float maxWidth = (float) (area.xpoints[1]-area.xpoints[5]);
        int maxLines = (area.ypoints[2]-area.ypoints[1])/
                g2.getFontMetrics(this.descriptionFont).getHeight();
        maxLines = Math.max(maxLines, 1);
        
        // BUGFIX
        // If the following if-statement is removed, the description text
        // of the cell at (0,0) will be more narrow when compared to the
        // descriptions texts of the other cells.
        // The first if-statement draws the text in the same color
        // as the cell itself, thus doing nothing visible to the user.
        // A cleaner bugfix could probably be done by examining the inner
        // workings of createTextBlock(), but this one seems to do the job. 
        if (message != null) {
            TextBlock block = TextUtilities.createTextBlock(
                message, 
                this.descriptionFont, 
                item.getColor(),
                this.DESCRIPTION_WIDTH_RATIO * maxWidth,
                maxLines,
                new G2TextMeasurer(g2)
            );
            block.draw(
                g2, (float) area.xpoints[5] + maxWidth/2, 
                (float) area.ypoints[5], TextBlockAnchor.TOP_CENTER
            );
        }
        
        if (message != null) {
            TextBlock block = TextUtilities.createTextBlock(
                message, 
                this.descriptionFont, 
                contrastingColor(item.getColor()),
                this.DESCRIPTION_WIDTH_RATIO * maxWidth,
                maxLines,
                new G2TextMeasurer(g2)
            );
            block.draw(
                g2, (float) area.xpoints[5] + maxWidth/2, 
                (float) area.ypoints[5], TextBlockAnchor.TOP_CENTER
            );
        }
        g2.setClip(savedClip);

    }
    
    /**
     * Creates an entity that is associated with a certain area
     * in the SOM-map ie. a cell.
     *
     * @param x  the x-coordinate.
     * @param y  the y-coordinate.
     * @param shape  the shape in Java2D space.
     * @param info  collects info about the drawing.
     */
    private void createEntity(int x, int y, Shape shape, 
            PlotRenderingInfo info) {
        
        if (info != null) {
            EntityCollection entities = info.getOwner().getEntityCollection();
            if (entities != null) {
                String tip = null;
                if (this.toolTipGenerator != null) {
                    tip = this.toolTipGenerator.generateToolTip(
                            this.dataset, x, y);
                }
                
                // SOMPlot does not have a text generator for html image maps
                String url = null;
                
                entities.add(
                        new SOMItemEntity(shape, this.dataset, x, y, tip, url));
            }
        }
    }
    
    /**
     * Creates a new hexagon cell for the SOM-map.
     * Vertices are numbered according the following picture.
     * Vertex 0 is on the x-axis.
     * Vertices 4 and 5 are on the y-axis.
     *
     *         0
     *        / \
     *       /   \
     *    5 /     \ 1
     *     |       |
     *     |       |
     *     |       |
     *    4 \     / 2
     *       \   /
     *        \ /
     *         3
     *
     * @param edgeLength  the length of the edges.
     * @return  The hexagon.
     * @throws IllegalArgumentException  If edgeLength <= 0.
     */
    private Polygon createHexagon(double edgeLength) throws IllegalArgumentException {
        double sin30 = Math.sin(Math.PI/6);
        double cos30 = Math.cos(Math.PI/6);
        int[] xPoints = new int[6];
        int[] yPoints = new int[6];
        xPoints[0] = (int)(edgeLength*cos30);
        yPoints[0] = 0;
        xPoints[3] = xPoints[0];
        yPoints[3] = (int)(edgeLength + 2 * edgeLength * sin30);
        xPoints[1] = (int)(2 * edgeLength * cos30);
        yPoints[1] = (int)(edgeLength * sin30);
        xPoints[2] = xPoints[1];
        yPoints[2] = (int)(yPoints[1] + edgeLength);
        xPoints[4] = 0;
        yPoints[4] = yPoints[2];
        xPoints[5] = 0;
        yPoints[5] = yPoints[1];
        
        return new Polygon(xPoints, yPoints, 6);
    }
    
    /**
     * This method generates a new color that will contrast well
     * with the color given as the parameter.
     *
     * @param color  the color to contrast with.
     *
     * @return The new color.
     *
     * @throws NullPointerException if color is null.
     */
    private Color contrastingColor(Color color) {
        Color background = (Color)getBackgroundPaint();
        double distance = 0;
        double tmp;
        Color returnValue = Color.BLACK;
        
        tmp = colorDistance(Color.BLACK, background) + 
              colorDistance(Color.BLACK, color);
        if (tmp > distance) {
            distance = tmp;
            returnValue = Color.BLACK;
        }
        tmp = colorDistance(Color.WHITE, background) + 
              colorDistance(Color.WHITE, color);
        if (tmp > distance) {
            distance = tmp;
            returnValue = Color.WHITE;
        }
        tmp = colorDistance(Color.RED, background) + 
              colorDistance(Color.RED, color);
        if (tmp > distance) {
            distance = tmp;
            returnValue = Color.RED;
        }
        tmp = colorDistance(Color.GREEN, background) + 
              colorDistance(Color.GREEN, color);
        if (tmp > distance) {
            distance = tmp;
            returnValue = Color.GREEN;
        }
        tmp = colorDistance(Color.BLUE, background) + 
              colorDistance(Color.BLUE, color);
        if (tmp > distance) {
            distance = tmp;
            returnValue = Color.BLUE;
        }
        
        return returnValue;
    }
    
    /**
     * This method calculates the distance between two sets of integer color-values.
     *
     * @return  The difference in color-values.
     *
     * @throws NullPointerException  if any of the parameters is null.
     */
    private double colorDistance(Color color1, Color color2) {
        int r1 = color1.getRed();
        int g1 = color1.getGreen();
        int b1 = color1.getBlue();

        int r2 = color2.getRed();
        int g2 = color2.getGreen();
        int b2 = color2.getBlue();

        return Math.sqrt((r1-r2)*(r1-r2)+(g1-g2)*(g1-g2)+(b1-b2)*(b1-b2));
    }
    
    /**
     * Creates a new JPanel for adjusting cell colors.
     *
     * @return  The panel.
     */
    public JPanel getPanel() {
        ResourceBundle lr = ResourceBundle.getBundle("org.jfree.chart.editor.LocalizationBundle");
        JPanel panel = new JPanel();
        panel.setLayout(new GridBagLayout());
        GridBagConstraints c = new GridBagConstraints();
        c.fill = GridBagConstraints.HORIZONTAL;
        panel.setBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2));

        this.colorHueSlider.setPaintLabels(true);
        this.colorHueSlider.setPaintTicks(true);
        this.colorHueSlider.setMajorTickSpacing(90);
        this.colorHueSlider.setMinorTickSpacing(30);
        this.colorHueSlider.setPaintTrack(false);
        this.colorHueSlider.addChangeListener(this);
        
        c.weightx = 0.5;
        c.gridx = 0;
        c.gridy = 0;
        panel.add(new JLabel(lr.getString("Adjust_colors")), c);
        c.gridx = 1;
        c.gridy = 0;
        panel.add(this.colorHueSlider,c);
        
        this.distanceSlider.setPaintLabels(true);
        this.distanceSlider.setPaintTicks(true);
        this.distanceSlider.setMajorTickSpacing(100);
        this.distanceSlider.setMinorTickSpacing(25);
        this.distanceSlider.setPaintTrack(false);
        this.distanceSlider.addChangeListener(this);
        
        if (this.selectedCells.size() == 1) {
            this.distanceSlider.setEnabled(true);
        } else {
            this.distanceSlider.setEnabled(false);
        }
        
        c.insets = new Insets(30,0,0,0);
        c.weightx = 0.5;
        c.gridx = 0;
        c.gridy = 1;
        panel.add(new JLabel(lr.getString("Select_neighbors")), c);
        c.weightx = 0.5;
        c.gridx = 1;
        c.gridy = 1;
        panel.add(this.distanceSlider,c);
        
        return panel;
    }
    
    /**
     * Returns a name for the SOM specific properties panel.
     *
     * @return  Panel name as a string.
     */
    public String getPanelName() {
        ResourceBundle lr = ResourceBundle.getBundle("org.jfree.chart.editor.LocalizationBundle");
        return lr.getString("SOM_panel");
    }
    
    /**
     * Implements the ChartMouseListener interface. This
     * method does nothing.
     *
     * @param event  the mouse event.
     */ 
    public void chartMouseMoved(ChartMouseEvent event) {
        ;
    }
    
    /**
     * Implements the ChartMouseListener interface. Listens
     * for the mouse key and CTRL-key to be pressed on top of a 
     * SOMItemEntity.
     *
     * @param event  the mouse event.
     * @throws NullPointerException  if event is null.
     */
    public void chartMouseClicked(ChartMouseEvent event) {
        if (event.getEntity() != null &&
            event.getEntity() instanceof SOMItemEntity) {
            SOMItemEntity entity = (SOMItemEntity) event.getEntity();
            SOMDataItem item =
                    entity.getDataset().getValue(entity.getX(), entity.getY());
            
            if (event.getTrigger().isControlDown()) {
                handleCtrlClick(item);
            } else if (event.getTrigger().isShiftDown()) {
                handleShiftClick(item);
            } else {
                handleClick(item);
            }
        }
    }
    
    /**
     * Handels the on-screen selection of multiple cells using the CTRL-key.
     *
     * @param item  the item that is being selected on screen.
     *
     * @throws NullPointerException  if item is null.
     */
    private void handleCtrlClick(SOMDataItem item) {
        if (item.isSelected()) {
            this.selectedCells.remove(item);
            item.setSelected(false);
        } else {
            this.selectedCells.add(item);
            item.setSelected(true);
        }
    }

    /**
     * Handels the on-screen selection of multiple cells using the SHIFT-key.
     *
     * @param cornerItem1  the dataitem.
     *
     * @throws NullPointerException If cornerItem1 is null
     * @throws IllegalArgumentException If cornerItem1 does not belong
     * to the same dataset as this.
     */
    private void handleShiftClick(SOMDataItem cornerItem1) {
        SOMDataItem cornerItem2;
        
        if (this.selectedCells.isEmpty() == false) {
            // It is assumed that the cell which has been selected
            // for the longest time is first on the list
            cornerItem2 = (SOMDataItem) this.selectedCells.get(0);
        } else {
            cornerItem2 = this.dataset.getValue(0, 0);
        }
        
        List areaCells = dataset.getArea(cornerItem1, cornerItem2);
        // make sure the oldest selection stays first on the list
        // this might be unnecessary XXX
        areaCells.remove(cornerItem2);
        areaCells.add(0, cornerItem2);
        this.dataset.deselectAll();
        this.selectedCells = areaCells;
        
        Iterator i = areaCells.iterator();
        while (i.hasNext()) {
            SOMDataItem item = (SOMDataItem) i.next();
            item.setSelected(true);
        }
    }


    /**
     * Handles the on-screen selection of a single SOM-map cell.
     *
     * @param item  the dataitem.
     * @throws NullPointerException  if item is null.
     */    
    private void handleClick(SOMDataItem item) {
        if (item.isSelected()) {
            this.selectedCells.clear();
            this.dataset.deselectAll();
        } else {
            this.selectedCells.clear();
            this.dataset.deselectAll();
            this.selectedCells.add(item);
            item.setSelected(true);
        }
    }
    
    /**
     * Listens for the hue-values slider to be moved and
     * then changes the hue-values accordingly.
     *
     * @param e  the event.
     * @throws NullPointerException  if e is null.
     */
    public void stateChanged(ChangeEvent e) {
        if (e.getSource() == this.colorHueSlider) {
            int value = this.colorHueSlider.getValue();
            dataset.changeHueValues(this.colorHueAdjustment - value);
            this.colorHueAdjustment = value;
        }
        else if (e.getSource() == this.distanceSlider) {
            int value = this.distanceSlider.getValue();
            SOMDataItem center = (SOMDataItem)this.selectedCells.get(0);
            List l = this.dataset.getNeighbors(center, value , false);
            
            this.dataset.deselectAll();
            this.selectedCells.clear();
            
            Iterator i = l.iterator();
            while (i.hasNext()) {
                SOMDataItem item = (SOMDataItem)i.next();
                item.setSelected(true);
                this.selectedCells.add(item);
            }
            
            center.setSelected(true);
            this.selectedCells.add(0, center);
        }

        notifyListeners(new PlotChangeEvent(this));
    }
    
    /**
     * Tests this plot for equality with an arbitrary object.  Note that the 
     * plot's dataset is NOT included in the test for equality.
     *
     * @param obj  the object to test against (<code>null</code> permitted).
     *
     * @return <code>true</code> or <code>false</code>.
     */
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (!(obj instanceof SOMPlot)) {
            return false;
        }
        SOMPlot that = (SOMPlot) obj;
        
        if (!ObjectUtilities.equal(this.toolTipGenerator, 
                that.toolTipGenerator)) {
            return false;
        }
//        if (!ObjectUtilities.equal(this.urlGenerator, that.urlGenerator)) {
//            return false;
//        }
        
        if (this.colorHueAdjustment != that.colorHueAdjustment) {
            return false;
        }
        
        if (!ObjectUtilities.equal(this.descriptionFont, 
                that.descriptionFont)) {
            return false;
        }
    
        // can't find any difference...
        return true;
    }

    /**
     * Returns a clone of the plot.
     *
     * @return A clone.
     *
     * @throws CloneNotSupportedException if some component of the plot does 
     *         not support cloning.
     */
    public Object clone() throws CloneNotSupportedException {

        SOMPlot clone = (SOMPlot) super.clone();
        if (clone.dataset != null) {
            clone.dataset.addChangeListener(clone);
        }
        return clone;

    }

    /**
     * Provides serialization support.
     *
     * @param stream  the output stream.
     *
     * @throws IOException  if there is an I/O error.
     * @throws NullPointerException  if stream is null.
     */
    private void writeObject(ObjectOutputStream stream) throws IOException {
        stream.defaultWriteObject();
    }

    /**
     * Provides serialization support.
     *
     * @param stream  the input stream.
     *
     * @throws IOException  if there is an I/O error.
     * @throws ClassNotFoundException  if there is a classpath problem.
     * @throws NullPointerException  if stream is null.
     */
    private void readObject(ObjectInputStream stream) 
        throws IOException, ClassNotFoundException {
        stream.defaultReadObject();
    }

}

