/* ======================================================================= 
 * A visualisation library extension for JFreeChart. Please see JFreeChart
 * for further information.
 * =======================================================================
 * Copyright (C) 2006  University of Helsinki, Department of Computer Science
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * -----------------------------
 * Contact:  ohtu@cs.helsinki.fi
 * -----------------------------
 *
 */

package org.jfree.data.hc;

import java.lang.RuntimeException;

/**
* A class representing single node of a cluster tree. This is
* part of {@link HCDataset}.
* @author  viski project
*/
public class HCTreeNode {

    private HCTreeNode leftChild;
    private HCTreeNode rightChild;
    private HCTreeNode parent;
    private DataRange dataRange;
    private double height;
    private boolean finalized;

    /**
     * Creates a new HCTreeNode. This version of the constructor is
     * usable for the branch nodes.
     *
     * @param height  the height of the node.
     */
    public HCTreeNode(double height) throws IllegalArgumentException {

	if (height < 0) throw new IllegalArgumentException("height given to HCTreeNode() was negative.");
	this.height = height;
	this.dataRange = new DataRange(0,-1); // empty range.

    }

    /**
     * Creates a new HCTreeNode. This version of the constructor is
     * usable for the leaf nodes.
     *
     * @param height  the height of the node.
     * @param index  the index of the heatmap row/column this node
     * corresponds to.
     */
    public HCTreeNode(double height,int index) throws IllegalArgumentException {

	this.dataRange = new DataRange(index,index);

	this.height = height;
	this.finalized = false;

    }

    /**
     * Checks if this tree is already finalized.
     *
     * @throws RuntimeException  if the tree is already finalized.
     */
    private void assertFinalized() {

	if (this.finalized) throw new RuntimeException();

    }

    /**
     * Returns the parent of this node.
     *
     * @return  parent node, or null, if the node has no parents.
     */
    public HCTreeNode getParent() {

	return this.parent;

    }

    /**
     * Returns the left child of this node.
     *
     * @return  left child node, or null, if the node has no left child.
     */
    public HCTreeNode getLeftChild() {

	return this.leftChild;

    }

    /**
     * Returns the right child of this node.
     *
     * @return  right child node, or null, if the node has no right child.
     */
    public HCTreeNode getRightChild() {

	return this.rightChild;

    }

    /**
     * Returns the data range of this node and the corresponding subtree.
     *
     * @return  a DataRange object.
     */
    public DataRange getDataRange() {

	return this.dataRange;

    }

    /**
     * Returns the height of this node.
     *
     * @return  the height.
     */
    public double getHeight() {

	return this.height;

    }

    /**
     * Returns the root node of this tree.
     *
     * @return  a HCTreeNode object specifying the root node.
     */
    public HCTreeNode getRoot() {

	// nothing checked. hope the tree is not broken.
	if (this.parent == null) return this;
	else return this.parent.getRoot();

    }

    /**
     * Returns a specified leaf node.
     *
     * @param index  data range index that specifies a leaf node.
     *
     * @throws IndexOutOfBoundsException, if the specified node
     * cannot be found.
     *
     * @return  the specified HCTreeNode object.
     */
    public HCTreeNode getLeafNodeByIndex(int index)
	throws IndexOutOfBoundsException {

	// nothing checked. hope the tree is not broken.

	if (this.dataRange.contains(index)) {

	    // if this is a leaf node we just return it.
	    if ((this.leftChild == null) && (this.rightChild == null)) {

		return this;

	    // if it is in left child datarange, we search from there.
	    } else if ((this.leftChild != null) && 
		(this.leftChild.getDataRange().contains(index))) {
		    return this.leftChild.getLeafNodeByIndex(index);
	    }
	    // otherwise we search from the right child.
	    else return this.rightChild.getLeafNodeByIndex(index);

	} else {

	    // index is not in the range of this node, so
	    // a) it does not exist, or
	    if (this.parent == null) throw new IndexOutOfBoundsException(
		    "The specified node does not exist in this tree.");
	    // b) it is a child of grandparents of this node.
	    return this.parent.getLeafNodeByIndex(index);

	}

    }

    /**
     * Updates the dataranges of this node and all the affected parents.
     *
     * @throws DataRangeMismatchException  if the data ranges of
     * the children are not next to each other.
     * @throws RuntimeException  if this node is already finalized.
     */
    private void updateDataRange() throws DataRangeMismatchException {

	DataRange newRange;
	DataRange oldRange;

	assertFinalized();

	if (this.leftChild != null)
	    newRange = (DataRange)this.leftChild.getDataRange().clone();
	else
	    newRange = new DataRange(0,-1);
	if (this.rightChild != null)
	    newRange.add(this.rightChild.getDataRange());

	oldRange = this.dataRange;
	this.dataRange = newRange;

	// if we changed things, we still need to update parents.
	// luckily nothing can go wrong with the parents, so we
	// don't have to worry about undoing changes here.
	if ((!newRange.equals(oldRange)) && (this.parent != null))
	    this.parent.updateDataRange();
    }

    /**
     * Sets the left child of this node.
     * This can indirectly affect four nodes (in this order.)
     * 1. the leftChild pointer is set as asked.
     * 2. if the new child already had a parent, the pointers of the old parent
     * will be nulled.
     * 3. the parent of the new child is set to point to this node.
     * 4. if this node already had a left child, the parent of the old
     * child is nulled.
     * If an exception happens, all changes will be undone.
     *
     * @param node  the new child.
     *
     * @throws DataRangeMismatchException  if the data range of the new
     * child is not adjacent to the datarange of the other child.
     * @throws RuntimeException  if the node is already the right child
     * of this node.
     * @throws RuntimeException  if this node is already finalized.
     */
    public void setLeftChild(HCTreeNode node) throws DataRangeMismatchException {

	HCTreeNode oldChild;

	assertFinalized(); // may throw.

	// if this is already the left child, we are done.
	// We need to return here, to avoid problems later on.
	if (this.leftChild == node) return;

	// if this is already the right child, this is illegal.
	if ((node != null) && (this.rightChild == node))
	    throw new RuntimeException();

	oldChild = this.leftChild;
	this.leftChild = node;

	try {
	    if (node != null) node.setParent(this);
	} catch (NotAChildException e) {
	    ; // this never happens.
	}

	try {
	    if (oldChild != null) oldChild.setParent(null);
	} catch (NotAChildException e) {
	    ; // this never happens.
	}

	this.updateDataRange();
    }

    /**
     * Sets the left child of this node.
     * This can indirectly affect four nodes (in this order.)
     * 1. the leftChild pointer is set as asked.
     * 2. if the new child already had a parent, the pointers of the old parent
     * will be nulled.
     * 3. the parent of the new child is set to point to this node.
     * 4. if this node already had a left child, the parent of the old
     * child is nulled.
     * If an exception happens, all changes will be undone.
     *
     * @param node  the new child.
     *
     * @throws DataRangeMismatchException  if the data range of the new
     * child is not adjacent to the datarange of the other child.
     * @throws RuntimeException  if the node is already the right child
     * of this node.
     * @throws RuntimeException  if this node is already finalized.
     */
    public void setRightChild(HCTreeNode node) throws DataRangeMismatchException {

	HCTreeNode oldChild;

	assertFinalized();

	// if this is already the right child, we are done.
	// We need to return here, to avoid problems later on.
	if (this.rightChild == node) return;

	// if this is already the left child, this is illegal.
	if ((node != null) && (this.leftChild == node))
	    throw new RuntimeException();


	oldChild = this.rightChild;
	this.rightChild = node;

	try {
	    if (node != null) node.setParent(this);
	} catch (NotAChildException e) {
	    ; // this never happens.
	}

	try {
	    if (oldChild != null) oldChild.setParent(null);
	} catch (NotAChildException e) {
	    ; // this never happens.
	}

	this.updateDataRange();
    }

    /**
     * Sets the parent of this node.
     * This method is not usable on its own. The only legal way to call
     * it is from the setLeftChild() and setRightChild() methods.
     * I.e. this method can be called only after the child of the parent
     * is already set to point to this node.
     *
     * The method can indirectly affect two nodes (in this order.)
     * 1. the reference to this node from the old parent is removed.
     * 2. The parent reference of this node is set as asked.
     *
     * If an exception happens, all changes will be undone.
     *
     * @param parent  the new parent.
     *
     * @throws NotAChildException  if this node is not a child of the
     * specified parent.
     * @throws RuntimeException  if this node is already finalized.
     */
    public void setParent(HCTreeNode parent) throws NotAChildException {

	assertFinalized();

	if (
	    (parent != null) &&
	    (parent.getRightChild() != this) &&
	    (parent.getLeftChild() != this)
	)
	    throw new NotAChildException(
		    "parent given to setParent() doesn't have this child.");

	// clean up old parent.
	if (this.parent != null) {
	    try {
		if (this.parent.getLeftChild() == this)
		    this.parent.setLeftChild(null);
		else if (this.parent.getRightChild() == this)
		    this.parent.setRightChild(null);
	    } catch (DataRangeMismatchException e) {

		// this never happens. The data ranges cannot mismatch
		// as there will be maximum one child.
		;


	    }
	}

	this.parent = parent;
    }

    /**
     * Finalizes the tree. After executing this
     *   - every node either has either no or exactly two children.
     *   - height of a node is higher than the heights of its children.
     *   - height of leaf nodes is zero.
     *   - every leaf node has width 1 datarange.
     *   - the data ranges of leaf nodes are in correct order, i.e.
     *     the first child counting from left is 0, then 1, etc.
     * This method really does nothing, but throws exceptions if necessary.
     *
     * @throws IllegalArgumentException  if this node is not a root node of
     * a tree.
     * @throws IllegalArgumentException  if any node in this root
     * doesn't meet all the criteria mentioned above.
     * a tree.
     */
    public void finalizeTree() throws IllegalArgumentException {

	if (this.parent != null) throw new IllegalArgumentException(
		"You may only finalize full trees.");

	this.finalizeTreeRecursively(0);
    }

    /**
     * A helper for finalizeTree()
     *
     * @param index  the index expected for the next child.
     *
     * @throws IllegalArgumentException  if this node does not
     * meet the criteria described in the documentation of
     * {@link finalizeTree}.
     *
     * @return  the index expected for the next child.
     */
    private int finalizeTreeRecursively(int index) {

	if ((this.leftChild == null) && (this.rightChild == null)) {

	    // A leaf node
	    if (this.height != 0) throw new IllegalArgumentException(
		"Height of a clustering tree leaf node is " + this.height
	    );
	    if (!this.dataRange.equals(new DataRange(index,index)))
		throw new IllegalArgumentException(
		    "Expecting index "+index+", but got range "+this.dataRange);
	    return index + 1;

	} else if ((this.leftChild != null) && (this.rightChild != null)) {

	    if (
		(this.height < this.leftChild.getHeight()) ||
		(this.height < this.rightChild.getHeight())
	    ) throw new IllegalArgumentException(
		"A height of a node is lower than the height of its child.");

	    index = this.leftChild.finalizeTreeRecursively(index);
	    index = this.rightChild.finalizeTreeRecursively(index);

	    this.finalized = true;

	    return index;

	} else throw new IllegalArgumentException(
	    "A clustering tree node must have either zero or two children."
	);

    }

}
