/* =======================================================================
 * A visualisation library extension for JFreeChart. Please see JFreeChart
 * for further information.
 * =======================================================================
 * Copyright (C) 2006  University of Helsinki, Department of Computer Science
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * -----------------------------
 * Contact:  ohtu@cs.helsinki.fi
 * -----------------------------
 *
 */

package org.jfree.chart.plot;

import java.awt.geom.Rectangle2D;
import java.awt.Point;
import java.awt.Rectangle;

import org.jfree.chart.entity.HCTreeNodeEntity;
import org.jfree.data.hc.HCTreeNode;
import org.jfree.chart.renderer.RendererState;

import org.jfree.ui.RectangleEdge;

/**
 * A state object used with a {@link HCPlot}.
 *
 * The class also implements some convenience methods for calculating
 * useful things based on plot state.
 *
 * @author  viski project
 */
public class HCPlotState extends RendererState {

    private int blockMinX;
    private int blockMinY;
    private int sizeOfNodeSymbol;

    private double blockWidth;
    private double blockHeight;
    private double columnTreeHeightUnitInPixels;
    private double rowTreeHeightUnitInPixels;


    /** 
     * Creates a new state object.
     *
     * @param info  xxx
     * @param blockMinX  xxx
     * @param blockMinY  xxx
     * @param sizeOfNodeSymbol  xxx
     * @param blockWidth  xxx
     * @param blockHeight  xxx
     * @param columnTreeHeightUnitInPixels  xxx
     * @param rowTreeHeightUnitInPixels  xxx
     */
    public HCPlotState(
	    PlotRenderingInfo info,
	    int blockMinX,
	    int blockMinY,
	    int sizeOfNodeSymbol,

	    double blockWidth,
	    double blockHeight,
	    double columnTreeHeightUnitInPixels,
	    double rowTreeHeightUnitInPixels ) {

        super(info);

	this.blockMinX = blockMinX;
	this.blockMinY = blockMinY;
	this.sizeOfNodeSymbol = sizeOfNodeSymbol;
	this.blockWidth = blockWidth;
	this.blockHeight = blockHeight;
	this.columnTreeHeightUnitInPixels = columnTreeHeightUnitInPixels;
	this.rowTreeHeightUnitInPixels = rowTreeHeightUnitInPixels;

    }
    
    /**
     * Returns the size of a visual symbol of a clustering tree node.
     *
     * @return  the width and height of a node symbol in pixels.
     */
    public int getSizeOfNodeSymbol() {

	return this.sizeOfNodeSymbol;

    }

    /**
     * Returns  the width of a heatmap block.
     *
     * @return the width of a heatmap block in pixels. The return
     * value is a double.
     */
    public double getBlockWidth() {

	return this.blockWidth;

    }

    /**
     * Returns the height of a heatmap block.
     *
     * @return  the height of a heatmap block in pixels. The return
     * value is a double.
     */
    public double getBlockHeight() {

	return this.blockHeight;

    }

    /**
     * Returns the number of pixels one column clustering tree height unit
     * translates to.
     *
     * @return  the size in pixels of a height unit.
     */
    public double getColumnTreeHeightUnitInPixels() {

	return this.columnTreeHeightUnitInPixels;

    }

    /**
     * Returns the number of pixels one row clustering tree height unit
     * translates to.
     *
     * @return  the size in pixels of a height unit.
     */
    public double getRowTreeHeightUnitInPixels() {

	return this.rowTreeHeightUnitInPixels;

    }

    /**
     * Returns the X-coordinate of the specified heatmap column.
     *
     * @param column  the heatmap column
     * @return  X-coordinate of the left edge of the column in pixels.
     */
    public int getHeatMapXCoordinate(int column) {

	return (int)(column * this.blockWidth + this.blockMinX);

    }

    /**
     * Returns the Y-coordinate of the specified heatmap row.
     *
     * @param row  the heatmap row
     * @return  T-coordinate of the top edge of the row in pixels.
     */
    public int getHeatMapYCoordinate(int row) {

	return (int)(row * this.blockHeight + this.blockMinY);

    }

    /**
     * Returns the area usable for drawing a subtree of a clustering tree.
     *
     * @param totalArea  the area reserved for the whole tree.
     * @param usedArea  the area already used for other parts of the tree.
     * @param edge  the edge of the heatmap where this tree is being drawn.
     *
     * @throws IllegalArgumentException  if the given edge parameter is illegal.
     *
     * @return  the area as a Rectangle object.
     */
    public Rectangle calculateSubTreeArea(
	    Rectangle2D totalArea,
	    Rectangle2D usedArea,
	    RectangleEdge edge) {

	if (edge == RectangleEdge.TOP) {

	    return new Rectangle (
		(int)(totalArea.getMinX()+usedArea.getWidth()),
		(int)totalArea.getMinY(),
		(int)(totalArea.getWidth()-usedArea.getWidth()),
		(int)totalArea.getHeight()
	    );

	} else if (edge == RectangleEdge.BOTTOM) {

	    return new Rectangle (
		(int)(totalArea.getMinX()+usedArea.getWidth()),
		(int)totalArea.getMinY(),
		(int)(totalArea.getWidth()-usedArea.getWidth()),
		(int)totalArea.getHeight()
	    );

	}
	else if (edge == RectangleEdge.LEFT) {

	    return new Rectangle (
		(int)totalArea.getMinX(),
		(int)(totalArea.getMinY()+usedArea.getHeight()),
		(int)totalArea.getWidth(),
		(int)(totalArea.getHeight()-usedArea.getHeight())
	    );

	}
	else if (edge == RectangleEdge.RIGHT) {

	    return new Rectangle (
		(int)totalArea.getMinX(),
		(int)(totalArea.getMaxY()+usedArea.getHeight()),
		(int)totalArea.getWidth(),
		(int)(totalArea.getHeight()-usedArea.getHeight())
	    );

	}
	else throw new IllegalArgumentException("Invalid edge.");

    }

    /**
     * Returns the center point of a branch node of a clustering tree.
     *
     * @param height  height of this node.
     * @param leftEntity  entity describing the node representing the
     * left child.
     * @param rightEntity  entity describing the node representing the
     * right child.
     * @param area  area where the tree is being drawn.
     * @param edge  the edge of the heatmap where this tree is being drawn.
     *
     * @throws IllegalArgumentException  if the given edge parameter is illegal.
     *
     * @return  the center of a branch node.
     */
    public Point calculateBranchNodeCenter(
	    double height,
	    HCTreeNodeEntity leftEntity,
	    HCTreeNodeEntity rightEntity,
	    Rectangle2D area,
	    RectangleEdge edge) {

	if (edge == RectangleEdge.TOP) {

	    return new Point (
		(int)((leftEntity.getCenter().getX()+rightEntity.getCenter().getX())/2),
		(int)(area.getMaxY()-height*this.columnTreeHeightUnitInPixels)
	    );

	} else if (edge == RectangleEdge.BOTTOM) {

	    return new Point (
		(int)((leftEntity.getCenter().getX()+rightEntity.getCenter().getX())/2),
		(int)(area.getMinY()+height*this.columnTreeHeightUnitInPixels)
	    );

	}
	else if (edge == RectangleEdge.LEFT) {

	    return new Point (
		(int)(area.getMaxX()-height*this.rowTreeHeightUnitInPixels),
		(int)((leftEntity.getCenter().getY()+rightEntity.getCenter().getY())/2)
	    );

	}
	else if (edge == RectangleEdge.RIGHT) {

	    return new Point (
		(int)(area.getMinX()+height*this.rowTreeHeightUnitInPixels),
		(int)((leftEntity.getCenter().getY()+rightEntity.getCenter().getY())/2)
	    );

	}
	else throw new IllegalArgumentException("Invalid edge.");
    }

    /**
     * Returns the area reserved by a branch node and the corresponding
     * subtree of a clustering tree.
     *
     * @param height  height of this node.
     * @param leftEntity  entity describing the node representing the
     * left child.
     * @param rightEntity  entity describing the node representing the
     * right child.
     * @param area  area where the tree is being drawn.
     * @param edge  the edge of the heatmap where this tree is being drawn.
     *
     * @throws IllegalArgumentException  if the given edge parameter is illegal.
     *
     * @return  the area as a Rectangle object.
     */
    public Rectangle calculateBranchNodeArea(
	    double height,
	    HCTreeNodeEntity leftEntity,
	    HCTreeNodeEntity rightEntity,
	    Rectangle2D area,
	    RectangleEdge edge) {

	if (edge == RectangleEdge.TOP) {

	    return new Rectangle (
		(int)(leftEntity.getCenter().getX()),
		(int)(area.getMaxY()
		      - height*this.columnTreeHeightUnitInPixels),
		(int)(rightEntity.getCenter().getX()
		      - leftEntity.getCenter().getX()),
		(int)(height*this.columnTreeHeightUnitInPixels)
	    );

	} else if (edge == RectangleEdge.BOTTOM) {

	    return new Rectangle (
		(int)(leftEntity.getCenter().getX()),
		(int)(area.getMinY()),
		(int)(rightEntity.getCenter().getX()
		      - leftEntity.getCenter().getX()),
		(int)(height*this.columnTreeHeightUnitInPixels)
	    );

	} else if (edge == RectangleEdge.LEFT) {

	    return new Rectangle (
		(int)(area.getMaxX()
		      - height*this.rowTreeHeightUnitInPixels),
		(int)(leftEntity.getCenter().getY()),
		(int)(height*this.rowTreeHeightUnitInPixels),
		(int)(rightEntity.getCenter().getY()
		      - leftEntity.getCenter().getY())
	    );

	} else if (edge == RectangleEdge.RIGHT) {

	    return new Rectangle (
		(int)(area.getMinX()),
		(int)(leftEntity.getCenter().getY()),
		(int)(height*this.rowTreeHeightUnitInPixels),
		(int)(rightEntity.getCenter().getY()
		      - leftEntity.getCenter().getY())
	    );

	}
	else throw new IllegalArgumentException("Invalid edge.");

    }
    
    /**
     * Returns the area reserved for a clustering tree leaf. 
     *
     * @param area  area where the tree is being drawn.
     * @param info  the object describing this node.
     * @param edge  the edge of the heatmap where this tree is being drawn.
     *
     * @throws IllegalArgumentException  if the given edge parameter is illegal.
     *
     * @return  the area.
     */
    public Rectangle calculateLeafNodeArea(
	    Rectangle2D area,
	    HCTreeNodeInfo info,
	    RectangleEdge edge) {

	int index;
	HCTreeNode node = info.getNode();

	try {
	    index = info.getVisibleDataRange().getLeftBound();
	} catch (Exception e) {
	    index = 0;
	    // This can never happen.
	}

	if (edge == RectangleEdge.TOP) {

	    return new Rectangle(
		getHeatMapXCoordinate(index),
		(int)(area.getMaxY()-
		      node.getHeight()*this.columnTreeHeightUnitInPixels),
		(int)getBlockWidth(),
		(int)(node.getHeight()*this.columnTreeHeightUnitInPixels)
	    );

	} else if (edge == RectangleEdge.BOTTOM) {

	    return new Rectangle(
		getHeatMapXCoordinate(index),
		(int)(area.getMinY()),
		(int)getBlockWidth(),
		(int)(node.getHeight()*this.columnTreeHeightUnitInPixels)
	    );

	}
	else if (edge == RectangleEdge.LEFT) {

	    return new Rectangle(
		(int)(area.getMaxX()-
		      (int)(node.getHeight()*this.rowTreeHeightUnitInPixels)),
		getHeatMapYCoordinate(index),
		(int)(node.getHeight()*this.rowTreeHeightUnitInPixels),
		(int)getBlockHeight()
	    );

	} else if (edge == RectangleEdge.RIGHT) {

	    return new Rectangle(
		(int)(area.getMinX()),
		getHeatMapYCoordinate(index),
		(int)(node.getHeight()*this.rowTreeHeightUnitInPixels),
		(int)getBlockHeight()
	    );

	} else throw new IllegalArgumentException("Invalid edge.");
	
    }

    /**
     * Returns the center point of a clustering tree leaf node.
     *
     * @param area  area where the leaf node is being drawn.
     * @param edge  the edge of the heatmap where this tree is being drawn.
     *
     * @throws IllegalArgumentException  if the given edge parameter is illegal.
     *
     * @return  the center of a leaf node.
     */
    public Point calculateLeafNodeCenter(Rectangle2D area, RectangleEdge edge) {
	if (edge == RectangleEdge.TOP) {

	    return new Point (
		(int)(area.getCenterX()),
		(int)(area.getMaxY())
	    );

	} else if (edge == RectangleEdge.BOTTOM) {

	    return new Point (
		(int)(area.getCenterX()),
		(int)(area.getMinY())
	    );

	} else if (edge == RectangleEdge.LEFT) {

	    return new Point (
		(int)(area.getMaxX()),
		(int)(area.getCenterY())
	    );

	} else if (edge == RectangleEdge.RIGHT) {

	    return new Point (
		(int)(area.getMinX()),
		(int)(area.getCenterY())
	    );

	} else throw new IllegalArgumentException("Invalid edge.");

    }

    /**
     * Returns the center point of a closed branch node.
     *
     * @param area  area where the tree is being drawn.
     * @param edge  the edge of the heatmap where this tree is being drawn.
     * @param info  an object describing the node.
     *
     * @throws IllegalArgumentException  if the given edge parameter is illegal.
     *
     * @return  the center of a closed branch node.
     */
    public Rectangle calculateClosedNodeArea(
	    Rectangle2D area,
	    HCTreeNodeInfo info,
	    RectangleEdge edge) {

	return calculateLeafNodeArea(area,info,edge);

    }

    /**
     * Returns the center point of a closed branch node.
     *
     * @param height  height of this node.
     * @param area  area where the subtree is being drawn.
     * @param edge  the edge of the heatmap where this tree is being drawn.
     *
     * @throws IllegalArgumentException  if the given edge parameter is illegal.
     *
     * @return  the center of a closed branch node.
     */
    public Point calculateClosedNodeCenter(
	    Rectangle2D area, double height, RectangleEdge edge
    ) {

	if (edge == RectangleEdge.TOP) {

	    return new Point (
		(int)(area.getCenterX()),
		(int)(area.getMaxY()-height*this.columnTreeHeightUnitInPixels)
	    );

	} else if (edge == RectangleEdge.BOTTOM) {

	    return new Point (
		(int)(area.getCenterX()),
		(int)(area.getMinY()+height*this.columnTreeHeightUnitInPixels)
	    );

	} else if (edge == RectangleEdge.LEFT) {

	    return new Point (
		(int)(area.getMaxX()-height*this.rowTreeHeightUnitInPixels),
		(int)(area.getCenterY())
	    );

	} else if (edge == RectangleEdge.RIGHT) {

	    return new Point (
		(int)(area.getMinX()+height*this.rowTreeHeightUnitInPixels),
		(int)(area.getCenterY())
	    );

	}
	else throw new IllegalArgumentException("Invalid edge.");

    }

}

