/* ======================================================================= 
 * A visualisation library extension for JFreeChart. Please see JFreeChart
 * for further information.
 * =======================================================================
 * Copyright (C) 2006  University of Helsinki, Department of Computer Science
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * -----------------------------
 * Contact:  ohtu@cs.helsinki.fi
 * -----------------------------
 *
 */


package org.jfree.data.hc;

import org.jfree.data.general.AbstractDataset;
import org.jfree.data.hc.HeatMap;
import org.jfree.data.hc.HCTreeNode;

import java.lang.Exception;

/**
 * A class representing data for the hierarchical clustering
 * and heatmap visualisation.
 * @author  viski project
 */
public class HCDataset extends AbstractDataset {

    private HeatMap heatMap;
    private HCTreeNode rowTree;
    private HCTreeNode columnTree;

    /**
     * Creates a new dataset.
     *
     * @param heatMap  the heatmap of this dataset.
     * @param columnTree  the column clustering tree of this dataset or null.
     * @param rowTree  the row clustering tree of this dataset or null.
     *
     * @throws IllegalArgumentException  if the row or column clustering
     * trees don't match the size of the heatmaps. The row clustering tree
     * must have exactly as many leaf nodes as there are rows in the heatmap,
     * and the row clustering tree must have exactly as many leaf nodes as
     * there are rows in the heatmap,
     * @throws NullPointerException  if the heatmap is null.
     * @throws IllegalArgumentException  if either of the trees does
     * not meet the criteria of finalizeTree.
     */
    public HCDataset(HeatMap heatMap, HCTreeNode rowTree, HCTreeNode columnTree)
	throws IllegalArgumentException, NullPointerException {

	if (heatMap == null) throw new NullPointerException(
		"HeatMap given to HCDataset() was null.");

	this.rowTree = rowTree;
	this.columnTree = columnTree;
	this.heatMap = heatMap;

	// does left tree match with heatmap height?
	if (
	    (this.rowTree != null) &&
	    (this.heatMap.getRowCount()
	    != this.rowTree.getDataRange().getWidth())
	) throw new IllegalArgumentException(
	    "Left tree doesn't match heatmap height.");

	// does top tree match with heatmap width
	if (
	    (this.columnTree != null) &&
	    (this.heatMap.getColumnsCount()
	    != this.columnTree.getDataRange().getWidth())
	) throw new IllegalArgumentException(
	    "Top tree doesn't match heatmap width.");

	// these throw exceptions if something is wrong.
	if (this.rowTree != null) this.rowTree.finalizeTree();
	if (this.columnTree != null) this.columnTree.finalizeTree();

    }

    /**
     * Returns the heatmap of this dataset.
     *
     * @return  the heatmap.
     */
    public HeatMap getHeatMap() {

	return this.heatMap;

    }

    /**
     * Returns the column clustering tree.
     *
     * @return  the clustering tree, or null, if the tree does not exist.
     */
    public HCTreeNode getColumnClusteringTree() {

	return this.columnTree;

    }

    /**
     * Returns the row clustering tree.
     *
     * @return  the clustering tree, or null, if the tree does not exist.
     */
    public HCTreeNode getRowClusteringTree() {

	return this.rowTree;

    }

}
