
package org.jfree.chart.plot.junit;

import java.awt.Rectangle;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.geom.Rectangle2D;
import java.awt.event.InputEvent;
import java.awt.event.MouseEvent;
import java.awt.Point;

import javax.swing.event.ChangeEvent;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.jfree.data.hc.DataRange;

import org.jfree.chart.plot.HCPlot;
import org.jfree.chart.plot.HCTreeNodeInfo;
import org.jfree.chart.plot.AbstractHCClusteringInfo;
import org.jfree.chart.plot.GradientColorPalette;
import org.jfree.chart.plot.StandardHCClusteringInfo;
import org.jfree.chart.plot.DummyHCClusteringInfo;
import org.jfree.chart.plot.PlotState;
import org.jfree.chart.plot.PlotRenderingInfo;
import org.jfree.chart.event.PlotChangeListener;
import org.jfree.chart.event.PlotChangeEvent;
import org.jfree.chart.event.SelectionChangeEvent;
import org.jfree.chart.event.ClusteringTreeChangeEvent;
import org.jfree.chart.entity.HCTreeNodeEntity;
import org.jfree.chart.entity.HeatMapBlockEntity;
import org.jfree.chart.labels.StandardHCToolTipGenerator;
import org.jfree.data.hc.*;

import org.jfree.chart.JFreeChart;
import org.jfree.chart.BioChartFactory;
import org.jfree.chart.ChartRenderingInfo;
import org.jfree.chart.ChartMouseEvent;
import org.jfree.chart.ChartPanel;


/**
 * Tests for the {@link HCPlot} class.
 */
public class HCPlotTests extends TestCase {

    private HCPlot plot;
    private HCPlot plotNoRowTree;
    private HCPlot plotNoColumnTree;
    private HCPlot plotNoTrees;

	private HCTestListener listener;

	private class HCTestListener implements PlotChangeListener {

		public Rectangle selection;
		public HCTreeNodeInfo info;
		public boolean event_received;

		public HCTestListener() {
			this.init();
		}

		public void init() {
			this.selection = null;
			this.info = null;
			this.event_received = false;
		}

		public void plotChanged(PlotChangeEvent event) {
			this.event_received = true;
			if (event instanceof SelectionChangeEvent) {
				HCPlot plot = (HCPlot)(event.getPlot());
				this.selection = plot.getSelection();
			}
			else if (event instanceof ClusteringTreeChangeEvent) {
				this.info = ((ClusteringTreeChangeEvent)event).getTreeNodeInfo();
			}
		}

	}

    /**
     * Returns the tests as a test suite.
     *
     * @return The test suite.
     */
    public static Test suite() {
        return new TestSuite(HCPlotTests.class);
    }

    /**
     * Constructs a new set of tests.
     *
     * @param name  the name of the tests.
     */
    public HCPlotTests(String name) throws Exception {

        super(name);

        this.plot = new HCPlot(createDataset(3,5,true,true));
        this.plotNoRowTree = new HCPlot(createDataset(3,5,false,true));
        this.plotNoColumnTree = new HCPlot(createDataset(3,5,true,false));
        this.plotNoTrees = new HCPlot(createDataset(3,5,false,false));
        
		this.listener = new HCTestListener();
		this.plot.addChangeListener(this.listener);

    }

    /**
     * Creates a test dataset.
     *
     * @param rows  number of rows in the dataset.
     * @param columns  number of columns in the dataset.
     * @param includeRowTree  true, if the dataset should have a row tree.
     * @param includeColumnTree  true, if the dataset should have a column tree.
     *
     * @throws Exception  if there was something wrong creating
     * the dataset.
     *
     * @return  a HCDataset.
     */
    public static HCDataset createDataset(
	    int rows,
	    int columns,
	    boolean includeRowTree,
	    boolean includeColumnTree) throws Exception {

	HCTreeNode rowTree;
	HCTreeNode columnTree;
	HeatMap heatmap;

	if(includeRowTree) rowTree = createTree(rows,0);
	else rowTree = null;
	if(includeColumnTree) columnTree = createTree(columns,0);
	else columnTree = null;

	heatmap = new HeatMap("",rows,columns);

	return new HCDataset(heatmap,rowTree,columnTree);

    }

    /**
     * Creates a test clustering tree of specified width.
     * The trees created are pretty balanced and simple.
     *
     * @param rows  number of rows in the dataset.
     * @param index  counter for the recursion. This is always zero
     * when the method is called by someone else than the method itself.
     *
     * @throws Exception  if there was something wrong creating the
     * dataset.
     *
     * @return  a HCTreeNode or null, if width == 0.
     */
    public static HCTreeNode createTree(int width, int index) throws Exception {

	int i;

	if (width == 0) return null;

	// just return a leaf node if width = 1.
	if (width == 1) return new HCTreeNode(0,index);

	// or recursively create the tree.
	HCTreeNode root = new HCTreeNode(100); // the root node.
	
	HCTreeNode leftTree = createTree(width / 2, index);
	index += width / 2;
	HCTreeNode rightTree = createTree(width - (width / 2), index);

	root.setLeftChild(leftTree);
	root.setRightChild(rightTree);

	return root;

    }

    /**
     * Test constructors.
     */
    public void testCreation() throws Exception {

	HCPlot plot;
        plot = new HCPlot(createDataset(3,5,true,true));
        plot = new HCPlot(createDataset(5,3,false,false));
        
    }

    /**
     * Test simple set-get-methods.
     */
    public void testSimpleMethods() {

	AbstractHCClusteringInfo info;

	try {
		this.plot.setRowTreeSize(0);
	} catch (IllegalArgumentException e) {
		assertTrue(false);
	}
	try {
		this.plot.setRowTreeSize(1);
	} catch (IllegalArgumentException e) {
		assertTrue(false);
	}
	try {
		this.plot.setColumnTreeSize(0);
	} catch (IllegalArgumentException e) {
		assertTrue(false);
	}
	try {
		this.plot.setColumnTreeSize(1);
	} catch (IllegalArgumentException e) {
		assertTrue(false);
	}
	this.plot.setRowTreeSize(0.3);
	assertTrue(this.plot.getRowTreeSize() == 0.3);
	this.plot.setColumnTreeSize(0.3);
	assertTrue(this.plot.getColumnTreeSize() == 0.3);
	try {
		this.plot.setRowTreeSize(-1);
		assertTrue(false);
	} catch (IllegalArgumentException e) {
		IllegalArgumentException f = new IllegalArgumentException("Size of a tree cannot be negative.");
		assertTrue(e.toString().equals(f.toString()));
	}
	assertTrue(this.plot.getRowTreeSize() == 0.3);
	try {
		this.plot.setRowTreeSize(-Double.MIN_VALUE);
		assertTrue(false);
	} catch (IllegalArgumentException e) {
		IllegalArgumentException f = new IllegalArgumentException("Size of a tree cannot be negative.");
		assertTrue(e.toString().equals(f.toString()));
	}
	assertTrue(this.plot.getRowTreeSize() == 0.3);
	try {
		this.plot.setRowTreeSize(-Double.MAX_VALUE);
		assertTrue(false);
	} catch (IllegalArgumentException e) {
		IllegalArgumentException f = new IllegalArgumentException("Size of a tree cannot be negative.");
		assertTrue(e.toString().equals(f.toString()));
	}
	assertTrue(this.plot.getRowTreeSize() == 0.3);
	try {
		this.plot.setRowTreeSize(2);
		assertTrue(false);
	} catch (IllegalArgumentException e) {
		IllegalArgumentException f = new IllegalArgumentException("Size of a tree cannot be more than 1.");
		assertTrue(e.toString().equals(f.toString()));
	}
	assertTrue(this.plot.getRowTreeSize() == 0.3);
	try {
		this.plot.setRowTreeSize(Double.MAX_VALUE);
		assertTrue(false);
	} catch (IllegalArgumentException e) {
		IllegalArgumentException f = new IllegalArgumentException("Size of a tree cannot be more than 1.");
		assertTrue(e.toString().equals(f.toString()));
	}
	assertTrue(this.plot.getRowTreeSize() == 0.3);
	try {
		this.plot.setColumnTreeSize(-1);
		assertTrue(false);
	} catch (IllegalArgumentException e) {
		IllegalArgumentException f = new IllegalArgumentException("Size of a tree cannot be negative.");
		assertTrue(e.toString().equals(f.toString()));
	}
	assertTrue(this.plot.getColumnTreeSize() == 0.3);
	try {
		this.plot.setColumnTreeSize(-Double.MIN_VALUE);
		assertTrue(false);
	} catch (IllegalArgumentException e) {
		IllegalArgumentException f = new IllegalArgumentException("Size of a tree cannot be negative.");
		assertTrue(e.toString().equals(f.toString()));
	}
	assertTrue(this.plot.getColumnTreeSize() == 0.3);
	try {
		this.plot.setColumnTreeSize(-Double.MAX_VALUE);
		assertTrue(false);
	} catch (IllegalArgumentException e) {
		IllegalArgumentException f = new IllegalArgumentException("Size of a tree cannot be negative.");
		assertTrue(e.toString().equals(f.toString()));
	}
	assertTrue(this.plot.getColumnTreeSize() == 0.3);
	try {
		this.plot.setColumnTreeSize(2);
		assertTrue(false);
	} catch (IllegalArgumentException e) {
		IllegalArgumentException f = new IllegalArgumentException("Size of a tree cannot be more than 1.");
		assertTrue(e.toString().equals(f.toString()));
	}
	assertTrue(this.plot.getColumnTreeSize() == 0.3);
	try {
		this.plot.setColumnTreeSize(Double.MAX_VALUE);
		assertTrue(false);
	} catch (IllegalArgumentException e) {
		IllegalArgumentException f = new IllegalArgumentException("Size of a tree cannot be more than 1.");
		assertTrue(e.toString().equals(f.toString()));
	}
	assertTrue(this.plot.getColumnTreeSize() == 0.3);

	this.plot.setRowNamesSize(0.3);
	assertTrue(this.plot.getRowNamesSize() == 0.3);
	this.plot.setColumnNamesSize(0.3);
	assertTrue(this.plot.getColumnNamesSize() == 0.3);
	try {
		this.plot.setRowNamesSize(-1);
		assertTrue(false);
	} catch (IllegalArgumentException e) {
		IllegalArgumentException f = new IllegalArgumentException("Size of names cannot be negative.");
		assertTrue(e.toString().equals(f.toString()));
	}
	assertTrue(this.plot.getRowNamesSize() == 0.3);
	try {
		this.plot.setRowNamesSize(2);
		assertTrue(false);
	} catch (IllegalArgumentException e) {
		IllegalArgumentException f = new IllegalArgumentException("Size of names cannot be more than 1.");
		assertTrue(e.toString().equals(f.toString()));
	}
	assertTrue(this.plot.getRowNamesSize() == 0.3);
	try {
		this.plot.setColumnNamesSize(-1);
		assertTrue(false);
	} catch (IllegalArgumentException e) {
		IllegalArgumentException f = new IllegalArgumentException("Size of names cannot be negative.");
		assertTrue(e.toString().equals(f.toString()));
	}
	assertTrue(this.plot.getColumnNamesSize() == 0.3);
	try {
		this.plot.setColumnNamesSize(2);
		assertTrue(false);
	} catch (IllegalArgumentException e) {
		IllegalArgumentException f = new IllegalArgumentException("Size of names cannot be more than 1.");
		assertTrue(e.toString().equals(f.toString()));
	}
	assertTrue(this.plot.getColumnNamesSize() == 0.3);

	this.plot.showRowTree();
	assertTrue(this.plot.getRowTreeVisibility());
	this.plot.hideRowTree();
	assertFalse(this.plot.getRowTreeVisibility());
	try {
		this.plotNoRowTree.showRowTree();
		assertTrue(false);
	} catch (NullPointerException e) {
		NullPointerException f
			= new NullPointerException("Cannot show a null tree.");
		assertTrue(e.toString().equals(f.toString()));
	}
	assertFalse(this.plotNoRowTree.getRowTreeVisibility());
	try {
		this.plotNoRowTree.hideRowTree();
		assertTrue(false);
	} catch (NullPointerException e) {
		NullPointerException f
			= new NullPointerException("Cannot hide a null tree.");
		assertTrue(e.toString().equals(f.toString()));
	}
	assertFalse(this.plotNoRowTree.getRowTreeVisibility());
	this.plotNoColumnTree.showRowTree();
	assertTrue(this.plotNoColumnTree.getRowTreeVisibility());
	this.plotNoColumnTree.hideRowTree();
	assertFalse(this.plotNoColumnTree.getRowTreeVisibility());

	this.plot.showColumnTree();
	assertTrue(this.plot.getColumnTreeVisibility());
	this.plot.hideColumnTree();
	assertFalse(this.plot.getColumnTreeVisibility());
	this.plotNoRowTree.showColumnTree();
	assertTrue(this.plotNoRowTree.getColumnTreeVisibility());
	this.plotNoRowTree.hideColumnTree();
	assertFalse(this.plotNoRowTree.getColumnTreeVisibility());
	try {
		this.plotNoColumnTree.showColumnTree();
		assertTrue(false);
	} catch (NullPointerException e) {
		NullPointerException f
			= new NullPointerException("Cannot show a null tree.");
		assertTrue(e.toString().equals(f.toString()));
	}
	assertFalse(this.plotNoColumnTree.getColumnTreeVisibility());
	try {
		this.plotNoColumnTree.hideColumnTree();
		assertTrue(false);
	} catch (NullPointerException e) {
		NullPointerException f
			= new NullPointerException("Cannot hide a null tree.");
		assertTrue(e.toString().equals(f.toString()));
	}

	// dummy tests for these.
	info = this.plot.getColumnClusteringInfo();
	info = this.plot.getRowClusteringInfo();
	this.plot.getPlotType();

	// palette setget.
	GradientColorPalette p = new GradientColorPalette(-1,1);
	this.plot.setColoring(p);
	assertTrue (this.plot.getColoring() == p);

	// selection set get.
	Rectangle r = new Rectangle(0,0,1,1);
	this.plot.setSelection(r);
	assertTrue (this.plot.getSelection() == r);

	r = new Rectangle(-2,0,1,1);
	try {
	    this.plot.setSelection(r);
	    assertTrue(false);
	} catch (IndexOutOfBoundsException e) {
	    ;
	}
	r = new Rectangle(0,-2,1,1);
	try {
	    this.plot.setSelection(r);
	    assertTrue(false);
	} catch (IndexOutOfBoundsException e) {
	    ;
	}
	r = new Rectangle(0,0,this.plot.getDataset().getHeatMap().getColumnsCount()+1,1);
	try {
	    this.plot.setSelection(r);
	    assertTrue(false);
	} catch (IndexOutOfBoundsException e) {
	    ;
	}
	r = new Rectangle(0,0,1,this.plot.getDataset().getHeatMap().getRowCount()+1);
	try {
	    this.plot.setSelection(r);
	    assertTrue(false);
	} catch (IndexOutOfBoundsException e) {
	    ;
	}

	// the rest of the methods.

	this.listener.init();
	this.plot.setMouseOverHighlight(true);
	assertTrue(this.listener.event_received);
	assertTrue(this.plot.getMouseOverHighlight());

	this.listener.init();
	this.plot.setMouseOverHighlight(false);
	assertTrue(this.listener.event_received);
	assertFalse(this.plot.getMouseOverHighlight());

	this.listener.init();
	this.plot.setSelectionHighlight(false);
	assertTrue(this.listener.event_received);
	assertFalse(this.plot.getSelectionHighlight());

	this.listener.init();
	this.plot.setSelectionHighlight(true);
	assertTrue(this.listener.event_received);
	assertTrue(this.plot.getSelectionHighlight());

	this.listener.init();
	this.plot.setAverageHighlight(false);
	assertTrue(this.listener.event_received);
	assertFalse(this.plot.getAverageHighlight());

	this.listener.init();
	this.plot.setAverageHighlight(true);
	assertTrue(this.listener.event_received);
	assertTrue(this.plot.getAverageHighlight());

	this.listener.init();
	this.plot.setRowTreeHighlight(0);
	assertTrue(this.listener.event_received);
	assertTrue(this.plot.getRowTreeHighlight() == 0);

	this.listener.init();
	this.plot.setRowTreeHighlight(-1);
	assertTrue(this.listener.event_received);
	assertTrue(this.plot.getRowTreeHighlight() < 0);

	this.listener.init();
	this.plot.setColumnTreeHighlight(0);
	assertTrue(this.listener.event_received);
	assertTrue(this.plot.getColumnTreeHighlight() == 0);

	this.listener.init();
	this.plot.setColumnTreeHighlight(-1);
	assertTrue(this.listener.event_received);
	assertTrue(this.plot.getColumnTreeHighlight() < 0);

	StandardHCToolTipGenerator tooltipgen = new StandardHCToolTipGenerator();
	this.plotNoColumnTree.setToolTipGenerator(tooltipgen);
	assertTrue(this.plotNoColumnTree.getToolTipGenerator() == tooltipgen);
	this.plotNoColumnTree.setToolTipGenerator(null);
	assertTrue(this.plotNoColumnTree.getToolTipGenerator() == null);

	assertTrue(this.plot.getOptionsPanel() != null);
	assertTrue(this.plot.getPalettePanel() != null);

	assertTrue(this.plot.getOptionsPanelName() != null);
	assertTrue(this.plot.getPalettePanelName() != null);

	DataRange range = this.plot.getDatasetRowsAtHeatMapRow(0);
	assertTrue(range.equals(new DataRange(0, 0)));
	range = this.plot.getDatasetColumnsAtHeatMapColumn(0);
	assertTrue(range.equals(new DataRange(0, 0)));

    }
    /**
     * Check the draw method.
     */
    public void testDraw() {

	// this.plot.draw();
	// this.plotNoRowTree.draw();
	// this.plotNoColumnTree.draw();
	// this.plotNoTrees.draw();

        try {
			HCDataset dataset = createDataset(3, 5, true, true);
			HCDataset datasetNoRowTree = createDataset(3, 5, false, true);
			HCDataset datasetNoColumnTree = createDataset(3, 5, true, false);
            JFreeChart chart = BioChartFactory.createHCChart(
                    "HC-test", // title
                    dataset,
                    true, // tooltips?
                    false // URLs?
                    );
			JFreeChart chartNoRowTree = BioChartFactory.createHCChart(
					"HC-test without Row Tree", // title
					datasetNoRowTree,
					true, // tooltips?
					false // URLs?
					);
			JFreeChart chartNoColumnTree = BioChartFactory.createHCChart(
					"HC-test without Column Tree", // title
					datasetNoColumnTree,
					false, // tooltips?
					false // URLs?
					);
            ChartPanel panel = new ChartPanel(chart);
			ChartPanel panelNoRowTree = new ChartPanel(chartNoRowTree);
			ChartPanel panelNoColumnTree = new ChartPanel(chartNoColumnTree);
            BufferedImage image = new BufferedImage(200, 200,
                            BufferedImage.TYPE_INT_RGB);
            Graphics2D g = image.createGraphics();
            ChartRenderingInfo chartInfo = new ChartRenderingInfo();

			// draw with default state
            chart.draw(g, new Rectangle2D.Double(0, 0, 200, 200), chartInfo);
            assertTrue(image != null);

            HCPlot plot = (HCPlot)chart.getPlot();
			HCPlot plotNoRowTree = (HCPlot)chartNoRowTree.getPlot();
			HCPlot plotNoColumnTree = (HCPlot)chartNoColumnTree.getPlot();
			plot.setAverageHighlight(true);

			// test selection
            plot.setSelection(new Rectangle(0, 0, 1, 1));
			Rectangle rect = new Rectangle(0, 0, 1, 1);
			assertTrue(rect.equals(plot.getSelection()));

			// draw with selection
            chart.draw(g, new Rectangle2D.Double(0, 0, 200, 200), chartInfo);
            assertTrue(image != null);
			plot.setSelection(new Rectangle(0, 0, 0, 0));

			// create MouseEvents for moving and clicking
			// (clicking also with shift and control modifiers)
			MouseEvent me_move = new MouseEvent(panel, 0, 0,
						0, // no modifiers
						1, 1,
						0, // no click
						false);
			MouseEvent me_click = new MouseEvent(panel, 0, 0,
						InputEvent.BUTTON1_MASK,
						1, 1,
						1, // click
						false);
			MouseEvent me_shift_click = new MouseEvent(panel, 0, 0,
						InputEvent.BUTTON1_MASK | InputEvent.SHIFT_MASK,
						1, 1,
						1, // click
						false);
			MouseEvent me_ctrl_click = new MouseEvent(panel, 0, 0,
						InputEvent.BUTTON1_MASK | InputEvent.CTRL_MASK,
						1, 1,
						1, // click
						false);

			// apply MouseEvents
			plot.chartMouseMoved(new ChartMouseEvent(chart, me_move, null));
			plot.chartMouseClicked(new ChartMouseEvent(chart, me_click, null));
			plot.chartMouseClicked(new ChartMouseEvent(chart, me_shift_click, null));
			plot.chartMouseClicked(new ChartMouseEvent(chart, me_ctrl_click, null));

			// get information used with clicking parts of the visualization
			HCTreeNode node = dataset.getColumnClusteringTree();
			HCTreeNodeInfo colInfo = plot.getColumnClusteringInfo().getRootNode();
			HCTreeNodeEntity colEntity = new
					HCTreeNodeEntity(new Rectangle(0, 0, 2, 2),
							"", // tooltiptext
							"", // urltext
							new Point(0, 0),
							new Rectangle(0, 0, 1, 1),
							colInfo);
			HCTreeNodeInfo rowInfo = plot.getRowClusteringInfo().getRootNode();
			HCTreeNodeEntity rowEntity = new
					HCTreeNodeEntity(new Rectangle(0, 0, 2, 2),
							"", // tooltiptext
							"", // urltext
							new Point(0, 0),
							new Rectangle(0, 0, 1, 1),
							rowInfo);
			plot.chartMouseMoved(new ChartMouseEvent(chart, me_move, colEntity));

			// test closing a column node
			plot.chartMouseClicked(new ChartMouseEvent(chart, me_click, colEntity));
            chart.draw(g, new Rectangle2D.Double(0, 0, 200, 200), chartInfo);
            assertTrue(image != null);
			plot.chartMouseClicked(new ChartMouseEvent(chart, me_click, colEntity));

			// test closing a row node
			plot.chartMouseClicked(new ChartMouseEvent(chart, me_click, rowEntity));
            chart.draw(g, new Rectangle2D.Double(0, 0, 200, 200), chartInfo);
            assertTrue(image != null);

			plot.chartMouseClicked(new ChartMouseEvent(chart, me_shift_click, colEntity));
			plot.chartMouseClicked(new ChartMouseEvent(chart, me_shift_click, rowEntity));
			plot.chartMouseClicked(new ChartMouseEvent(chart, me_ctrl_click, colEntity));
			plot.chartMouseClicked(new ChartMouseEvent(chart, me_ctrl_click, rowEntity));
			HeatMapBlockEntity hentity = new
					HeatMapBlockEntity(new Rectangle(0, 0, 2, 2),
							"", // tooltiptext
							"", // urltext
							0, 0);
			plot.chartMouseMoved(new ChartMouseEvent(chart, me_move, hentity));

			// test clicking a HeatMapBlock
			plot.chartMouseClicked(new ChartMouseEvent(chart, me_click, hentity));
            chart.draw(g, new Rectangle2D.Double(0, 0, 200, 200), chartInfo);
            assertTrue(image != null);

			// test drawing with different proportions
			image = new BufferedImage(200, 100, BufferedImage.TYPE_INT_RGB);
			g = image.createGraphics();
            chart.draw(g, new Rectangle2D.Double(0, 0, 200, 100), chartInfo);
            assertTrue(image != null);
			plot.chartMouseClicked(new ChartMouseEvent(chart, me_shift_click, hentity));
			plot.chartMouseClicked(new ChartMouseEvent(chart, me_ctrl_click, hentity));

			// test drawing with different proportions
			image = new BufferedImage(100, 200, BufferedImage.TYPE_INT_RGB);
			g = image.createGraphics();
			chartNoRowTree.draw(g, new Rectangle2D.Double(0, 0, 100, 200), chartInfo);
			assertTrue(image != null);
			chartNoColumnTree.draw(g, new Rectangle2D.Double(0, 0, 100, 200), chartInfo);
			assertTrue(image != null);

			plot.stateChanged(new ChangeEvent(colInfo));
			plot.stateChanged(new ChangeEvent(rowInfo));

        } catch (Exception e) {
            assertTrue(false);
        }

		// illegal and/or strange stuff

		try {
			new HCPlot(null);
			assertTrue(false);
		} catch (NullPointerException e) {
		}
		assertTrue(this.plotNoTrees.getColumnClusteringInfo()
				instanceof DummyHCClusteringInfo);
		assertTrue(this.plotNoTrees.getRowClusteringInfo()
				instanceof DummyHCClusteringInfo);
		try {
			this.plotNoTrees.setColoring(null);
			assertTrue(false);
		} catch (Exception e) {
		}
		try {
			this.plotNoTrees.setSelection(null);
			assertTrue(false);
		} catch (Exception e) {
		}
		try {
			this.plotNoColumnTree.getDatasetRowsAtHeatMapRow(-1);
			assertTrue(false);
		} catch (Exception e) {
		}
		try {
			this.plotNoRowTree.getDatasetColumnsAtHeatMapColumn(-1);
			assertTrue(false);
		} catch (Exception e) {
		}
		try {
			this.plotNoColumnTree.getDatasetRowsAtHeatMapRow(Integer.MIN_VALUE);
			assertTrue(false);
		} catch (Exception e) {
		}
		try {
			this.plotNoRowTree.getDatasetColumnsAtHeatMapColumn(Integer.MIN_VALUE);
			assertTrue(false);
		} catch (Exception e) {
		}
		try {
			this.plotNoColumnTree.getDatasetRowsAtHeatMapRow(Integer.MAX_VALUE);
			assertTrue(false);
		} catch (Exception e) {
		}
		try {
			this.plotNoRowTree.getDatasetColumnsAtHeatMapColumn(Integer.MAX_VALUE);
			assertTrue(false);
		} catch (Exception e) {
		}
		BufferedImage image2 = new BufferedImage(100, 100, BufferedImage.TYPE_INT_RGB);
		Graphics2D g2 = image2.createGraphics();
		Rectangle area2 = new Rectangle(0, 0, 100, 100);
		Point point2 = new Point();
		PlotState ps2 = new PlotState();
		PlotRenderingInfo pri2 = new PlotRenderingInfo(new ChartRenderingInfo());
		try {
			this.plot.draw(null, area2, point2, ps2, pri2);
			assertTrue(false);
		} catch (Exception e) {
		}
		try {
			this.plot.draw(g2, null, point2, ps2, pri2);
			assertTrue(false);
		} catch (Exception e) {
		}
		try {
			this.plot.draw(g2, area2, null, ps2, pri2);
			assertTrue(false);
		} catch (Exception e) {
		}
		try {
			this.plot.draw(g2, area2, point2, null, pri2);
			assertTrue(false);
		} catch (Exception e) {
		}
		try {
			this.plot.draw(g2, area2, point2, ps2, null);
			assertTrue(false);
		} catch (Exception e) {
		}
		
		try {
			plot.stateChanged(null);
			assertTrue(false);
		} catch (Exception e) {
		}

    }

    /**
     * Runs the test suite using JUnit's text-based runner.
     * 
     * @param args  ignored.
     */
    public static void main(String[] args) {
        junit.textui.TestRunner.run(suite());
    }

}
