/* =======================================================================
 * A visualisation library extension for JFreeChart. Please see JFreeChart
 * for further information.
 * =======================================================================
 * Copyright (C) 2006  University of Helsinki, Department of Computer Science
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * -----------------------------
 * Contact:  ohtu@cs.helsinki.fi
 * -----------------------------
 *
 */

package org.jfree.chart.plot;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.jfree.data.hc.HCTreeNode;
import org.jfree.data.hc.DataRange;

/**
 * A class that stores information about the state of a clustering tree.
 * used in {@link HCPlot}. This is organized as a tree with the same topology
 * as the original clustering tree.
 *
 * @author  viski project
 */
public abstract class AbstractHCClusteringInfo {

    protected LinkedList listeners;
    protected int location;
    protected String[] names;

    /**
     * Return the names of visible rows/columns corresponding to
     * this clustering info object.
     *
     * @return  a list object containing the names.
     */
    public abstract List getNames();

    /**
     * Return the HCTreeNodeInfo corresponding to the root node
     * of the clustering tree.
     *
     * @return  the root node, or null, if the clustering tree does not exist.
     */
    public abstract HCTreeNodeInfo getRootNode();

    /**
     * Returns the indices of the dataset that
     * correspond to specified visible index
     *
     * @param row  the row.
     * @return  The datarange specifying the indices.
     */
    public abstract DataRange getDataRangeForVisibleIndex(int row);

    /**
     * Returns the visible data range. i.e. number of rows/columns
     * of a heatmap currently seen on the screen.
     *
     * @return  The datarange specifying the visible indices.
     */
    public abstract int getNumberOfVisibleItems();

    /**
     * Returns a visible name for specified datarange.
     *
     * @param range  The datarange.
     * @return  The name.
     */
    public String getName(DataRange range) {

	int leftBound;
	int rightBound;
	String leftName;
	String rightName;
	try {

	    leftBound = range.getLeftBound();
	    rightBound = range.getRightBound();

	} catch (Exception e) {

	    return "[]"; // this doesn't happen if used properly

	}
	try {

	    leftName = names[leftBound];
	    rightName = names[rightBound];

	} catch (Exception e) {

	    leftName = ""+leftBound;
	    rightName = ""+rightBound;

	}
	if (leftBound == rightBound) return leftName;
	else return leftName + " - " + rightName;

    }

    /**
     * Returns the location of this clustering tree.
     *
     * @return  The location. Either HCPlot.LEFT or HCPlot.TOP.
     */
    public int getLocation() {
      
        return this.location;

    }

    /**
     * Adds a change listener
     *
     * @param listener  the listener.
     */
    public void addChangeListener(ChangeListener listener) {

	this.listeners.add((Object)listener);

    }

    /**
     * Removes a change listener
     *
     * @param listener  the listener.
     */
    public void removeChangeListener(ChangeListener listener) {

	this.listeners.remove((Object)listener);

    }

    /**
     * Notifies listeners of an event
     *
     * @param event  the event.
     */
    public void notifyChangeListeners(ChangeEvent event) {

	Iterator iterator = listeners.iterator();
	ChangeListener listener;

	while(iterator.hasNext()) {

	    listener = (ChangeListener)(iterator.next());
	    listener.stateChanged (event);

	}

    }

}
