

package org.jfree.data.som.junit;

import java.awt.Color;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInput;
import java.io.ObjectInputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import org.jfree.data.som.SOMDataItem;
import org.jfree.data.som.SOMDataset;

/**
 * Tests for the {@link org.jfree.data.general.PieDataset} class.
 */
public class SOMDatasetTests extends TestCase {
    
    private SOMDataset dataset;
    
    /**
     * Returns the tests as a test suite.
     *
     * @return The test suite.
     */
    public static Test suite() {
        return new TestSuite(SOMDatasetTests.class);
    }

    /**
     * Constructs a new set of tests.
     *
     * @param name  the name of the tests.
     */
    public SOMDatasetTests(String name) {
        super(name);
        
    }

    protected void setUp() throws Exception {
        this.dataset = new SOMDataset(2, 2);
        
        String[] s = new String[] {"A", "B"};
        double[] w = new double[] {1.0, 2.3, 2.7};
        this.dataset.addValue(0, 0, new Color(0,0,100), s, w);
        
        s = new String[] {"C"};
        w = new double[] {1.1, 2.5};
        this.dataset.addValue(1, 0, new Color(100,0,0), s, w);
        
        s = new String[] {"D", "E"};
        w = new double[] {1.1, 2.5, 1.9, 4.4};
        this.dataset.addValue(0, 1, new Color(255,255,255), s, w);
        
        s = new String[] {"F", "G", "H"};
        w = new double[] {1.1, 2.5, 1.9, 4.4};
        this.dataset.addValue(1, 1, new Color(0,0,100), s, w);
    }
    
    public void testConstructor() {
        try {
            new SOMDataset(0, 1);
            assertTrue(false);
        } catch (IllegalArgumentException e) {
        }
        try {
            new SOMDataset(1, 0);
            assertTrue(false);
        } catch (IllegalArgumentException e) {
        }
        try {
            new SOMDataset(0, 0);
            assertTrue(false);
        } catch (IllegalArgumentException e) {
        }
        try {
            new SOMDataset(-1, 1);
            assertTrue(false);
        } catch (IllegalArgumentException e) {
        }
        try {
            new SOMDataset(1, -1);
            assertTrue(false);
        } catch (IllegalArgumentException e) {
        }
        try {
            new SOMDataset(-1, -1);
            assertTrue(false);
        } catch (IllegalArgumentException e) {
        }
        try {
            new SOMDataset(-2, 3);
            assertTrue(false);
        } catch (IllegalArgumentException e) {
        }
        try {
            new SOMDataset(1, 1);
        } catch (IllegalArgumentException e) {
            assertTrue(false);
        }
        try {
            new SOMDataset(2000, 3000);
        } catch (IllegalArgumentException e) {
            assertTrue(false);
        }
    }
    
    public void testAddGetValue() {
        // negative matrix x-coordinate
        try {
            new SOMDataset(1,1).addValue(-1,0,Color.BLACK,new String[] {"A"}, new double[] {1.0});
            assertTrue(false);
        } catch (IndexOutOfBoundsException e) {
        }
        // negative matrix y-coordinate
        try {
            new SOMDataset(1,1).addValue(0,-1,Color.BLACK,new String[] {"A"}, new double[] {1.0});
            assertTrue(false);
        } catch (IndexOutOfBoundsException e) {
        }
        // too high matrix x-coordinate
        try {
            new SOMDataset(1,1).addValue(2,0,Color.BLACK,new String[] {"A"}, new double[] {1.0});
            assertTrue(false);
        } catch (IndexOutOfBoundsException e) {
        }
        // too high matrix y-coordinate
        try {
            new SOMDataset(1,1).addValue(0,2,Color.BLACK,new String[] {"A"}, new double[] {1.0});
            assertTrue(false);
        } catch (IndexOutOfBoundsException e) {
        }
        // color == null
        try {
            new SOMDataset(1,1).addValue(0,0,null,new String[] {"A"}, new double[] {1.0});
            assertTrue(false);
        } catch (NullPointerException e) {
        }
        // descriptions == null
        try {
            new SOMDataset(1,1).addValue(0,0,Color.BLACK,null, new double[] {1.0});
            assertTrue(false);
        } catch (NullPointerException e) {
        }
        // descriptions.length == 0
        try {
            new SOMDataset(1,1).addValue(0,0,Color.BLACK,new String[] {}, new double[] {1.0});
            assertTrue(false);
        } catch (IllegalArgumentException e) {
        }
        // neuronWeights == null
        try {
            new SOMDataset(1,1).addValue(0,0,Color.BLACK,new String[] {"A"}, null);
            assertTrue(false);
        } catch (NullPointerException e) {
        }
        // neuronWeights.length == 0
        try {
            new SOMDataset(1,1).addValue(0,0,Color.BLACK,new String[] {"A"}, new double[] {});
            assertTrue(false);
        } catch (IllegalArgumentException e) {
        }
        
        // item == null
        try {
            new SOMDataset(1,1).addValue(0,0,null);
            assertTrue(false);
        } catch (NullPointerException e) {
        }
        
        SOMDataset set = new SOMDataset(1,1);
        SOMDataItem item = new SOMDataItem(Color.BLACK,new String[] {"A"}, new double[] {1.0});
        set.addValue(0,0,Color.BLACK,new String[] {"A"}, new double[] {1.0});
        assertEquals(set.getValue(0,0), item);
        
        try {
            set.addValue(0,0,item);
        } catch (Exception e) {
            assertTrue(false);
        }
    }
    
    public void testRowCount() {
        assertEquals(new SOMDataset(1,1).getRowCount(), 1);
        assertEquals(new SOMDataset(2,1).getRowCount(), 1);
        assertEquals(new SOMDataset(1,10).getRowCount(), 10);
        assertEquals(new SOMDataset(333,19999).getRowCount(), 19999);
    }
    
    public void testColumnCount() {
        assertEquals(new SOMDataset(1,1).getColumnCount(), 1);
        assertEquals(new SOMDataset(1,2).getColumnCount(), 1);
        assertEquals(new SOMDataset(10,1).getColumnCount(), 10);
        assertEquals(new SOMDataset(3233,2344).getColumnCount(), 3233);
    }
    
    public void testGetNeighbors() {
        try {
            this.dataset.getNeighbors(null, 0, false);
            assertTrue(false);
        } catch (NullPointerException e) {
        }
        
        try {
            this.dataset.getNeighbors(0,0, -1, false);
            assertTrue(false);
        } catch (IllegalArgumentException e) {
        }
        try {
            this.dataset.getNeighbors(0,0, -1, true);
            assertTrue(false);
        } catch (IllegalArgumentException e) {
        }
        try {
            this.dataset.getNeighbors(0,0, 0, false);
            this.dataset.getNeighbors(0,1, 1, false);
            this.dataset.getNeighbors(1,0, 2, false);
            this.dataset.getNeighbors(1,1, 3, false);
            this.dataset.getNeighbors(0,0, 100, false);
            this.dataset.getNeighbors(0,1, 400, false);
            this.dataset.getNeighbors(1,0, 500, false);
            this.dataset.getNeighbors(1,1, 0, true);
            this.dataset.getNeighbors(0,0, 1, true);
            this.dataset.getNeighbors(0,1, 2, true);
            this.dataset.getNeighbors(1,0, 3, true);
            this.dataset.getNeighbors(1,1, 100, true);
            this.dataset.getNeighbors(0,0, 400, true);
            this.dataset.getNeighbors(0,1, 500, true);
            assertTrue(true);
        } catch (Exception e) {
        }
        
        // zero distance. Only the center and the cell that is
        // exactly the same color as the center are selected.
        List list = this.dataset.getNeighbors(0,0, 0, true);
        assertTrue(list.contains(this.dataset.getValue(0,0)));
        assertTrue(list.contains(this.dataset.getValue(1,1)));
        
        // zero distance. Only the cell that is
        // exactly the same color as the center is selected.
        list = this.dataset.getNeighbors(0,0, 0, false);
        assertFalse(list.contains(this.dataset.getValue(0,0)));
        assertTrue(list.contains(this.dataset.getValue(1,1)));
        
        // distance greater that maximum possible distance,
        // all cells will be selected.
        list = this.dataset.getNeighbors(0,0, 500, true);
        assertTrue(list.contains(this.dataset.getValue(0,0)));
        assertTrue(list.contains(this.dataset.getValue(0,1)));
        assertTrue(list.contains(this.dataset.getValue(1,0)));
        assertTrue(list.contains(this.dataset.getValue(1,1)));
        
        // distance greater that maximum possible distance,
        // all cells will be selected with the exception of the
        // center cell.
        list = this.dataset.getNeighbors(0,0, 500, false);
        assertFalse(list.contains(this.dataset.getValue(0,0)));
        assertTrue(list.contains(this.dataset.getValue(0,1)));
        assertTrue(list.contains(this.dataset.getValue(1,0)));
        assertTrue(list.contains(this.dataset.getValue(1,1)));
        
        // distance is not large enought to select other cells
        // than the cell that is of the same color as the center.
        list = this.dataset.getNeighbors(0,0, 141, false);
        assertTrue(list.contains(this.dataset.getValue(1,1)));
        
        // distance is large enought to select cells of different color.
        list = this.dataset.getNeighbors(0,0, 142, false);
        assertTrue(list.contains(this.dataset.getValue(1,1)));
        assertTrue(list.contains(this.dataset.getValue(1,0)));
    }
    
    public void testGetArea() {
        // null arguments
        try {
            this.dataset.getArea(null, this.dataset.getValue(0,0));
            assertTrue(false);
        } catch (NullPointerException e) {
        }
        try {
            this.dataset.getArea(this.dataset.getValue(0,0), null);
            assertTrue(false);
        } catch (NullPointerException e) {
        }
        
        // dataset does not contain first argument
        try {
            this.dataset.getArea(
                    new SOMDataItem(Color.BLACK,new String[] {"A"}, new double[] {1.0}), 
                            this.dataset.getValue(0,0));
            assertTrue(false);
        } catch (IllegalArgumentException e) {
        }
        // dataset does not contain second argument
        try {
            this.dataset.getArea(
                    this.dataset.getValue(0,0), 
                    new SOMDataItem(Color.BLACK,new String[] {"A"}, new double[] {1.0}));
            assertTrue(false);
        } catch (IllegalArgumentException e) {
        }
        
        List list = this.dataset.getArea(
                this.dataset.getValue(0,0), 
                this.dataset.getValue(0,0));
        assertTrue(list.size() == 1);
        list = this.dataset.getArea(
                this.dataset.getValue(1,0), 
                this.dataset.getValue(1,0));
        assertTrue(list.size() == 1);
        list = this.dataset.getArea(
                this.dataset.getValue(1,0), 
                this.dataset.getValue(1,1));
        assertTrue(list.size() == 2);
        list = this.dataset.getArea(
                this.dataset.getValue(0,1), 
                this.dataset.getValue(1,1));
        assertTrue(list.size() == 2);
        list = this.dataset.getArea(
                this.dataset.getValue(0,0), 
                this.dataset.getValue(1,1));
        assertTrue(list.size() == 4);
        list = this.dataset.getArea(
                this.dataset.getValue(1,1), 
                this.dataset.getValue(0,0));
        assertTrue(list.size() == 4);
    }
    
    public void testChangeHueValues() {
        try {
            this.dataset.changeHueValues(-361);
            assertTrue(false);
        } catch (IllegalArgumentException e) {
        }
        try {
            this.dataset.changeHueValues(361);
            assertTrue(false);
        } catch (IllegalArgumentException e) {
        }
        
        try {
            this.dataset.changeHueValues(360);
            this.dataset.changeHueValues(-360);
            this.dataset.changeHueValues(0);
            this.dataset.changeHueValues(1);
            this.dataset.changeHueValues(10);
            this.dataset.changeHueValues(101);
            this.dataset.changeHueValues(-1);
            this.dataset.changeHueValues(-12);
        } catch (Exception e) {
            assertTrue(false);
        }
    }
    
    public void testDeselectAll() {
        Iterator i = this.dataset.iterator();
        while (i.hasNext()) {
            ((SOMDataItem) i.next()).setSelected(true);
        }
        this.dataset.deselectAll();
        i = this.dataset.iterator();
        while (i.hasNext()) {
            assertFalse(((SOMDataItem) i.next()).isSelected());
        }
        
        try {
            new SOMDataset(1, 1).deselectAll();
            assertTrue(false);
        } catch (NullPointerException e) {
        }
    }
    
    public void testIterator() {
        List aList = new LinkedList();
        List bList = new LinkedList();
        
        aList.add(this.dataset.getValue(0, 0));
        aList.add(this.dataset.getValue(1, 0));
        aList.add(this.dataset.getValue(0, 1));
        aList.add(this.dataset.getValue(1, 1));
        
        Iterator i = this.dataset.iterator();
        while (i.hasNext()) {
            bList.add(i.next());
        }
        
        assertEquals(aList, bList);
        
        // call to remove should throw an UnsupportedOperationException
        i = this.dataset.iterator();
        try {
            i.next();
            i.remove();
            assertTrue(false);
        } catch (UnsupportedOperationException e) {
        } catch (Exception e) {
            assertTrue(false);
        }
    }
    
    public void testEquals() {
        assertEquals(this.dataset, this.dataset);
        assertFalse(this.dataset.equals(new LinkedList()));
        assertFalse(this.dataset.equals(new SOMDataset(1, 23)));
        assertEquals(new SOMDataset(1, 1), new SOMDataset(1, 1));
        SOMDataset set = new SOMDataset(2, 1);
        set.addValue(1,0, new SOMDataItem(Color.BLACK, new String[] {"A"}, new double[] {0.0}));
        assertFalse(set.equals(new SOMDataset(2, 1)));
        
    }
    
    /**
     * Confirm that cloning works.
     */
    public void testCloning() {
        
        SOMDataset d2 = null;
        try {
            d2 = (SOMDataset) this.dataset.clone();
        }
        catch (CloneNotSupportedException e) {
            System.err.println("Failed to clone.");
        }
        assertTrue(this.dataset != d2);
        assertTrue(this.dataset.getClass() == d2.getClass());
        assertTrue(this.dataset.equals(d2));
    }

    /**
     * Serialize an instance, restore it, and check for equality.
     */
    public void testSerialization() {
        SOMDataset d2 = null;

        try {
            ByteArrayOutputStream buffer = new ByteArrayOutputStream();
            ObjectOutput out = new ObjectOutputStream(buffer);
            out.writeObject(this.dataset);
            out.close();

            ObjectInput in = new ObjectInputStream(
                new ByteArrayInputStream(buffer.toByteArray())
            );
            d2 = (SOMDataset) in.readObject();
            in.close();
        }
        catch (Exception e) {
            System.out.println(e.toString());
        }
        assertEquals(this.dataset, d2);

    }

}
