
package threed;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.event.*;
import java.util.*;

public class ScatterPlot {
    private JLabel label;
    private Point clickPoint, cursorPoint;

    private void buildUI(Container container) {
        container.setLayout(new BoxLayout(container,
                                          BoxLayout.PAGE_AXIS));

        CoordinateArea coordinateArea = new CoordinateArea(this);
        container.add(coordinateArea);

        label = new JLabel();
        container.add(label);

        //Align the left edges of the components.
        coordinateArea.setAlignmentX(Component.LEFT_ALIGNMENT);
        label.setAlignmentX(Component.LEFT_ALIGNMENT); //redundant
    }
    
    /**
     * Create the GUI and show it.  For thread safety, 
     * this method should be invoked from the 
     * event-dispatching thread.
     */
    private static void createAndShowGUI() {
        //Make sure we have nice window decorations.
        JFrame.setDefaultLookAndFeelDecorated(true);

        //Create and set up the window.
        JFrame frame = new JFrame("3DPlot");
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

        //Set up the content pane.
        ScatterPlot controller = new ScatterPlot();
        controller.buildUI(frame.getContentPane());

        //Display the window.
        frame.pack();
        frame.setVisible(true);
    }

    /**
     * 
     * @param args 
     */
    public static void main(String[] args) {
        //Schedule a job for the event-dispatching thread:
        //creating and showing this application's GUI.
        javax.swing.SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                createAndShowGUI(); 
            }
        });
    }

    public static class CoordinateArea extends JComponent 
            implements MouseInputListener, MouseWheelListener, ActionListener  {
        
        LinkedList<DataPoint> selectedPoints;
        Projection projection;
        Worker worker;
        AutomatedMovement movement;
        JMenuItem setOrigin;
        JMenuItem resetOrigin;
        JMenuItem rotateX180;
        JMenuItem rotateY180;
        JMenuItem rotateZ180;
        JMenuItem hideSelected;
        JMenuItem showAll;
        JMenuItem invertSelection;
        
        ScatterPlot controller;
        Dimension preferredSize = new Dimension(600,600);
        
        final double ANGLE_INCREMENT = Math.PI/180;
    
        /**
         * 
         * @param controller 
         */
        public CoordinateArea(ScatterPlot controller) {
            this.controller = controller;
           
            createPopupMenu();
                                                      
            addMouseListener(this);
            addMouseMotionListener(this);
            addMouseWheelListener(this);
            setBackground(Color.WHITE);
            setOpaque(true);
            
            setToolTipText("");
            ToolTipManager.sharedInstance().setDismissDelay(60*1000);
            
            setFocusable(true);
            
            selectedPoints = new LinkedList<DataPoint>();
            projection = new Projection();
            worker = new Worker(this, projection);
            worker.start();
            worker.workRequest();
            movement = new AutomatedMovement(projection, worker);
            movement.start();
            movement.addRotationTask(Math.PI, 0, 0, 2000, 24.0);
            movement.addRotationTask(0, Math.PI*0.25, 0, 1000, 24.0);
            movement.addRotationTask(Math.PI*0.1, 0, 0, 500, 24.0);
        }
    
        /**
         * 
         * @return 
         */
        public Dimension getPreferredSize() {
            return preferredSize;
        }
    
        /**
         * 
         * @param g 
         */
        protected void paintComponent(Graphics g) {
            //Paint background if we're opaque.
            if (isOpaque()) {
                g.setColor(Color.BLACK);//getBackground());
                g.fillRect(0, 0, getWidth(), getHeight());
            }
            
            java.util.PriorityQueue<Drawable> points = projection.getResultPoints();
            if (points != null && points.isEmpty() == false) {
                Drawable p = null;
                while ((p = points.poll()) != null) {
                    p.draw(g, getWidth(), getHeight());
                }
            }
            else {
                worker.workRequest();
            }
        }
        
        //Methods required by the MouseInputListener interface.
        /**
         * 
         * @param e 
         */
        public void mouseClicked(MouseEvent e) {
            double deg = 0;
            switch (e.getButton()) {
                case MouseEvent.BUTTON1:
                    if (e.isControlDown()) {
                        addToSelections(e);
                    }
                    else {
                        selectOne(e);
                    }
                    break;
                case MouseEvent.BUTTON2:
                    deg = projection.getZAxisRotation();
                    projection.setZAxisRotation(/*deg - */-ANGLE_INCREMENT);
                    break;
                case MouseEvent.BUTTON3:
                    break;
                default:
            }
            
            //repaint();
            worker.workRequest();
        }

        /**
         * 
         * @param e 
         */
        public void mouseMoved(MouseEvent e) {
            
            if (e.isShiftDown()) {
                double deg = projection.getYAxisRotation();
                if (mousePressX - e.getX() < 0) {
                    projection.setYAxisRotation(/*deg - */-ANGLE_INCREMENT);
                } else if (mousePressX - e.getX() > 0) {
                    projection.setYAxisRotation(/*deg + */ANGLE_INCREMENT);
                }
                
                deg = projection.getXAxisRotation();
                if (mousePressY - e.getY() < 0) {
                    projection.setXAxisRotation(/*deg + */ANGLE_INCREMENT);
                } else if (mousePressY - e.getY() > 0) {
                    projection.setXAxisRotation(/*deg - */-ANGLE_INCREMENT);
                }
                
                mousePressX = e.getX();
                mousePressY = e.getY();
                
                //repaint();
                worker.workRequest();
            }
        }

        /**
         * 
         * @param e 
         */
        public void mouseExited(MouseEvent e) { }
        
        /**
         * 
         * @param e 
         */
        public void mouseEntered(MouseEvent e) { }
        
        private int mousePressX = 0;
        private int mousePressY = 0;
        
        /**
         * 
         * @param e 
         */
        public void mousePressed(MouseEvent e) {
            mousePressX = e.getX();
            mousePressY = e.getY();
        }
        
        boolean mouseDragged = false;
        /**
         * 
         * @param e 
         */
        public void mouseDragged(MouseEvent e) {
            mouseDragged = true;
        }
        
        /**
         * 
         * @param e 
         */
        public void mouseReleased(MouseEvent e) {
            if (e.getButton() == MouseEvent.BUTTON1 && mouseDragged == true) {
                selectGroup(e, mousePressX, mousePressY);
                mouseDragged = false;
                //repaint();
                worker.workRequest();
            }
        }
        
        /**
         * 
         * @param e 
         */
        public void mouseWheelMoved(MouseWheelEvent e) {
            double[] pov = projection.getPointOfView();
            if (e.getWheelRotation() < 0) {
                pov[2] += 0.1;
                projection.setPointOfView(pov);
            } else {
                pov[2] -= 0.1;
                projection.setPointOfView(pov);
            }
            
            //repaint();
            worker.workRequest();
        }
        
        /**
         * 
         * @param event 
         * @return 
         */
        public String getToolTipText(MouseEvent event) {
            LinkedList<DataPoint> l = DataPoint.getNearest(
                    event.getX(), event.getY(), projection.points, 4);
            
            if (l.isEmpty() == false) {
                DataPoint dp = l.getFirst();
                if (dp != null)
                    return "<html>" + dp.dataCoords[0][0] + " " + dp.dataCoords[0][1] + " " + dp.dataCoords[0][2] + "<br>" + dp.distanceFromCamera[0] + "<br>" + dp.text + "</html>";
            }
            
            return null;
        }
        
        private void clearSelections() {
            if (selectedPoints == null)
                return;
            
            for (DataPoint dp : selectedPoints) {
                    if (dp != null)
                        dp.selected = false;
            }
        }
        
        private void selectOne(MouseEvent e) {
            clearSelections();
            selectedPoints =
                    DataPoint.getNearest(e.getX(), e.getY(), projection.points, 4);
            for (DataPoint dp : selectedPoints) {
                if (dp != null)
                    dp.selected = true;
            }
        }
        
        private void addToSelections(MouseEvent e) {
            LinkedList<DataPoint> newSelection =
                    DataPoint.getNearest(e.getX(), e.getY(), projection.points, 4);
            for (DataPoint dp : newSelection) {
                if (dp != null) {
                    if (dp.selected == true) {
                        dp.selected = false;
                        selectedPoints.remove(dp);
                    }
                    else {
                        dp.selected = true;
                        selectedPoints.add(dp);
                    }
                }
            }
        }
        
        private void selectGroup(MouseEvent e, int x1, int y1) {
            clearSelections();
            selectedPoints =
                    DataPoint.getGroup(x1, y1, 
                    e.getX(), e.getY(), projection.points);
            for (DataPoint dp : selectedPoints) {
                if (dp != null)
                    dp.selected = true;
            }
            
        }
        
        private DataPoint popupSelection = null;
        
        class PopupListener extends MouseAdapter {
            JPopupMenu popup;
            
            PopupListener(JPopupMenu popupMenu) {
                popup = popupMenu;
            }
            
            public void mousePressed(MouseEvent e) {
                maybeShowPopup(e);
            }
            
            public void mouseReleased(MouseEvent e) {
                maybeShowPopup(e);
            }
            
            private void maybeShowPopup(MouseEvent e) {
                if (e.isPopupTrigger()) {
                    int x = e.getX();
                    int y = e.getY();
                    LinkedList<DataPoint> l;
                    l = DataPoint.getNearest(x, y, projection.points);
                    if (l.isEmpty()) {
                        setOrigin.setEnabled(false);
                    }
                    else {
                        setOrigin.setEnabled(true);
                        popupSelection = l.getFirst();
                    }
                    
                    if (selectedPoints.isEmpty()) {
                        hideSelected.setEnabled(false);
                        invertSelection.setEnabled(false);
                    }
                    else {
                        hideSelected.setEnabled(true);
                        invertSelection.setEnabled(true);
                    }
                    popup.show(e.getComponent(), x, y);
                }
            }
        }
        
        /**
         * 
         * @param e 
         */
        public void actionPerformed(ActionEvent e) {
            if (e.getSource() == setOrigin) {
                //projection.setNewOrigin(popupSelection.visualisationCoords[0]);
                movement.addTranslationTask(popupSelection.visualisationCoords[0], 1000,  24);
                popupSelection = null;
            }
            else if (e.getSource() == resetOrigin) {
                movement.addTranslationTask(projection.getDataOrigin().visualisationCoords[0], 1000,  24);
            }
            else if (e.getSource() == rotateX180) {
                movement.addRotationTask(Math.PI, 0, 0, 1800,  24);
            }
            else if (e.getSource() == rotateY180) {
                movement.addRotationTask(0, Math.PI, 0, 1800,  24);
            }
            else if (e.getSource() == rotateZ180) {
                movement.addRotationTask(0, 0, Math.PI, 1800,  24);
            }
            else if (e.getSource() == hideSelected) {
                for (Drawable d : selectedPoints) {
                    d.hidden = true;
                }
            }
            else if (e.getSource() == showAll) {
                for (Drawable d : projection.points) {
                    d.hidden = false;
                }
            }
            else if (e.getSource() == invertSelection) {
                selectedPoints = new LinkedList<DataPoint>();
                for (Drawable d : projection.points) {
                    d.selected = !d.selected;
                    if (d instanceof DataPoint && d.selected)
                        selectedPoints.add((DataPoint)d);
                }
            }
            //repaint();
            worker.workRequest();
        }
        
        public void createPopupMenu() {
            
            //Create the popup menu.
            JPopupMenu popup = new JPopupMenu();
            setOrigin = new JMenuItem("Rotate around this point");
            setOrigin.addActionListener(this);
            popup.add(setOrigin);
            resetOrigin = new JMenuItem("Reset origin");
            resetOrigin.addActionListener(this);
            popup.add(resetOrigin);
            rotateX180 = new JMenuItem("Rotate around camera x-axis by 180 degrees");
            rotateX180.addActionListener(this);
            popup.add(rotateX180);
            rotateY180 = new JMenuItem("Rotate around camera y-axis by 180 degrees");
            rotateY180.addActionListener(this);
            popup.add(rotateY180);
            rotateZ180 = new JMenuItem("Rotate around camera z-axis by 180 degrees");
            rotateZ180.addActionListener(this);
            popup.add(rotateZ180);
            hideSelected = new JMenuItem("Hide selected");
            hideSelected.addActionListener(this);
            popup.add(hideSelected);
            showAll = new JMenuItem("Show all points");
            showAll.addActionListener(this);
            popup.add(showAll);
            invertSelection = new JMenuItem("InvertSelection");
            invertSelection.addActionListener(this);
            popup.add(invertSelection);
            
            
            //Add listener to the text area so the popup menu can come up.
            MouseListener popupListener = new PopupListener(popup);
            this.addMouseListener(popupListener);
        }
        
        
    }

}

