/* ======================================================================= 
 * A visualisation library extension for JFreeChart. Please see JFreeChart
 * for further information.
 * =======================================================================
 * Copyright (C) 2006  University of Helsinki, Department of Computer Science
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * -----------------------------
 * Contact:  ohtu@cs.helsinki.fi
 * -----------------------------
 *
 */


package org.jfree.data.som;

import java.awt.Color;
import java.io.Serializable;

/**
 * A simple data-storage object to be used in a
 * {@link org.jfree.data.som.SOMDataset}. This class collects the 
 * data
 * from a single SOM-map cell.
 */
public class SOMDataItem implements Cloneable, Serializable {

    /** The color associated with a dataitem. */
    private Color color;
    
    /** Textual descriptions and names of a dataitem. */
    private String[] descriptions;

    /** Numerical data contained in a dataitem. */
    private double[] neuronWeights;

    /** Identifier for on-screen cell-selection. */
    private boolean selected;


    /**
     * Creates a new SOMDataItem.
     *
     * @param color  the color of the cell.
     * @param descriptions  the textual descriptions and/or names of this dataitem.
     * @param neuronWeights  the numerical data contained in a cell.
     * @throws NullPointerException If any parameter is null.
     * @throws IllegalArgumentException If descriptions or neuronWeights do not contain any items.
     */
    public SOMDataItem(Color color, String descriptions[], double[] neuronWeights) 
    throws NullPointerException {
        if (color == null || descriptions == null || neuronWeights == null)
            throw new NullPointerException("color/descriptions/neuronWeights given to SOMDataItem was null.");
        if (descriptions.length == 0)
            throw new IllegalArgumentException("descriptions given to SOMDataItem did not contain any strings.");
        if (neuronWeights.length == 0)
            throw new IllegalArgumentException("neuronWeights given to SOMDataItem did not contain any values.");
        this.color = color;
        this.descriptions = descriptions;
        this.neuronWeights = neuronWeights;
        this.selected = false;
    }

    /**
     * Returns the color of a dataitem.
     *
     * @return  The dataitem color.
     */
    public Color getColor() {
        return color;
    }

    /**
     * Returns the textual descriptions of a dataitem.
     *
     * @return  The textual descriptions.
     */
    public String[] getDescriptions() {
        return descriptions;
    }

    /**
     * Returns the numerical data of a dataitem.
     *
     * @return  The numerical data
     */ 
    public double[] getNeuronWeights() {
        return neuronWeights;
    }

    /**
     * Sets the color of a dataitem.
     *
     * @param color  the color.
     */
    public void setColor(Color color) {
        if (color == null)
            throw new NullPointerException("color given to setColor() was null.");
        this.color = color;
    }

    /**
     * Sets the textual descriptions of a dataitem.
     *
     * @param descriptions  the descriptions or names.
     */
    public void setDescriptions(String[] descriptions) {
        if (descriptions == null)
            throw new NullPointerException("descriptions given to setDescriptions() was null.");
        this.descriptions = descriptions;
    }

    /**
     * Sets the numerical datavalues of a dataitem.
     *
     * @param neuronWeights  the numerical data.
     */
    public void setNeuronWeights(double[] neuronWeights) {
        if (neuronWeights == null)
            throw new NullPointerException("neuronWeights given to setNeuronWeights() was null.");
        this.neuronWeights = neuronWeights;
    }

    /** 
     * Returns the boolean value indicating whether a dataitem is
     * selected on-screen or not.
     *
     * @return A boolean.
     */
    public boolean isSelected() {
        return selected;
    }

    /**
     * Sets the boolean value indicating whether a dataitem is
     * selected on-screen or not.
     *
     * @param selected  A boolean.
     */
    public void setSelected(boolean selected) {
        this.selected = selected;
    }

    /**
     * Tests if this object is equal to another.
     *
     * @param obj  the other object.
     *
     * @return A boolean.
     */
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }

        if (!(obj instanceof SOMDataItem)) {
            return false;
        }
        
        SOMDataItem that = (SOMDataItem) obj;
        
        if (this.color.equals(that.color) == false)
            return false;
        
        if (this.selected != that.selected)
            return false;
        
        if (this.descriptions.length != that.descriptions.length)
            return false;
        
        for (int i=0; i < this.descriptions.length; ++i) {
            if (this.descriptions[i] == null && that.descriptions[i] == null) {
                ;
            }
            else if (this.descriptions[i] == null || 
                     that.descriptions[i] == null || 
                     this.descriptions[i].equals(that.descriptions[i]) == false)
                return false;
        }
        
        if (this.neuronWeights.length != that.neuronWeights.length)
            return false;
        
        for (int i=0; i < this.neuronWeights.length; ++i) {
            if (this.neuronWeights[i] != that.neuronWeights[i])
                return false;
        }
        
        return true;
    }
    
    /**
     * Returns a clone of the dataset.
     * 
     * @return A clone.
     * 
     * @throws CloneNotSupportedException This class will not throw this 
     *         exception, but subclasses (if any) might.
     */
    public Object clone() throws CloneNotSupportedException {
        SOMDataItem clone = (SOMDataItem) super.clone();
        clone.descriptions = (String[]) this.descriptions.clone();
        clone.neuronWeights = (double[]) this.neuronWeights.clone();
        return clone;    
    }
    
}

