/*
 * Decompiled with CFR 0.152.
 */
package net.sf.picard.analysis.directed;

import java.io.File;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import net.sf.picard.analysis.directed.HsMetrics;
import net.sf.picard.sam.DuplicationMetrics;
import net.sf.picard.util.Interval;
import net.sf.picard.util.IntervalList;
import net.sf.picard.util.Log;
import net.sf.picard.util.OverlapDetector;
import net.sf.samtools.AlignmentBlock;
import net.sf.samtools.SAMFileReader;
import net.sf.samtools.SAMRecord;
import net.sf.samtools.SAMSequenceRecord;
import net.sf.samtools.util.CoordMath;

public class HsMetricsCalculator {
    private static final int NEAR_BAIT_DISTANCE = 250;
    private static final Log log = Log.getInstance(HsMetricsCalculator.class);
    private SAMFileReader sam;
    private final File baitFile;
    private final File targetFile;
    private final IntervalList baits;
    private final IntervalList targets;
    private final OverlapDetector<Interval> targetDetector = new OverlapDetector(0, 0);
    private final OverlapDetector<Interval> baitDetector = new OverlapDetector(-250, 0);
    private final Map<Interval, Coverage> coverageByTarget;
    private final HsMetrics metrics = new HsMetrics();
    private double PF_BASES = 0.0;
    private long PF_SELECTED_PAIRS;
    private long PF_SELECTED_UNIQUE_PAIRS;
    private long ON_TARGET_FROM_PAIR_BASES = 0L;

    public HsMetricsCalculator(File baits, File targets) {
        this.baitFile = baits;
        this.targetFile = targets;
        this.baits = IntervalList.fromFile(baits);
        this.targets = IntervalList.fromFile(targets);
        this.metrics.BAIT_SET = baits.getName();
        int tmp = this.metrics.BAIT_SET.indexOf(".");
        if (tmp > 0) {
            this.metrics.BAIT_SET = this.metrics.BAIT_SET.substring(0, tmp);
        }
        List<Interval> uniqueBaits = this.baits.getUniqueIntervals();
        this.baitDetector.addAll(uniqueBaits, uniqueBaits);
        this.metrics.BAIT_TERRITORY = Interval.countBases(uniqueBaits);
        List<Interval> uniqueTargets = this.targets.getUniqueIntervals();
        this.targetDetector.addAll(uniqueTargets, uniqueTargets);
        this.metrics.TARGET_TERRITORY = Interval.countBases(uniqueTargets);
        for (SAMSequenceRecord seq : this.baits.getHeader().getSequenceDictionary().getSequences()) {
            this.metrics.GENOME_SIZE += (long)seq.getSequenceLength();
        }
        this.coverageByTarget = new HashMap<Interval, Coverage>(uniqueTargets.size() * 2, 0.5f);
        for (Interval target : uniqueTargets) {
            this.coverageByTarget.put(target, new Coverage(target, 0));
        }
    }

    public void analyze(Iterator<SAMRecord> records) {
        int i = 0;
        while (records.hasNext()) {
            this.analyze(records.next());
            if (++i % 1000000 != 0) continue;
            log.info("Processed " + i + " records so far.");
        }
    }

    public void analyze(SAMRecord rec) {
        boolean mappedInPair;
        Collection<Interval> baits;
        Collection<Interval> targets;
        if (rec.getNotPrimaryAlignmentFlag()) {
            return;
        }
        ++this.metrics.TOTAL_READS;
        if (!rec.getReadUnmappedFlag()) {
            Interval read = new Interval(rec.getReferenceName(), rec.getAlignmentStart(), rec.getAlignmentEnd());
            targets = this.targetDetector.getOverlaps(read);
            baits = this.baitDetector.getOverlaps(read);
        } else {
            targets = null;
            baits = null;
        }
        if (rec.getReadFailsVendorQualityCheckFlag()) {
            return;
        }
        ++this.metrics.PF_READS;
        this.PF_BASES += (double)rec.getReadLength();
        if (rec.getReadPairedFlag() && rec.getFirstOfPairFlag() && !rec.getReadUnmappedFlag() && !rec.getMateUnmappedFlag() && baits != null && !baits.isEmpty()) {
            ++this.PF_SELECTED_PAIRS;
            if (!rec.getDuplicateReadFlag()) {
                ++this.PF_SELECTED_UNIQUE_PAIRS;
            }
        }
        if (rec.getDuplicateReadFlag()) {
            return;
        }
        ++this.metrics.PF_UNIQUE_READS;
        if (rec.getReadUnmappedFlag() || rec.getMappingQuality() == 0) {
            return;
        }
        ++this.metrics.PF_UQ_READS_ALIGNED;
        for (AlignmentBlock block : rec.getAlignmentBlocks()) {
            this.metrics.PF_UQ_BASES_ALIGNED += (long)block.getLength();
        }
        boolean bl = mappedInPair = rec.getReadPairedFlag() && !rec.getMateUnmappedFlag();
        if (targets != null && !targets.isEmpty()) {
            for (Interval target : targets) {
                Coverage coverage = this.coverageByTarget.get(target);
                for (AlignmentBlock block : rec.getAlignmentBlocks()) {
                    int end = CoordMath.getEnd((int)block.getReferenceStart(), (int)block.getLength());
                    for (int pos = block.getReferenceStart(); pos <= end; ++pos) {
                        if (pos < target.getStart() || pos > target.getEnd()) continue;
                        ++this.metrics.ON_TARGET_BASES;
                        if (mappedInPair) {
                            ++this.ON_TARGET_FROM_PAIR_BASES;
                        }
                        coverage.addBase(pos - target.getStart());
                    }
                }
            }
        }
        int mappedBases = 0;
        for (AlignmentBlock block : rec.getAlignmentBlocks()) {
            mappedBases += block.getLength();
        }
        int onBaitBases = 0;
        if (baits != null && !baits.isEmpty()) {
            for (Interval bait : baits) {
                for (AlignmentBlock block : rec.getAlignmentBlocks()) {
                    int end = CoordMath.getEnd((int)block.getReferenceStart(), (int)block.getLength());
                    for (int pos = block.getReferenceStart(); pos <= end; ++pos) {
                        if (pos < bait.getStart() || pos > bait.getEnd()) continue;
                        ++onBaitBases;
                    }
                }
            }
            this.metrics.ON_BAIT_BASES += (long)onBaitBases;
            this.metrics.NEAR_BAIT_BASES += (long)(mappedBases - onBaitBases);
        } else {
            this.metrics.OFF_BAIT_BASES += (long)mappedBases;
        }
    }

    public HsMetrics getMetrics() {
        this.metrics.PCT_USABLE_BASES_ON_BAIT = (double)this.metrics.ON_BAIT_BASES / this.PF_BASES;
        this.metrics.PCT_USABLE_BASES_ON_TARGET = (double)this.metrics.ON_TARGET_BASES / this.PF_BASES;
        this.metrics.HS_LIBRARY_SIZE = DuplicationMetrics.estimateLibrarySize(this.PF_SELECTED_PAIRS, this.PF_SELECTED_UNIQUE_PAIRS);
        this.metrics.calculateDerivedMetrics();
        this.calculateTargetCoverageMetrics();
        this.metrics.HS_PENALTY_10X = this.calculateHsPenalty(10);
        this.metrics.HS_PENALTY_20X = this.calculateHsPenalty(20);
        this.metrics.HS_PENALTY_30X = this.calculateHsPenalty(30);
        return this.metrics;
    }

    private void calculateTargetCoverageMetrics() {
        short[] depths = new short[(int)this.metrics.TARGET_TERRITORY];
        int zeroCoverageTargets = 0;
        int depthIndex = 0;
        double totalCoverage = 0.0;
        int basesConsidered = 0;
        for (Coverage c : this.coverageByTarget.values()) {
            if (!c.hasCoverage()) {
                ++zeroCoverageTargets;
                continue;
            }
            short[] targetDepths = c.getDepths();
            basesConsidered += targetDepths.length;
            for (short depth : targetDepths) {
                depths[depthIndex++] = depth;
                totalCoverage += (double)depth;
            }
        }
        this.metrics.MEAN_TARGET_COVERAGE = totalCoverage / (double)basesConsidered;
        Arrays.sort(depths);
        int indexOf80thPercentile = depths.length - 1 - basesConsidered + (int)((double)basesConsidered * 0.2);
        short coverageAt80thPercentile = depths[indexOf80thPercentile];
        this.metrics.FOLD_80_BASE_PENALTY = this.metrics.MEAN_TARGET_COVERAGE / (double)coverageAt80thPercentile;
        this.metrics.ZERO_CVG_TARGETS_PCT = (double)zeroCoverageTargets / (double)this.targets.getIntervals().size();
        int totalTargetBases = 0;
        int targetBases2x = 0;
        int targetBases10x = 0;
        int targetBases20x = 0;
        int targetBases30x = 0;
        for (Coverage c : this.coverageByTarget.values()) {
            for (short depth : c.getDepths()) {
                ++totalTargetBases;
                if (depth < 2) continue;
                ++targetBases2x;
                if (depth < 10) continue;
                ++targetBases10x;
                if (depth < 20) continue;
                ++targetBases20x;
                if (depth < 30) continue;
                ++targetBases30x;
            }
        }
        this.metrics.PCT_TARGET_BASES_2X = (double)targetBases2x / (double)totalTargetBases;
        this.metrics.PCT_TARGET_BASES_10X = (double)targetBases10x / (double)totalTargetBases;
        this.metrics.PCT_TARGET_BASES_20X = (double)targetBases20x / (double)totalTargetBases;
        this.metrics.PCT_TARGET_BASES_30X = (double)targetBases30x / (double)totalTargetBases;
    }

    private double calculateHsPenalty(int coverageGoal) {
        double uniquePairGoalMultiplier;
        if (this.metrics.HS_LIBRARY_SIZE == null) {
            return 0.0;
        }
        double meanCoverage = (double)this.ON_TARGET_FROM_PAIR_BASES / (double)this.metrics.TARGET_TERRITORY;
        double fold80 = this.metrics.FOLD_80_BASE_PENALTY;
        long hsLibrarySize = this.metrics.HS_LIBRARY_SIZE;
        long pairs = this.PF_SELECTED_PAIRS;
        long uniquePairs = this.PF_SELECTED_UNIQUE_PAIRS;
        double onTargetPct = (double)this.metrics.ON_TARGET_BASES / (double)this.metrics.PF_UQ_BASES_ALIGNED;
        double pairMultiplier = uniquePairGoalMultiplier = (double)coverageGoal / meanCoverage * fold80;
        double increment = 1.0;
        boolean goingUp = uniquePairGoalMultiplier >= 1.0;
        double finalPairMultiplier = -1.0;
        for (int i = 0; i < 10000; ++i) {
            double uniquePairMultiplier = DuplicationMetrics.estimateRoi(hsLibrarySize, pairMultiplier, pairs, uniquePairs);
            if (Math.abs(uniquePairMultiplier - uniquePairGoalMultiplier) <= 0.01) {
                finalPairMultiplier = pairMultiplier;
                break;
            }
            if (uniquePairMultiplier > uniquePairGoalMultiplier && goingUp || uniquePairMultiplier < uniquePairGoalMultiplier && !goingUp) {
                increment /= 2.0;
                goingUp = !goingUp;
            }
            pairMultiplier += goingUp ? increment : -increment;
        }
        if (finalPairMultiplier == -1.0) {
            return -1.0;
        }
        double uniqueFraction = (double)uniquePairs * uniquePairGoalMultiplier / ((double)pairs * finalPairMultiplier);
        return 1.0 / uniqueFraction * fold80 * (1.0 / onTargetPct);
    }

    public static class Coverage {
        private final Interval interval;
        private final short[] depths;

        public Coverage(Interval i, int padding) {
            this.interval = i;
            this.depths = new short[this.interval.length() + 2 * padding];
        }

        public void addBase(int offset) {
            if (offset >= 0 && offset < this.depths.length) {
                int n = offset;
                this.depths[n] = (short)(this.depths[n] + 1);
            }
        }

        public boolean hasCoverage() {
            for (short s : this.depths) {
                if (s <= 1) continue;
                return true;
            }
            return false;
        }

        public short[] getDepths() {
            return this.depths;
        }
    }
}

