/*
 * Decompiled with CFR 0.152.
 */
package net.sf.picard.analysis;

import java.io.File;
import java.text.NumberFormat;
import java.util.Collection;
import java.util.TreeMap;
import net.sf.picard.analysis.GcBiasDetailMetrics;
import net.sf.picard.analysis.GcBiasSummaryMetrics;
import net.sf.picard.cmdline.CommandLineProgram;
import net.sf.picard.cmdline.Option;
import net.sf.picard.io.IoUtil;
import net.sf.picard.metrics.MetricsFile;
import net.sf.picard.reference.ReferenceSequence;
import net.sf.picard.reference.ReferenceSequenceFile;
import net.sf.picard.reference.ReferenceSequenceFileFactory;
import net.sf.picard.util.Log;
import net.sf.picard.util.PeekableIterator;
import net.sf.picard.util.QualityUtil;
import net.sf.picard.util.RExecutor;
import net.sf.samtools.SAMFileReader;
import net.sf.samtools.SAMRecord;
import net.sf.samtools.SAMSequenceDictionary;
import net.sf.samtools.util.SequenceUtil;
import net.sf.samtools.util.StringUtil;

public class CollectGcBiasMetrics
extends CommandLineProgram {
    private static final Log LOG = Log.getInstance(CollectGcBiasMetrics.class);
    private static final String R_SCRIPT = "net/sf/picard/analysis/gcBias.R";
    @Option(shortName="R", doc="The reference sequence fasta file.")
    public File REFERENCE_SEQUENCE;
    @Option(shortName="I", doc="The BAM or SAM file containing aligned reads.")
    public File INPUT;
    @Option(shortName="O", doc="The text file to write the metrics table to.")
    public File OUTPUT;
    @Option(shortName="CHART", doc="The PDF file to render the chart to.")
    public File CHART_OUTPUT;
    @Option(doc="The text file to write summary metrics to.", optional=true)
    public File SUMMARY_OUTPUT;
    @Option(doc="The size of windows on the genome that are used to bin reads.")
    public int WINDOW_SIZE = 100;
    @Option(doc="For summary metrics, exclude GC windows that include less than this fraction of the genome.")
    public double MINIMUM_GENOME_FRACTION = 1.0E-5;
    private int totalClusters = 0;
    private int totalAlignedReads = 0;

    public static void main(String[] args) {
        System.exit(new CollectGcBiasMetrics().instanceMain(args));
    }

    @Override
    protected int doWork() {
        IoUtil.assertFileIsReadable(this.REFERENCE_SEQUENCE);
        IoUtil.assertFileIsReadable(this.INPUT);
        IoUtil.assertFileIsWritable(this.OUTPUT);
        IoUtil.assertFileIsWritable(this.CHART_OUTPUT);
        if (this.SUMMARY_OUTPUT != null) {
            IoUtil.assertFileIsWritable(this.SUMMARY_OUTPUT);
        }
        int[] windowsByGc = new int[101];
        int[] readsByGc = new int[101];
        long[] basesByGc = new long[101];
        long[] errorsByGc = new long[101];
        SAMFileReader sam = new SAMFileReader(this.INPUT);
        PeekableIterator iterator = new PeekableIterator(sam.iterator());
        ReferenceSequenceFile referenceFile = ReferenceSequenceFileFactory.getReferenceSequenceFile(this.REFERENCE_SEQUENCE);
        SAMSequenceDictionary referenceDictionary = referenceFile.getSequenceDictionary();
        SAMSequenceDictionary samFileDictionary = sam.getFileHeader().getSequenceDictionary();
        if (referenceDictionary != null && samFileDictionary != null) {
            SequenceUtil.assertSequenceDictionariesEqual((SAMSequenceDictionary)referenceDictionary, (SAMSequenceDictionary)samFileDictionary);
        }
        ReferenceSequence ref = null;
        while ((ref = referenceFile.nextSequence()) != null) {
            byte[] refBases = ref.getBases();
            StringUtil.toUpperCase((byte[])refBases);
            int refLength = refBases.length;
            int lastWindowStart = refLength - this.WINDOW_SIZE;
            byte[] gc = null;
            int countSeq = 0;
            while (iterator.hasNext() && ((SAMRecord)iterator.peek()).getReferenceIndex().intValue() == ref.getContigIndex()) {
                SAMRecord rec = (SAMRecord)iterator.next();
                if (countSeq == 0) {
                    gc = this.calculateAllGcs(refBases, windowsByGc, lastWindowStart);
                }
                if (!rec.getReadPairedFlag() || rec.getFirstOfPairFlag()) {
                    ++this.totalClusters;
                }
                if (!rec.getReadUnmappedFlag()) {
                    byte windowGc;
                    int pos = rec.getReadNegativeStrandFlag() ? rec.getAlignmentEnd() - this.WINDOW_SIZE : rec.getAlignmentStart();
                    ++this.totalAlignedReads;
                    if (pos > 0 && (windowGc = gc[pos]) >= 0) {
                        byte by = windowGc;
                        readsByGc[by] = readsByGc[by] + 1;
                        byte by2 = windowGc;
                        basesByGc[by2] = basesByGc[by2] + (long)rec.getReadLength();
                        byte by3 = windowGc;
                        errorsByGc[by3] = errorsByGc[by3] + (long)SequenceUtil.countMismatches((SAMRecord)rec, (byte[])refBases);
                    }
                }
                ++countSeq;
            }
            LOG.info("Processed: " + ref.getName());
        }
        while (iterator.hasNext()) {
            SAMRecord rec = (SAMRecord)iterator.next();
            if (rec.getReadPairedFlag() && !rec.getFirstOfPairFlag()) continue;
            ++this.totalClusters;
        }
        MetricsFile metricsFile = this.getMetricsFile();
        double totalWindows = this.sum(windowsByGc);
        double totalReads = this.sum(readsByGc);
        double meanReadsPerWindow = totalReads / totalWindows;
        double minimumWindowsToCountInSummary = totalWindows * this.MINIMUM_GENOME_FRACTION;
        for (int i = 0; i < windowsByGc.length; ++i) {
            if (windowsByGc[i] == 0) continue;
            GcBiasDetailMetrics m = new GcBiasDetailMetrics();
            m.GC = i;
            m.WINDOWS = windowsByGc[i];
            m.READ_STARTS = readsByGc[i];
            if (errorsByGc[i] > 0L) {
                m.MEAN_BASE_QUALITY = QualityUtil.getPhredScoreFromObsAndErrors(basesByGc[i], errorsByGc[i]);
            }
            m.NORMALIZED_COVERAGE = (double)m.READ_STARTS / (double)m.WINDOWS / meanReadsPerWindow;
            m.ERROR_BAR_WIDTH = Math.sqrt(m.READ_STARTS) / (double)m.WINDOWS / meanReadsPerWindow;
            metricsFile.addMetric(m);
        }
        metricsFile.write(this.OUTPUT);
        if (this.SUMMARY_OUTPUT != null) {
            MetricsFile summaryMetricsFile = this.getMetricsFile();
            GcBiasSummaryMetrics summary = new GcBiasSummaryMetrics();
            summary.WINDOW_SIZE = this.WINDOW_SIZE;
            summary.TOTAL_CLUSTERS = this.totalClusters;
            summary.ALIGNED_READS = this.totalAlignedReads;
            summary.LOW_GC_BIAS = this.calculateSummaryMetric(metricsFile.getMetrics(), minimumWindowsToCountInSummary, 0, 33);
            summary.MID_GC_BIAS = this.calculateSummaryMetric(metricsFile.getMetrics(), minimumWindowsToCountInSummary, 34, 66);
            summary.HIGH_GC_BIAS = this.calculateSummaryMetric(metricsFile.getMetrics(), minimumWindowsToCountInSummary, 67, 100);
            summary.TOTAL_BIAS = this.calculateSummaryMetric(metricsFile.getMetrics(), minimumWindowsToCountInSummary, 0, 100);
            summary.JAFFE_BIAS_METRIC = this.calculateJaffeMetric(metricsFile.getMetrics(), minimumWindowsToCountInSummary, 0, 100);
            summaryMetricsFile.addMetric(summary);
            if (this.SUMMARY_OUTPUT != null) {
                summaryMetricsFile.write(this.SUMMARY_OUTPUT);
            }
        }
        NumberFormat fmt = NumberFormat.getIntegerInstance();
        fmt.setGroupingUsed(true);
        String subtitle = "Total clusters: " + fmt.format(this.totalClusters) + ", Aligned reads: " + fmt.format(this.totalAlignedReads);
        RExecutor.executeFromClasspath(R_SCRIPT, this.OUTPUT.getAbsolutePath(), this.CHART_OUTPUT.getAbsolutePath(), this.INPUT.getName().replace(".duplicates_marked", "").replace(".aligned.bam", ""), subtitle, String.valueOf(this.WINDOW_SIZE));
        return 0;
    }

    private double sum(int[] values) {
        int length = values.length;
        double total = 0.0;
        for (int i = 0; i < length; ++i) {
            total += (double)values[i];
        }
        return total;
    }

    private double calculateSummaryMetric(Collection<GcBiasDetailMetrics> details, double minimumWindows, int minGc, int maxGc) {
        int[][] loopvars;
        double retval = 0.0;
        double totalBins = 0.0;
        TreeMap<Integer, GcBiasDetailMetrics> metricsByGc = new TreeMap<Integer, GcBiasDetailMetrics>();
        for (GcBiasDetailMetrics detail : details) {
            metricsByGc.put(detail.GC, detail);
        }
        for (int[] limits : loopvars = new int[][]{{50, -1, -1}, {51, 101, 1}}) {
            double last = 0.0;
            for (int gc = limits[0]; gc != limits[1]; gc += limits[2]) {
                GcBiasDetailMetrics detail = (GcBiasDetailMetrics)metricsByGc.get(gc);
                if (detail == null || !((double)detail.WINDOWS >= minimumWindows)) continue;
                double current = detail.NORMALIZED_COVERAGE + detail.ERROR_BAR_WIDTH;
                if (current == 0.0) {
                    current = last;
                } else {
                    last = current;
                }
                if (detail.GC < minGc || detail.GC > maxGc) continue;
                totalBins += 1.0;
                if (!(current > 0.0) || !(current < 1.0)) continue;
                retval += 1.0 / current;
            }
        }
        if (totalBins > 0.0) {
            return retval / totalBins;
        }
        return 0.0;
    }

    private double calculateJaffeMetric(Collection<GcBiasDetailMetrics> details, double minimumWindows, int minGc, int maxGc) {
        double jaffeMetric = 0.0;
        double binsCounted = 0.0;
        GcBiasDetailMetrics previous = null;
        for (GcBiasDetailMetrics detail : details) {
            if (previous == null) continue;
            double h = detail.GC - previous.GC;
            double fx0 = Math.pow(1.0 - previous.NORMALIZED_COVERAGE, 2.0);
            double fx1 = Math.pow(1.0 - detail.NORMALIZED_COVERAGE, 2.0);
            jaffeMetric += h / 2.0 * (fx0 + fx1);
            previous = detail;
        }
        return 100.0 * Math.sqrt(jaffeMetric) / (double)details.size();
    }

    private byte[] calculateAllGcs(byte[] refBases, int[] windowsByGc, int lastWindowStart) {
        int refLength = refBases.length;
        byte[] gc = new byte[refLength + 1];
        CalculateGcState state = new CalculateGcState();
        for (int i = 1; i < lastWindowStart; ++i) {
            int windowEnd = i + this.WINDOW_SIZE;
            int windowGc = this.calculateGc(refBases, i, windowEnd, state);
            gc[i] = (byte)windowGc;
            if (windowGc == -1) continue;
            int n = windowGc;
            windowsByGc[n] = windowsByGc[n] + 1;
        }
        return gc;
    }

    private int calculateGc(byte[] bases, int startIndex, int endIndex, CalculateGcState state) {
        if (state.init) {
            state.init = false;
            state.gcCount = 0;
            state.nCount = 0;
            for (int i = startIndex; i < endIndex; ++i) {
                byte base = bases[i];
                if (base == 71 || base == 67) {
                    ++state.gcCount;
                    continue;
                }
                if (base != 78) continue;
                ++state.nCount;
            }
        } else {
            byte newBase = bases[endIndex - 1];
            if (newBase == 71 || newBase == 67) {
                ++state.gcCount;
            } else if (newBase == 78) {
                ++state.nCount;
            }
            if (state.priorBase == 71 || state.priorBase == 67) {
                --state.gcCount;
            } else if (state.priorBase == 78) {
                --state.nCount;
            }
        }
        state.priorBase = bases[startIndex];
        if (state.nCount > 4) {
            return -1;
        }
        return state.gcCount * 100 / (endIndex - startIndex);
    }

    class CalculateGcState {
        boolean init = true;
        int nCount;
        int gcCount;
        byte priorBase;

        CalculateGcState() {
        }
    }
}

