#include <iostream.h>
#include <fstream.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#define HAVE_MESA 1

#if HAVE_MESA
#  include <GL/osmesa.h>
#endif

#include <GL/gl.h>
#include <GL/glut.h>

#include "sppc.H"
#include "opengl.H"

extern int conv24;		// defined in xv24to8.C, possible values 5,6,7 (fast,slow,best, default 6)

static const char *psfilename = "dataplot.ps";
const greal gl_fontheight_title = 18;
const greal gl_fontheight_label = 13;
const greal gl_fontheight_axis = 11;
const greal gl_fontheight_annot = 12;

static Tsppc S;
static int Width=555, Height=740, win = -1;
static real paperwidth_pts, paperheight_pts;
static bool BufferIsInvalid = true;
static bool DrawCopyrightNotice = true;

enum TOutputType {OUTPUT_PS, OUTPUT_GL, OUTPUT_PPM, OUTPUT_GIF};

static TOutputType outputtype = OUTPUT_PS;
static char *paper = "A4";
static bool landscape = false;

#define USE_PIPES 0		/* in Linux, if USE_PIPES=1, lpr doesn't send to printer until sppc is closed */
// if USE_PIPES=0, uses tmp-PS file instead of a pipe, which works but is not so elegant (but who cares)

#if USE_PIPES
#  include <pfstream.h>	// piped streams
#endif

static void DrawPlotGL()
{
	glMatrixMode(GL_PROJECTION);  // Start modifying the projection matrix
	glLoadIdentity();             // Reset project matrix
	glOrtho(0,Width-1,0,Height-1,-1,1);
	glViewport(0,0,Width,Height);
	glClearColor(1,1,1,0);
	glClear(GL_COLOR_BUFFER_BIT /*| GL_DEPTH_BUFFER_BIT*/);
	glColor3f(0,0,0);
	TOpenGL g(Width,Height,gl_fontheight_title,gl_fontheight_label,gl_fontheight_axis,gl_fontheight_annot);
	S.plot(g,Width,Height,gl_fontheight_title,gl_fontheight_label,gl_fontheight_axis,gl_fontheight_annot,DrawCopyrightNotice);
}

static void Display()
{
	if (BufferIsInvalid) {
//		cerr << "Display, Width=" << Width << ", Height=" << Height << "\n";
		DrawPlotGL();
		BufferIsInvalid = false;
	}
//	cerr << "Display without redraw\n";
	glutSwapBuffers();
}

static void Reshape(int w, int h)
{
//	cerr << "Reshape\n";
	Width = w;
	Height = h;
	BufferIsInvalid = true;
	Display();
}

static void Key(unsigned char key, int x, int y)
{
	switch (key) {
	case 's': case 'S':
	{
		ofstream o(psfilename);
		S.plotPS(o,paperwidth_pts,paperheight_pts,landscape,paper);
		o.close();
		cerr << "sppc: Wrote " << psfilename << "\n";
	}
	break;
	case 'p': case 'P':
	{
#		if USE_PIPES
		opfstream o("|lpr -h");
		if (o.good()) {
			S.plotPS(o,paperwidth_pts,paperheight_pts,landscape,paper);
			o.close();
			cerr << "sppc: Sent page to printer\n";
		} else
			cerr << "*** Failure to open pipe to lpr\n";
#		else
		char *const pstmp1 = tmpnam(0);
		char *const pstmp = strdup(pstmp1);
		ofstream o(pstmp);
		if (o.good()) {
			S.plotPS(o,paperwidth_pts,paperheight_pts,landscape,paper);
			o.close();
		} else {
			cerr << "*** Failure to open tmp PS file \"" << pstmp << "\"\n";
		}
		char *const s = new char [80+2*strlen(pstmp)];
		sprintf(s,"(lpr -h %s; rm -f %s)&",pstmp,pstmp);
		system(s);
		delete [] s;
		free(pstmp);
#		endif
	}
	break;
	case 'g': case 'G':
	{
		char *const tmpfile1 = tmpnam(0);
		char *const tmpfile = strdup(tmpfile1);
		ofstream o(tmpfile);
		S.plotPS(o,paperwidth_pts,paperheight_pts,landscape,paper);
		o.close();
		char *const s = new char [80+2*strlen(tmpfile)];
		sprintf(s,"(ghostview %s; rm -f %s)&",tmpfile,tmpfile);
		system(s);
		delete [] s;
		free(tmpfile);
	}
	break;
	case 27: case 'q': case 'Q':
		exit(0);
	default:;
	}
}

static void usage()
{
	cout << "usage: sppc [-gl] [-ps] [-ppm] [-gif]\n";
	cout << "  [-landscape] [-portrait] [-papertype type]\n";
	cout << "  [-fast] [-slow] [-best]\n";
	cout << "  [<]file.sppc [>file.ps]\n";
	cout << "reads a SPPC (Simple Panel Plot Composer) file from stdin\n";
	cout << "and either displays it in an OpenGL window, or writes the corresponding\n";
	cout << "PostScript code to stdout.\n";
	cout << "Options:\n";
	cout << "  -ps          Write PostScript file to stdout (the default)\n";
	cout << "  -gl          Display file in OpenGL window\n";
	cout << "  -ppm         Write PPM file to stdout\n";
	cout << "  -gif         Write GIF file to stdout\n";
	cout << "  -papertype a4|a3|b5|letter|legal|tabloid\n";
	cout << "  -papertype 12.3x15[cm|in|mm|pt]   Various ways to select paper size\n";
	cout << "                                    format: widthxheight\n";
	cout << "  -landscape   Select 'landscape' document mode (reversing width,height)\n";
	cout << "  -portrait    Select 'portrait' document mode (the default)\n";
	cout << "  -nocopyright Suppress copyright line in GL/PPM image\n";
	cout << "  -fast        (GIF output only:) Set 'fast' 24->8 conversion\n";
	cout << "  -slow        (GIF output only:) Set 'slow' conversion (default)\n";
	cout << "  -best        (GIF output only:) Set 'best' conversion (slowest!)\n";
}

static void ParseArgs(int& argc, char *argv[], int& a)
{
	for (a=1; a<argc; a++) {
		if (!strcmp(argv[a],"-gl")) {
			outputtype = OUTPUT_GL;
		} else if (!strcmp(argv[a],"-ps")) {
			outputtype = OUTPUT_PS;
		} else if (!strcmp(argv[a],"-ppm")) {
			outputtype = OUTPUT_PPM;
		} else if (!strcmp(argv[a],"-gif")) {
			outputtype = OUTPUT_GIF;
		} else if (!strcmp(argv[a],"-landscape")) {
			landscape = true;
		} else if (!strcmp(argv[a],"-portrait")) {
			landscape = false;
		} else if (!strcmp(argv[a],"-fast")) {
			conv24 = 5;
		} else if (!strcmp(argv[a],"-slow")) {
			conv24 = 6;
		} else if (!strcmp(argv[a],"-best")) {
			conv24 = 7;
		} else if (!strcasecmp(argv[a],"-help") || !strcasecmp(argv[a],"--help") || !strcmp(argv[a],"-h")) {
			usage();
			exit(0);
		} else if (!strcmp(argv[a],"-nocopyright")) {
			DrawCopyrightNotice = false;
		} else if (!strcmp(argv[a],"-papertype")) {
			if (a+1>=argc) {
				cerr << "*** sppc: -papertype option requires a value\n";
			} else {
				paper = strdup(argv[a+1]);
				a++;
			}
		} else if (argv[a][0] != '-') {
			break;
		} else {
			cerr << "sppc warning: ignoring arg \"" << argv[a] << "\", hope OpenGL recognises it\n";
		}
	}
}

static void WritePPM(const GLubyte buffer[], int width, int height, FILE *f)
{
	int j, x, y, k;
	const int stride = 3;
	fprintf(f,"P6\n# ppm-file created by sppc\n%d %d\n255\n", width,height);
	for (y=0; y<height; y++) fwrite(&buffer[y*width*stride],3,width,f);
}

extern bool WriteGIF(FILE *fp,
					 unsigned char *pic,
					 int ptype,			/* 0 for 8-bit, 1 for 24-bit */
					 int w, int h,
					 unsigned char rmap[], unsigned char gmap[], unsigned char bmap[], int numcols,
					 int colorstyle, char *comment);

int main(int argc,char *argv[])
{
	int a;
	ParseArgs(argc,argv,a);
	istream* instream_ptr = &cin;
	ifstream input_file;
	if (a < argc) {
		// argv[a] is input file
		input_file.open(argv[a]);
		if (!input_file.good()) {
			cerr << "*** sppc: cannot open input file \"" << argv[a] << "\"\n";
			exit(1);
		}
		instream_ptr = &input_file;
	}
	if (!strcasecmp(paper,"A4")) {
		paperwidth_pts = 597.6;
		paperheight_pts = 842.4;
	} else if (!strcasecmp(paper,"letter")) {
		paperwidth_pts = 612.0;
		paperheight_pts = 792.0;
	} else if (!strcasecmp(paper,"legal")) {
		paperwidth_pts = 612.0;
		paperheight_pts = 1072.0;
	} else if (!strcasecmp(paper,"b5")) {
		paperwidth_pts = 496.8;
		paperheight_pts = 705.6;
	} else if (!strcasecmp(paper,"tabloid")) {
		paperwidth_pts = 792.0;
		paperheight_pts = 1224.0;
	} else if (!strcasecmp(paper,"a3")) {
		paperwidth_pts = 842.4;
		paperheight_pts = 1188.0;
	} else {
		double w,h;
		if (sscanf(paper,"%lfx%lf",&w,&h) == 2) {
			const int L = strlen(paper);
			double unit = 1;
			if (paper[L-2] == 'c' && paper[L-1] == 'm') {
				unit = 72.0/2.54;
			} else if (paper[L-2] == 'm' && paper[L-1] == 'm') {
				unit = 0.1*72.0/2.54;
			} else if (paper[L-2] == 'i' && paper[L-1] == 'n') {
				unit = 72.0;
			} else if (paper[L-2] == 'p' && paper[L-1] == 't') {
				unit = 1;
			} else if (paper[L-3] == 'p' && paper[L-2] == 't' && paper[L-1] == 's') {
				unit = 1;
			}
			paperwidth_pts = w*unit;
			paperheight_pts = h*unit;
			paper = "";
		} else {
			cerr << "*** sppc: Unknown paper type \"" << paper << "\", using A4\n";
			paperwidth_pts = 597.6;
			paperheight_pts = 842.4;
			paper = "A4";
		}
	}
	if (landscape) {
		const real tmp = paperwidth_pts;
		paperwidth_pts = paperheight_pts;
		paperheight_pts = tmp;
	}
	switch (outputtype) {
	case OUTPUT_GL:
	{
		glutInit(&argc, argv);
		const int margin_top = 28;
		const int margin_side = 2*28;
		const int screen_width = /*1024*/ glutGet(GLenum(GLUT_SCREEN_WIDTH));
		const int screen_height = /*768*/ glutGet(GLenum(GLUT_SCREEN_HEIGHT));
		const double max_screen_area = 0.5;
		if (paperwidth_pts/paperheight_pts < double(screen_width-margin_top)/double(screen_height-margin_side)) {
			// Height of screen is minimum
			Height = screen_height - margin_top;
			Width = (int)((paperwidth_pts/paperheight_pts)*Height + 0.5);
		} else {
			// Width of screen is minimum
			Width = screen_width - margin_side;
			Height = (int)((paperheight_pts/paperwidth_pts)*Width + 0.5);
		}
		if (double(Height)*double(Width) > max_screen_area*double(screen_width)*double(screen_height)) {
			const double reducefactor =
				sqrt(max_screen_area*double(screen_width)*double(screen_height)/(double(Height)*double(Width)));
			Width = (int)(reducefactor*Width + 0.5);
			Height = (int)(reducefactor*Height + 0.5);
		}
		glutInitWindowSize(Width,Height);
		const bool DoubleBuffering = true;
		if (DoubleBuffering) glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE /*| GLUT_ALPHA*/ /*| GLUT_DEPTH*/);
		S.read(*instream_ptr);
		win = glutCreateWindow("SPPC [S=SavePS,G=Ghostview,P=Print,Q=Quit]");
		glutSetWindow(win);
		glutDisplayFunc(&Display);
		glutReshapeFunc(&Reshape);
		glutKeyboardFunc(&Key);
		glutMainLoop();
		return 0;
	}
	case OUTPUT_PS:
	{
		if (!S.read(*instream_ptr)) return 1;
		if (!S.plotPS(cout,paperwidth_pts,paperheight_pts,landscape,paper)) return 1;
		return 0;
	}
	case OUTPUT_PPM:
	case OUTPUT_GIF:
	{
#		if HAVE_MESA
		Width = int(paperwidth_pts + 0.5);
		Height = int(paperheight_pts + 0.5);
		GLubyte *buff = new GLubyte [Width*Height*4];
		if (!buff) {
			cerr << "*** sppc: cannot allocate image buffer for PPM output\n";
			exit(2);
		}
//		glutInit(&argc, argv);		// unnecessary, since we don't actually use GLUT anywhere in DrawPlotGL()
//		glutInitWindowSize(Width,Height);
		OSMesaContext ctx = OSMesaCreateContext(OSMESA_RGB,0);
		if (!ctx) {
			cerr << "*** sppc: cannot create Mesa context for PPM output\n";
			exit(4);
		}
		OSMesaMakeCurrent(ctx,buff,GL_UNSIGNED_BYTE,Width,Height);
		OSMesaPixelStore(GLint(OSMESA_Y_UP),0);
		GLint wh[2];
		glGetIntegerv(GLenum(GL_MAX_VIEWPORT_DIMS),wh);
		if (Width > wh[0] || Height > wh[1]) {
			cerr << "*** sppc: requested size " << Width << "x" << Height << " exceeds OpenGL limits\n";
			exit(5);
		}
//		cerr << "queried max width=" << wh[0] << ", height=" << wh[1] << "\n";
		S.read(*instream_ptr);
		DrawPlotGL();
		if (outputtype == OUTPUT_GIF) {
			unsigned char rmap[256],gmap[256],bmap[256];
			WriteGIF(stdout,buff,1,Width,Height,rmap,gmap,bmap,256,0,"SPPC output");
		} else
			WritePPM(buff,Width,Height,stdout);
		OSMesaDestroyContext(ctx);
		delete [] buff;
#		else
		cerr << "*** This sppc version does not support the -ppm switch because it is linked with non-Mesa GL\n";
#		endif
	}
	break;
	}
	return 0;
}
