BeginPackage["HTriangles`", {"HPolygons`", "HDraw`"}]

(* 
  
  This package defines some routines for calculating and visualizing
  Euclidean and hyperbolic geometry in the unit disk.
  
  Requires Mathematica version 2.0 or greater.
  
  The newest version of this package is available for anonymous ftp at
  nic.funet.fi, directory "pub/math/riemann/mathematica".
  
  Version 0.1. Last modified: Nov 1st, 1994.
  
  Send comments and bug reports to 
  
    Juha Haataja (e-mail: jhaataja@csc.fi)
    Center for Scientific Computing
    PO Box 405
    FIN-02101 Espoo, Finland
  
*)

(*----------------------------------------------------------------------*)

Unprotect[TriAngle, TriangleFromSidesAndAngle, TriangleFromAngles,
  TriangleGroup];

(*----------------------------------------------------------------------*)

(* Descriptions of functions *)

TriAngle::usage =
"TriAngle[opside, adjside1, adjside2]";

TriangleFromSidesAndAngle::usage = 
"TriangleFromSidesAndAngle[s1, s2, angle]. Construct a hyperbolic triangle,
when the length of two sides and the angle between are given.";

TriangleFromAngles::usage = 
"TriangleFromAngles[a1, a2, a3]. Construct a triangle, when all three
angles are given.";

TriangleGroup::usage =
"TriangleGroup[n1, n2, n3, level]";

(*----------------------------------------------------------------------*)

Begin["`Private`"]

(*----------------------------------------------------------------------*)

(* Construction of triangles and polygons *)

TriAngle[opside_,adjside1_,adjside2_] := 
  ArcCos[(Cosh[adjside1]*Cosh[adjside2]-Cosh[opside])/
                         (Sinh[adjside1]*Sinh[adjside2])];  

(* Construct a triangle, when two sides and the angle between are given *)

TriangleFromSidesAndAngle[dist1_, dist2_, angle_]:= 
  Polygon[{ZPoint[0], MovePoint[dist1], MovePoint[dist2,angle]}] 

(* Construct a triangle, when all angles are given, and first is
   non-zero *)

TriangleFromAngles[a1_/;NGP[a1]!=0, a2_, a3_]:= 
  Module[{dist1, dist2},
  dist1 = If[NGP[a3]==0, Infinity,
           ArcCosh[(Cos[a1]*Cos[a3]+Cos[a2])/(Sin[a1]*Sin[a3])]];
        dist2 = If[NGP[a2]==0, Infinity,
           ArcCosh[(Cos[a1]*Cos[a2]+Cos[a3])/(Sin[a1]*Sin[a2])]];
  Polygon[{ZPoint[0], MovePoint[dist1], MovePoint[dist2,a1]}]]

(* Construct a triangle, when all angles are given *)

TriangleFromAngles[a1_, a2_, a3_]:= 
  If[NGP[a1]==0 && NGP[a2]==0 
    && NGP[a3]==0,
    {-I, Exp[I*Pi/6], Exp[I*5*Pi/6]},
    If[NGP[a1]!=0, TriangleFromAngles[a1, a2, a3], 
      TriangleFromAngles[a2, a3, a1]]]

(*----------------------------------------------------------------------*)


TriangleGroup[n1_, n2_, n3_, level_Integer:1] :=
  Module[{tri},
    tri = NGP[TriangleFromAngles[Pi/n1, Pi/n2, Pi/n3]];
    PolygonInvGroup[tri, level]
  ];

(*----------------------------------------------------------------------*)

End[];

(*----------------------------------------------------------------------*)

Protect[TriAngle, TriangleFromSidesAndAngle, TriangleFromAngles,
  TriangleGroup];

(*----------------------------------------------------------------------*)

EndPackage[];

(*----------------------------------------------------------------------*)

