/*
 * random.c
 * RLaB interface to RANLIB functions
 */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "bltin.h"
#include "util.h"
#include "symbol.h"
#include "scalar.h"
#include "matop1.h"
#include "r_string.h"

#include "random.h"

static F_INT iseed1 = 1;
static F_INT iseed2 = 1;
static int once = 1;

#define DRANF     1
#define DBETA     2
#define DCHI      3
#define DEXP      4
#define DF        5
#define DGAMMA    6
#define DMULN     7		/* not used yet */
#define DNONCCHI  8
#define DNONCF    9
#define DNORM    10
#define DUNIF    11
#define DBIN     12
#define DPOIS    13

static int dist_type;
static F_REAL dp1, dp2, dp3;

/*
 * We have our own cover package for rand() functionality.
 * This allows us to generate random numbers based upon
 * stored distribution type and parameters.
 */

double
rrand ()
{
  F_INT ip1;
  double d = 0.0;

  switch (dist_type)
  {
  case DRANF:
    d = (F_DOUBLE) RANF ();
    break;
  case DBETA:
    d = (F_DOUBLE) GENBET (&dp1, &dp2);
    break;
  case DCHI:
    d = (F_DOUBLE) GENCHI (&dp1);
    break;
  case DEXP:
    d = (F_DOUBLE) GENEXP (&dp1);
    break;
  case DF:
    d = (F_DOUBLE) GENF (&dp1, &dp2);
    break;
  case DGAMMA:
    d = (F_DOUBLE) GENGAM (&dp1, &dp2);
    break;
  case DNONCCHI:
    d = (F_DOUBLE) GENCHI (&dp1);
    break;
  case DNONCF:
    d = (F_DOUBLE) GENF (&dp1, &dp2, &dp3);
    break;
  case DNORM:
    d = (F_DOUBLE) GENNOR (&dp1, &dp2);
    break;
  case DUNIF:
    d = (F_DOUBLE) GENUNF (&dp1, &dp2);
    break;
  case DBIN:
    ip1 = (F_INT) dp1;
    d = (F_DOUBLE) IGNBIN (&ip1, &dp2);
    break;
  case DPOIS:
    d = (F_DOUBLE) IGNPOI (&dp1);
    break;
  }
  return (d);
}

void
Rand (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  double d, d1, d2;
  Matrix *m;
  ListNode *A1;

  /*
   * Initialize random number generators.
   * Do this only once. User can reset via srand().
   */

  if (once)
  {
    SETALL (&iseed1, &iseed2);
    dist_type = DRANF;
    dp1 = dp2 = dp3 = 0.0;
    once = 0;
  }

  /* get args from list */

  if (n_args == 0)
  {
    /* return a randomly generated scalar */
    d = rrand ();
    *return_ptr = (VPTR) scalar_Create (d);
    return;
  }
  else if (n_args == 1)
  {
    A1 = bltin_get_matrix ("rand", d_arg, 1);
    
    if (MTYPE (e_data (A1)) == REAL)
    {
      if (MNR (e_data (A1)) * MNC (e_data (A1)) != 2)
	error_1 (e_name (A1),
		 "must supply rand with 2-element MATRIX");

      m = matrix_Rand ((int) MATrv (e_data (A1), 0), 
		       (int) MATrv (e_data (A1), 1));
      *return_ptr = (VPTR) m;
      remove_tmp_destroy (A1);
      return;
    }
    else if (MTYPE (e_data (A1)) == COMPLEX)
    {
      error_1 (e_name (A1), "COMPLEX MATRIX not allowed");
    }
    else if (MTYPE (e_data (A1)) == STRING)
    {
      if (!strcmp ("default", MATs (e_data (A1), 1, 1)))
      {
	dist_type = DRANF;
	*return_ptr = (VPTR) scalar_Create (1.0);
	remove_tmp_destroy (A1);
	return;
      }
      else
	error_1 (e_name (A1), "invalid STRING for rand()");
    }
  }
  else if (n_args >= 2)
  {
    /*
       * The 1st arg can either be a SCALAR, or a STRING. If the
       * 1st arg is a SCALAR, the 2nd arg must be a SCALAR.
       * If the 1st arg is a STRING. The string describes the
       * distribution. The remaining args are the parameters
       * required to define the distribution.
     */

    A1 = bltin_get_matrix ("rand", d_arg, 1);
    if (MTYPE (e_data (A1)) == STRING)
    {
      if (!strcmp ("beta", MATs (e_data (A1), 1, 1)))
      {
	if (n_args != 3)
	  error_1 ("rand() requires 3 args when arg 1 is \"beta\"", 0);
	dp1 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 2);
	dp2 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 3);
	dist_type = DBETA;
      }
      else if (!strcmp ("chi", MATs (e_data (A1), 1, 1)))
      {
	if (n_args != 2)
	  error_1 ("rand() requires 2 args when arg 1 is \"chi\"", 0);
	dist_type = DCHI;
	dp1 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 2);
      }
      else if (!strcmp ("exp", MATs (e_data (A1), 1, 1)))
      {
	if (n_args != 2)
	  error_1 ("rand() requires 2 args when arg 1 is \"exp\"", 0);
	dp1 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 2);
	dist_type = DEXP;
      }
      else if (!strcmp ("f", MATs (e_data (A1), 1, 1)))
      {
	if (n_args != 3)
	  error_1 ("rand() requires 3 args when arg 1 is \"f\"", 0);
	dp1 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 2);
	dp2 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 3);
	dist_type = DF;
      }
      else if (!strcmp ("gamma", MATs (e_data (A1), 1, 1)))
      {
	if (n_args != 3)
	  error_1 ("rand() requires 3 args when arg 1 is \"gamma\"", 0);
	dp1 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 2);
	dp2 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 3);
	dist_type = DGAMMA;
      }
      else if (!strcmp ("nchi", MATs (e_data (A1), 1, 1)))
      {
	if (n_args != 3)
	  error_1 ("rand() requires 3 args when 1st arg is \"nchi\"", 0);
	dp1 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 2);
	dp2 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 3);
	dist_type = DNONCCHI;
      }
      else if (!strcmp ("nf", MATs (e_data (A1), 1, 1)))
      {
	if (n_args != 4)
	  error_1 ("rand() requires 4 args when 1st arg is \"nf\"", 0);
	dp1 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 2);
	dp2 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 3);
	dp3 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 4);
	dist_type = DNONCF;
      }
      else if (!strcmp ("normal", MATs (e_data (A1), 1, 1)))
      {
	if (n_args != 3)
	  error_1 ("rand() requires 3 args when 1st arg is \"normal\"", 0);
	dp1 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 2);
	dp2 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 3);
	dist_type = DNORM;
      }
      else if (!strcmp ("uniform", MATs (e_data (A1), 1, 1)))
      {
	if (n_args != 3)
	  error_1 ("rand() requires 3 args when arg 1 is \"uniform\"", 0);
	dp1 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 2);
	dp2 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 3);
	dist_type = DUNIF;
      }
      else if (!strcmp ("bin", MATs (e_data (A1), 1, 1)))
      {
	if (n_args != 3)
	  error_1 ("rand() requires 3 args when 1st arg is \"bin\"", 0);
	dp1 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 2);
	dp2 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 3);
	dist_type = DBIN;
      }
      else if (!strcmp ("poisson", MATs (e_data (A1), 1, 1)))
      {
	if (n_args != 2)
	  error_1 ("rand() requires 2 args when arg 1 is \"poissom\"", 0);
	dp1 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 2);
	dist_type = DPOIS;
      }
      else
      {
	error_1 ("invalid distribution string, see `help rand'", 0);
      }
      remove_tmp_destroy (A1);
      *return_ptr = (VPTR) scalar_Create (1.0);
      return;
    }
    else if (MTYPE (e_data (A1)) == REAL)
    {
      if (n_args > 2)
      {
	remove_tmp_destroy (A1);
	error_1 ("rand: 2 arguments allowed in this context", 0);
      }
      d1 = MAT (e_data (A1), 1, 1);
      d2 = (F_REAL) bltin_get_numeric_double ("rand", d_arg, 2);
      m = matrix_Rand ((int) d1, (int) d2);
      remove_tmp_destroy (A1);
      *return_ptr = (VPTR) m;
      return;
    }
  }
}

#include <sys/types.h>

#ifdef __STDC__
#include <time.h>
#endif /* __STDC__ */

#ifdef HAVE_TIME_H
#include <time.h>
#endif

void
SRand (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  long int init;
  ListNode *S;

  /* Check n_args */
  if (!(n_args == 0 || n_args == 1))
    error_1 ("srand: 1 or 2 args required", 0);

  if (once)
  {
    SETALL (&iseed1, &iseed2);
    dist_type = DRANF;
    dp1 = dp2 = dp3 = 0.0;
    once = 0;
  }

  if (n_args == 0)
  {
    /* Reset seed to defaults */
    init = -1;
    INITGN (&init);
  }
  else if (n_args == 1)
  {
    S = bltin_get_matrix ("srand", d_arg, 1);
    if (MTYPE (e_data (S)) == STRING)
    {
      if (!strcmp ("clock", MATs (e_data (S), 1, 1)))
      {
#ifdef THINK_C
	/* mac uses 1904 as the stating year for counting, 
	   it will give negative value here, so... */
	iseed1 = (long int) (time (0)) - 2081376000L;
#else
	iseed1 = (long int) time (0);
#endif
	SETALL (&iseed1, &iseed2);
	}
      else
	error_1 ("\"clock\" is the only valid string-arg to srand()", 0);
    }
    else if (MTYPE (e_data (S)) == REAL)
    {
      iseed1 = (long int) MAT (e_data (S), 1, 1);
      if (iseed1 == 0)
	error_1 ("cannot call srand() with argument = 0", 0);
      SETALL (&iseed1, &iseed2);
    }
    else
    {
      error_1 (e_name (S), "invalid type for srand()");
    }
  }
  *return_ptr = (VPTR) scalar_Create (1.0);
}
