/* r_string.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "mem.h"
#include "r_string.h"
#include "util.h"

#include <stdio.h>
#include <string.h>

/* **************************************************************
 * Create a STRING, and initialize it.
 * ************************************************************** */

String *
string_Create (s)
     char *s;
{
  String *new = (String *) MALLOC (sizeof (String));

  new->type = STRING;
  new->name = 0;
  if (s == 0)
  {				/* Create the NULL string */
    new->string = MALLOC (1 * sizeof (char));
    new->string[0] = '\0';
  }
  else
    new->string = s;

  return (new);
}

/* **************************************************************
 * Destroy an instance of a string.
 * ************************************************************** */

void
string_Destroy (s)
     String *s;
{
  ASSERT (s);
  {
    s->type = 0;
    FREE (s->name);
    FREE (s->string);
    FREE (s);
  }
}

void
string_Set (s, cptr)
     String *s;
     char *cptr;
{
  ASSERT (s);
  {
    FREE (s->string);
    s->string = cptr;
  }
}

/* **************************************************************
 * Copy a string, return a pointer to the new string.
 * ************************************************************** */

String *
string_Copy (s)
     String *s;
{
  char *string;
  String *new;

  string = cpstr (string_GetString (s));
  new = string_Create (string);
  return (new);
}

void
string_memcpy (s1, s2, len)
     String *s1;
     VPTR s2;
     int len;
{
  ASSERT (s1);
  {
    FREE (s1->string);
    s1->string = (char *) MALLOC (sizeof (char) * (len + 1));
    memcpy (s1->string, s2, (size_t) len);
    s1->string[len] = '\0';
  }
}

String *
string_Add (s1, s2)
     String *s1, *s2;
{
  ASSERT (s1);
  ASSERT (s2);
  {
    int len1, len2;
    String *new;

    len1 = string_GetLength (s1);
    len2 = string_GetLength (s2);

    new = (String *) MALLOC (sizeof (String));
    new->type = STRING;
    new->name = 0;
    new->string = (char *) MALLOC (sizeof (char) * (len1 + len2 + 1));

    strncpy (new->string, s1->string, len1);
    strcpy (&(new->string[len1]), s2->string);

    return (new);
  }
}

/* **************************************************************
 * Return the length of the String, similar to strlen(3C).
 * ************************************************************** */
int
string_GetLength (string)
     String *string;
{
  return (strlen (string->string));
}

/* **************************************************************
 * Compare two Strings, similar to strcmp(3C).
 * ************************************************************** */
int
string_Compare (s1, s2)
     String *s1, *s2;
{
  return (strcmp (s1->string, s2->string));
}


void
string_SetName (s, name)
     String *s;
     char *name;
{
  ASSERT (s);
  {
    FREE (s->name);
    s->name = name;
  }
}

/* **************************************************************
 * Write a string to file.
 * ************************************************************** */
void
string_Write (s, fn)
     String *s;
     FILE *fn;
{
  ASSERT (s);
  {
    if (string_GetName (s) == 0)
      fprintf (fn, "# string : %s Length : %d\n%s\n", "STRING",
	       string_GetLength (s), string_GetString (s));
    else
      fprintf (fn, "# string : %s Length : %d\n%s\n", string_GetName (s),
	       string_GetLength (s), string_GetString (s));
    fflush (fn);
  }
}

/* **************************************************************
 * Read a string from file that was written by string_Write().
 * ************************************************************** */
String *
string_Read (fn)
     FILE *fn;
{
  int i, length;
  char name[80], *string;
  String *snew;

  /* Get string name */
  fscanf (fn, "%s Length : %d\n", name, &length);

  /* Now read the string */

  string = (char *) MALLOC ((size_t) sizeof (char) * (length + 1));

  for (i = 0; i < length; i++)
  {
    string[i] = (char) fgetc (fn);
  }
  string[length] = '\0';

  snew = string_Create (cpstr (string));
  string_SetName (snew, cpstr (name));

  FREE (string);

  if (length > 0)
    fscanf (fn, "\n");

  return (snew);
}

/* **************************************************************
 * Add two character strings together, return ptr to the newly
 * created string.
 * ************************************************************** */

char *
string_add (s1, s2)
     char *s1, *s2;
{
  char *snew;
  int l1, l2;

  l1 = strlen (s1);
  l2 = strlen (s2);

  snew = (char *) CALLOC ((l1 + l2 + 1), sizeof (char));

  if (l1 != 0)
    strcpy (snew, s1);
  if (l2 != 0)
    strcpy (snew + l1, s2);

  return (snew);
}

/* **************************************************************
 * Print a String entity.
 * ************************************************************** */
void
string_Print (str, fn)
     String *str;
     FILE *fn;
{
  ASSERT (str);
  {
    if (str->name != 0 && strncmp (str->name, "-", 1))
      fprintf (fn, " %s =\n", str->name);

    fprintf (fn, "%s\n", string_GetString (str));
  }
}
