/* op_mat.c */
/* Matrix class operations for the RLaB machine. */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1994, 1995  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "code.h"
#include "scop1.h"
#include "matrix.h"
#include "matop1.h"
#include "matop2.h"
#include "r_string.h"
#include "util.h"

/* **************************************************************
 * Create a matrix given a Datum as input, return a new Datum that
 * contains the newly created matrix.
 * ************************************************************** */
Datum
matrix_create (d)
     Datum d;
{
  int n_row, n_col;
  Datum new;
  Matrix *m;

  switch (d.type)
  {
  case CONSTANT:
    m = matrix_Create (1, 1);
    MAT (m, 1, 1) = d.u.val;
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
  case iCONSTANT:
    m = matrix_CreateC (1, 1);
    MATr (m, 1, 1) = 0.0;
    MATi (m, 1, 1) = d.u.val;
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      if (SVALi (e_data (d.u.ent)) == 0.0)
      {
	m = matrix_Create (1, 1);
	MAT (m, 1, 1) = SVALr (e_data (d.u.ent));
      }
      else
      {
	m = matrix_CreateC (1, 1);
	MATr (m, 1, 1) = SVALr (e_data (d.u.ent));
	MATi (m, 1, 1) = SVALi (e_data (d.u.ent));
      }
      new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
      new.type = ENTITY;
      break;
    case MATRIX:
      /*
       * Maybe this part should just return the original
       * matrix, do we really need to copy it?
       */
      n_col = MNC (e_data (d.u.ent));
      n_row = MNR (e_data (d.u.ent));
      m = matrix_Copy (e_data (d.u.ent));
      new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
      new.type = ENTITY;
      break;
    case STRING:
      m = matrix_CreateS (1, 1);
      MATs (m, 1, 1) = cpstr (string_GetString (e_data (d.u.ent)));
      new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
      new.type = ENTITY;
      break;
    case UNDEF:
      error_1 (e_name (d.u.ent), "UNDEFINED, cannot create MATRIX");
      break;
    default:
      error_1 (e_name (d.u.ent), "cannot create a MATRIX from this?");
      break;
    }
    remove_tmp_destroy (d.u.ent);
    break;
  }
  return (new);
}

/* **************************************************************
 * Stack a new row onto the bottom of an existing matrix.
 * d1 is the element we will append to ( an existing matrix ).
 * d2 is the element we will append ( the new row ).
 * [ d1 ; d2 ]
 *
 * Note: This function does NOT copy d1, since to do a stack-op we
 * have had to pass though matrix_create() 1st, and matrix_create()
 * copies whatever is inside the [ ].
 * ************************************************************** */
Datum
matrix_stack (d1, d2)
     Datum d1, d2;
{
  int i, j, n_row, n_row_base, n_col;

  switch (d2.type)
  {
  case CONSTANT:
    if (MNC (e_data (d1.u.ent)) > 1)
      error_1 (e_name (d1.u.ent),
	       "must be a column matrix to stack a SCALAR");
    n_col = 1;
    n_row = MNR (e_data (d1.u.ent)) + 1;
    if (MTYPE (e_data (d1.u.ent)) == REAL)
    {
      matrix_AppendRowR (e_data (d1.u.ent), 1);
      MAT (e_data (d1.u.ent), n_row, 1) = d2.u.val;
    }
    else if (MTYPE (e_data (d1.u.ent)) == COMPLEX)
    {
      matrix_AppendRowC (e_data (d1.u.ent), 1);
      MATr (e_data (d1.u.ent), n_row, 1) = d2.u.val;
    }
    else
      error_1 (e_name (d1.u.ent),
	       "String matrix not allowed in nuermic stack-op");
    break;

  case iCONSTANT:
    if (MNC (e_data (d1.u.ent)) > 1)
      error_1 (e_name (d1.u.ent),
	       "must be a column matrix to stack a scalar");
    n_col = 1;
    n_row = MNR (e_data (d1.u.ent)) + 1;
    if (MTYPE (e_data (d1.u.ent)) == STRING)
      error_1 (e_name (d1.u.ent),
	       "String matrix not allowed in nuermic stack-op");
    matrix_AppendRowC (e_data (d1.u.ent), 1);
    MATi (e_data (d1.u.ent), n_row, 1) = d2.u.val;
    break;

  case ENTITY:
    switch (e_type (d2.u.ent))
    {
    case SCALAR:
      if (MNC (e_data (d1.u.ent)) > 1)
	error_1 (e_name (d2.u.ent),
		 "must be a column matrix to stack a scalar");
      n_col = 1;
      n_row = MNR (e_data (d1.u.ent)) + 1;
      if (MTYPE (e_data (d1.u.ent)) == REAL
	  && SVALi (e_data (d2.u.ent)) == 0.0)
      {
	matrix_AppendRowR (e_data (d1.u.ent), 1);
	MAT (e_data (d1.u.ent), n_row, 1) = SVALr (e_data (d2.u.ent));
      }
      else if (MTYPE (e_data (d1.u.ent)) == COMPLEX
	       || SVALi (e_data (d2.u.ent)) != 0.0)
      {
	matrix_AppendRowC (e_data (d1.u.ent), 1);
	MATr (e_data (d1.u.ent), n_row, 1) = SVALr (e_data (d2.u.ent));
	MATi (e_data (d1.u.ent), n_row, 1) = SVALi (e_data (d2.u.ent));
      }
      else if (MTYPE (e_data (d1.u.ent)) == STRING)
	error_1 (e_name (d1.u.ent), "cannot stack a NUMERIC and a STRING");

      remove_tmp_destroy (d2.u.ent);
      break;
    case MATRIX:
      /*
         * Check for the special case where either d1, or d2
         * are NULL objects, in either case shortcut to the
         * correct answer.
       */
      if (MNC (e_data (d1.u.ent)) == 0 && MNR (e_data (d1.u.ent)) == 0)
      {
	/* In case they're both NULL break out, leave d1 alone */
	if (MNC (e_data (d2.u.ent)) == 0 && MNR (e_data (d2.u.ent)) == 0)
	  break;

	remove_tmp_destroy (d1.u.ent);
	d1.u.ent = install_tmp (MATRIX,
				matrix_Copy (e_data (d2.u.ent)),
				matrix_Destroy);
      }
      else if (MNC (e_data (d2.u.ent)) == 0 && MNR (e_data (d2.u.ent)) == 0)
	break;
      else
      {
	n_col = MNC (e_data (d1.u.ent));
	n_row_base = MNR (e_data (d1.u.ent));
	n_row = n_row_base + MNR (e_data (d2.u.ent));

	if (MNC (e_data (d2.u.ent)) != n_col && n_col != 0)
	  error_2 (e_name (d1.u.ent), e_name (d2.u.ent),
		   "column dimensions must match");

	if (MTYPE (e_data (d1.u.ent)) == REAL &&
	    MTYPE (e_data (d2.u.ent)) == REAL)
	{
	  matrix_AppendRowR (e_data (d1.u.ent), n_row - n_row_base);
	  /* now load in new row(s) */
	  for (i = n_row_base + 1; i <= n_row; i++)
	    for (j = 1; j <= n_col; j++)
	      MAT (e_data (d1.u.ent), i, j) =
		MAT (e_data (d2.u.ent), (i - n_row_base), j);
	}
	else if (MTYPE (e_data (d1.u.ent)) == COMPLEX ||
		 MTYPE (e_data (d2.u.ent)) == COMPLEX)
	{
	  matrix_AppendRowC (e_data (d1.u.ent), n_row - n_row_base);
	  if (MTYPE (e_data (d2.u.ent)) == REAL)
	  {
	    for (i = n_row_base + 1; i <= n_row; i++)
	      for (j = 1; j <= n_col; j++)
	      {
		MATr (e_data (d1.u.ent), i, j) =
		  MAT (e_data (d2.u.ent), (i - n_row_base), j);
		MATi (e_data (d1.u.ent), i, j) = 0.0;
	      }
	  }
	  else
	  {
	    for (i = n_row_base + 1; i <= n_row; i++)
	      for (j = 1; j <= n_col; j++)
	      {
		MATr (e_data (d1.u.ent), i, j) =
		  MATr (e_data (d2.u.ent), (i - n_row_base), j);
		MATi (e_data (d1.u.ent), i, j) =
		  MATi (e_data (d2.u.ent), (i - n_row_base), j);
	      }
	  }
	}
	else if (MTYPE (e_data (d1.u.ent)) == STRING &&
		 MTYPE (e_data (d2.u.ent)) == STRING)
	{
	  matrix_AppendRowS (e_data (d1.u.ent), n_row - n_row_base);
	  /* now load in new row(s) */
	  for (i = n_row_base + 1; i <= n_row; i++)
	  {
	    for (j = 1; j <= n_col; j++)
	      MATs (e_data (d1.u.ent), i, j) =
		cpstr (MATs (e_data (d2.u.ent), (i - n_row_base), j));
	  }
	}
      }
      remove_tmp_destroy (d2.u.ent);
      break;
    case STRING:		/* d2 */
      if (MNC (e_data (d1.u.ent)) > 1)
	error_1 (e_name (d1.u.ent),
		 "must be a column matrix to stack a STRING");
      if (MNR (e_data (d1.u.ent)) == 0)
      {
	matrix_Destroy (e_data (d1.u.ent));
	e_data (d1.u.ent) = (VPTR) matrix_CreateS (0, 0);
      }
      n_row = MNR (e_data (d1.u.ent));
      n_col = 1;
      matrix_AppendRowS (e_data (d1.u.ent), 1);
      MATs (e_data (d1.u.ent), n_row + 1, 1) =
	cpstr (string_GetString (e_data (d2.u.ent)));
      break;
    case UNDEF:
      error_1 (e_name (d2.u.ent), "UNDEFINED");
      break;
    default:
      error_1 (e_name (d2.u.ent), "cannot stack this type ?");
      break;
    }
  }
  return (d1);
}

Datum con_con_mat_promote _PROTO ((Matrix * m, double d1, double d2));
Datum con_ent_mat_promote _PROTO ((Matrix * m, double d, ListNode * ent));
Datum ent_con_mat_promote _PROTO ((Matrix * m, ListNode * ent, double d));
Datum ent_ent_mat_promote _PROTO ((Matrix * m, ListNode * e1, ListNode * e2));

/* **************************************************************
 * Create a sub-matrix. Promote a part of an existing matrix.
 *   i_flag = 1: both row and column indices.
 *            2: row index only.
 *            3: column index only.
 *
 *   var '[' i1 ';' i2 ']'
 * ************************************************************** */
Datum
matrix_sub (i_flag, i1, i2, var)
     int i_flag;
     Datum i1, i2, var;
{
  Datum new;
  Matrix *matrix, *mrow, *mcol, *mnew;
  Scalar *s;
  String *str;

  mnew = 0;			/* Initialize */

  /* Try and cover errors allowed by grammar */
  if (var.type != ENTITY)
    error_1 ("cannot use scalar CONSTANT in MATRIX context", (char *) 0);
  if (e_type (var.u.ent) != MATRIX &&
      e_type (var.u.ent) != SCALAR &&
      e_type (var.u.ent) != STRING)
    error_1 (e_name (var.u.ent),
	     "must be a MATRIX, SCALAR or STRING in matrix context");

  var = convert_all_to_matrix (var);
  matrix = (Matrix *) e_data (var.u.ent);

  switch (i_flag)
  {
  case 1:			/* row and column indices */

    i1 = convert_to_scalar (i1);
    i2 = convert_to_scalar (i2);

    switch (e_type (i1.u.ent))
    {
    case SCALAR:
      switch (e_type (i2.u.ent))
      {
      case SCALAR:
	mnew = matrix_ExtractElement (matrix, (int) SVALr (e_data (i1.u.ent)),
				      (int) SVALr (e_data (i2.u.ent)));
	break;
      case MATRIX:
	mcol = (Matrix *) e_data (i2.u.ent);
	mnew = matrix_ExtractRow (matrix, (int) SVALr (e_data (i1.u.ent)),
				  mcol);
	break;
      case STRING:
	error_1 (e_name (i2.u.ent), "string invalid as matrix index");
      default:
	error_1 (e_name (i2.u.ent), "invalid type for matrix index");
      }
      break;
    case MATRIX:
      switch (e_type (i2.u.ent))
      {
      case SCALAR:
	mrow = (Matrix *) e_data (i1.u.ent);
	mnew = matrix_ExtractColumn (matrix, (int) SVALr (e_data (i2.u.ent)),
				     mrow);
	break;
      case MATRIX:
	mrow = (Matrix *) e_data (i1.u.ent);
	mcol = (Matrix *) e_data (i2.u.ent);
	mnew = matrix_ExtractSubMatrix (matrix, mrow, mcol);
	break;
      case STRING:
	error_1 (e_name (i2.u.ent), "string invalid as matrix index");
      default:
	error_1 (e_name (i2.u.ent), "invalid type for matrix index");
      }
      break;
    case STRING:
      error_1 (e_name (i1.u.ent), "string invalid as matrix index");
    default:
      error_1 (e_name (i1.u.ent), "invalid type for matrix index");
    }

    remove_tmp_destroy (i1.u.ent);
    remove_tmp_destroy (i2.u.ent);
    remove_tmp_destroy (var.u.ent);

    break;

  case 2:			/* row index (ices) only, get ALL columns */

    i1 = convert_to_matrix_d (i1);

    mrow = (Matrix *) e_data (i1.u.ent);

    if (MTYPE (mrow) != REAL)
      error_1 (e_name (var.u.ent), "row indices must be REAL");

    mnew = matrix_ExtractColMatrix (matrix, mrow);

    remove_tmp_destroy (i1.u.ent);
    remove_tmp_destroy (var.u.ent);

    break;

  case 3:			/* column index (ices) only */

    i2 = convert_to_matrix_d (i2);

    mcol = (Matrix *) e_data (i2.u.ent);

    if (MTYPE (mcol) != REAL)
      error_1 (e_name (var.u.ent), "row indices must be REAL");

    mnew = matrix_ExtractRowMatrix (matrix, mcol);

    remove_tmp_destroy (i2.u.ent);
    remove_tmp_destroy (var.u.ent);

    break;
  }

  /* If the new MATRIX is a 1-by-1, make it a SCALAR */
  if (MNR (mnew) == 1 && MNC (mnew) == 1)
  {
    if (MTYPE (mnew) == REAL)
    {
      s = scalar_Create (MAT (mnew, 1, 1));
      matrix_Destroy (mnew);
      new.u.ent = install_tmp (SCALAR, s, scalar_Destroy);
    }
    else if (MTYPE (mnew) == COMPLEX)
    {
      s = scalar_CreateC (MATr (mnew, 1, 1), MATi (mnew, 1, 1));
      matrix_Destroy (mnew);
      new.u.ent = install_tmp (SCALAR, s, scalar_Destroy);
    }
    else if (MTYPE (mnew) == STRING)
    {
      str = string_Create (cpstr (MATs (mnew, 1, 1)));
      matrix_Destroy (mnew);
      new.u.ent = install_tmp (STRING, str, string_Destroy);
    }
  }
  else
  {
    new.u.ent = install_tmp (MATRIX, mnew, matrix_Destroy);
  }

  new.type = ENTITY;
  return (new);
}

/*
 * Allow indexing of a MATRIX or SCALAR like a VECTOR
 * d1 [ d2 ]
 */

Datum
matrix_vector_sub (d1, d2)
     Datum d1, d2;
{
  Datum new;
  Matrix *m, *mi, *mnew;
  Scalar *s;
  String *str;

  /* Try and cover errors allowed by grammar */
  if (d1.type != ENTITY)
    error_1 ("cannot use CONSTANT in MATRIX context", (char *) 0);
  if (e_type (d1.u.ent) != MATRIX &&
      e_type (d1.u.ent) != SCALAR &&
      e_type (d1.u.ent) != STRING)
    error_1 (e_name (d1.u.ent),
	     "must be a MATRIX, SCALAR or STRING in array context");

  d1 = convert_all_to_matrix (d1);

  m = (Matrix *) e_data (d1.u.ent);

  d2 = convert_to_matrix_d (d2);

  mi = (Matrix *) e_data (d2.u.ent);

  if (MTYPE (mi) != REAL)
    error_1 (e_name (d1.u.ent), "row indices must be REAL");

  mnew = matrix_ExtractVector (m, mi);
  new.u.ent = install_tmp (MATRIX, mnew, matrix_Destroy);

  /* If the new MATRIX is a 1-by-1, make it a SCALAR */
  if (MNR (mnew) == 1 && MNC (mnew) == 1)
  {
    if (MTYPE (mnew) == REAL)
    {
      s = scalar_Create (MAT (mnew, 1, 1));
      listNode_DestroyDataOnly (new.u.ent);
      listNode_AttachData (new.u.ent, SCALAR, s, scalar_Destroy);
    }
    else if (MTYPE (mnew) == COMPLEX)
    {
      s = scalar_CreateC (MATr (mnew, 1, 1), MATi (mnew, 1, 1));
      listNode_DestroyDataOnly (new.u.ent);
      listNode_AttachData (new.u.ent, SCALAR, s, scalar_Destroy);
    }
    else if (MTYPE (mnew) == STRING)
    {
      str = string_Create (cpstr (MATs (mnew, 1, 1)));
      listNode_DestroyDataOnly (new.u.ent);
      listNode_AttachData (new.u.ent, STRING, str, string_Destroy);
    }
  }

  remove_tmp_destroy (d1.u.ent);
  remove_tmp_destroy (d2.u.ent);

  new.type = ENTITY;
  return (new);
}

/* **************************************************************
 * Assign new values to the element(s) of a matrix.
 * i_flag = 1: user specified row and column
 *          2: user specified row only (use all columns)
 *          3: user specified column only (use all rows)
 * i1:  row index(s).
 * i2:  col index(s).
 * a:   the new value(s).
 * var: The matrix that will be modified.
 *
 *      var [ i1; i2 ] = a
 *
 * Take a different approach than vector_assign. First figure out
 * what the rhs is, then tackle the indices one at a time.
 * ************************************************************** */
Datum
matrix_assign (i_flag, var, i1, i2, a)
     int i_flag;
     Datum var, i1, i2, a;
{
  ListNode *tmpm;
  Matrix *m, *mrow, *mcol;

  m = 0;
  tmpm = 0;			/* Initialize */

  a = convert_to_rhs_matrix (a);

  /* Make sure we are assigning to a matrix */
  if (var.type == ENTITY)
  {
    if (e_type (var.u.ent) != MATRIX)
    {
      /*
       * Do not destroy the old entity yet...
       * Wait until we make it through assign details.
       * Set the lhs-type based upon rhs-type. Check for
       * scalar 1st.
       */

      if (e_type (var.u.ent) == SCALAR)
      {
	if (SVALi (e_data (var.u.ent)) == 0.0)
	{
	  m = matrix_Create (1, 1);
	  MAT (m, 1, 1) = SVALr (e_data (var.u.ent));
	}
	else
	{
	  m = matrix_CreateC (1, 1);
	  MATr (m, 1, 1) = SVALr (e_data (var.u.ent));
	  MATi (m, 1, 1) = SVALi (e_data (var.u.ent));
	}
	tmpm = install_tmp (MATRIX, m, matrix_Destroy);
      }
      else if (e_type (var.u.ent) == STRING)
      {
	m = matrix_CreateS (1, 1);
	MATsv (m, 0) = cpstr (string_GetString (e_data (var.u.ent)));
	tmpm = install_tmp (MATRIX, m, matrix_Destroy);
      }
      else
	/* Overwrite whatever var.u.ent is */
      {
	if (MTYPE (e_data (a.u.ent)) == REAL)
	  tmpm = install_tmp (MATRIX, m = matrix_Create (0, 0),
			      matrix_Destroy);
	if (MTYPE (e_data (a.u.ent)) == COMPLEX)
	  tmpm = install_tmp (MATRIX, m = matrix_CreateC (0, 0),
			      matrix_Destroy);
	if (MTYPE (e_data (a.u.ent)) == STRING)
	  tmpm = install_tmp (MATRIX, m = matrix_CreateS (0, 0),
			      matrix_Destroy);
	matrix_Zero (m);
      }
    }
    else
      /* var is already a MATRIX, do some checking */
    {
      char *nm;
      m = (Matrix *) e_data (var.u.ent);
      if (MNR (m) == 0)
      {
	/* Make the LHS same type as RHS */
	if (MTYPE (e_data (a.u.ent)) == REAL && MTYPE (m) != REAL)
	{
	  nm = cpstr (matrix_GetName (m));
	  matrix_Destroy (m);
	  m = matrix_Create (0, 0);
	  matrix_SetName (m, nm);
	  e_data (var.u.ent) = (VPTR) m;
	}
	else if (MTYPE (e_data (a.u.ent)) == COMPLEX && MTYPE (m) != COMPLEX)
	{
	  nm = cpstr (matrix_GetName (m));
	  matrix_Destroy (m);
	  m = matrix_CreateC (0, 0);
	  matrix_SetName (m, nm);
	  e_data (var.u.ent) = (VPTR) m;
	}
	else if (MTYPE (e_data (a.u.ent)) == STRING && MTYPE (m) != STRING)
	{
	  nm = cpstr (matrix_GetName (m));
	  matrix_Destroy (m);
	  m = matrix_CreateS (0, 0);
	  matrix_SetName (m, nm);
	  e_data (var.u.ent) = (VPTR) m;
	}
      }
    }
  }
  else
    error_1 ("cannot use CONSTANT in MATRIX LHS context", 0);

  switch (i_flag)
  {
  case 1:			/* Both row and column indices */
    
    /*
     * Check for scalar assign. This can speed things up a bit
     * if we are in a loop, and using the loop idices.
     */

    if (i1.type == ENTITY && e_type (i1.u.ent) == SCALAR)
    {
      if (i2.type == ENTITY && e_type (i2.u.ent) == SCALAR)
      {
	int ii1, ii2;
	ii1 = (int) SVALr (e_data (i1.u.ent));
	ii2 = (int) SVALr (e_data (i2.u.ent));

	matrix_Assign_el (m, ii1, ii2, e_data (a.u.ent));
	remove_tmp_destroy (i1.u.ent);
	remove_tmp_destroy (i2.u.ent);
	remove_tmp_destroy (a.u.ent);
	break;
      }
    }

    i1 = convert_to_matrix_d (i1);
    i2 = convert_to_matrix_d (i2);

    mrow = (Matrix *) e_data (i1.u.ent);
    mcol = (Matrix *) e_data (i2.u.ent);

    if (MTYPE (mrow) != REAL)
      error_1 (e_name (var.u.ent), "row indices must be REAL");
    if (MTYPE (mcol) != REAL)
      error_1 (e_name (var.u.ent), "column indices must be REAL");

    matrix_gt_zero (mrow);
    matrix_gt_zero (mcol);

    if (m != (Matrix *) e_data (a.u.ent))
    {
      matrix_Assign (m, mrow, mcol, e_data (a.u.ent));
    }
    else
    {
      Matrix *mtmp;
      mtmp = matrix_Copy (e_data (a.u.ent));
      matrix_Assign (m, mrow, mcol, mtmp);
      matrix_Destroy (mtmp);
    }

    remove_tmp_destroy (i1.u.ent);
    remove_tmp_destroy (i2.u.ent);
    remove_tmp_destroy (a.u.ent);

    break;

  case 2:			/* Row indices only */

    i1 = convert_to_matrix_d (i1);

    mrow = (Matrix *) e_data (i1.u.ent);

    if (MTYPE (mrow) != REAL)
      error_1 (e_name (var.u.ent), "row indices must be REAL");

    matrix_gt_zero (mrow);

    if (m != (Matrix *) e_data (a.u.ent))
    {
      matrix_AssignCol (m, mrow, e_data (a.u.ent));
    }
    else
    {
      Matrix *mtmp;
      mtmp = matrix_Copy (e_data (a.u.ent));
      matrix_AssignCol (m, mrow, mtmp);
      matrix_Destroy (mtmp);
    }

    remove_tmp_destroy (i1.u.ent);
    remove_tmp_destroy (a.u.ent);

    break;

  case 3:			/* Column indices only */

    i2 = convert_to_matrix_d (i2);

    mcol = (Matrix *) e_data (i2.u.ent);

    if (MTYPE (mcol) != REAL)
      error_1 (e_name (var.u.ent), "col indices must be REAL");

    matrix_gt_zero (mcol);

    if (m != (Matrix *) e_data (a.u.ent))
    {
      matrix_AssignRow (m, mcol, e_data (a.u.ent));
    }
    else
    {
      Matrix *mtmp;
      mtmp = matrix_Copy (e_data (a.u.ent));
      matrix_AssignRow (m, mcol, mtmp);
      matrix_Destroy (mtmp);
    }

    remove_tmp_destroy (i2.u.ent);
    remove_tmp_destroy (a.u.ent);

    break;
  }

  /* Now take care of old entity destruction if necessary */
  if (e_type (var.u.ent) != MATRIX)
  {
    remove_tmp (tmpm);
    matrix_SetName (m, cpstr (e_name (var.u.ent)));
    listNode_DestroyDataOnly (var.u.ent);
    listNode_AttachData (var.u.ent, MATRIX, m, matrix_Destroy);
  }
  return (var);
}

Datum
matrix_vector_assign (lhs, ind, rhs)
     Datum lhs, ind, rhs;
{
  ListNode *tmp;
  Matrix *m, *mi, *mrhs;

  tmp = 0;
  m = 0;			/* Initialize */

  rhs = convert_to_rhs_matrix (rhs);
  mrhs = (Matrix *) e_data (rhs.u.ent);

  /* Make sure we are assigning to a matrix */
  if (lhs.type == ENTITY)
  {
    if (e_type (lhs.u.ent) != MATRIX)
    {
      /*
       * Do not destroy the old entity yet...
       * Wait until we make it through assign details.
       * Set the lhs-type based upon the rhs-type.
       * Check for SCALAR 1st.
       */
      if (e_type (lhs.u.ent) == SCALAR)
      {
	if (SVALi (e_data (lhs.u.ent)) == 0.0)	/* REAL */
	{
	  m = matrix_Create (1, 1);
	  MAT (m, 1, 1) = SVALr (e_data (lhs.u.ent));
	}
	else
	  /* COMPLEX */
	{
	  m = matrix_CreateC (1, 1);
	  MATr (m, 1, 1) = SVALr (e_data (lhs.u.ent));
	  MATi (m, 1, 1) = SVALi (e_data (lhs.u.ent));
	}
	tmp = install_tmp (MATRIX, m, matrix_Destroy);
      }
      else if (e_type (lhs.u.ent) == STRING)
      {
	m = matrix_CreateS (1, 1);
	MATsv (m, 0) = cpstr (string_GetString (e_data (lhs.u.ent)));
	tmp = install_tmp (MATRIX, m, matrix_Destroy);
      }
      else
      {
	if (MTYPE (mrhs) == REAL)
	  tmp = install_tmp (MATRIX, m = matrix_Create (0, 0),
			     matrix_Destroy);
	else if (MTYPE (mrhs) == COMPLEX)
	  tmp = install_tmp (MATRIX, m = matrix_CreateC (0, 0),
			     matrix_Destroy);
	else if (MTYPE (mrhs) == STRING)
	  tmp = install_tmp (MATRIX, m = matrix_CreateS (0, 0),
			     matrix_Destroy);
	matrix_Zero (m);
      }
    }
    else
      /* lhs is already a MATRIX, do some checking */
    {
      char *nm;
      m = (Matrix *) e_data (lhs.u.ent);
      if (MNR (m) == 0)
      {
	/* Make the LHS same type as RHS */
	if (MTYPE (mrhs) == REAL && MTYPE (m) != REAL)
	{
	  nm = cpstr (matrix_GetName (m));
	  matrix_Destroy (m);
	  m = matrix_Create (0, 0);
	  matrix_SetName (m, nm);
	  e_data (lhs.u.ent) = (VPTR) m;
	}
	else if (MTYPE (mrhs) == COMPLEX && MTYPE (m) != COMPLEX)
	{
	  nm = cpstr (matrix_GetName (m));
	  matrix_Destroy (m);
	  m = matrix_CreateC (0, 0);
	  matrix_SetName (m, nm);
	  e_data (lhs.u.ent) = (VPTR) m;
	}
	else if (MTYPE (mrhs) == STRING && MTYPE (m) != STRING)
	{
	  nm = cpstr (matrix_GetName (m));
	  matrix_Destroy (m);
	  m = matrix_CreateS (0, 0);
	  matrix_SetName (m, nm);
	  e_data (lhs.u.ent) = (VPTR) m;
	}
      }
    }
  }
  else
    error_1 ("cannot use CONSTANT in MATRIX context", 0);

  ind = convert_to_matrix_d (ind);
  mi = (Matrix *) e_data (ind.u.ent);

  if (MTYPE (mi) != REAL)
    error_1 (e_name (lhs.u.ent), "col indices must be REAL");

  matrix_gt_zero (mi);
  if (m != mrhs)
  {
    matrix_AssignVector (m, mi, mrhs);
  }
  else
  {
    Matrix *mtmp;
    mtmp = matrix_Copy (mrhs);
    matrix_AssignVector (m, mi, mtmp);
    matrix_Destroy (mtmp);
  }

  remove_tmp_destroy (ind.u.ent);
  remove_tmp_destroy (rhs.u.ent);

  /* Now take care of old entity destruction if necessary */
  if (e_type (lhs.u.ent) != MATRIX)
  {
    remove_tmp (tmp);
    matrix_SetName (m, cpstr (e_name (lhs.u.ent)));
    listNode_DestroyDataOnly (lhs.u.ent);
    listNode_AttachData (lhs.u.ent, MATRIX, m, matrix_Destroy);
  }
  return (lhs);
}

/* **************************************************************
 * Create and return the transpose of a matrix.
 * ************************************************************** */
Datum
matrix_transpose (d)
     Datum d;
{
  Datum new;

  d = convert_to_scalar (d);

  switch (e_type (d.u.ent))
  {
  case SCALAR:
    new.u.ent = install_tmp (SCALAR,
			     scalar_CreateC (SVALr (e_data (d.u.ent)),
					     -SVALi (e_data (d.u.ent))),
			     scalar_Destroy);
    new.type = ENTITY;
    break;
  case MATRIX:
    new.u.ent = install_tmp (MATRIX, matrix_Transpose (e_data (d.u.ent)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;
  case STRING:
    {
      String *stmp;
      char *str = string_GetString (e_data (d.u.ent));
      stmp = string_Create (cpstr (str));
      new.u.ent = install_tmp (STRING, stmp, string_Destroy);
      new.type = ENTITY;
    }
    break;
  case UNDEF:
    error_1 (e_name (d.u.ent), "UNDEFINED");
    break;
  default:
    error_1 (e_name (d.u.ent), "wrong type for transpose");
    break;
  }
  remove_tmp_destroy (d.u.ent);
  return (new);
}

/*
 * Matrix non-conjugate transpose
 */

Datum
matrix_el_transpose (d)
     Datum d;
{
  Datum new;

  d = convert_to_scalar (d);

  switch (e_type (d.u.ent))
  {
  case SCALAR:
    new.u.ent = install_tmp (SCALAR,
			     scalar_CreateC (SVALr (e_data (d.u.ent)),
					     SVALi (e_data (d.u.ent))),
			     scalar_Destroy);
    new.type = ENTITY;
    break;
  case MATRIX:
    new.u.ent = install_tmp (MATRIX, matrix_ElTranspose (e_data (d.u.ent)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;
  case STRING:
    {
      String *stmp;
      char *str = string_GetString (e_data (d.u.ent));
      stmp = string_Create (cpstr (str));
      new.u.ent = install_tmp (STRING, stmp, string_Destroy);
      new.type = ENTITY;
    }
    break;
  case UNDEF:
    error_1 (e_name (d.u.ent), "UNDEFINED");
    break;
  default:
    error_1 (e_name (d.u.ent), "wrong type for non-conj transpose");
    break;
  }
  remove_tmp_destroy (d.u.ent);
  return (new);
}

/* **************************************************************
 * Special element-by-element multiply. We must also do
 * conventional el-by-el multiply, scalar*vector to make it easier
 * for users to do: `s # v' where s can be scalar or vector.
 * ************************************************************** */
Datum
element_multiply (d1, d2)
     Datum d1, d2;
{
  Datum new;
  Matrix *m;
  Scalar *s;

  d1 = convert_to_scalar (d1);
  d2 = convert_to_scalar (d2);

  switch (e_type (d1.u.ent))
  {
  case SCALAR:
    switch (e_type (d2.u.ent))
    {
    case SCALAR:
      s = scalar_Multiply (e_data (d1.u.ent), e_data (d2.u.ent));
      new.u.ent = install_tmp (SCALAR, s, scalar_Destroy);
      break;
    case MATRIX:
      m = matrix_scalar_mul (e_data (d2.u.ent), e_data (d1.u.ent));
      new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
      break;
    default:
      error_1 (e_name (d2.u.ent), "improper type for multiply");
      break;
    }
    break;
  case MATRIX:
    switch (e_type (d2.u.ent))
    {
    case SCALAR:
      m = matrix_scalar_mul (e_data (d1.u.ent), e_data (d2.u.ent));
      new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
      break;
    case MATRIX:
      m = matrix_El_Mul (e_data (d1.u.ent), e_data (d2.u.ent));
      new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
      break;
    default:
      error_1 (e_name (d2.u.ent), "improper type for el-matrix multiply");
      break;
    }
  }
  new.type = ENTITY;
  remove_tmp_destroy (d1.u.ent);
  remove_tmp_destroy (d2.u.ent);
  return (new);
}

/* **************************************************************
 * Reshape a arbitrarily sized matrix into a column matrix
 * ************************************************************** */

Datum
matrix_reshape_col (d)
     Datum d;
{
  int nrow;
  Datum new;
  Matrix *m, *mnew;

  /* Make sure the Datum holds a matrix */
  d = convert_all_to_matrix_d (d);
  m = (Matrix *) e_data (d.u.ent);

  /* Try and avoid operation */
  if (MNC (m) == 1)
  {
    new = d;
  }
  else if (MNR (m) == 0 && MNC (m) == 0)
  {
    new = d;
  }
  else
  {
    /* Now do the reshape */
    nrow = MNR (m) * MNC (m);
    mnew = matrix_Reshape (m, nrow, 1);

    /* Now prepare return value, and clean-up */
    new.u.ent = install_tmp (MATRIX, mnew, matrix_Destroy);
    new.type = ENTITY;

    remove_tmp_destroy (d.u.ent);
  }
  return (new);
}
