/* matop2.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1994  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "mem.h"
#include "symbol.h"
#include "matrix.h"
#include "complex.h"
#include "matop1.h"
#include "scop1.h"
#include "r_string.h"
#include "fi_1.h"
#include "fi_2.h"
#include "bltin.h"
#include "mathl.h"
#include "util.h"

#include <math.h>

int matrix_is_positive _PROTO ((Matrix * m));

/* **************************************************************
 * Return the REAL part of a matrix.
 * ************************************************************** */
Matrix *
matrix_Real (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i;
    Matrix *new;
    new = matrix_Create (MNR (m), MNC (m));

    matrix_screen_string (m);

    if (MTYPE (m) == REAL)
    {
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = m->val.mr[i];
    }
    else
    {
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = m->val.mc[i].r;
    }
    return (new);
  }
}

/* **************************************************************
 * Return the IMAGINARY part of a matrix.
 * ************************************************************** */
Matrix *
matrix_Imag (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i;
    Matrix *new;
    new = matrix_Create (MNR (m), MNC (m));

    matrix_screen_string (m);

    if (MTYPE (m) == REAL)
    {
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = 0.0;
    }
    else
    {
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = m->val.mc[i].i;
    }
    return (new);
  }
}

Matrix *
matrix_Conj (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i;
    Matrix *new;

    matrix_screen_string (m);

    if (MTYPE (m) == REAL)
      new = matrix_Copy (m);
    else
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	new->val.mc[i].r = m->val.mc[i].r;
	new->val.mc[i].i = -m->val.mc[i].i;
      }
    }
    return (new);
  }
}

Matrix *
matrix_Sin (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i;
    Matrix *new;

    matrix_screen_string (m);

    if (MTYPE (m) == REAL)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = errcheck (sin (m->val.mr[i]), "sin");
    }
    else
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mc[i] = complex_Sin (m->val.mc[i]);
    }
    return (new);
  }
}

Matrix *
matrix_Cos (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i;
    Matrix *new;

    matrix_screen_string (m);

    if (MTYPE (m) == REAL)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = errcheck (cos (m->val.mr[i]), "cos");
    }
    else
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mc[i] = complex_Cos (m->val.mc[i]);
    }
    return (new);
  }
}

Matrix *
matrix_Tan (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i;
    Matrix *new;

    matrix_screen_string (m);

    if (MTYPE (m) == REAL)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = errcheck (tan (m->val.mr[i]), "tan");
    }
    else
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mc[i] = complex_Tan (m->val.mc[i]);
    }
    return (new);
  }
}

extern do_cmplx_1 _PROTO ((Matrix * m));

Matrix *
matrix_Asin (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i;
    int dc = 0;
    Matrix *new = 0;

    matrix_screen_string (m);

    if (MTYPE (m) == REAL && !(dc = do_cmplx_1 (m)))
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = errcheck (asin (m->val.mr[i]), "asin");
    }
    else if (MTYPE (m) == REAL && dc)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mc[i] = complex_asin (m->val.mr[i], 0.0);
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mc[i] = complex_Asin (m->val.mc[i]);
    }
    return (new);
  }
}

Matrix *
matrix_Acos (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i;
    int dc = 0;
    Matrix *new = 0;

    matrix_screen_string (m);

    if (MTYPE (m) == REAL && !(dc = do_cmplx_1 (m)))
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = errcheck (acos (m->val.mr[i]), "acos");
    }
    else if (MTYPE (m) == REAL && dc)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mc[i] = complex_acos (m->val.mr[i], 0.0);
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mc[i] = complex_Acos (m->val.mc[i]);
    }
    return (new);
  }
}

Matrix *
matrix_Atan (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i;
    Matrix *new;

    matrix_screen_string (m);

    if (MTYPE (m) == REAL)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = errcheck (atan (m->val.mr[i]), "atan");
    }
    else
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mc[i] = complex_Atan (m->val.mc[i]);
    }
    return (new);
  }
}

Matrix *
matrix_Atan2 (m1, m2)
     Matrix *m1, *m2;
{
  ASSERT (m1);
  ASSERT (m2);
  {
    register int i;
    Matrix *new = 0;

    matrix_screen_string (m1);
    matrix_screen_string (m2);

    if (MTYPE (m1) == REAL && MTYPE (m2) == REAL)
    {
      if (MNR (m1) != MNR (m2) || MNC (m1) != MNC (m2))
      {
	if (MNR (m1) == 1 && MNC (m1) == 1)
	{
	  new = matrix_Create (MNR (m2), MNC (m2));
	  for (i = 0; i < MNR (new) * MNC (new); i++)
	    new->val.mr[i] = errcheck (atan2 (m1->val.mr[0],
					      m2->val.mr[i]),
				       "atan2");
	}
	else if (MNR (m2) == 1 && MNC (m2) == 1)
	{
	  new = matrix_Create (MNR (m1), MNC (m1));
	  for (i = 0; i < MNR (new) * MNC (new); i++)
	    new->val.mr[i] = errcheck (atan2 (m1->val.mr[i],
					      m2->val.mr[0]),
				       "atan2");
	}
	else
	  error_2 (matrix_GetName (m1), matrix_GetName (m2),
		   "atan2: matrix args must be same size");
      }
      else
      {
	new = matrix_Create (MNR (m1), MNC (m1));
	for (i = 0; i < MNR (m1) * MNC (m1); i++)
	  new->val.mr[i] = errcheck (atan2 (m1->val.mr[i],
					    m2->val.mr[i]),
				     "atan2");
      }
    }
    else
    {
      error_1 ("atan2: complex matrices not allowed", 0);
    }
    return (new);
  }
}

Matrix *
matrix_Sign (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i, size;
    Matrix *new;

    matrix_screen_string (m);
    size = MNR (m) * MNC (m);

    if (MTYPE (m) == REAL)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	if (MATrv (m, i) > 0)
	  MATrv (new, i) = 1.0;
	else if (MATrv (m, i) < 0)
	  MATrv (new, i) = -1.0;
	else
	  MATrv (new, i) = 0.0;
      }
    }
    else
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	MATcv (new, i) = complex_div (MATcvr (m, i), MATcvi (m, i),
				      complex_Abs (MATcv (m, i)), 0.0);
      }
    }
    return (new);
  }
}

Matrix *
matrix_Max (m)
     Matrix *m;
{
  ASSERT (m);
  {
    double maxr;
    Complex maxc;
    Matrix *mmax = 0;
    int i, j;

    matrix_screen_string (m);

    if (MTYPE (m) == REAL)
    {
      if (MNR (m) == 1)
      {
	maxr = MAT0 (m, 0, 0);
	for (i = 1; i < MNC (m); i++)
	{
	  if (MATrv (m, i) > maxr)
	    maxr = MATrv (m, i);
	}
	mmax = matrix_Create (1, 1);
	MAT0 (mmax, 0, 0) = maxr;
      }
      else
      {
	mmax = matrix_Create (1, MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  maxr = MAT0 (m, 0, i);
	  for (j = 1; j < MNR (m); j++)
	    if (MAT0 (m, j, i) > maxr)
	      maxr = MAT0 (m, j, i);
	  MAT0 (mmax, 0, i) = maxr;
	}
      }
    }
    else if (MTYPE (m) == COMPLEX)
    {
      if (MNR (m) == 1)
      {
	maxc.r = MATcvr (m, 0);
	maxc.i = MATcvi (m, 0);
	for (i = 1; i < MNC (m); i++)
	{
	  if (complex_Abs (MATcv (m, i)) > complex_Abs (maxc))
	    maxc = MATcv (m, i);
	}
	mmax = matrix_CreateC (1, 1);
	MATc0 (mmax, 0, 0) = maxc;
      }
      else
      {
	mmax = matrix_CreateC (1, MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  maxc.r = MATr0 (m, 0, i);
	  maxc.i = MATi0 (m, 0, i);
	  for (j = 1; j < MNR (m); j++)
	    if (complex_Abs (MATc0 (m, j, i)) > complex_Abs (maxc))
	      maxc = MATc0 (m, j, i);
	  MATc0 (mmax, 0, i) = maxc;
	}
      }
    }
    else
      error_1 (matrix_GetName (m), "type not supported");

    return (mmax);
  }
}

Matrix *
matrix_Maxi (m)
     Matrix *m;
{
  ASSERT (m);
  {
    double maxr;
    Complex maxc;
    Matrix *mmax = 0;
    int i, j;
    int ind, size;

    matrix_screen_string (m);
    ind = 0;
    size = MNR (m) * MNC (m);

    if (MTYPE (m) == REAL)
    {
      if (MNR (m) == 1)
      {
	maxr = MAT0 (m, 0, 0);
	ind = 1;
	for (i = 1; i < MNC (m); i++)
	{
	  if (MATrv (m, i) > maxr)
	  {
	    maxr = MATrv (m, i);
	    ind = i + 1;
	  }
	}
	mmax = matrix_Create (1, 1);
	MAT0 (mmax, 0, 0) = (double) ind;
      }
      else
      {
	mmax = matrix_Create (1, MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  maxr = MAT0 (m, 0, i);
	  ind = 1;
	  for (j = 1; j < MNR (m); j++)
	    if (MAT0 (m, j, i) > maxr)
	    {
	      maxr = MAT0 (m, j, i);
	      ind = j + 1;
	    }
	  MAT0 (mmax, 0, i) = (double) ind;
	}
      }
    }
    else if (MTYPE (m) == COMPLEX)
    {
      if (MNR (m) == 1)
      {
	maxc.r = MATcvr (m, 0);
	maxc.i = MATcvi (m, 0);
	ind = 1;
	for (i = 1; i < MNC (m); i++)
	{
	  if (complex_Abs (MATcv (m, i)) > complex_Abs (maxc))
	  {
	    maxc = MATcv (m, i);
	    ind = i + 1;
	  }
	}
	mmax = matrix_Create (1, 1);
	MAT0 (mmax, 0, 0) = (double) ind;
      }
      else
      {
	mmax = matrix_Create (1, MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  maxc.r = MATr0 (m, 0, i);
	  maxc.i = MATi0 (m, 0, i);
	  ind = 1;
	  for (j = 1; j < MNR (m); j++)
	    if (complex_Abs (MATc0 (m, j, i)) > complex_Abs (maxc))
	    {
	      maxc = MATc0 (m, j, i);
	      ind = j + 1;
	    }
	  MAT0 (mmax, 0, i) = (double) ind;
	}
      }
    }
    else
      error_1 (matrix_GetName (m), "type not supported");

    return (mmax);
  }
}

Matrix *
matrix_Min (m)
     Matrix *m;
{
  ASSERT (m);
  {
    double maxr;
    Complex maxc;
    Matrix *mmax = 0;
    int i, j;

    matrix_screen_string (m);

    if (MTYPE (m) == REAL)
    {
      if (MNR (m) == 1)
      {
	maxr = MAT0 (m, 0, 0);
	for (i = 1; i < MNC (m); i++)
	{
	  if (MATrv (m, i) < maxr)
	    maxr = MATrv (m, i);
	}
	mmax = matrix_Create (1, 1);
	MAT0 (mmax, 0, 0) = maxr;
      }
      else
      {
	mmax = matrix_Create (1, MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  maxr = MAT0 (m, 0, i);
	  for (j = 1; j < MNR (m); j++)
	    if (MAT0 (m, j, i) < maxr)
	      maxr = MAT0 (m, j, i);
	  MAT0 (mmax, 0, i) = maxr;
	}
      }
    }
    else if (MTYPE (m) == COMPLEX)
    {
      if (MNR (m) == 1)
      {
	maxc.r = MATcvr (m, 0);
	maxc.i = MATcvi (m, 0);
	for (i = 1; i < MNC (m); i++)
	{
	  if (complex_Abs (MATcv (m, i)) < complex_Abs (maxc))
	    maxc = MATcv (m, i);
	}
	mmax = matrix_CreateC (1, 1);
	MATc0 (mmax, 0, 0) = maxc;
      }
      else
      {
	mmax = matrix_CreateC (1, MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  maxc.r = MATr0 (m, 0, i);
	  maxc.i = MATi0 (m, 0, i);
	  for (j = 1; j < MNR (m); j++)
	    if (complex_Abs (MATc0 (m, j, i)) < complex_Abs (maxc))
	      maxc = MATc0 (m, j, i);
	  MATc0 (mmax, 0, i) = maxc;
	}
      }
    }
    else
      error_1 (matrix_GetName (m), "type not supported");

    return (mmax);
  }
}

Matrix *
matrix_Mini (m)
     Matrix *m;
{
  ASSERT (m);
  {
    double maxr;
    Complex maxc;
    Matrix *mmax = 0;
    int i, j;
    int ind, size;

    matrix_screen_string (m);
    ind = 0;
    size = MNR (m) * MNC (m);

    if (MTYPE (m) == REAL)
    {
      if (MNR (m) == 1)
      {
	maxr = MAT0 (m, 0, 0);
	ind = 1;
	for (i = 1; i < MNC (m); i++)
	{
	  if (MATrv (m, i) < maxr)
	  {
	    maxr = MATrv (m, i);
	    ind = i + 1;
	  }
	}
	mmax = matrix_Create (1, 1);
	MAT0 (mmax, 0, 0) = (double) ind;
      }
      else
      {
	mmax = matrix_Create (1, MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  maxr = MAT0 (m, 0, i);
	  ind = 1;
	  for (j = 1; j < MNR (m); j++)
	    if (MAT0 (m, j, i) < maxr)
	    {
	      maxr = MAT0 (m, j, i);
	      ind = j + 1;
	    }
	  MAT0 (mmax, 0, i) = (double) ind;
	}
      }
    }
    else if (MTYPE (m) == COMPLEX)
    {
      if (MNR (m) == 1)
      {
	maxc.r = MATcvr (m, 0);
	maxc.i = MATcvi (m, 0);
	ind = 1;
	for (i = 1; i < MNC (m); i++)
	{
	  if (complex_Abs (MATcv (m, i)) < complex_Abs (maxc))
	  {
	    maxc = MATcv (m, i);
	    ind = i + 1;
	  }
	}
	mmax = matrix_Create (1, 1);
	MAT0 (mmax, 0, 0) = (double) ind;
      }
      else
      {
	mmax = matrix_Create (1, MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  maxc.r = MATr0 (m, 0, i);
	  maxc.i = MATi0 (m, 0, i);
	  ind = 1;
	  for (j = 1; j < MNR (m); j++)
	    if (complex_Abs (MATc0 (m, j, i)) < complex_Abs (maxc))
	    {
	      maxc = MATc0 (m, j, i);
	      ind = j + 1;
	    }
	  MAT0 (mmax, 0, i) = (double) ind;
	}
      }
    }
    else
      error_1 (matrix_GetName (m), "type not supported");

    return (mmax);
  }
}

/* **************************************************************
 * Functions for element-wise matrix powers.
 * ************************************************************** */

/*
 * MATRIX .^ SCALAR
 */

Matrix *
matrix_Pow1 (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new = 0;

    matrix_screen_string (m);

    /* Zero power */
    if (SVALr (s) == 0.0 && SVALi (s) == 0.0)
    {
      new = matrix_Create (MNR (m), MNC (m));
      /* Fill with ones */
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = 1.0;
    }

    else if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* integer power */
      if (floor (SVALr (s)) == SVALr (s))
      {
	new = matrix_Create (MNR (m), MNC (m));
	for (i = 0; i < MNR (m) * MNC (m); i++)
	  MATrv (new, i) = errcheck (pow (MATrv (m, i), SVALr (s)), "pow");
      }

      /* non-integer power, positive operand */
      else if (matrix_is_positive (m))
      {
	new = matrix_Create (MNR (m), MNC (m));
	for (i = 0; i < MNR (m) * MNC (m); i++)
	  MATrv (new, i) = errcheck (pow (MATrv (m, i), SVALr (s)), "pow");
      }

      /* non-integer power, some negative operand elements */
      else
      {
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m) * MNC (m); i++)
	  MATcv (new, i) = complex_pow (MATrv (m, i), 0.0, SVALr (s), SVALi (s));
      }
    }
    else if (MTYPE (m) == COMPLEX || SVALi (s) != 0.0)
    {
      if (MTYPE (m) == REAL)
      {
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m) * MNC (m); i++)
	  MATcv (new, i) = complex_pow (MATrv (m, i), 0.0, SVALr (s), SVALi (s));
      }
      else if (MTYPE (m) == COMPLEX)
      {
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m) * MNC (m); i++)
	  MATcv (new, i) = complex_pow (MATcvr (m, i), MATcvi (m, i), SVALr (s), SVALi (s));
      }
    }

    return (new);
  }
}

/*
 * SCALAR .^ MATRIX
 */

Matrix *
matrix_Pow2 (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    int i, size;
    int type = 1;
    Matrix *new = 0;

    matrix_screen_string (m);
    size = MNR (m) * MNC (m);

    /* We must do some checking 1st */
    if (SVALr (s) > 0.0)
    {
      /* Positive operand, use real power function */
      type = 0;
    }
    else
    {
      /* Now check for integer powers */
      type = 0;
      for (i = 0; i < size; i++)
      {
	if (floor (MATrv (m, i)) != MATrv (m, i))
	{
	  type = 1;
	  break;
	}
      }
    }

    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      if (type == 0)
      {
	new = matrix_Create (MNR (m), MNC (m));
	for (i = 0; i < MNR (m) * MNC (m); i++)
	  MATrv (new, i) = errcheck (pow (SVALr (s), MATrv (m, i)), "pow");
      }
      else
      {
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m) * MNC (m); i++)
	  MATcv (new, i) = complex_pow (SVALr (s), 0.0, MATrv (m, i), 0.0);
      }
    }
    else if (MTYPE (m) == COMPLEX || SVALi (s) != 0.0)
    {
      if (MTYPE (m) == REAL)
      {
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m) * MNC (m); i++)
	  MATcv (new, i) = complex_pow (SVALr (s), SVALi (s),
					MATrv (m, i), 0.0);
      }
      else
      {
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m) * MNC (m); i++)
	  MATcv (new, i) = complex_pow (SVALr (s), SVALi (s),
					MATcvr (m, i), MATcvi (m, i));
      }
    }

    return (new);
  }
}

/*
 * m1 .^ m2
 */

Matrix *
matrix_matrix_el_pow (m1, m2)
     Matrix *m1, *m2;
{
  ASSERT (m1);
  ASSERT (m2);
  {
    int i;
    int type = 0;
    Matrix *new = 0;

    matrix_screen_string (m1);
    matrix_screen_string (m2);

    /* Check sizes 1st. */

    /* Special case: 1-by-1 .^ Matrix */
    if (MNR (m1) == 1 && MNC (m1) == 1)
    {
      Scalar *stmp = 0;
      if (MTYPE (m1) == REAL)
	stmp = scalar_Create (MAT (m1, 1, 1));
      else if (MTYPE (m1) == COMPLEX)
	stmp = scalar_CreateC (MATr (m1, 1, 1), MATi (m1, 1, 1));
      new = matrix_Pow2 (m2, stmp);
      scalar_Destroy (stmp);
      return (new);
    }

    /* Special case: Matrix .^ 1-by-1 */
    if (MNR (m2) == 1 && MNC (m2) == 1)
    {
      Scalar *stmp = 0;
      if (MTYPE (m2) == REAL)
	stmp = scalar_Create (MAT (m2, 1, 1));
      else if (MTYPE (m2) == COMPLEX)
	stmp = scalar_CreateC (MATr (m2, 1, 1), MATi (m2, 1, 1));
      new = matrix_Pow1 (m1, stmp);
      scalar_Destroy (stmp);
      return (new);
    }

    if (MNR (m1) != MNR (m2) || MNC (m1) != MNC (m2))
      error_2 (matrix_GetName (m1), matrix_GetName (m2),
	       "row and column dimensions must match for .^ operation");

    if (MTYPE (m1) == REAL && MTYPE (m2) == REAL)
    {
      /* Do some operand checking */
      for (i = 0; i < MNR (m1) * MNC (m1); i++)
      {
	if (floor (MATrv (m2, i)) != MATrv (m2, i))
	{
	  type = 1;
	  break;
	}
      }
      if (type == 0)
      {
	if (!matrix_is_positive (m1))
	  type = 1;
      }

      if (type == 0)
      {
	new = matrix_Create (MNR (m1), MNC (m1));
	for (i = 0; i < MNR (m1) * MNC (m1); i++)
	  MATrv (new, i) = errcheck (pow (MATrv (m1, i), MATrv (m2, i)), "pow");
      }
      else
      {
	new = matrix_CreateC (MNR (m1), MNC (m1));
	for (i = 0; i < MNR (m1) * MNC (m1); i++)
	  MATcv (new, i) = complex_pow (MATrv (m1, i), 0.0,
					MATrv (m2, i), 0.0);
      }
    }
    else if (MTYPE (m1) == COMPLEX || MTYPE (m2) == COMPLEX)
    {
      if (MTYPE (m1) == REAL && MTYPE (m2) == COMPLEX)
      {
	new = matrix_CreateC (MNR (m1), MNC (m1));
	for (i = 0; i < MNR (m1) * MNC (m1); i++)
	  MATcv (new, i) = complex_pow (MATrv (m1, i), 0.0,
					MATcvr (m2, i), MATcvi (m2, i));
      }
      else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == REAL)
      {
	new = matrix_CreateC (MNR (m1), MNC (m1));
	for (i = 0; i < MNR (m1) * MNC (m1); i++)
	  MATcv (new, i) = complex_pow (MATcvr (m1, i), MATcvi (m1, i),
					MATrv (m2, i), 0.0);
      }
      else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == COMPLEX)
      {
	new = matrix_CreateC (MNR (m1), MNC (m1));
	for (i = 0; i < MNR (m1) * MNC (m1); i++)
	  MATcv (new, i) = complex_pow (MATcvr (m1, i), MATcvi (m1, i),
					MATcvr (m2, i), MATcvi (m2, i));
      }
    }

    return (new);
  }
}

/*
 * Read a "generic" matrix from fn.
 * We cannot read in COMPLEX or STRING matrices
 * cause there is no way of distinguish them
 * from REAL matrices.
 */

static char *endp;

Matrix *
matrix_ReadGeneric (fn, block_size)
     FILE *fn;
     int block_size;
{
  ASSERT (fn);
  {
    int j, nrow, ncol, rv;
    ListNode *tmp;
    Matrix *new;

    double dnum;
    int c, nchar;
    char num[256];

    /*
     * Read the 1st line, counting number of columns.
     */

    nchar = 0;
    ncol = 0;
    tmp = install_tmp (MATRIX, new = matrix_Create (0, 0),
		       matrix_Destroy);

    while ((c = fgetc (fn)) != EOF)
    {
      if (c == ' ' && nchar != 0)
      {
	num[nchar] = '\0';
	ncol++;
	matrix_AppendColR (new, 1);
	MAT (new, 1, ncol) = strtod (num, (char **) &endp);
	nchar = 0;
      }
      else if (c == ' ' && nchar == 0)
      {
	continue;
      }
      else if (c == '\n' && nchar != 0)
      {
	num[nchar] = '\0';
	ncol++;
	matrix_AppendColR (new, 1);
	MAT (new, 1, ncol) = strtod (num, (char **) &endp);
	nchar = 0;
	break;
      }
      else if (c == '\n' && nchar == 0)
      {
	if (ncol == 0)
	  continue;
	break;
      }
      else
      {
	num[nchar] = (char) c;
	nchar++;
      }
    }

    /*
     * Check for and handle A 1-liner without a newline.
     */

    if (c == EOF)
    {
      if (nchar != 0)
      {
	ncol++;
	matrix_AppendColR (new, 1);
	MAT (new, 1, ncol) = strtod (num, (char **) &endp);
      }
      matrix_Truncate (new, 1, ncol);
      remove_tmp (tmp);
      return (new);
    }

    /*
     * Now read each new row, one at a time.
     * Play some tricks to try and make this faster.
     * Use a heuristic to grow the matrix in blocks.
     * Let the user override.
     */

    if (!block_size)
    {
      if (ncol < 3)
	block_size = 1000 * ncol;	/* .072 mega-bytes */
      else if (ncol < 10)
	block_size = 100 * ncol;	/* .080 mega-bytes */
      else if (ncol < 50)
	block_size = 5 * ncol;	        /* .100 mega-bytes */
      else if (ncol < 100)
	block_size = 2 * ncol;	        /* .160 mega-bytes */
      else
	block_size = 1;
    }

    matrix_AppendRowR (new, block_size);

    j = 0;
    nrow = 2;
    while ((rv = fscanf (fn, "%lf", &dnum)) != EOF && rv != 0)
    {
      j++;
      if (j == 1)
      {
	/* Add new row(s) */
	if (nrow > MNR (new))
	  matrix_AppendRowR (new, 2 * MNR (new));

	MAT (new, nrow, j) = dnum;
	if (j == ncol)
	{
	  j = 0;
	  nrow++;
	}
      }
      else if (j == ncol)
      {
	/* Add LAST element */
	MAT (new, nrow, j) = dnum;
	j = 0;
	nrow++;
      }
      else
      {
	/* Add element */
	MAT (new, nrow, j) = dnum;
      }
    }

    /* Check rv. */
    if (rv == 0)
      warning_1 ("readm: premature end", 0);
    
    /*
     * Shrink matrix down to nrow-by-ncol
     */

    matrix_Truncate (new, nrow - 1, ncol);
    remove_tmp (tmp);
    return (new);
  }
}

/*
 * Write out a matrix in a generic sort of format
 * Handles all types of matrices.
 */

void
matrix_WriteGeneric (m, fn)
     Matrix *m;
     FILE *fn;
{
  ASSERT (m);
  ASSERT (fn);
  {
    int i, j, fwidth, fprec;

    fwidth = get_fwidth ();
    fprec = get_fprec ();

    if (MTYPE (m) == REAL)
    {
      for (i = 0; i < MNR (m); i++)
      {
	for (j = 0; j < MNC (m); j++)
	{
	  fprintf (fn, " %*.*g", fwidth, fprec, MAT0 (m, i, j));
	}
	fprintf (fn, "\n");
      }
    }
    else if (MTYPE (m) == COMPLEX)
    {
      for (i = 0; i < MNR (m); i++)
      {
	for (j = 0; j < MNC (m); j++)
	{
	  fprintf (fn, " %*.*g %*.*g", fwidth, fprec, MATr0 (m, i, j),
		   fwidth, fprec, MATi0 (m, i, j));
	}
	fprintf (fn, "\n");
      }
    }
    else if (MTYPE (m) == STRING)
    {
      for (i = 0; i < MNR (m); i++)
      {
	for (j = 0; j < MNC (m); j++)
	{
	  fprintf (fn, " %s", MATs0 (m, i, j));
	}
	fprintf (fn, "\n");
      }
    }
  }
}

/* **************************************************************
 * Create a COMPLEX MATRIX, and load it with the contents of vr
 * and vi.
 * ************************************************************** */
Matrix *
matrix_CreateCLoad (mr, mi)
     Matrix *mr, *mi;
{
  ASSERT (mr);
  ASSERT (mi);
  {
    int i;
    Matrix *new;

    /* Checks */
    if (MNR (mr) != MNR (mi))
      error_2 (matrix_GetName (mr), matrix_GetName (mi),
	       "incompatible sizes for matrix creation");
    if (MTYPE (mr) != REAL && MTYPE (mi) != REAL)
      error_2 (matrix_GetName (mr), matrix_GetName (mi),
	       "args must be REAL matrices");

    new = matrix_CreateC (MNR (mr), MNC (mr));
    for (i = 0; i < MNR (mr) * MNC (mr); i++)
    {
      MATcvr (new, i) = MATrv (mr, i);
      MATcvi (new, i) = MATrv (mi, i);
    }
    return (new);
  }
}

/*
 * m1 / m2 or B / A
 * same as B*inv(A)
 */

Matrix *
matrix_Rdivide (m1, m2)
     Matrix *m1, *m2;
{
  Matrix *new, *tnew, *t1, *t2;

  matrix_screen_string (m1);
  matrix_screen_string (m2);

  /*
   * Check for special case where denominator is a scalar.
   * The only thing we gain if m2 is a scalar, is speed.
   */
  if (MNR (m2) == 1 && MNC (m2) == 1)
  {
    Scalar *stmp;
    if (MTYPE (m2) == REAL)
      stmp = scalar_Create (MAT (m2, 1, 1));
    else
      stmp = scalar_CreateC (MATr (m2, 1, 1), MATi (m2, 1, 1));

    new = matrix_scalar_div1 (m1, stmp);
    scalar_Destroy (stmp);
    return (new);
  }
  else if (MNR (m1) == 0 || MNR (m2) == 0)
  {
    new = matrix_Create (0, 0);
    return (new);
  }

  t1 = matrix_Transpose (m1);
  t2 = matrix_Transpose (m2);

  /* Check row dimensions */
  if (MNR (t1) != MNR (t2))
  {
    matrix_Destroy (t1);
    matrix_Destroy (t2);
    error_2 (matrix_GetName (m1), matrix_GetName (m2),
	     "column dimensions must agree");
  }

  if (MNR (t2) == MNC (t2))
    tnew = solve_eq (t2, t1);
  else
    tnew = matrix_LS (t2, t1);

  new = matrix_Transpose (tnew);
  matrix_Destroy (t1);
  matrix_Destroy (t2);
  matrix_Destroy (tnew);

  return (new);
}

/*
 * m1 \ m2 or A \ B
 * Ax = B
 */
Matrix *
matrix_Ldivide (m1, m2)
     Matrix *m1, *m2;
{
  Matrix *new;

  matrix_screen_string (m1);
  matrix_screen_string (m2);

  /*
   * Check for special case where denominator is a scalar.
   * The only thing we gain if m2 is a scalar, is speed.
   */
  if (MNR (m1) == 1 && MNC (m1) == 1)
  {
    Scalar *stmp;
    if (MTYPE (m1) == REAL)
      stmp = scalar_Create (MAT (m1, 1, 1));
    else
      stmp = scalar_CreateC (MATr (m1, 1, 1), MATi (m1, 1, 1));

    new = matrix_scalar_div1 (m2, stmp);
    scalar_Destroy (stmp);
    return (new);
  }
  else if (MNR (m1) == 0 || MNR (m2) == 0)
  {
    new = matrix_Create (0, 0);
    return (new);
  }

  /* Check dimensions */
  if (MNR (m1) != MNR (m2))
    error_2 (matrix_GetName (m1), matrix_GetName (m2),
	     "RHS row dim. must match LHS row dim.");

  if (MNR (m1) == MNC (m1))
    new = solve_eq (m1, m2);
  else
    new = matrix_LS (m1, m2);

  return (new);
}

/* **************************************************************
 * Check that ALL of the matrix elements are >= 1.
 * ************************************************************** */
void
matrix_gt_zero (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i;

    matrix_screen_string (m);

    for (i = 0; i < MNR (m); i++)
    {
      if (MATrv (m, i) <= 0)
	error_1 (matrix_GetName (m), "Index less than 1 not allowed");
    }
  }
}

Matrix *
matrix_Find (m)
     Matrix *m;
{
  register int i, j = 0;
  Matrix *mfind;

  /* Make mfind as big as it could be, reduce later */
  mfind = matrix_Create (1, MNR (m) * MNC (m));

  if (MTYPE (m) == REAL)
  {
    j = 0;
    for (i = 0; i < MNR (m) * MNC (m); i++)
    {
      if (MATrv (m, i) != 0.0)
	MATrv (mfind, j++) = i + 1;
    }
  }
  else if (MTYPE (m) == COMPLEX)
  {
    j = 0;
    for (i = 0; i < MNR (m) * MNC (m); i++)
    {
      if (MATcvr (m, i) != 0.0 || MATcvi (m, i) != 0.0)
	MATrv (mfind, j++) = i + 1;
    }
  }
  else if (MTYPE (m) == STRING)
  {
    j = 0;
    for (i = 0; i < MNR (m) * MNC (m); i++)
    {
      if (MATsv (m, i) != 0 && strncmp (MATsv (m, i), "\0", 1))
	MATrv (mfind, j++) = i + 1;
    }
  }
  /* Now reduce mfind to correct size */
  if (j == 0)
  {
    /* No reduction here, just make dimensions correct */
    mfind->nrow = 0;
    mfind->ncol = 0;
  }
  else
  {
    mfind->val.mr = (double *) REALLOC (mfind->val.mr, sizeof (double) * j);
    mfind->ncol = j;
  }

  return (mfind);
}

Matrix *
matrix_Mod (m1, m2)
     Matrix *m1, *m2;
{
  ASSERT (m1);
  ASSERT (m2);
  {
    register int i;
    Matrix *m = 0;

    matrix_screen_string (m1);
    matrix_screen_string (m2);

    if (MNR (m1) != MNR (m2) || MNC (m1) != MNC (m2))
    {
      if (MNR (m1) == 1 && MNC (m1) == 1)	/* SCALAR / MATRIX */
      {
	if (MTYPE (m1) == REAL && MTYPE (m2) == REAL)
	{
	  m = matrix_Create (MNR (m2), MNC (m2));
	  for (i = 0; i < MNR (m) * MNC (m); i++)
	    MATrv (m, i) = errcheck (fmod (MATrv (m1, 0), MATrv (m2, i)), "fmod");
	}
	else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == REAL)
	{
	  m = matrix_CreateC (MNR (m2), MNC (m2));
	  for (i = 0; i < MNR (m) * MNC (m); i++)
	    MATcv (m, i) = complex_mod (MATcvr (m1, 0), MATcvi (m1, 0),
					MATrv (m2, i), 0.0);
	}
	else if (MTYPE (m1) == REAL && MTYPE (m2) == COMPLEX)
	{
	  m = matrix_CreateC (MNR (m2), MNC (m2));
	  for (i = 0; i < MNR (m) * MNC (m); i++)
	    MATcv (m, i) = complex_mod (MATrv (m1, 0), 0.0,
					MATcvr (m2, i), MATcvi (m2, i));
	}
	else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == COMPLEX)
	{
	  m = matrix_CreateC (MNR (m2), MNC (m2));
	  for (i = 0; i < MNR (m) * MNC (m); i++)
	    MATcv (m, i) = complex_Mod (MATcv (m1, 0), MATcv (m2, i));
	}
      }
      else if (MNR (m2) == 1 && MNC (m2) == 1)	/* MATRIX / SCALAR */
      {
	if (MTYPE (m1) == REAL && MTYPE (m2) == REAL)
	{
	  m = matrix_Create (MNR (m1), MNC (m1));
	  for (i = 0; i < MNR (m) * MNC (m); i++)
	    MATrv (m, i) = errcheck (fmod (MATrv (m1, i), MATrv (m2, 0)), "fmod");
	}
	else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == REAL)
	{
	  m = matrix_CreateC (MNR (m1), MNC (m1));
	  for (i = 0; i < MNR (m) * MNC (m); i++)
	    MATcv (m, i) = complex_mod (MATcvr (m1, i), MATcvi (m1, i),
					MATrv (m2, 0), 0.0);
	}
	else if (MTYPE (m1) == REAL && MTYPE (m2) == COMPLEX)
	{
	  m = matrix_CreateC (MNR (m1), MNC (m1));
	  for (i = 0; i < MNR (m) * MNC (m); i++)
	    MATcv (m, i) = complex_mod (MATrv (m1, i), 0.0,
					MATcvr (m2, 0), MATcvi (m2, 0));
	}
	else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == COMPLEX)
	{
	  m = matrix_CreateC (MNR (m1), MNC (m1));
	  for (i = 0; i < MNR (m) * MNC (m); i++)
	    MATcv (m, i) = complex_Mod (MATcv (m1, i), MATcv (m2, 0));
	}
      }
      else
	error_2 (matrix_GetName (m1), matrix_GetName (m2),
		 "mod: matrices must have same dimension");
    }
    else
    {
      if (MTYPE (m1) == REAL && MTYPE (m2) == REAL)
      {
	m = matrix_Create (MNR (m1), MNC (m1));
	for (i = 0; i < MNR (m) * MNC (m); i++)
	  MATrv (m, i) = errcheck (fmod (MATrv (m1, i), MATrv (m2, i)), "fmod");
      }
      else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == REAL)
      {
	m = matrix_CreateC (MNR (m1), MNC (m1));
	for (i = 0; i < MNR (m) * MNC (m); i++)
	  MATcv (m, i) = complex_mod (MATcvr (m1, i), MATcvi (m1, i),
				      MATrv (m2, i), 0.0);
      }
      else if (MTYPE (m1) == REAL && MTYPE (m2) == COMPLEX)
      {
	m = matrix_CreateC (MNR (m1), MNC (m1));
	for (i = 0; i < MNR (m) * MNC (m); i++)
	  MATcv (m, i) = complex_mod (MATrv (m1, i), 0.0, MATcvr (m2, i),
				      MATcvi (m2, i));
      }
      else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == COMPLEX)
      {
	m = matrix_CreateC (MNR (m1), MNC (m1));
	for (i = 0; i < MNR (m) * MNC (m); i++)
	  MATcv (m, i) = complex_Mod (MATcv (m1, i), MATcv (m2, i));
      }
    }
    return (m);
  }
}

VPTR
matrix_Sum (m)
     Matrix *m;
{
  ASSERT (m);
  {
    int i, j;
    VPTR retval = 0;

    if (MTYPE (m) == REAL)
    {
      if (MNR (m) == 1)
      {
	double d = 0.0;
	for (i = 0; i < MNC (m); i++)
	  d = d + MATrv (m, i);
	retval = (VPTR) scalar_Create (d);
      }
      else
      {
	Matrix *new = matrix_Create (1, MNC (m));
	matrix_Zero (new);
	for (i = 0; i < MNC (m); i++)
	  for (j = 0; j < MNR (m); j++)
	    MATrv (new, i) = MATrv (new, i) + MAT0 (m, j, i);
	retval = (VPTR) new;
      }
    }
    else if (MTYPE (m) == COMPLEX)
    {
      if (MNR (m) == 1)
      {
	Complex c;
	c.r = 0.0;
	c.i = 0.0;
	for (i = 0; i < MNC (m); i++)
	{
	  c.r = c.r + MATcvr (m, i);
	  c.i = c.i + MATcvi (m, i);
	}
	retval = (VPTR) scalar_CreateC (c.r, c.i);
      }
      else
      {
	Matrix *new = matrix_CreateC (1, MNC (m));
	matrix_Zero (new);
	for (i = 0; i < MNC (m); i++)
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATcvr (new, i) = MATcvr (new, i) + MATr0 (m, j, i);
	    MATcvi (new, i) = MATcvi (new, i) + MATi0 (m, j, i);
	  }
	retval = (VPTR) new;
      }
    }
    else
      error_1 (matrix_GetName (m), "invalid type matrix for sum()");

    return (retval);
  }
}

Matrix *
matrix_CumSum (m)
     Matrix *m;
{
  ASSERT (m);
  {
    int i, j;
    Matrix *new = 0;

    if (MTYPE (m) == REAL)
    {
      if (MNR (m) == 1 || MNC (m) == 1)
      {
	int N = MNR (m) * MNC (m);
	new = matrix_Create (MNR (m), MNC (m));
	MATrv (new, 0) = MATrv (m, 0);
	for (i = 1; i < N; i++)
	{
	  MATrv (new, i) = MATrv (new, i-1) + MATrv (m, i);
	}
      }
      else
      {
	new = matrix_Create (MNR (m), MNC (m));
	/* Initialize first row. */
	for (i = 0; i < MNC (m); i++)
	  MAT0 (new, 0, i) = MAT0 (m, 0, i);

	/* Now compute running sum. */
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 1; j < MNR (m); j++)
	  {
	    MAT0 (new, j, i) = MAT0 (new, j-1, i) + MAT0 (m, j, i);
	  }
	}
      }
    }
    else if (MTYPE (m) == COMPLEX)
    {
      if (MNR (m) == 1 || MNC (m) == 1)
      {
	int N = MNR (m) * MNC (m);
	new = matrix_CreateC (MNR (m), MNC (m));
	MATcvr (new, 0) = MATcvr (m, 0);
	MATcvi (new, 0) = MATcvi (m, 0);
	
	for (i = 1; i < N; i++)
	{
	  MATcvr (new, i) = MATcvr (new, i-1) + MATcvr (m, i);
	  MATcvi (new, i) = MATcvi (new, i-1) + MATcvi (m, i);
	}
      }
      else
      {
	new = matrix_CreateC (MNR (m), MNC (m));
	/* Initialize first row. */
	for (i = 0; i < MNC (m); i++)
	{
	  MATr0 (new, 0, i) = MATr0 (m, 0, i);
	  MATi0 (new, 0, i) = MATi0 (m, 0, i);
	}

	/* Now compute running sum. */
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 1; j < MNR (m); j++)
	  {
	    MATr0 (new, j, i) = MATr0 (new, j-1, i) + MATr0 (m, j, i);
	    MATi0 (new, j, i) = MATi0 (new, j-1, i) + MATi0 (m, j, i);
	  }
	}
      }
    }
    else
      error_1 (matrix_GetName (m), "invalid type matrix for cumsum");

    return (new);
  }
}

VPTR
matrix_Prod (m)
     Matrix *m;
{
  ASSERT (m);
  {
    int i, j;
    VPTR retval = 0;

    if (MTYPE (m) == REAL)
    {
      if (MNR (m) == 1)
      {
	double d = MATrv (m, 0);
	for (i = 1; i < MNC (m); i++)
	  d = d * MATrv (m, i);
	retval = (VPTR) scalar_Create (d);
      }
      else
      {
	Matrix *new = matrix_Create (1, MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  MATrv (new, i) = MAT0 (m, 0, i);
	  for (j = 1; j < MNR (m); j++)
	    MATrv (new, i) = MATrv (new, i) * MAT0 (m, j, i);
	}
	retval = (VPTR) new;
      }
    }
    else if (MTYPE (m) == COMPLEX)
    {
      if (MNR (m) == 1)
      {
	Complex c;
	c = MATcv (m, 0);
	for (i = 1; i < MNC (m); i++)
	{
	  c = complex_Multiply (c.r, c.i, MATcvr (m, i), MATcvi (m, i));
	}
	retval = (VPTR) scalar_CreateC (c.r, c.i);
      }
      else
      {
	Matrix *new = matrix_CreateC (1, MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  MATcvr (new, i) = MATr0 (m, 0, i);
	  MATcvi (new, i) = MATi0 (m, 0, i);
	  for (j = 1; j < MNR (m); j++)
	  {
	    MATcv (new, i) = complex_Multiply (MATcvr (new, i), MATcvi (new, i),
					    MATr0 (m, j, i), MATi0 (m, j, i));
	  }
	}
	retval = (VPTR) new;
      }
    }
    else
      error_1 (matrix_GetName (m), "invalid type matrix for prod()");

    return (retval);
  }
}

Matrix *
matrix_CumProd (m)
     Matrix *m;
{
  ASSERT (m);
  {
    int i, j;
    Matrix *new = 0;

    if (MTYPE (m) == REAL)
    {
      if (MNR (m) == 1 || MNC (m) == 1)
      {
	int N = MNR (m) * MNC (m);
	new = matrix_Create (MNR (m), MNC (m));
	MATrv (new, 0) = MATrv (m, 0);
	for (i = 1; i < N; i++)
	{
	  MATrv (new, i) = MATrv (new, i-1) * MATrv (m, i);
	}
      }
      else
      {
	new = matrix_Create (MNR (m), MNC (m));
	/* Initialize first row. */
	for (i = 0; i < MNC (m); i++)
	  MAT0 (new, 0, i) = MAT0 (m, 0, i);

	/* Now compute running sum. */
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 1; j < MNR (m); j++)
	  {
	    MAT0 (new, j, i) = MAT0 (new, j-1, i) * MAT0 (m, j, i);
	  }
	}
      }
    }
    else if (MTYPE (m) == COMPLEX)
    {
      if (MNR (m) == 1 || MNC (m) == 1)
      {
	int N = MNR (m) * MNC (m);
	new = matrix_CreateC (MNR (m), MNC (m));
	MATcvr (new, 0) = MATcvr (m, 0);
	MATcvi (new, 0) = MATcvi (m, 0);
	
	for (i = 1; i < N; i++)
	{
	  MATcv (new, i) = complex_Multiply (MATcvr (new,i-1), MATcvi (new,i-1),
					     MATcvr (m, i), MATcvi (m, i));
	}
      }
      else
      {
	new = matrix_CreateC (MNR (m), MNC (m));
	/* Initialize first row. */
	for (i = 0; i < MNC (m); i++)
	{
	  MATr0 (new, 0, i) = MATr0 (m, 0, i);
	  MATi0 (new, 0, i) = MATi0 (m, 0, i);
	}

	/* Now compute running sum. */
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 1; j < MNR (m); j++)
	  {
	    MATc0 (new,j,i) = complex_Multiply(MATr0(new,j-1,i), MATi0(new,j-1,i),
					       MATr0(m,j,i), MATi0(m,j,i));
	  }
	}
      }
    }
    else
      error_1 (matrix_GetName (m), "invalid type matrix for cumprod");

    return (new);
  }
}

/*
 * matrix ^ matrix
 * Only handle 1-by-1 ^ 1-by-1 case,
 * and 1-by-1 ^ matrix,
 * and matrix ^ 1-by-1.
 */

Matrix * matrix_scalar_Pow1 _PROTO ((Matrix *m, Scalar *s));
Matrix * matrix_scalar_Pow2 _PROTO ((Matrix *m, Scalar *s));

Matrix *
matrix_matrix_Pow (m1, m2)
     Matrix *m1, *m2;
{
  ASSERT (m1);
  ASSERT (m2);
  {
    Matrix *new = 0;
    Scalar *s1, *s2, *snew;

    if (MNR (m1) == 1 && MNC (m1) ==1 && MNR (m2) ==1 && MNC (m2) == 1)
    {
      if (MTYPE (m1) == REAL)
	s1 = scalar_Create (MAT (m1, 1, 1));
      else
	s1 = scalar_CreateC (MATr (m1, 1, 1), MATi (m1, 1, 1));

      if (MTYPE (m2) == REAL)
	s2 = scalar_Create (MAT (m2, 1, 1));
      else
	s2 = scalar_CreateC (MATr (m2, 1, 1), MATi (m2, 1, 1));

      snew = scalar_Pow (s1, s2);

      if (SVALi (snew) == 0.0)
      {
	new = matrix_Create (1, 1);
	MAT (new, 1, 1) = SVALr (snew);
      }
      else
      {
	new = matrix_CreateC (1, 1);
	MATr (new, 1, 1) = SVALr (snew);
	MATi (new, 1, 1) = SVALi (snew);
      }
      scalar_Destroy (s1);
      scalar_Destroy (s2);
      scalar_Destroy (snew);
    }
    else if (MNR (m1) == 1 && MNC (m1) == 1)
    {
      if (MTYPE (m1) == REAL)
	s1 = scalar_Create (MAT (m1, 1, 1));
      else
	s1 = scalar_CreateC (MATr (m1, 1, 1), MATi (m1, 1, 1));
      
      new = matrix_scalar_Pow2 (m2, s1);
      scalar_Destroy (s1);
    }
    else if (MNR (m2) == 1 && MNC (m2) == 1)
    {
      if (MTYPE (m2) == REAL)
	s2 = scalar_Create (MAT (m2, 1, 1));
      else
	s2 = scalar_CreateC (MATr (m2, 1, 1), MATi (m2, 1, 1));

      new = matrix_scalar_Pow1 (m1, s2);
      scalar_Destroy (s2);
    }
    else
      error_2 (matrix_GetName (m1), matrix_GetName (m2),
	       "matrix^matrix invalid operation");
    return (new);
  }
}

/*
 * matrix ^ scalar
 * If scalar is an integer, use matrix multiplies.
 * If scalar is not integer use eigenvalues/vectors.
 */

Matrix *
matrix_scalar_Pow1 (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    int i;
    Matrix *eval, *evec, *mtmp1, *mtmp2, *mtmp3, *dum;

    matrix_screen_string (m);

    if (MNR (m) != MNC (m))
      error_1 (matrix_GetName (m), "must be square matrix");

    if (MNR (m) ==1 && MNC (m) == 1 )
    {
      Scalar *stmp = 0;
      Scalar *snew = 0;
      if (MTYPE (m) == REAL)
	stmp = scalar_Create (MAT (m, 1, 1));
      else if (MTYPE (m) == COMPLEX)
	stmp = scalar_CreateC (MATr (m, 1, 1), MATi (m, 1, 1));
      snew = scalar_Pow (stmp, s);
      if (SVALi (snew) == 0.0)
      {
	mtmp1 = matrix_Create (1, 1);
	MAT (mtmp1, 1, 1) = SVALr (snew);
      }
      else
      {
	mtmp1 = matrix_CreateC (1, 1);
	MATr (mtmp1, 1, 1) = SVALr (snew);
	MATi (mtmp1, 1, 1) = SVALi (snew);
      }
      scalar_Destroy (stmp);
      scalar_Destroy (snew);
      return (mtmp1);
    }

    if (SVALi (s) == 0.0 && floor (SVALr (s)) == SVALr (s))
    {
      /*
       * Use matrix Multiply method.
       * Note that we could use a more efficient
       * algorithm that only involves about half the
       * multiplies.
       */

      if (SVALr (s) == 0.0)
      {
	/* Return I */
	mtmp1 = matrix_Create (MNR (m), MNC (m));
	matrix_Zero (mtmp1);
	for (i = 0; i < MNR (m); i++)
	  MAT0 (mtmp1, i, i) = 1.0;
      }
      else if (SVALr (s) < 0.0)
      {
	if (SVALr (s) == -1.0)
	  mtmp1 = matrix_Copy (m);
	else
	  mtmp1 = matrix_Multiply (m, m);

	for (i = 1; i < -SVALr (s) - 1; i++)
	{
	  mtmp2 = matrix_Multiply (mtmp1, m);
	  matrix_Destroy (mtmp1);
	  mtmp1 = mtmp2;
	}
	/* Invert result */
	mtmp2 = matrix_Inverse (mtmp1);
	matrix_Destroy (mtmp1);
	mtmp1 = mtmp2;
      }
      else
      {
	/* Positive integer exponent */
	if (SVALr (s) == 1.0)
	  mtmp1 = matrix_Copy (m);
	else
	  mtmp1 = matrix_Multiply (m, m);

	for (i = 1; i < SVALr (s) - 1; i++)
	{
	  mtmp2 = matrix_Multiply (mtmp1, m);
	  matrix_Destroy (mtmp1);
	  mtmp1 = mtmp2;
	}
      }
    }
    else
    {
      /*
       * Use matrix eigenvalues method.
       */
      signal (SIGINT, intcatch);
      matrix_Eig_NEP (m, &eval, &evec, &dum, 0);
      signal (SIGINT, intcatch_wait);

      mtmp1 = matrix_Pow1 (eval, s);

      /* Now form diagonal matrix from mtmp */
      mtmp2 = matrix_CreateC (MNC (mtmp1), MNC (mtmp1));
      matrix_Zero (mtmp2);
      for (i = 1; i <= MNC (mtmp2); i++)
      {
	MATr (mtmp2, i, i) = MATcvr1 (mtmp1, i);
	MATi (mtmp2, i, i) = MATcvi1 (mtmp1, i);
      }
      matrix_Destroy (mtmp1);

      /* Now do matrix multiplies to get results */
      mtmp3 = matrix_MultiplyCC (evec, mtmp2);
      mtmp1 = matrix_Rdivide (mtmp3, evec);

      /* Clean-Up */
      matrix_Destroy (mtmp2);
      matrix_Destroy (mtmp3);
      matrix_Destroy (evec);
      matrix_Destroy (eval);
    }
    return (mtmp1);
  }
}

/*
 * scalar ^ matrix
 */

Matrix *
matrix_scalar_Pow2 (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    int i;
    Matrix *eval, *evec, *mtmp1, *mtmp2, *mtmp3, *dum;

    matrix_screen_string (m);

    if (MNR (m) != MNC (m))
      error_1 (matrix_GetName (m), "must be square matrix");

    /* Special case... */
    if (MNR (m) == 1 && MNC (m) == 1)
    {
      Scalar *stmp = 0;
      Scalar *snew = 0;
      if (MTYPE (m) == REAL)
	stmp = scalar_Create (MAT (m, 1, 1));
      else if (MTYPE (m) == COMPLEX)
	stmp = scalar_CreateC (MATr (m, 1, 1), MATi (m, 1, 1));
      snew = scalar_Pow (s, stmp);
      if (SVALi (snew) == 0.0)
      {
	mtmp1 = matrix_Create (1, 1);
	MAT (mtmp1, 1, 1) = SVALr (snew);
      }
      else
      {
	mtmp1 = matrix_CreateC (1, 1);
	MATr (mtmp1, 1, 1) = SVALr (snew);
	MATi (mtmp1, 1, 1) = SVALi (snew);
      }
      scalar_Destroy (stmp);
      scalar_Destroy (snew);
      return (mtmp1);
    }

    matrix_Eig_NEP (m, &eval, &evec, &dum, 0);
    mtmp1 = matrix_Pow2 (eval, s);

    /* Now form diagonal matrix from mtmp */
    mtmp2 = matrix_CreateC (MNC (mtmp1), MNC (mtmp1));
    matrix_Zero (mtmp2);
    for (i = 1; i <= MNC (mtmp2); i++)
    {
      MATr (mtmp2, i, i) = MATcvr1 (mtmp1, i);
      MATi (mtmp2, i, i) = MATcvi1 (mtmp1, i);
    }
    matrix_Destroy (mtmp1);

    /* Now do matrix multiplies to get results */
    mtmp3 = matrix_MultiplyCC (evec, mtmp2);
    mtmp1 = matrix_Rdivide (mtmp3, evec);

    /* Clean-Up */
    matrix_Destroy (mtmp2);
    matrix_Destroy (mtmp3);
    matrix_Destroy (evec);
    matrix_Destroy (eval);

    return (mtmp1);
  }
}

/* **************************************************************
 * MATRIX - STRING Relational/Logical Operations.
 * ************************************************************** */

Matrix *
matrix_string_eq (m, str)
     Matrix *m;
     String *str;
{
  ASSERT (m);
  ASSERT (str);
  {
    int i, size;
    Matrix *new;

    if (MTYPE (m) == STRING)
    {
      new = matrix_Create (MNR (m), MNC (m));
      size = MNR (m) * MNC (m);
      for (i = 0; i < size; i++)
      {
	MATrv (new, i) = (double) (!strcmp (MATsv (m, i), STR (str)));
      }
    }
    else
    {
      new = matrix_Create (1, 1);
      MAT (new, 1, 1) = 0.0;
    }
    return (new);
  }
}

Matrix *
matrix_string_ne (m, str)
     Matrix *m;
     String *str;
{
  ASSERT (m);
  ASSERT (str);
  {
    int i, size;
    Matrix *new;

    if (MTYPE (m) == STRING)
    {
      new = matrix_Create (MNR (m), MNC (m));
      size = MNR (m) * MNC (m);
      for (i = 0; i < size; i++)
      {
	MATrv (new, i) = (double) (!!strcmp (MATsv (m, i), STR (str)));
      }
    }
    else
    {
      new = matrix_Create (1, 1);
      MAT (new, 1, 1) = 1.0;
    }
    return (new);
  }
}

/*
 * M <= str
 */

Matrix *
matrix_string_le1 (m, str)
     Matrix *m;
     String *str;
{
  ASSERT (m);
  ASSERT (str);
  {
    int i, size;
    Matrix *new = 0;

    if (MTYPE (m) == STRING)
    {
      new = matrix_Create (MNR (m), MNC (m));
      size = MNR (m) * MNC (m);
      for (i = 0; i < size; i++)
      {
	MATrv (new, i) = (double) (strcmp (MATsv (m, i), STR (str)) <= 0);
      }
    }
    else
      error_1 (matrix_GetName (m), "not a string matrix");
    return (new);
  }
}

/*
 * str <= M
 */

Matrix *
matrix_string_le2 (m, str)
     Matrix *m;
     String *str;
{
  ASSERT (m);
  ASSERT (str);
  {
    int i, size;
    Matrix *new = 0;

    if (MTYPE (m) == STRING)
    {
      new = matrix_Create (MNR (m), MNC (m));
      size = MNR (m) * MNC (m);
      for (i = 0; i < size; i++)
      {
	MATrv (new, i) = (double) (strcmp (STR (str), MATsv (m, i)) <= 0);
      }
    }
    else
      error_1 (matrix_GetName (m), "not a string matrix");
    return (new);
  }
}

/*
 * M >= str
 */

Matrix *
matrix_string_ge1 (m, str)
     Matrix *m;
     String *str;
{
  ASSERT (m);
  ASSERT (str);
  {
    int i, size;
    Matrix *new = 0;

    if (MTYPE (m) == STRING)
    {
      new = matrix_Create (MNR (m), MNC (m));
      size = MNR (m) * MNC (m);
      for (i = 0; i < size; i++)
      {
	MATrv (new, i) = (double) (strcmp (MATsv (m, i), STR (str)) >= 0);
      }
    }
    else
      error_1 (matrix_GetName (m), "not a string matrix");
    return (new);
  }
}

/*
 * str >= M
 */

Matrix *
matrix_string_ge2 (m, str)
     Matrix *m;
     String *str;
{
  ASSERT (m);
  ASSERT (str);
  {
    int i, size;
    Matrix *new = 0;

    if (MTYPE (m) == STRING)
    {
      new = matrix_Create (MNR (m), MNC (m));
      size = MNR (m) * MNC (m);
      for (i = 0; i < size; i++)
      {
	MATrv (new, i) = (double) (strcmp (STR (str), MATsv (m, i)) >= 0);
      }
    }
    else
      error_1 (matrix_GetName (m), "not a string matrix");
    return (new);
  }
}

/*
 * M < str
 */

Matrix *
matrix_string_lt1 (m, str)
     Matrix *m;
     String *str;
{
  ASSERT (m);
  ASSERT (str);
  {
    int i, size;
    Matrix *new = 0;

    if (MTYPE (m) == STRING)
    {
      new = matrix_Create (MNR (m), MNC (m));
      size = MNR (m) * MNC (m);
      for (i = 0; i < size; i++)
      {
	MATrv (new, i) = (double) (strcmp (MATsv (m, i), STR (str)) < 0);
      }
    }
    else
      error_1 (matrix_GetName (m), "not a string matrix");
    return (new);
  }
}

/*
 * str < M
 */

Matrix *
matrix_string_lt2 (m, str)
     Matrix *m;
     String *str;
{
  ASSERT (m);
  ASSERT (str);
  {
    int i, size;
    Matrix *new = 0;

    if (MTYPE (m) == STRING)
    {
      new = matrix_Create (MNR (m), MNC (m));
      size = MNR (m) * MNC (m);
      for (i = 0; i < size; i++)
      {
	MATrv (new, i) = (double) (strcmp (STR (str), MATsv (m, i)) < 0);
      }
    }
    else
      error_1 (matrix_GetName (m), "not a string matrix");
    return (new);
  }
}

/*
 * M > str
 */

Matrix *
matrix_string_gt1 (m, str)
     Matrix *m;
     String *str;
{
  ASSERT (m);
  ASSERT (str);
  {
    int i, size;
    Matrix *new = 0;

    if (MTYPE (m) == STRING)
    {
      new = matrix_Create (MNR (m), MNC (m));
      size = MNR (m) * MNC (m);
      for (i = 0; i < size; i++)
      {
	MATrv (new, i) = (double) (strcmp (MATsv (m, i), STR (str)) > 0);
      }
    }
    else
      error_1 (matrix_GetName (m), "not a string matrix");
    return (new);
  }
}

/*
 * str > M
 */

Matrix *
matrix_string_gt2 (m, str)
     Matrix *m;
     String *str;
{
  ASSERT (m);
  ASSERT (str);
  {
    int i, size;
    Matrix *new = 0;

    if (MTYPE (m) == STRING)
    {
      new = matrix_Create (MNR (m), MNC (m));
      size = MNR (m) * MNC (m);
      for (i = 0; i < size; i++)
      {
	MATrv (new, i) = (double) (strcmp (STR (str), MATsv (m, i)) > 0);
      }
    }
    else
      error_1 (matrix_GetName (m), "not a string matrix");
    return (new);
  }
}

Matrix *
matrix_string_and (m, str)
     Matrix *m;
     String *str;
{
  ASSERT (m);
  ASSERT (str);
  {
    int i, size;
    Matrix *new = 0;
    String *stmp;

    if (MTYPE (m) == STRING)
    {
      new = matrix_Create (MNR (m), MNC (m));
      stmp = string_Create (0);
      size = MNR (m) * MNC (m);
      for (i = 0; i < size; i++)
      {
	MATrv (new, i) = (double) ((strcmp (STR (str), STR (stmp)) != 0) &&
				   (strcmp (MATsv (m, i), STR (stmp)) != 0));
      }
      string_Destroy (stmp);
    }
    else
      error_1 (matrix_GetName (m), "not a string matrix");
    return (new);
  }
}

Matrix *
matrix_string_or (m, str)
     Matrix *m;
     String *str;
{
  ASSERT (m);
  ASSERT (str);
  {
    int i, size;
    Matrix *new = 0;
    String *stmp;

    if (MTYPE (m) == STRING)
    {
      new = matrix_Create (MNR (m), MNC (m));
      stmp = string_Create (0);
      size = MNR (m) * MNC (m);
      for (i = 0; i < size; i++)
      {
	MATrv (new, i) = (double) ((strcmp (STR (str), STR (stmp)) != 0) ||
				   (strcmp (MATsv (m, i), STR (stmp)) != 0));
      }
      string_Destroy (stmp);
    }
    else
      error_1 (matrix_GetName (m), "not a string matrix");
    return (new);
  }
}

/* **************************************************************
 * Add a STRING Matrix, and a scalar string.
 * ************************************************************** */

Matrix *
matrix_string_add (m, str, flag)
     Matrix *m;
     String *str;
     int flag;
{
  int i;
  Matrix *new = 0;

  if (MTYPE (m) == STRING)
  {
    new = matrix_CreateS (MNR (m), MNC (m));
    if (flag == 1)
    {				/* Matrix 1st */
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.ms[i] = string_add (m->val.ms[i], string_GetString (str));
    }
    else if (flag == 2)
    {				/* Matrix 2nd */
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.ms[i] = string_add (string_GetString (str), m->val.ms[i]);
    }
  }
  else
    error_1 (matrix_GetName (m), "must be type string for add-op");

  return (new);
}

/* 
 * Matrix_is_positive:
 * Return TRUE if ALL elements are >= 0.
 * This function is not meant to be used with
 * complex, or string matrices.
 */

int
matrix_is_positive (m)
     Matrix *m;
{
  int i;
  if (MTYPE (m) == REAL)
  {
    for (i = 0; i < MNR (m) * MNC (m); i++)
      if (MATrv (m, i) < 0.0)
	return (0);
  }
  return (1);
}

/* **************************************************************
 * Add matrices and vectors...
 * m : Matrix
 * v : Vector
 * If v is a row-vector, then its column dimension must match 
 * the column dimension of m. Otherwise, the row dimension of
 * v must match the row dimension of m.
 * ************************************************************** */

/*
 * v is a row-vector, column dimension must match
 * m's column dimension.
 */

Matrix *
matrix_VectorAddRow (m, v)
     Matrix *m, *v;
{
  ASSERT (m);
  ASSERT (v);
  {
    int i, j;
    Matrix *new = 0;

    switch (MTYPE (m))
    {
    case REAL:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_Create (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MAT0 (new, i, j) = MAT0 (m, i, j) + MAT0 (v, 0, j);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MATr0 (new, i, j) = MAT0 (m, i, j) + MATr0 (v, 0, j);
	    MATi0 (new, i, j) = MATi0 (v, 0, j);
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot add string and numeric matrices");
	break;
      }
      break;
    case COMPLEX:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MATr0 (new, i, j) = MATr0 (m, i, j) + MAT0 (v, 0, j);
	    MATi0 (new, i, j) = MATi0 (m, i, j);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MATr0 (new, i, j) = MATr0 (m, i, j) + MATr0 (v, 0, j);
	    MATi0 (new, i, j) = MATr0 (m, i, j) + MATi0 (v, 0, j);
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot add string and numeric matrices");
	break;
      }
      break;
    case STRING:
      error_2 (matrix_GetName (m), matrix_GetName (v),
	       "cannot row-wise add string and numeric matrices");
      break;
    default:
      error_1 ("terrible matrix type-error!", 0);
      break;
    }
    return (new);
  }
}

/*
 * v is a column-vector, row dimension must match
 * m's row dimension.
 */

Matrix *
matrix_VectorAddCol (m, v)
     Matrix *m, *v;
{
  ASSERT (m);
  ASSERT (v);
  {
    int i, j;
    Matrix *new = 0;

    switch (MTYPE (m))
    {
    case REAL:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_Create (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MAT0 (new, j, i) = MAT0 (m, j, i) + MAT0 (v, j, 0);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATr0 (new, j, i) = MAT0 (m, j, i) + MATr0 (v, j, 0);
	    MATi0 (new, j, i) = MATi0 (v, j, 0);
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot add string and numeric matrices");
	break;
      }
      break;
    case COMPLEX:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATr0 (new, j, i) = MATr0 (m, j, i) + MAT0 (v, j, 0);
	    MATi0 (new, j, i) = MATi0 (m, j, i);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATr0 (new, j, i) = MATr0 (m, j, i) + MATr0 (v, j, 0);
	    MATi0 (new, j, i) = MATr0 (m, j, i) + MATi0 (v, j, 0);
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot add string and numeric matrices");
	break;
      }
      break;
    case STRING:
      error_2 (matrix_GetName (m), matrix_GetName (v),
	       "cannot col-wise add string and numeric matrices");
      break;
    default:
      error_1 ("terrible matrix type-error!", 0);
      break;
    }
    return (new);
  }
}

/*
 * v is a row-vector, column dimension must match
 * m's column dimension.
 */

Matrix *
matrix_VectorSubRow1 (m, v)
     Matrix *m, *v;
{
  ASSERT (m);
  ASSERT (v);
  {
    int i, j;
    Matrix *new = 0;

    switch (MTYPE (m))
    {
    case REAL:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_Create (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MAT0 (new, i, j) = MAT0 (m, i, j) - MAT0 (v, 0, j);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MATr0 (new, i, j) = MAT0 (m, i, j) - MATr0 (v, 0, j);
	    MATi0 (new, i, j) = -MATi0 (v, 0, j);
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot sub string and numeric matrices");
	break;
      }
      break;
    case COMPLEX:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MATr0 (new, i, j) = MATr0 (m, i, j) - MAT0 (v, 0, j);
	    MATi0 (new, i, j) = MATi0 (m, i, j);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MATr0 (new, i, j) = MATr0 (m, i, j) - MATr0 (v, 0, j);
	    MATi0 (new, i, j) = MATr0 (m, i, j) - MATi0 (v, 0, j);
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot sub string and numeric matrices");
	break;
      }
      break;
    case STRING:
      error_2 (matrix_GetName (m), matrix_GetName (v),
	       "cannot row-wise sub string and numeric matrices");
      break;
    default:
      error_1 ("terrible matrix type-error!", 0);
      break;
    }
    return (new);
  }
}

/*
 * V - M
 */

Matrix *
matrix_VectorSubRow2 (m, v)
     Matrix *m, *v;
{
  ASSERT (m);
  ASSERT (v);
  {
    int i, j;
    Matrix *new = 0;

    switch (MTYPE (m))
    {
    case REAL:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_Create (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MAT0 (new, i, j) = -MAT0 (m, i, j) + MAT0 (v, 0, j);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MATr0 (new, i, j) = -MAT0 (m, i, j) + MATr0 (v, 0, j);
	    MATi0 (new, i, j) = MATi0 (v, 0, j);
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot sub string and numeric matrices");
	break;
      }
      break;
    case COMPLEX:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MATr0 (new, i, j) = -MATr0 (m, i, j) + MAT0 (v, 0, j);
	    MATi0 (new, i, j) = -MATi0 (m, i, j);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MATr0 (new, i, j) = -MATr0 (m, i, j) + MATr0 (v, 0, j);
	    MATi0 (new, i, j) = -MATr0 (m, i, j) + MATi0 (v, 0, j);
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot sub string and numeric matrices");
	break;
      }
      break;
    case STRING:
      error_2 (matrix_GetName (m), matrix_GetName (v),
	       "cannot row-wise sub string and numeric matrices");
      break;
    default:
      error_1 ("terrible matrix type-error!", 0);
      break;
    }
    return (new);
  }
}

/*
 * v is a column-vector, row dimension must match
 * m's row dimension.
 */

Matrix *
matrix_VectorSubCol1 (m, v)
     Matrix *m, *v;
{
  ASSERT (m);
  ASSERT (v);
  {
    int i, j;
    Matrix *new = 0;

    switch (MTYPE (m))
    {
    case REAL:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_Create (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MAT0 (new, j, i) = MAT0 (m, j, i) - MAT0 (v, j, 0);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATr0 (new, j, i) = MAT0 (m, j, i) - MATr0 (v, j, 0);
	    MATi0 (new, j, i) = -MATi0 (v, j, 0);
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot add string and numeric matrices");
	break;
      }
      break;
    case COMPLEX:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATr0 (new, j, i) = MATr0 (m, j, i) - MAT0 (v, j, 0);
	    MATi0 (new, j, i) = MATi0 (m, j, i);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATr0 (new, j, i) = MATr0 (m, j, i) - MATr0 (v, j, 0);
	    MATi0 (new, j, i) = MATr0 (m, j, i) - MATi0 (v, j, 0);
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot sub string and numeric matrices");
	break;
      }
      break;
    case STRING:
      error_2 (matrix_GetName (m), matrix_GetName (v),
	       "cannot col-wise sub string and numeric matrices");
      break;
    default:
      error_1 ("terrible matrix type-error!", 0);
      break;
    }
    return (new);
  }
}

Matrix *
matrix_VectorSubCol2 (m, v)
     Matrix *m, *v;
{
  ASSERT (m);
  ASSERT (v);
  {
    int i, j;
    Matrix *new = 0;

    switch (MTYPE (m))
    {
    case REAL:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_Create (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MAT0 (new, j, i) = -MAT0 (m, j, i) + MAT0 (v, j, 0);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATr0 (new, j, i) = -MAT0 (m, j, i) + MATr0 (v, j, 0);
	    MATi0 (new, j, i) = MATi0 (v, j, 0);
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot add string and numeric matrices");
	break;
      }
      break;
    case COMPLEX:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATr0 (new, j, i) = -MATr0 (m, j, i) + MAT0 (v, j, 0);
	    MATi0 (new, j, i) = -MATi0 (m, j, i);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATr0 (new, j, i) = -MATr0 (m, j, i) + MATr0 (v, j, 0);
	    MATi0 (new, j, i) = -MATr0 (m, j, i) + MATi0 (v, j, 0);
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot sub string and numeric matrices");
	break;
      }
      break;
    case STRING:
      error_2 (matrix_GetName (m), matrix_GetName (v),
	       "cannot col-wise sub string and numeric matrices");
      break;
    default:
      error_1 ("terrible matrix type-error!", 0);
      break;
    }
    return (new);
  }
}

/* **************************************************************
 * Multiply / Divide matrices and vectors...
 * m : Matrix
 * v : Vector
 * If v is a row-vector, then its column dimension must match 
 * the column dimension of m. Otherwise, the row dimension of
 * v must match the row dimension of m.
 * ************************************************************** */

/*
 * v is a row-vector, column dimension must match
 * m's column dimension.
 */

Matrix *
matrix_VectorMulRow (m, v)
     Matrix *m, *v;
{
  ASSERT (m);
  ASSERT (v);
  {
    int i, j;
    Matrix *new = 0;

    switch (MTYPE (m))
    {
    case REAL:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_Create (MNR (m), MNC (m));
 	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MAT0 (new, j, i) = MAT0 (m, j, i) * MAT0 (v, 0, i);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MATc0 (new, i, j) = complex_Multiply (MAT0 (m, i, j),
						  0.0,
						  MATr0 (v, 0, j),
						  MATi0 (v, 0, j));
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot mul string and numeric matrices");
	break;
      }
      break;
    case COMPLEX:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MATc0 (new, i, j) = complex_Multiply (MATr0 (m, i, j),
						  MATi0 (m, i, j),
						  MAT0 (v, 0, j),
						  0.0);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MATc0 (new, i, j) = complex_Multiply (MATr0 (m, i, j),
						  MATi0 (m, i, j),
						  MATr0 (v, 0, j),
						  MATi0 (v, 0, j));
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot mul string and numeric matrices");
	break;
      }
      break;
    case STRING:
      error_2 (matrix_GetName (m), matrix_GetName (v),
	       "cannot row-wise mul string and numeric matrices");
      break;
    default:
      error_1 ("terrible matrix type-error!", 0);
      break;
    }
    return (new);
  }
}

/*
 * v is a column-vector, row dimension must match
 * m's row dimension.
 */

Matrix *
matrix_VectorMulCol (m, v)
     Matrix *m, *v;
{
  ASSERT (m);
  ASSERT (v);
  {
    int i, j;
    Matrix *new = 0;

    switch (MTYPE (m))
    {
    case REAL:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_Create (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MAT0 (new, j, i) = MAT0 (m, j, i) * MAT0 (v, j, 0);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATc0 (new, j, i) = complex_Multiply (MAT0 (m, j, i),
						  0.0,
						  MATr0 (v, j, 0),
						  MATi0 (v, j, 0));
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot mul string and numeric matrices");
	break;
      }
      break;
    case COMPLEX:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATc0 (new, j, i) = complex_Multiply (MATr0 (m, j, i),
						  MATi0 (m, j, i),
						  MAT0 (v, j, 0),
						  0.0);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATc0 (new, j, i) = complex_Multiply (MATr0 (m, j, i),
						  MATi0 (m, j, i),
						  MATr0 (v, j, 0),
						  MATi0 (v, j, 0));
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot mul string and numeric matrices");
	break;
      }
      break;
    case STRING:
      error_2 (matrix_GetName (m), matrix_GetName (v),
	       "cannot col-wise mul string and numeric matrices");
      break;
    default:
      error_1 ("terrible matrix type-error!", 0);
      break;
    }
    return (new);
  }
}

/*
 * v is a row-vector, column dimension must match
 * m's column dimension.
 *
 * M ./ V
 */

Matrix *
matrix_VectorDivRow1 (m, v)
     Matrix *m, *v;
{
  ASSERT (m);
  ASSERT (v);
  {
    int i, j;
    Matrix *new = 0;

    switch (MTYPE (m))
    {
    case REAL:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_Create (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MAT0 (new, i, j) = MAT0 (m, i, j) / MAT0 (v, 0, j);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MATc0 (new, i, j) = complex_div (MAT0 (m, i, j),
					     0.0,
					     MATr0 (v, 0, j),
					     MATi0 (v, 0, j));
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot div string and numeric matrices");
	break;
      }
      break;
    case COMPLEX:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MATc0 (new, i, j) = complex_div (MATr0 (m, i, j),
					     MATi0 (m, i, j),
					     MAT0 (v, 0, j),
					     0.0);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MATc0 (new, i, j) = complex_div (MATr0 (m, i, j),
					     MATi0 (m, i, j),
					     MATr0 (v, 0, j),
					     MATi0 (v, 0, j));
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot div string and numeric matrices");
	break;
      }
      break;
    case STRING:
      error_2 (matrix_GetName (m), matrix_GetName (v),
	       "cannot row-wise div string and numeric matrices");
      break;
    default:
      error_1 ("terrible matrix type-error!", 0);
      break;
    }
    return (new);
  }
}

/*
 * V ./ M
 */

Matrix *
matrix_VectorDivRow2 (m, v)
     Matrix *m, *v;
{
  ASSERT (m);
  ASSERT (v);
  {
    int i, j;
    Matrix *new = 0;

    switch (MTYPE (m))
    {
    case REAL:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_Create (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MAT0 (new, i, j) = MAT0 (v, 0, j) / MAT0 (m, i, j);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MATc0 (new, i, j) = complex_div (MATr0 (v, 0, j),
					     MATi0 (v, 0, j),
					     MAT0 (m, i, j),
					     0.0);
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot div string and numeric matrices");
	break;
      }
      break;
    case COMPLEX:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MATc0 (new, i, j) = complex_div (MAT0 (v, 0, j),
					     0.0,
					     MATr0 (m, i, j),
					     MATi0 (m, i, j));
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNR (m); i++)
	{
	  for (j = 0; j < MNC (m); j++)
	  {
	    MATc0 (new, i, j) = complex_div (MATr0 (v, 0, j),
					     MATi0 (v, 0, j),
					     MATr0 (m, i, j),
					     MATi0 (m, i, j));
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot divide string and numeric matrices");
	break;
      }
      break;
    case STRING:
      error_2 (matrix_GetName (m), matrix_GetName (v),
	       "cannot row-wise divide string and numeric matrices");
      break;
    default:
      error_1 ("terrible matrix type-error!", 0);
      break;
    }
    return (new);
  }
}

/*
 * v is a column-vector, row dimension must match
 * m's row dimension.
 *
 * M ./ V
 */

Matrix *
matrix_VectorDivCol1 (m, v)
     Matrix *m, *v;
{
  ASSERT (m);
  ASSERT (v);
  {
    int i, j;
    Matrix *new = 0;

    switch (MTYPE (m))
    {
    case REAL:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_Create (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MAT0 (new, j, i) = MAT0 (m, j, i) / MAT0 (v, j, 0);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATc0 (new, j, i) = complex_div (MAT0 (m, j, i),
					     0.0,
					     MATr0 (v, j, 0),
					     MATi0 (v, j, 0));
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot divide string and numeric matrices");
	break;
      }
      break;
    case COMPLEX:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATc0 (new, j, i) = complex_div (MATr0 (m, j, i),
					     MATi0 (m, j, i),
					     MAT0 (v, j, 0),
					     0.0);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATc0 (new, j, i) = complex_div (MATr0 (m, j, i),
					     MATi0 (m, j, i),
					     MATr0 (v, j, 0),
					     MATi0 (v, j, 0));
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot divide string and numeric matrices");
	break;
      }
      break;
    case STRING:
      error_2 (matrix_GetName (m), matrix_GetName (v),
	       "cannot col-wise divide string and numeric matrices");
      break;
    default:
      error_1 ("terrible matrix type-error!", 0);
      break;
    }
    return (new);
  }
}

/*
 * V ./ M
 */

Matrix *
matrix_VectorDivCol2 (m, v)
     Matrix *m, *v;
{
  ASSERT (m);
  ASSERT (v);
  {
    int i, j;
    Matrix *new = 0;

    switch (MTYPE (m))
    {
    case REAL:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_Create (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MAT0 (new, j, i) = MAT0 (v, j, 0) / MAT0 (m, j, i);
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATc0 (new, j, i) = complex_div (MATr0 (v, j, 0),
					     MATi0 (v, j, 0),
					     MAT0 (m, j, i),
					     0.0);
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot divide string and numeric matrices");
	break;
      }
      break;
    case COMPLEX:
      switch (MTYPE (v))
      {
      case REAL:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATc0 (new, j, i) = complex_div (MAT0 (v, j, 0),
					     0.0,
					     MATr0 (m, j, i),
					     MATi0 (m, j, i));
	  }
	}
	break;
      case COMPLEX:
	new = matrix_CreateC (MNR (m), MNC (m));
	for (i = 0; i < MNC (m); i++)
	{
	  for (j = 0; j < MNR (m); j++)
	  {
	    MATc0 (new, j, i) = complex_div (MATr0 (v, j, 0),
					     MATi0 (v, j, 0),
					     MATr0 (m, j, i),
					     MATi0 (m, j, i));
	  }
	}
	break;
      case STRING:
	error_2 (matrix_GetName (m), matrix_GetName (v),
		 "Cannot divide string and numeric matrices");
	break;
      }
      break;
    case STRING:
      error_2 (matrix_GetName (m), matrix_GetName (v),
	       "cannot col-wise divide string and numeric matrices");
      break;
    default:
      error_1 ("terrible matrix type-error!", 0);
      break;
    }
    return (new);
  }
}
