/* bltin1.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1994  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "symbol.h"
#include "mem.h"
#include "list.h"
#include "btree.h"
#include "bltin.h"
#include "scop1.h"
#include "matop1.h"
#include "matop2.h"
#include "r_string.h"
#include "util.h"
#include "mathl.h"
#include "function.h"
#include "print.h"

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

/* scan.l */
extern int new_file _PROTO ((char *file_name));

#define rabs(x) ((x) >= 0 ? (x) : -(x))

/* **************************************************************
 * Return a list of the current built-in and user functions in the
 * symbol-table.
 * ************************************************************** */
void
What (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *L;
  Matrix *m = 0;

  if (n_args == 0)
    m = (Matrix *) print_function_list (0);
  else if (n_args == 1)
  {
    L = bltin_get_list ("what", d_arg, 1);
    m = (Matrix *) print_function_list (e_data (L));
    remove_tmp_destroy (L);
  }
  else
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("what: 0 or 1 arguments allowed", 0);
  }

  *return_ptr = (VPTR) m;
}

/* **************************************************************
 * Return a list of current objects in the symbol table. NFY
 * ************************************************************** */
void
Who (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *L;
  Matrix *m = 0;

  if (n_args == 0)
    m = (Matrix *) print_object_list (0);
  else if (n_args == 1)
  {
    L = bltin_get_list ("who", d_arg, 1);
    m = (Matrix *) print_object_list (e_data (L));
    remove_tmp_destroy (L);
  }
  else
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("who: 0 or 1 arguments allowed", 0);
  }

  *return_ptr = (VPTR) m;
}

void
Members (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *L = 0;
  Matrix *m = 0;

  if (n_args == 1)
  {
    L = bltin_get_list ("members", d_arg, 1);
    m = (Matrix *) btree_members ((e_data (L)));
  }
  else
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("members: 1 argument allowed", 0);
  }

  *return_ptr = (VPTR) m;
  remove_tmp_destroy (L);

}

/* **************************************************************
 * Load an RLaB file.
 * ************************************************************** */

extern int run_program _PROTO ((char *ptr));

void
Load (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *string;
  ListNode *S;

  /* Check n_args */
  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    warning_1 ("load: 1 argument allowed", 0);
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }

  /* get arg from list */
  S = bltin_get_string ("load", d_arg, 1);
  string = string_GetString (e_data (S));

  /* Run rlab on the file */
  run_program (string);

  close_file_ds (string);

  *return_ptr = (VPTR) scalar_Create (1.0);
  remove_tmp_destroy (S);
}

/* **************************************************************
 * Return an object filled with zeros.
 * ************************************************************** */
void
Zero (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  double d[2];
  ListNode *M;

  Matrix *m = 0;

  if (n_args == 1)
  {
    M = bltin_get_numeric_matrix ("zeros", d_arg, 1);
    if (MNR (e_data (M)) * MNC (e_data (M)) != 2)
      error_1 ("must supply a matrix with n = 2 to zeros()", 0);
    matrix_screen_string (e_data (M));
    m = matrix_Create ((int) MATrv (e_data (M), 0),
		       (int) MATrv (e_data (M), 1));
    remove_tmp_destroy (M);
  }
  else if (n_args == 2)
  {
    d[0] = bltin_get_numeric_double ("zeros", d_arg, 1);
    d[1] = bltin_get_numeric_double ("zeros", d_arg, 2);
    m = matrix_Create ((int) d[0], (int) d[1]);
  }
  else
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("zeros() requires 1 or 2 args", 0);
  }

  matrix_Zero (m);
  *return_ptr = (VPTR) m;
}

void
rOnes (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  double d[2];
  int i, size;
  ListNode *M;
  Matrix *m = 0;

  if (n_args == 1)
  {
    M = bltin_get_numeric_matrix ("ones", d_arg, 1);
    if (MNR (e_data (M)) * MNC (e_data (M)) != 2)
      error_1 ("must supply a matrix with n = 2 to ones()", 0);
    m = matrix_Create ((int) MATrv (e_data (M), 0),
		       (int) MATrv (e_data (M), 1));
    remove_tmp_destroy (M);
  }
  else if (n_args == 2)
  {
    d[0] = bltin_get_numeric_double ("ones", d_arg, 1);
    d[1] = bltin_get_numeric_double ("ones", d_arg, 2);
    m = matrix_Create ((int) d[0], (int) d[1]);
  }
  else
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("ones: requires 1 or 2 args", 0);
  }

  /*
   * Now fill the matrix with 1s
   */
  
  size = MNR (m) * MNC (m);
  for (i = 0; i < size; i++)
  {
    MATrv (m, i) = 1.0;
  }
  *return_ptr = (VPTR) m;
}

void
Diag (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int i, k, smax, smin, size;
  ListNode *M;
  Matrix *m, *marg;

  m = 0;			/* Initialize */

  if (n_args != 1 && n_args != 2)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("diag: 1 or 2 arguments allowed", 0);
  }

  M = bltin_get_numeric_matrix ("diag", d_arg, 1);
  if (n_args == 1)
  {
    k = 0;
  }
  else
  {
    k = bltin_get_numeric_double ("diag", d_arg, 2);
  }
  marg = (Matrix *) e_data (M);

  if (MNR (marg) == 1 || MNC (marg) == 1)
  {
    /* Create a diagonal matrix */
    size = max (MNR (marg), MNC (marg)) + rabs (k);
    if (MTYPE (marg) == REAL)
    {
      m = matrix_Create (size, size);
      matrix_Zero (m);
      if (k < 0)
      {
	for (i = 1 - k; i <= size; i++)
	  MAT (m, i, (i + k)) = MATrv1 (marg, (i + k));
      }
      else
      {
	for (i = 1; i <= size - k; i++)
	  MAT (m, i, (i + k)) = MATrv1 (marg, i);
      }
    }
    else if (MTYPE (marg) == COMPLEX)
    {
      m = matrix_CreateC (size, size);
      matrix_Zero (m);
      if (k < 0)
      {
	for (i = 1 - k; i <= size; i++)
	{
	  MATr (m, i, (i + k)) = MATcvr1 (marg, (i + k));
	  MATi (m, i, (i + k)) = MATcvi1 (marg, (i + k));
	}
      }
      else
      {
	for (i = 1; i <= size - k; i++)
	{
	  MATr (m, i, (i + k)) = MATcvr1 (marg, i);
	  MATi (m, i, (i + k)) = MATcvi1 (marg, i);
	}
      }
    }
  }
  else
  {
    /* Extract the diagonal elements */
    smin = min (MNR (marg), MNC (marg));
    smax = max (MNR (marg), MNC (marg));

    if (k >= 0)
    {
      if (MNR (marg) >= MNC (marg))
	size = smin - k;
      else
	size = smin - max (0, k - (MNC (marg) - MNR (marg)));
    }
    else
    {
      if (MNR (marg) >= MNC (marg))
	size = smin - max (0, -k - (MNR (marg) - MNC (marg)));
      else
	size = smin + k;
    }
    if (size <= 0)
      size = 0;

    if (MTYPE (marg) == REAL)
    {
      m = matrix_Create (size, 1);
      matrix_Zero (m);
      if (k >= 0)
      {
	for (i = 1; i <= size; i++)
	  MAT (m, i, 1) = MAT (marg, i, i + k);
      }
      else
      {
	for (i = 1; i <= size; i++)
	  MAT (m, i, 1) = MAT (marg, i - k, i);
      }
    }
    else if (MTYPE (marg) == COMPLEX)
    {
      m = matrix_CreateC (size, 1);
      matrix_Zero (m);
      if (k >= 0)
      {
	for (i = 1; i <= size; i++)
	{
	  MATr (m, i, 1) = MATr (marg, i, i + k);
	  MATi (m, i, 1) = MATi (marg, i, i + k);
	}
      }
      else
      {
	for (i = 1; i <= size; i++)
	{
	  MATr (m, i, 1) = MATr (marg, i - k, i);
	  MATi (m, i, 1) = MATi (marg, i - k, i);
	}
      }
    }
  }
  remove_tmp_destroy (M);
  *return_ptr = (VPTR) m;
}

/* **************************************************************
 * Return the size of an entity.
 * ************************************************************** */
void
Size (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;
  Matrix *m;

  /* Check n_args */
  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("size: 1 argument allowed", 0);
  }

  arg = bltin_get_arg ("size", d_arg, 1);

  switch (arg.type)
  {
  case CONSTANT:
  case iCONSTANT:
    m = matrix_Create (1, 2);
    MAT (m, 1, 1) = 1.0;
    MAT (m, 1, 2) = 1.0;
    *return_ptr = (VPTR) m;
    break;

  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
    case STRING:
      m = matrix_Create (1, 2);
      MAT (m, 1, 1) = 1.0;
      MAT (m, 1, 2) = 1.0;
      *return_ptr = (VPTR) m;
      break;

    case MATRIX:
      m = matrix_Create (1, 2);
      MAT (m, 1, 1) = (double) MNR (e_data (arg.u.ent));
      MAT (m, 1, 2) = (double) MNC (e_data (arg.u.ent));
      *return_ptr = (VPTR) m;
      break;

    case BTREE:
      *return_ptr = (VPTR)
	scalar_Create ((double) btree_GetRealNumNodes (e_data (arg.u.ent)));
      break;

    case UNDEF:
      clean_bltin_args (d_arg, n_args);
      error_1 (e_name (arg.u.ent), "UNDEFINED");
      break;

    default:
      clean_bltin_args (d_arg, n_args);
      error_1 (e_name (arg.u.ent), "invalid type for size()");
      break;
    }
    remove_tmp_destroy (arg.u.ent);
    break;
  }
}

/* **************************************************************
 * Return the length of an entity.
 * ************************************************************** */
void

#ifdef THINK_C
Lengthx (return_ptr, n_args, d_arg)
#else
Length (return_ptr, n_args, d_arg)
#endif

     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;

  /* Check n_args */
  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("length: 1 argument allowed", 0);
  }

  /* get arg from list */
  arg = bltin_get_arg ("length", d_arg, 1);

  switch (arg.type)
  {
  case CONSTANT:
  case iCONSTANT:
    *return_ptr = (VPTR) scalar_Create (1.0);
    break;

  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      *return_ptr = (VPTR) scalar_Create (1.0);
      break;

    case MATRIX:
      if (MNR (e_data (arg.u.ent)) >= MNC (e_data (arg.u.ent)))
	*return_ptr = (VPTR) scalar_Create ((double) MNR (e_data (arg.u.ent)));
      else
	*return_ptr = (VPTR) scalar_Create ((double) MNC (e_data (arg.u.ent)));
      break;

    case STRING:
      *return_ptr = (VPTR)
	scalar_Create ((double) string_GetLength (e_data (arg.u.ent)));
      break;

    case BTREE:
      *return_ptr = (VPTR)
	scalar_Create ((double) btree_GetRealNumNodes (e_data (arg.u.ent)));
      break;

    default:
      error_1 (e_name (arg.u.ent), "invalid type for length()");
      break;
    }
    remove_tmp_destroy (arg.u.ent);
  }
}

/* **************************************************************
 * Return the "class" of an object as a character string.
 * ************************************************************** */
void
Class (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *str_tmp;
  Datum arg;

  str_tmp = 0;			/* Initialize */

  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("class() takes a single argument", (char *) 0);
  }
  
  arg = bltin_get_arg ("class", d_arg, 1);

  switch (arg.type)
  {
  case CONSTANT:
    str_tmp = cpstr ("num");
    break;

  case iCONSTANT:
    str_tmp = cpstr ("num");
    break;

  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      str_tmp = cpstr ("num");
      break;

    case MATRIX:
      if (MTYPE (e_data (arg.u.ent)) != STRING)
	str_tmp = cpstr ("num");
      else
	str_tmp = cpstr ("string");
      break;

    case BTREE:
      str_tmp = cpstr ("list");

      break;
    case STRING:
      str_tmp = cpstr ("string");
      break;

    case BLTIN:
      str_tmp = cpstr ("function");
      break;

    case U_FUNCTION:
      str_tmp = cpstr ("function");
      break;

    default:
      error_1 ("invalid type for class()", 0);
      break;
    }
    remove_tmp_destroy (arg.u.ent);
    break;

  default:
    error_1 ("class: invalid argument", 0);
    break;
  }

  /*
   * Do NOT free() str_tmp, its value is used by string_Create().
   */

  *return_ptr = (VPTR) string_Create (str_tmp);
}

/* **************************************************************
 * Change the numerical output format.
 * format()    should print the current values of the formats
 * ************************************************************** */

/* Control output print formats */
static int FWIDTH_DEFAULT = 9;
static int FPREC_DEFAULT = 3;
static int fwidth = 9;
static int fprec = 3;

void
Format (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{

  if (n_args > 2)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("format: takes two args at most", 0);
  }

  if (n_args == 1)
  {
    /* Set format from user input */
    fprec = (int) bltin_get_numeric_double ("format", d_arg, 1);
  }
  else if (n_args == 2)
  {
    fwidth = (int) bltin_get_numeric_double ("format", d_arg, 1);
    fprec = (int) bltin_get_numeric_double ("format", d_arg, 2);
  }
  else
  {
    /* Reset format to default */
    fwidth = FWIDTH_DEFAULT;
    fprec = FPREC_DEFAULT;
  }
  *return_ptr = (VPTR) scalar_Create (1.0);
}

int
get_fwidth ()
{
  return (fwidth);
}

int
get_fprec ()
{
  return (fprec);
}

/* **************************************************************
 * Open a diary file.
 * ************************************************************** */
#include <sys/types.h>

#ifdef __STDC__
#include <time.h>
#endif /* __STDC__ */

#ifdef HAVE_TIME_H
#include <time.h>
#endif

static int write_diary;
static FILE *diary_file_ptr;
static char *diary_filenm;

void
Diary (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *string;
  FILE *fn;
  time_t r_time;
  ListNode *FN = 0;

  if (n_args > 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("diary: 1 argument allowed", 0);
  }

  /* Handle diary() args */
  if (n_args == 1)
  {
    if (write_diary)
      error_1 ("diary: only one diary file may be open at a time", 0);
    FN = bltin_get_string ("diary", d_arg, 1);
    string = string_GetString (e_data (FN));
  }
  else
  {				/* No args */
    if (write_diary)
    {
      /*
       * n_args = 0, and a diary file is already open
       * close the existing diary file, and return
       */
      close_file_ds (diary_filenm);
      write_diary = 0;
      diary_file_ptr = 0;
      *return_ptr = (VPTR) scalar_Create (1.0);
      return;
    }
    else
      /*
       * n_args = 0, and no diary file open, open the
       * default diary file "DIARY"
       */
      string = cpstr ("DIARY");
  }

  if ((fn = get_file_ds (string, "w", 0)) == 0)
  {
    warning_1 (string, "cannot open for write");
    *return_ptr = (VPTR) scalar_Create (0.0);
    if (FN)
      remove_tmp_destroy (FN);
    return;
  }

  /* Set static variables */
  write_diary = 1;
  diary_file_ptr = fn;
  diary_filenm = cpstr (string);

  /* Write out a header to diary file */
  r_time = time (0);
  fprintf (fn, "RLaB diary file: %s. Opened %s\n",
	   string, ctime (&r_time));

  *return_ptr = (VPTR) scalar_Create (1.0);

  if (FN)
    remove_tmp_destroy (FN);
}

int
get_write_diary ()
{
  return (write_diary);
}

FILE *
get_diary_file_ptr ()
{
  return (diary_file_ptr);
}

void btree_write _PROTO ((Btree * btree, FILE * fn));

/* **************************************************************
 * Write an object to a file.
 * ************************************************************** */
void
Write (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int i;
  char *string;
  Datum arg;
  ListNode *S;
  FILE *fn;

  /* Check n_args */
  if (n_args > 1)
  {
    /* Standard user-specified write() */
  }
  else
  {
    warning_1 ("write: at least 2 arguments required", 0);
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }

  /* get file name from argument list */
  S = bltin_get_string ("write", d_arg, 1);
  string = string_GetString (e_data (S));

  if ((fn = get_file_ds (string, "w", 0)) == 0)
  {
    warning_1 (string, "cannot open for write");
    *return_ptr = (VPTR) scalar_Create (0.0);
    remove_tmp_destroy (S);
    return;
  }

  /* Now move through arg list writing objects to file */
  for (i = 2; i <= n_args; i++)
  {
    arg = bltin_get_arg ("write", d_arg, i);
    switch (arg.type)
    {
    case CONSTANT:
      scalar_Write (scalar_Create (arg.u.val), fn);
      break;
    case iCONSTANT:
      scalar_Write (scalar_CreateC (0.0, arg.u.val), fn);
      break;
    case ENTITY:
      switch (e_type (arg.u.ent))
      {
      case SCALAR:
	scalar_Write (e_data (arg.u.ent), fn);
	break;
      case MATRIX:
	matrix_Write (e_data (arg.u.ent), fn);
	break;
      case BTREE:
	btree_write (e_data (arg.u.ent), fn);
	break;
      case STRING:
	string_Write (e_data (arg.u.ent), fn);
	break;
      case UNDEF:
	break;
      default:
	warning_1 (e_name (arg.u.ent), "invalid object for write()");
	break;
      }
      remove_tmp_destroy (arg.u.ent);
      break;
    default:
      warning_1 ("invalid object for write()", (char *) 0);
      break;
    }
  }
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
btree_write (btree, fn)
     Btree *btree;
     FILE *fn;
{
  btree_Write (btree, fn);
  fflush (fn);
}

/* **************************************************************
 * Read RLaB objects from a file. The default behavior is to read
 * ALL of the objects in the file, and install them on the GLOBAL
 * list. If the file contains a LIST, it is also put on the GLOBAL
 * list.
 * ************************************************************** */
void
Read (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *name, *string, type[20];
  FILE *fn;
  Btree *btree, *symtab;
  Datum arg2;
  ListNode *lnode, *FN;
  Matrix *m;
  Scalar *s;
  String *str;

  /* Check n_args */
  if (n_args > 2)
    error_1 ("read: wrong number of input arguments", 0);

  /* get file name from argument list, always the 1st argument */
  FN = bltin_get_string ("read", d_arg, 1);
  string = string_GetString (e_data (FN));

  if (n_args == 1)
  {
    /* Use the global symbol table */
    symtab = get_symtab_ptr ();
  }
  else
  {
    /* Use a user-supplied list */
    arg2 = d_arg[1];

    /* Check to see if variable exists, etc... */
    if (arg2.type != ENTITY)
      error_1 ("read: 2nd argument must be a variable", 0);
    if (e_type (arg2.u.ent) != UNDEF)
      listNode_DestroyDataOnly (arg2.u.ent);
    symtab = btree_Create ();
    btree_SetName (symtab, cpstr (e_name (arg2.u.ent)));
    listNode_AttachData (arg2.u.ent, BTREE, symtab, btree_Destroy);
  }

  if ((fn = get_file_ds (string, "r", 0)) == 0)
  {
    warning_1 (string, "cannot open for read");
    *return_ptr = (VPTR) scalar_Create (0.0);
    remove_tmp_destroy (FN);
    return;
  }

  /* 
   * Save current position in file, we will need to go back to
   * this point when we hand the read task off to the object
   * libraries
   */
  while (fscanf (fn, "# %s :", type) != EOF)
  {
    if (!strcmp ("scalar", type))
    {
      s = scalar_Read (fn);
      name = scalar_GetName (s);
      if ((lnode = btree_FindNode (symtab, name)) != 0)
      {
	listNode_DestroyDataOnly (lnode);
	listNode_AttachData (lnode, SCALAR, s, scalar_Destroy);
      }
      else
	install (symtab, cpstr (name), SCALAR, s);
    }
    else if (!strcmp ("matrix", type))
    {
      m = matrix_Read (fn);
      name = matrix_GetName (m);
      if ((lnode = btree_FindNode (symtab, name)) != 0)
      {
	listNode_DestroyDataOnly (lnode);
	listNode_AttachData (lnode, MATRIX, m, matrix_Destroy);
      }
      else
	install (symtab, cpstr (name), MATRIX, m);
    }
    else if (!strcmp ("string", type))
    {
      str = string_Read (fn);
      name = string_GetName (str);
      if ((lnode = btree_FindNode (symtab, name)) != 0)
      {
	listNode_DestroyDataOnly (lnode);
	listNode_AttachData (lnode, STRING, str, string_Destroy);
      }
      else
	install (symtab, cpstr (name), STRING, str);
    }
    else if (!strcmp ("list", type))
    {
      btree = btree_Read (fn);
      name = btree_GetName (btree);
      if ((lnode = btree_FindNode (symtab, name)) != 0)
      {
	listNode_DestroyDataOnly (lnode);
	listNode_AttachData (lnode, BTREE, btree, btree_Destroy);
      }
      else
	install (symtab, cpstr (name), BTREE, btree);
    }
    else
    {
      close_file_ds (string);
      remove_tmp_destroy (FN);
      error_1 (type, "do not know how to read this type");
    }
  }

  close_file_ds (string);

  /* Maybe we should return the list we wrote to ? */
  *return_ptr = (VPTR) scalar_Create (1.0);
  remove_tmp_destroy (FN);
  return;
}

/* **************************************************************
 * Reshape a matrix.
 * ************************************************************** */
void
Reshape (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  double drow, dcol;
  ListNode *M;

  if (n_args != 3)
    error_1 ("reshape: 3 arguments required", 0);

  M = bltin_get_matrix ("reshape", d_arg, 1);
  drow = bltin_get_numeric_double ("reshape", d_arg, 2);
  dcol = bltin_get_numeric_double ("reshape", d_arg, 3);

  *return_ptr = (VPTR)
    matrix_Reshape (e_data (M), (int) drow, (int) dcol);

  remove_tmp_destroy (M);
  return;
}

/* **************************************************************
 * Show the attributes of an object.
 * ************************************************************** */
void
Show (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;
  char str[80];
  Matrix *m = 0;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("show: 1 argument allowed", 0);

  /* Get object from argument list */
  arg = bltin_get_arg ("show", d_arg, 1);

  if (arg.type != ENTITY)
    error_1 ("show takes an entity as an argument", 0);

  switch (e_type (arg.u.ent))
  {
  case SCALAR:
    m = matrix_CreateS (5, 1);

    if (e_name (arg.u.ent) == 0)
      sprintf (str, "   name:      %s", "NULL");
    else
      sprintf (str, "   name:      %s", e_name (arg.u.ent));
    MATs (m, 1, 1) = cpstr (str);

    sprintf (str, "   class:     num");
    MATs (m, 2, 1) = cpstr (str);

    if (SVALi (e_data (arg.u.ent)) == 0.0)
      sprintf (str, "   type:      real");
    else
      sprintf (str, "   type:      complex");
    MATs (m, 3, 1) = cpstr (str);

    sprintf (str, "     nr:      %d", 1);
    MATs (m, 4, 1) = cpstr (str);

    sprintf (str, "     nc:      %d", 1);
    MATs (m, 5, 1) = cpstr (str);

    break;

  case MATRIX:
    m = matrix_CreateS (5, 1);

    if (e_name (arg.u.ent) == 0)
      sprintf (str, "   name:      %s", "NULL");
    else
      sprintf (str, "   name:      %s", e_name (arg.u.ent));
    MATs (m, 1, 1) = cpstr (str);

    if (MTYPE (e_data (arg.u.ent)) != STRING)
    {
      sprintf (str, "   class:     num");
      MATs (m, 2, 1) = cpstr (str);

      if (MTYPE (e_data (arg.u.ent)) == REAL)
	sprintf (str, "   type:      real");
      else if (MTYPE (e_data (arg.u.ent)) == COMPLEX)
	sprintf (str, "   type:      complex");
      MATs (m, 3, 1) = cpstr (str);
    }
    else
    {
      sprintf (str, "   class:     string");
      MATs (m, 2, 1) = cpstr (str);

      sprintf (str, "   type:      string");
      MATs (m, 3, 1) = cpstr (str);
    }

    sprintf (str, "     nr:      %d", MNR (e_data (arg.u.ent)));
    MATs (m, 4, 1) = cpstr (str);

    sprintf (str, "     nc:      %d", MNC (e_data (arg.u.ent)));
    MATs (m, 5, 1) = cpstr (str);

    break;

  case STRING:
    m = matrix_CreateS (5, 1);

    if (e_name (arg.u.ent) == 0)
      sprintf (str, "   name:      %s", "NULL");
    else
      sprintf (str, "   name:      %s", e_name (arg.u.ent));
    MATs (m, 1, 1) = cpstr (str);

    sprintf (str, "   class:     string");
    MATs (m, 2, 1) = cpstr (str);

    sprintf (str, "    type:     string");
    MATs (m, 3, 1) = cpstr (str);

    sprintf (str, "     nr:      %d", 1);
    MATs (m, 4, 1) = cpstr (str);

    sprintf (str, "     nc:      %d", 1);
    MATs (m, 5, 1) = cpstr (str);

    break;

  case BTREE:
    m = matrix_CreateS (3, 1);

    if (e_name (arg.u.ent) == 0)
      sprintf (str, "   name:      %s", "NULL");
    else
      sprintf (str, "   name:      %s", e_name (arg.u.ent));
    MATs (m, 1, 1) = cpstr (str);

    sprintf (str, "   class:     list");
    MATs (m, 2, 1) = cpstr (str);

    sprintf (str, "       n:     %d",
	     btree_GetRealNumNodes (e_data (arg.u.ent)));
    MATs (m, 3, 1) = cpstr (str);

    break;

  case BLTIN:
    m = matrix_CreateS (3, 1);

    if (e_name (arg.u.ent) == 0)
      sprintf (str, "   name:      %s", "NULL");
    else
      sprintf (str, "   name:      %s", e_name (arg.u.ent));
    MATs (m, 1, 1) = cpstr (str);

    sprintf (str, "   class:     function");
    MATs (m, 2, 1) = cpstr (str);

    sprintf (str, "    type:     builtin");
    MATs (m, 3, 1) = cpstr (str);

    break;

  case U_FUNCTION:
    m = matrix_CreateS (3, 1);

    if (e_name (arg.u.ent) == 0)
      sprintf (str, "   name:      %s", "NULL");
    else
      sprintf (str, "   name:      %s", e_name (arg.u.ent));
    MATs (m, 1, 1) = cpstr (str);

    sprintf (str, "   class:     function");
    MATs (m, 2, 1) = cpstr (str);

    sprintf (str, "    type:     user");
    MATs (m, 3, 1) = cpstr (str);

    break;

  default:
    error_1 ("invalid type for show()", 0);
    break;
  }
  remove_tmp_destroy (arg.u.ent);
  *return_ptr = (VPTR) m;
}

/* **************************************************************
 * User's error() function. longjmp() back to prompt.
 * ************************************************************** */
void
Error (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *E;

  if (n_args == 0)
    /* Default error message */
    error_1 ("USER-RAISED-ERROR", 0);
  else
  {
    /*
     * We can't free the string here, cause we
     * need it for error_1().
     */
    E = bltin_get_string ("error", d_arg, 1);
    error_1 (string_GetString (e_data (E)), 0);
  }
}

/* **************************************************************
 * Read a generic matrix from a file. The format is:
 * All lines:    whitespace separated numbers.
 * ************************************************************** */
void
ReadM (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *string;
  int block_size = 0;
  FILE *fn;
  ListNode *FN;
  Matrix *m;

  /* Check n_args */
  if (n_args != 1 && n_args != 2)
    error_1 ("readm: 1 or 2 arguments allowed", 0);

  FN = bltin_get_string ("readm", d_arg, 1);
  string = string_GetString (e_data (FN));

  if (n_args == 2)
  {
    /* Get block_size from argument list */
    block_size = bltin_get_numeric_double ("rcond", d_arg, 2);
  }

  /* Try and open() the file */
  if ((fn = get_file_ds (string, "r", 0)) == 0)
  {
    warning_1 (string, "cannot open for read");
    *return_ptr = (VPTR) scalar_Create (0.0);
    remove_tmp_destroy (FN);
    return;
  }

  if ((m = matrix_ReadGeneric (fn, block_size)) == 0)
  {
    close_file_ds (string);
    remove_tmp_destroy (FN);
    error_1 ("readm: error during read", 0);
  }

  close_file_ds (string);
  *return_ptr = (VPTR) m;
  remove_tmp_destroy (FN);
}

void
WriteM (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *string;
  FILE *fn;
  ListNode *M, *S;
  Matrix *m;

  /* Check n_args */
  if (n_args != 2)
    error_1 ("writem: 2 arguments required", 0);

  S = bltin_get_string ("writem", d_arg, 1);
  M = bltin_get_numeric_matrix ("writem", d_arg, 2);
  string = string_GetString (e_data (S));
  m = (Matrix *) e_data (M);

  /* Try and open() the file */
  if ((fn = get_file_ds (string, "w", 0)) == 0)
  {
    warning_1 (string, "cannot open for write");
    *return_ptr = (VPTR) scalar_Create (0.0);
    remove_tmp_destroy (S);
    remove_tmp_destroy (M);
    return;
  }
  
  matrix_WriteGeneric (m, fn);
  remove_tmp_destroy (S);
  remove_tmp_destroy (M);
  *return_ptr = (VPTR) scalar_Create (1.0);
}

/* **************************************************************
 * Return 1 if any value of a matrix is non-zero.
 * Return 0 otherwise.
 * ************************************************************** */
void
Any (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *M;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("any: 1 argument allowed", 0);
  
  M = bltin_get_numeric_matrix ("any", d_arg, 1);
  *return_ptr = (VPTR) matrix_Any (e_data (M));
  remove_tmp_destroy (M);
}

/* **************************************************************
 * Return 1 if all values of a matrix are non-zero
 * Return 0 otherwise.
 * ************************************************************** */
void
All (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *M;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("all: 1 argument allowed", 0);

  M = bltin_get_numeric_matrix ("all", d_arg, 1);
  *return_ptr = (VPTR) matrix_All (e_data (M));
  remove_tmp_destroy (M);
}

/* **************************************************************
 * Clear a variable's data.
 * ************************************************************** */
void
Clear (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int i;
  Datum arg;

  /* Check n_args */
  if (n_args == 0)
    error_1 ("clear: 1 argument allowed", 0);

  for (i = 1; i <= n_args; i++)
  {
    arg = bltin_get_arg ("clear", d_arg, i);

    if ((arg.type == CONSTANT) || (arg.type == iCONSTANT))
      error_1 ("clear: will not operate on CONSTANTS", 0);

    if (e_type (arg.u.ent) == BLTIN)
      error_1 (e_name (arg.u.ent), "cannot clear() a built-in function");

    listNode_DestroyDataOnly (arg.u.ent);
    listNode_SetType (arg.u.ent, UNDEF);

    remove_tmp_destroy (arg.u.ent);
  }
  *return_ptr = (VPTR) scalar_Create ((double) i - 1);
}

/* **************************************************************
 * Get a matrix from the argument list of a bltin function.
 * These functions return the argument or a transformed copy.
 * The user is repsonsible for cleaning up the return'ed matrix
 * (see matrix_clean()).
 * ************************************************************** */

/*
 * Return a SCALAR or MATRIX if possible.
 */

ListNode *
bltin_get_numeric (bltin_name, arg_array, arg_number)
     char *bltin_name;
     Datum *arg_array;
     int arg_number;
{
  Datum d;
  ListNode *ent = 0;
  Matrix *m;

  /* get the requested arg */
  d = arg_array[arg_number - 1];

  switch (d.type)
  {
  case CONSTANT:
    m = matrix_Create (1, 1);
    MAT (m, 1, 1) = d.u.val;
    ent = install_tmp (MATRIX, m, matrix_Destroy);
    break;

  case iCONSTANT:
    m = matrix_CreateC (1, 1);
    matrix_Zero (m);
    MATi (m, 1, 1) = d.u.val;
    ent = install_tmp (MATRIX, m, matrix_Destroy);
    break;

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      ent = (ListNode *) d.u.ent;
      break;

    case MATRIX:
      if (MTYPE (e_data (d.u.ent)) == STRING)
	error_1 (bltin_name, "STRING matrix invalid as argument");
      ent = (ListNode *) d.u.ent;
      break;

    case STRING:
      error_1 (bltin_name, "STRING invalid as argument");
      break;

    case BLTIN:
      error_1 (bltin_name, "BUILTIN FUNCTION invalid as argument");
      break;

    case U_FUNCTION:
      error_1 (bltin_name, "USER FUNCTION invalid as argument");
      break;

    case BTREE:
      error_1 (bltin_name, "LIST invalid as argument");
      break;
    
    default:
      error_1 (bltin_name, "invalid argument");
      break;
    }
    break;

  default:
    error_1 (bltin_name, "invalid argument");
    break;
  }

  return (ent);
}

ListNode *
bltin_get_numeric_matrix (bltin_name, arg_array, arg_number)
     char *bltin_name;
     Datum *arg_array;
     int arg_number;
{
  Datum d;
  ListNode *ent = 0;
  Matrix *m;

  /* get the requested arg */
  d = arg_array[arg_number - 1];

  switch (d.type)
  {
  case CONSTANT:
    m = matrix_Create (1, 1);
    MAT (m, 1, 1) = d.u.val;
    ent = install_tmp (MATRIX, m, matrix_Destroy);
    break;

  case iCONSTANT:
    m = matrix_CreateC (1, 1);
    matrix_Zero (m);
    MATi (m, 1, 1) = d.u.val;
    ent = install_tmp (MATRIX, m, matrix_Destroy);
    break;

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      if (SVALi (e_data (d.u.ent)) == 0.0)
      {
	m = matrix_Create (1, 1);
	MAT (m, 1, 1) = SVALr (e_data (d.u.ent));
      }
      else
      {
	m = matrix_CreateC (1, 1);
	MATr (m, 1, 1) = SVALr (e_data (d.u.ent));
	MATi (m, 1, 1) = SVALi (e_data (d.u.ent));
      }
      ent = install_tmp (MATRIX, m, matrix_Destroy);
      remove_tmp_destroy (d.u.ent);
      break;

    case MATRIX:
      if (MTYPE (e_data (d.u.ent)) == STRING)
	error_1 (bltin_name, "STRING matrix invalid as argument");
      ent = (ListNode *) d.u.ent;
      break;

    case STRING:
      error_1 (bltin_name, "STRING invalid as argument");
      break;

    case BLTIN:
      error_1 (bltin_name, "BUILTIN FUNCTION invalid as argument");
      break;

    case U_FUNCTION:
      error_1 (bltin_name, "USER FUNCTION invalid as argument");
      break;

    case BTREE:
      error_1 (bltin_name, "LIST invalid as argument");
      break;
    
    default:
      error_1 (bltin_name, "invalid argument");
      break;
    }
    break;

  default:
    error_1 (bltin_name, "invalid argument");
    break;
  }

  return (ent);
}

/*
 * Get a builtin argument. Coerce it into a matrix if
 * possible.
 */

ListNode *
bltin_get_matrix (bltin_name, arg_array, arg_number)
     char *bltin_name;
     Datum *arg_array;
     int arg_number;
{
  Datum d;
  ListNode *ent = 0;
  Matrix *m;

  /* get the requested arg */
  d = arg_array[arg_number - 1];

  switch (d.type)
  {
  case CONSTANT:
    m = matrix_Create (1, 1);
    MAT (m, 1, 1) = d.u.val;
    ent = install_tmp (MATRIX, m, matrix_Destroy);
    break;

  case iCONSTANT:
    m = matrix_CreateC (1, 1);
    matrix_Zero (m);
    MATi (m, 1, 1) = d.u.val;
    ent = install_tmp (MATRIX, m, matrix_Destroy);
    break;

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      if (SVALi (e_data (d.u.ent)) == 0.0)
      {
	m = matrix_Create (1, 1);
	MAT (m, 1, 1) = SVALr (e_data (d.u.ent));
      }
      else
      {
	m = matrix_CreateC (1, 1);
	MATr (m, 1, 1) = SVALr (e_data (d.u.ent));
	MATi (m, 1, 1) = SVALi (e_data (d.u.ent));
      }
      ent = install_tmp (MATRIX, m, matrix_Destroy);
      remove_tmp_destroy (d.u.ent);
      break;

    case MATRIX:
      ent = (ListNode *) d.u.ent;
      break;

    case STRING:
      m = matrix_CreateS (1, 1);
      MATs (m, 1, 1) = cpstr (string_GetString (e_data (d.u.ent)));
      ent = install_tmp (MATRIX, m, matrix_Destroy);
      remove_tmp_destroy (d.u.ent);      
      break;

    case BLTIN:
      error_1 (bltin_name, "BUILTIN FUNCTION invalid as argument");
      break;

    case U_FUNCTION:
      error_1 (bltin_name, "USER FUNCTION invalid as argument");
      break;

    case BTREE:
      error_1 (bltin_name, "LIST invalid as argument");
      break;
    
    default:
      error_1 (bltin_name, "invalid argument");
      break;
    }
    break;

  default:
    error_1 (bltin_name, "invalid argument");
    break;
  }

  return (ent);
}


/*
 * Return a single double value.
 */

double
bltin_get_numeric_double (bltin_name, arg_array, arg_number)
     char *bltin_name;
     Datum *arg_array;
     int arg_number;
{
  Datum d;
  double dval = 0.0;

  /* get the requested arg */
  d = arg_array[arg_number - 1];

  switch (d.type)
  {
  case CONSTANT:
    dval = (double) d.u.val;
    break;

  case iCONSTANT:
    dval = (double) d.u.val;
    break;

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      if (SVALi (e_data (d.u.ent)) == 0.0)
      {
	dval = SVALr (e_data (d.u.ent));
      }
      else
      {
	dval = SVALr (e_data (d.u.ent));
      }
      remove_tmp_destroy (d.u.ent);
      break;

    case MATRIX:
      if (MTYPE (e_data (d.u.ent)) == REAL)
      {
	dval = (double) MAT (e_data (d.u.ent), 1, 1);
	remove_tmp_destroy (d.u.ent);
      }
      else if (MTYPE (e_data (d.u.ent)) == COMPLEX)
      {
	dval = (double) MATr (e_data (d.u.ent), 1, 1);
	remove_tmp_destroy (d.u.ent);
      }
      else if (MTYPE (e_data (d.u.ent)) == STRING)
	error_1 (bltin_name, "STRING invalid as argument");
      break;

    case STRING:
      error_1 (bltin_name, "STRING invalid as argument");
      break;

    case BLTIN:
      error_1 (bltin_name, "BUILTIN FUNCTION invalid as argument");
      break;

    case U_FUNCTION:
      error_1 (bltin_name, "USER FUNCTION invalid as argument");
      break;
      
    case BTREE:
      error_1 (bltin_name, "LIST invalid as argument");
      break;
      
    default:
      error_1 (bltin_name, "invalid argument");
      break;
    }
    break;
    
  default:
    error_1 (bltin_name, "invalid argument");
    break;
  }
  
  return (dval);
}

/*
 * Return a scalar string.
 */

ListNode *
bltin_get_string (bltin_name, arg_array, arg_number)
     char *bltin_name;
     Datum *arg_array;
     int arg_number;
{
  Datum d;
  ListNode *ent = 0;
  String *s;

  /* get the requested arg */
  d = arg_array[arg_number - 1];

  switch (d.type)
  {
  case CONSTANT:
    error_1 (bltin_name, "CONSTANT invalid as argument");
    break;

  case iCONSTANT:
    error_1 (bltin_name, "CONSTANT invalid as argument");
    break;

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      error_1 (bltin_name, "NUMERIC invalid as argument");
      break;

    case MATRIX:
      if (MTYPE (e_data (d.u.ent)) == STRING)
      {
	/* Convert to a STRING */
	s = string_Create (cpstr (MATs (e_data (d.u.ent), 1, 1)));
	ent = install_tmp (STRING, s, string_Destroy);
	remove_tmp_destroy (d.u.ent);
      }
      else
	error_1 (bltin_name, "NUMERIC invalid as argument");
      break;

    case STRING:
      ent = (ListNode *) d.u.ent;
      break;

    case BLTIN:
      error_1 (bltin_name, "BUILTIN FUNCTION invalid as argument");
      break;

    case U_FUNCTION:
      error_1 (bltin_name, "USER FUNCTION invalid as argument");
      break;

    case BTREE:
      error_1 (bltin_name, "LIST invalid as argument");
      break;
    
    default:
      error_1 (bltin_name, "invalid argument");
      break;
    }
    break;

  default:
    error_1 (bltin_name, "invalid argument");
    break;
  }
  
  return (ent);
}

/*
 * Return a string matrix.
 */

ListNode *
bltin_get_string_matrix (bltin_name, arg_array, arg_number)
     char *bltin_name;
     Datum *arg_array;
     int arg_number;
{
  Datum d;
  ListNode *ent = 0;
  Matrix *m;

  /* get the requested arg */
  d = arg_array[arg_number - 1];

  switch (d.type)
  {
  case CONSTANT:
    error_1 (bltin_name, "CONSTANT invalid as argument");
    break;

  case iCONSTANT:
    error_1 (bltin_name, "CONSTANT invalid as argument");
    break;

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      error_1 (bltin_name, "NUMERIC invalid as argument");
      break;

    case MATRIX:
      if (MTYPE (e_data (d.u.ent)) == STRING)
      {
	ent = (ListNode *) d.u.ent;
      }
      else
	error_1 (bltin_name, "NUMERIC invalid as argument");
      break;

    case STRING:
      m = matrix_CreateS (1, 1);
      MATs (m, 1, 1) = cpstr (string_GetString (e_data (d.u.ent)));
      ent = install_tmp (MATRIX, m, matrix_Destroy);
      remove_tmp_destroy (d.u.ent);
      break;

    case BLTIN:
      error_1 (bltin_name, "BUILTIN FUNCTION invalid as argument");
      break;

    case U_FUNCTION:
      error_1 (bltin_name, "USER FUNCTION invalid as argument");
      break;

    case BTREE:
      error_1 (bltin_name, "LIST invalid as argument");
      break;
    
    default:
      error_1 (bltin_name, "invalid argument");
      break;
    }
    break;

  default:
    error_1 (bltin_name, "invalid argument");
    break;
  }
  
  return (ent);
}

/*
 * Return a list.
 */

ListNode *
bltin_get_list (bltin_name, arg_array, arg_number)
     char *bltin_name;
     Datum *arg_array;
     int arg_number;
{
  Datum d;
  ListNode *ent = 0;

  /* get the requested arg */
  d = arg_array[arg_number - 1];

  switch (d.type)
  {
  case CONSTANT:
    error_1 (bltin_name, "CONSTANT invalid as argument");
    break;

  case iCONSTANT:
    error_1 (bltin_name, "CONSTANT invalid as argument");
    break;

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      error_1 (bltin_name, "NUMERIC invalid as argument");
      break;

    case MATRIX:
      error_1 (bltin_name, "NUMERIC invalid as argument");
      break;

    case STRING:
      error_1 (bltin_name, "STRING invalid as argument");
      break;

    case BLTIN:
      error_1 (bltin_name, "BUILTIN FUNCTION invalid as argument");
      break;

    case U_FUNCTION:
      error_1 (bltin_name, "USER FUNCTION invalid as argument");
      break;

    case BTREE:
      ent = (ListNode *) d.u.ent;
      break;
    
    default:
      error_1 (bltin_name, "invalid argument");
      break;
    }
    break;

  default:
    error_1 (bltin_name, "invalid argument");
    break;
  }
  
  return (ent);
}

Datum
bltin_get_arg (bltin_name, arg_array, arg_number)
     char *bltin_name;
     Datum *arg_array;
     int arg_number;
{
  Datum d;
  
  /* get the requested arg */
  d = arg_array[arg_number - 1];

  return (d);
}

/*
 * Return a Function, user or bltin.
 */

ListNode *
bltin_get_func (bltin_name, arg_array, arg_number)
     char *bltin_name;
     Datum *arg_array;
     int arg_number;
{
  Datum d;
  ListNode *ent = 0;

  /* get the requested arg */
  d = arg_array[arg_number - 1];

  switch (d.type)
  {
  case CONSTANT:
    error_1 (bltin_name, "CONSTANT invalid as argument");
    break;

  case iCONSTANT:
    error_1 (bltin_name, "CONSTANT invalid as argument");
    break;

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      error_1 (bltin_name, "NUMERIC invalid as argument");
      break;

    case MATRIX:
      error_1 (bltin_name, "NUMERIC invalid as argument");
      break;

    case STRING:
      error_1 (bltin_name, "STRING invalid as argument");
      break;

    case BLTIN:
      ent = (ListNode *) d.u.ent;
      break;

    case U_FUNCTION:
      ent = (ListNode *) d.u.ent;
      break;

    case BTREE:
      error_1 (bltin_name, "LIST invalid as argument");
      break;
    
    case UNDEF:
      error_1 (bltin_name, "UNDEF invalid as argument");
      break;
    
    default:
      error_1 (bltin_name, "invalid argument");
      break;
    }
    break;

  default:
    error_1 (bltin_name, "invalid argument");
    break;
  }
  
  return (ent);
}

/*
 * Return a User-Function
 */

ListNode *
bltin_get_ufunc (bltin_name, arg_array, arg_number)
     char *bltin_name;
     Datum *arg_array;
     int arg_number;
{
  Datum d;
  ListNode *ent = 0;

  /* get the requested arg */
  d = arg_array[arg_number - 1];

  switch (d.type)
  {
  case CONSTANT:
    error_1 (bltin_name, "CONSTANT invalid as argument");
    break;

  case iCONSTANT:
    error_1 (bltin_name, "CONSTANT invalid as argument");
    break;

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      error_1 (bltin_name, "NUMERIC invalid as argument");
      break;

    case MATRIX:
      error_1 (bltin_name, "NUMERIC invalid as argument");
      break;

    case STRING:
      error_1 (bltin_name, "STRING invalid as argument");
      break;

    case BLTIN:
      error_1 (bltin_name, "BUILTIN FUNCTION invalid as argument");
      break;

    case U_FUNCTION:
      ent = (ListNode *) d.u.ent;
      break;

    case BTREE:
      error_1 (bltin_name, "LIST invalid as argument");
      break;
    
    default:
      error_1 (bltin_name, "invalid argument");
      break;
    }
    break;

  default:
    error_1 (bltin_name, "invalid argument");
    break;
  }
  
  return (ent);
}

/*
 * Return an entity, convert anything possible to
 * a matrix. Copy the name of the entity as it is
 * converted. If the entity is a CONSTANT, then
 * give it a name...
 */

ListNode *
bltin_get_entity (bltin_name, arg_array, arg_number)
     char *bltin_name;
     Datum *arg_array;
     int arg_number;
{
  Datum d;
  ListNode *ent = 0;
  Matrix *m;

  /* get the requested arg */
  d = arg_array[arg_number - 1];

  switch (d.type)
  {
  case CONSTANT:
    m = matrix_Create (1, 1);
    MAT (m, 1, 1) = d.u.val;
    matrix_SetName (m, cpstr ("CONSTANT"));
    ent = install_tmp (MATRIX, m, matrix_Destroy);
    break;

  case iCONSTANT:
    m = matrix_CreateC (1, 1);
    matrix_Zero (m);
    MATi (m, 1, 1) = d.u.val;
    matrix_SetName (m, cpstr ("iCONSTANT"));
    ent = install_tmp (MATRIX, m, matrix_Destroy);
    break;

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      if (SVALi (e_data (d.u.ent)) == 0.0)
      {
	m = matrix_Create (1, 1);
	MAT (m, 1, 1) = SVALr (e_data (d.u.ent));
      }
      else
      {
	m = matrix_CreateC (1, 1);
	MATr (m, 1, 1) = SVALr (e_data (d.u.ent));
	MATi (m, 1, 1) = SVALi (e_data (d.u.ent));
      }
      matrix_SetName (m, cpstr (scalar_GetName (e_data (d.u.ent))));
      ent = install_tmp (MATRIX, m, matrix_Destroy);
      break;

    case MATRIX:
      ent = (ListNode *) d.u.ent;
      break;

    case STRING:
      m = matrix_CreateS (1, 1);
      MATs (m, 1, 1) = cpstr (string_GetString (e_data (d.u.ent)));
      matrix_SetName (m, cpstr (string_GetName (e_data (d.u.ent))));
      ent = install_tmp (MATRIX, m, matrix_Destroy);
      break;

    case BLTIN:
    case U_FUNCTION:
    case BTREE:
    case UNDEF:
      ent = (ListNode *) d.u.ent;
      break;
    
    default:
      error_1 (bltin_name, "invalid argument");
      break;
    }
    break;

  default:
    error_1 (bltin_name, "invalid argument");
    break;
  }

  return (ent);
}

/*
 * Used to clean up arguments in case of an error.
 */

void
clean_bltin_args (d_arg, n_args)
     Datum *d_arg;
     int n_args;
{
  int i;
  for (i = 0; i < n_args; i++)
  {
    if (d_arg[i].type == ENTITY)
      remove_tmp_destroy (d_arg[i].u.ent);
  }
}
