/* symbol.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "symbol.h"
#include "list.h"
#include "mem.h"
#include "scalar.h"
#include "matrix.h"
#include "r_string.h"
#include "function.h"
#include "btree.h"
#include "util.h"

#ifdef __STDC__
#include <stdlib.h>
#else
#include <malloc.h>
#endif

/*
 * We'll have to have the above until I find out how to fix all
 * of the config header files.
 */
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

Btree *global_symbol_table;	/* holds ptr to current global sym tab */
static Btree *symlist = 0;	/* sym table */
static List *tmplist = 0;	/* sym table for temporary stuff */
static List *string_list = 0;	/* sym table for machine strings */

/* **************************************************************
 * Create RLaB global symbol-table. Also create a List that hangs
 * on the global symbol table for stroing temporary entities.
 * ************************************************************** */
void
symbol_table_create ()
{
  symlist = btree_Create ();
  btree_SetName (symlist, cpstr ("$$"));

  tmplist = list_Create ();
  string_list = list_Create ();

  global_symbol_table = symlist;
}

Btree *
get_symtab_ptr ()
{
  return (global_symbol_table);
}

/* **************************************************************
 * Install a new entry in symbol table.
 * ************************************************************** */
ListNode *
install (sym_table, s, type, data)
     VPTR sym_table;
     char *s;			/* name, or symbol table key */
     int type;			/* type of data */
     VPTR data;			/* ptr to data */
{
  ListNode *listNode;

  /* Create the ListNode and set the key */
  listNode = listNode_Create ();
  listNode_SetKey (listNode, s);

  /* Decide how, and where to install the new ListNode */
  if (sym_table == 0)
    btree_AddNode (global_symbol_table, listNode);
  else if (sym_table != 0)
    btree_AddNode ((Btree *) sym_table, listNode);
  else
    error_1 ("unanticipated args to install", (char *) 0);

  switch (type)			/* do the switch to get proper (*dtor)() */
  {
  case SCALAR:
    listNode_AttachData (listNode, type, data, scalar_Destroy);
    break;
  case MATRIX:
    listNode_AttachData (listNode, type, data, matrix_Destroy);
    break;
  case BTREE:
    listNode_AttachData (listNode, type, data, btree_Destroy);
    break;
  case STRING:
    listNode_AttachData (listNode, type, data, string_Destroy);
    break;
  case U_FUNCTION:
    listNode_AttachData (listNode, type, data, function_Destroy);
    break;
  case UNDEF:
    listNode_AttachData (listNode, type, data, 0);
    break;
  case D_VOID:
    listNode_AttachData (listNode, type, data, free);
    break;
  default:
    listNode_AttachData (listNode, type, data, NULL);
    break;
  }
  return (listNode);
}

/*
 * Special function for install data in tmplist.
 * Try to be faster than install().
 */

ListNode *
install_tmp (type, data, freeFn)
     int type;			/* type of data */
     VPTR data;			/* ptr to data */
     void (*freeFn) ();		/* Destructor function */
{
  ListNode *listNode;

  /* Create the ListNode and set the key */
  listNode = listNode_Create ();
  list_PushNode (tmplist, listNode);
  listNode_AttachData (listNode, type, data, freeFn);

  return (listNode);
}

ListNode *
install_tmp_switch (type, data)
     int type;			/* type of data */
     VPTR data;			/* ptr to data */
{
  ListNode *listNode;

  /* Create the ListNode and set the key */
  listNode = listNode_Create ();
  list_PushNode (tmplist, listNode);

  switch (type)			/* do the switch to get proper (*dtor)() */
  {
  case SCALAR:
    listNode_AttachData (listNode, type, data, scalar_Destroy);
    break;
  case MATRIX:
    listNode_AttachData (listNode, type, data, matrix_Destroy);
    break;
  case BTREE:
    listNode_AttachData (listNode, type, data, btree_Destroy);
    break;
  case STRING:
    listNode_AttachData (listNode, type, data, string_Destroy);
    break;
  case U_FUNCTION:
    listNode_AttachData (listNode, type, data, function_Destroy);
    break;
  case UNDEF:
    listNode_AttachData (listNode, type, data, 0);
    break;
  case D_VOID:
    listNode_AttachData (listNode, type, data, free);
    break;
  default:
    listNode_AttachData (listNode, type, data, NULL);
    break;
  }
  return (listNode);
}

/* **************************************************************
 * Install and existing entity on one of the lists.
 * ************************************************************** */
ListNode *
install_existing_tmp_ent (ent)
     ListNode *ent;
{
  list_PushNode (tmplist, ent);
  return (ent);
}

/* **************************************************************
 * Delete RLaB symbol tables(s).
 * Walk list deleting entries untill we get to the end. Do not
 * delete built-in functions from the list since they are not
 * dynamically allocated.
 * ************************************************************** */
int
delete_symbol_table ()
{
  btree_DestroyExcept (global_symbol_table, BLTIN);
  list_Destroy (tmplist);
  list_Destroy (string_list);

  return (1);
}

/* **************************************************************
 * Find s in symbol table. Return 0 id we DO NOT find s.
 * ************************************************************** */
ListNode *
lookup (sym_table, s)
     List *sym_table;
     char *s;
{
  ListNode *listNode;

  if (sym_table == 0)
  {
    if ((listNode = btree_FindNode (global_symbol_table, s)))
      return (listNode);
  }
  else
  {
    if ((listNode = list_GetNodeByKey (sym_table, s)))
      return (listNode);
  }
  return (0);
}

/* **************************************************************
 * Clean up the temporary list (remove trash).
 * ************************************************************** */
void
clean_list ()
{
  /* fprintf(stderr, "garbage = %i\t collect\n", list_GetNumNodes(tmplist)); */
  list_DestroyAllNodes (tmplist);
}

/*
 * Print out information about the garbage list as a
 * debugging aid.
 */
void
garbage_print ()
{
  fprintf (stderr, "garbage = %i\n", list_GetNumNodes (tmplist));
}

/* **************************************************************
 * Remove a temporary entity from tmplist, return a pointer to the
 * entities data. Note, this function uses list_DetachNodeByAddr ()
 * which detaches a node from WHEREVER!. Thus if you pass remove_tmp
 * something that is not on the tmp-list you will be in TROUBLE!
 * ************************************************************** */

VPTR
remove_tmp (ent)
     ListNode *ent;
{
  VPTR tmp;

  if ((list_DetachNodeByAddr (tmplist, ent)) != 0)
  {
    tmp = e_data (ent);
    listNode_DestroyNodeOnly (ent);
    return (tmp);
  }
  return (0);
}

/* **************************************************************
 * Remove a temporary entity from tmplist, Destroy the node and
 * the data. NOTE, the data is only destroyed if the entity has
 * a NULL name.
 * ************************************************************** */
int
remove_tmp_destroy (ent)
     ListNode *ent;
{
  if (listNode_GetKey (ent) == 0)
  {
    if ((list_DetachNodeByAddr (tmplist, ent)) != 0)
      listNode_Destroy (ent);
    return (1);
  }
  return (0);
}

/* **************************************************************
 * Print the built-in functions contained in the main symbol-table.
 * ************************************************************** */
VPTR
print_function_list (root)
     Btree *root;
{
  Btree *table;

  if (root == 0)
    table = global_symbol_table;
  else
    table = root;

  return ((VPTR) btree_Print_What (table));
}

/* **************************************************************
 * Print the objects contained in the main symbol-table.
 * ************************************************************** */
VPTR
print_object_list (root)
     Btree *root;
{
  Btree *table;

  if (root == 0)
    table = global_symbol_table;
  else
    table = root;

  return ((VPTR) btree_Print_Who (table));
}

/* **************************************************************
 * Reset the pointer to the global symbol table.
 * ************************************************************** */
void
reset_global_symbol_table_ptr ()
{
  global_symbol_table = symlist;
}

/* **************************************************************
 * Log a string pointer into the string-list. Then when we quit
 * we can clean up the list and check the program for leaks.
 * ************************************************************** */
void
string_log (s)
     char *s;
{
  ListNode *lnode;

  lnode = listNode_Create ();
  listNode_AttachData (lnode, 0, (VPTR) s, free);
  list_PushNode (string_list, lnode);

}

/* **************************************************************
 * Return a ENTITY that contains a pointer to the global symbol
 * table. Call listNode_AttachData with a 0 last arg so that we
 * NEVER destroy the global symbol table.
 * ************************************************************** */

Var *
gst ()
{
  Var *retval;
  ListNode *new;

  new = listNode_Create ();
  new = listNode_AttachData (new, BTREE, global_symbol_table, 0);
  listNode_SetKey (new, cpstr ("$$"));

  retval = (Var *) MALLOC (sizeof (Var));

  retval->type = GLOBAL;
  retval->ent = new;
  retval->offset = 0;
  retval->name = 0;

  return (retval);
}
