/* scalar.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "mem.h"
#include "scalar.h"
#include "bltin.h"
#include "util.h"

#include <stdio.h>
#include <string.h>
#include <math.h>

/* ************************************************************************
 * Create an instance of a Scalar, Initialize it to zero.
 * ************************************************************************ */
Scalar *
scalar_Create (val)
     double val;
{
  Scalar *new = (Scalar *) MALLOC (sizeof (Scalar));
  new->type = SCALAR;
  new->name = 0;
  new->dtype = REAL;
  new->val.r = val;
  new->val.i = 0.0;
  return (new);
}

Scalar *
scalar_CreateC (vr, vi)
     double vr, vi;
{
  Scalar *new = (Scalar *) MALLOC (sizeof (Scalar));
  new->type = SCALAR;
  new->name = 0;
  if (vi == 0.0)
    new->dtype = REAL;
  else
    new->dtype = COMPLEX;
  new->val.r = vr;
  new->val.i = vi;
  return (new);
}

/* ************************************************************************
 * Copy the contents of scalarOrig into scalarCopy.
 * ************************************************************************ */
Scalar *
scalar_Copy (s)
     Scalar *s;
{
  ASSERT (s);
  {
    Scalar *new = scalar_CreateC (s->val.r, s->val.i);
    return (new);
  }
}

/* **************************************************************
 * Destroy an instance of a scalar object
 * ************************************************************** */
void
scalar_Destroy (scalar)
     Scalar *scalar;
{
  ASSERT (scalar);
  {
    scalar->type = 0;
    FREE (scalar->name);
    scalar->dtype = 0;
    FREE (scalar);
  }
}

/* **************************************************************
 * Set the value of a scalar
 * ************************************************************** */
void
scalar_SetVal (s, vr)
     Scalar *s;
     double vr;
{
  ASSERT ((s));
  {
    s->val.r = vr;
  }
}

void
scalar_SetValC (s, vr, vi)
     Scalar *s;
     double vr, vi;
{
  ASSERT ((s));
  {
    s->val.r = vr;
    s->val.i = vi;
  }
}

void
scalar_SetName (s, name)
     Scalar *s;
     char *name;
{
  ASSERT (s);
  FREE (s->name);
  s->name = name;
}

/* Not sure what we should do here yet, for now ALWAYS increment
   REAL part, inc imaginary part, if it's non-zero. */
void
scalar_Inc (s)
     Scalar *s;
{
  s->val.r++;
  if (s->val.i != 0.0)
    s->val.i++;
}

void
scalar_Dec (s)
     Scalar *s;
{
  s->val.r--;
  if (s->val.i != 0.0)
    s->val.i--;
}

/* **************************************************************
 * Write a scalar entity to a file.
 * ************************************************************** */
void
scalar_Write (s, fn)
     Scalar *s;
     FILE *fn;
{
  ASSERT (s);
  {
    if (scalar_GetName (s) == 0)
    {
      fprintf (fn, "# scalar : %s\n %-24.16g %-24.16g\n",
	       "SCALAR", SVALr (s), SVALi (s));
    }
    else
    {
      fprintf (fn, "# scalar : %s\n %-24.16g %-24.16g\n",
	       scalar_GetName (s), SVALr (s), SVALi (s));
    }
    fflush (fn);
  }
}

/* **************************************************************
 * Read a scalar entity from file.
 * ************************************************************** */
Scalar *
scalar_Read (fn)
     FILE *fn;
{
  char s_name[80];
  Scalar *new;

  new = scalar_CreateC (0.0, 0.0);
  fscanf (fn, "%s\n %le %le\n", s_name, &new->val.r, &new->val.i);

  scalar_SetName (new, cpstr (s_name));
  return (new);
}

void
scalar_Print (s, fn)
     Scalar *s;
     FILE *fn;
{
  ASSERT (s);
  {
    char tmp[100];
    int fwidth, fprec, width;

    fwidth = get_fwidth ();
    fprec = get_fprec ();

    if (s->name != 0 && strncmp (s->name, "-", 1))
      fprintf (fn, " %s =\n", s->name);

    if (s->val.i == 0.0)
      fprintf (fn, "%*.*g", fwidth, fprec, s->val.r);
    else
    {
      width = max (fwidth, fprec) + 4;
      if (width > 100)
      {
	error_1 ("format too large for COMPLEX print", 0);
      }

      if (SVALi (s) >= 0.0)
	sprintf (tmp, "%*.*g + %.*gi", fwidth, fprec, SVALr (s),
		 fprec, SVALi (s));
      else
	sprintf (tmp, "%*.*g - %.*gi", fwidth, fprec, SVALr (s),
		 fprec, -SVALi (s));
      fprintf (fn, "%*s", width, tmp);
    }
    fprintf (fn, "\n");
  }
}
