/* rfile.c
 * Handle RLaB rfiles */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1994  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "symbol.h"
#include "mem.h"
#include "r_string.h"
#include "util.h"

#ifdef __STDC__
#include <stdlib.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#include <sys/types.h>
#include <stdio.h>
#include <errno.h>
#ifdef HAVE_DIRENT
#include <dirent.h>
#else
#include <sys/dir.h>
#endif

#ifdef THINK_C
#include <myconsole16.h>
extern int mac_rows;		/* no. of rows of console (defined in RLAB_ROWS env.) */
static int line_counter;
static int show = 1;		/* show = 0 if q (quit) is pressed */
#endif

/* The RLAB_SEARCH_PATH separator character */

#ifdef unix
static char PATH_SEP = ':';
static char PATH_DELIM[] = "/";
#define cat_help_file cat_help_file_unix
#endif
#ifdef OS2
static char PATH_SEP = ':';
#define cat_help_file cat_help_file_unix
#endif
#ifdef DOS
static char PATH_SEP = ':';
#define cat_help_file cat_DOS
#endif
#ifdef THINK_C
static char PATH_SEP = ';';
static char PATH_DELIM[] = ":";
#define cat_help_file cat_help_file_unix
#endif

extern int rpclose _PROTO ((FILE * fp));
extern int run_program _PROTO ((char *name));

/*
 * A structure for holding the parsed up directory names.
 */

struct _path
{
  int ndir;
  char **dira;
};

typedef struct _path Path;

int rfile_test _PROTO ((char *filename));
void print_rfiles _PROTO ((char *dirname, FILE * fptr));
int find_open _PROTO ((char *filenm, char *dirname));
int rfile_dir _PROTO ((char *dir));
Path *path_split _PROTO ((char *path));
void path_Destroy _PROTO ((Path * path));

static void swap _PROTO ((char *v[], int i, int j));
void nsort _PROTO ((char *v[], int right, int left));
static int nlength _PROTO ((char *carray[], int n));
static void cat_help_file_unix _PROTO ((char *name));
static void cat_DOS _PROTO ((char *name));
static char *find_full_filename _PROTO ((char *rfilenm, char *dirname));

static char *l_search_path;
static char *get_rlab_search_path _PROTO ((void));

/* Make this static so we can close after catching a SIGPIPE */
static FILE *pfile;

/* **************************************************************
 * Some interface functions to avoid global variables.
 * ************************************************************** */

static char *search_path;

void
set_search_path (value)
     char *value;
{
  search_path = cpstr (value);
}

static char *help_dir;

void
set_help_dir (value)
     char *value;
{
  help_dir = cpstr (value);
}

static char *lib_dir;

void
set_lib_dir (value)
     char *value;
{
  lib_dir = cpstr (value);
}

static char *pager;

void
set_pager (value)
     char *value;
{
  pager = cpstr (value);
}

/* **************************************************************
 * Rfile interface to parser
 * ************************************************************** */

#ifdef unix
void
rfile ()
{
  int i;
  Path *path;

  if (pfile)
    rpclose (pfile);

  if ((pfile = popen (pager, "w")) == 0)
  {
    fprintf (stderr, "ERROR, could not open %s pager for write\n", pager);
    return;
  }

  /* Re-split the PATH, in case it has changed */
  if ((l_search_path = get_rlab_search_path ()) == 0)
    l_search_path = search_path;
  path = path_split (l_search_path);

  for (i = 0; i < path->ndir; i++)
  {
    fprintf (pfile, "%s :\n", path->dira[i]);
    print_rfiles (path->dira[i], pfile);
  }

  fflush (pfile);
  path_Destroy (path);
  rpclose (pfile);
  pfile = 0;
}
#endif /* unix */

#ifdef THINK_C
void
rfile ()
{
  int i;
  Path *path;

  /* Re-split the PATH, in case it has changed */
  if ((l_search_path = get_rlab_search_path ()) == 0)
    l_search_path = search_path;
  path = path_split (l_search_path);

  for (i = 0; i < path->ndir; i++)
  {
    show = more (stdout, "");
    if (show)
      fprintf (stdout, "%s :\n", path->dira[i]);
    print_rfiles (path->dira[i], stdout);
  }
  fflush (stdout);
  path_Destroy (path);
}
#endif /* THINK_C */

/*
 * Interpret an rfile (load it).
 * Argument `name' is destroyed by rfile_load.
 */

void
rfile_load (name)
     char *name;
{
  char tmp[100];
  int i;
  Path *path;

  /* Append the `.r' to the name */
  /* FIX */
  strcpy (tmp, name);
  strcat (tmp, ".r");

  /* Re-split the PATH, in case it has changed */
  if ((l_search_path = get_rlab_search_path ()) == 0)
    l_search_path = search_path;
  path = path_split (l_search_path);

  for (i = 0; i < path->ndir; i++)
  {
    if ((find_open (tmp, path->dira[i])) != 0)
    {
      FREE (name);
      path_Destroy (path);
      return;
    }
  }
  fprintf (stderr, "ERROR, could not find/open: %s\n", tmp);
  FREE (name);
  path_Destroy (path);
}

/*
 * Split a colon separated path into distinct character strings.
 * EX: .:/u1/ian/misc/rlab/examples:/usr/local/rlab
 * Return a pointer to an array of character pointers.
 */

Path *
path_split (path)
     char *path;
{
  int i, n, ndir, start;
  char **dira;
  Path *new = (Path *) MALLOC (sizeof (Path));

  ndir = 1;
  i = 0;
  /* Count the number of directories */
  while (path[i] != '\0')
  {
    if (path[i] == PATH_SEP)
      ndir++;
    i++;
  }
  /* Get space for char pts */
  dira = (char **) MALLOC (ndir * sizeof (char *));

  /* Now get directory strings */
  start = 0;
  for (i = 0; i < ndir; i++)
  {
    /* Count length of string */
    n = 0;
    while (path[start + n] != PATH_SEP && path[start + n] != '\0')
      n++;

    /* Now copy string into dira */
    dira[i] = (char *) MALLOC ((n + 1) * sizeof (char));
    strncpy (dira[i], &path[start], n);
    dira[i][n] = '\0';
    start += (n + 1);
  }
  new->ndir = ndir;
  new->dira = dira;
  return (new);
}

#ifdef THINK_C
static int
more (FILE * fptr, char *msg)
{
  int x, y, c;

  if (!show)
    return 0;

  fprintf (fptr, "More %s ('q' to quit) ... ", msg);
  cgetxy (&x, &y, fptr);	/* get cursor position */
  c = getchar ();

  if (c == '\r' || c == '\n')
    cgotoxy (1, y - 1, fptr);	/* goto cursor position col 1  row y-1 */
  else
    cgotoxy (1, y, fptr);
  fprintf (fptr, "                                    ");	/* erase more .. */

  if (c == '\r' || c == '\n')
    cgotoxy (1, y - 1, fptr);
  else
    cgotoxy (1, y, fptr);

  return c == 'q' ? 0 : 1;	/* return 0 if 'q' (quit) is pressed */
}

static int
page_break (FILE * fptr, int line_counter)
{
  if (line_counter % (mac_rows - 3) == 0)
    return more (fptr, "");
  else
    return show;
}

/*
 * my_pager
 * build a simple pager here 
 * t.s.yang  Nov/93  ucb
 */

void 
my_pager (char *name)
{
  FILE *fp;
  char line[101], msg[20];
  int counter;
  long csize, fsize;

  show = 1;
  if ((fp = fopen (name, "r")) == NULL)
  {
    fprintf (stderr, "No help available.\n");
    return;
  }
  fseek (fp, 0L, SEEK_END);
  fsize = ftell (fp);
  rewind (fp);
  csize = 0;
  counter = 0;
  while (fgets (line, 100, fp) != NULL)
  {
    fputs (line, stdout);
    counter++;
    csize += strlen (line);
    if ((counter % (mac_rows - 3)) == 0)
    {
      sprintf (msg, "(%d%%)", csize * 100 / fsize);
      if (!more (stdout, msg))
	break;
    }
  }
  fclose (fp);
}
#endif /* THINK_C */

/*
 * Print all of the rfiles in a given directory.
 */

void
print_rfiles (dirname, fptr)
     char *dirname;
     FILE *fptr;
{
  char **names;
  int i, length, ncol, nfile;
  DIR *dirp;
#ifdef HAVE_DIRENT
  struct dirent *direntp;
#else
  struct direct *direntp;
#endif

  nfile = 0;

  /* Create array of rfiles names in this directory */
  if ((dirp = opendir (dirname)) != 0)
  {
    /* Count rfiles */
    while ((direntp = readdir (dirp)) != NULL)
    {
      if (rfile_test (direntp->d_name))
	nfile++;
    }
    rewinddir (dirp);

    /* Create array of rfile names */
    if (!nfile)
    {
      fprintf (fptr, "\n");
      return;
    }
    names = (char **) MALLOC (nfile * sizeof (char *));
    i = 0;

    while ((direntp = readdir (dirp)) != NULL)
    {
      if (rfile_test (direntp->d_name))
      {
	names[i] = cpstr (direntp->d_name);

	/* remove `.r' */
	names[i][strlen (names[i]) - 2] = '\0';
	i++;
      }
    }
    closedir (dirp);
  }
  else
  {
    fprintf (stderr, "ERROR opening directory: %s\n", dirname);
    return;
  }

  /* Sort rfile names */
  nsort (names, 0, nfile - 1);

  length = nlength (names, nfile);
  ncol = TERM_WIDTH / (length + 2);

  /* Now print out rfile names */
  for (i = 1; i <= nfile; i++)
  {

#ifdef unix
    fprintf (fptr, "%*s ", length + 2, names[i - 1]);
#endif
#ifdef THINK_C
    if (show)
      fprintf (fptr, "%*s ", length + 2, names[i - 1]);
#endif /* THINK_C */

    if ((i % ncol) == 0)
    {

#ifdef unix
      fprintf (fptr, " \n");
      fflush (fptr);
#endif

#ifdef THINK_C
      if (show)
	fprintf (fptr, " \n");
      fflush (fptr);
      line_counter++;
      show = page_break (fptr, line_counter);
#endif /* THINK_C */

    }
    FREE (names[i - 1]);
  }

#ifdef unix
  fprintf (fptr, "\n\n");
#endif
#ifdef THINK_C
  if (show)
    fprintf (fptr, "\n\n");
#endif /* THINK_C */

  fflush (fptr);

  FREE (names);
}

/*
 * Given a potential file name, and a directory name,
 * search the directory for the file. If it exists,
 * call new_file().
 *
 * Return TRUE if successful, FALSE if not.
 */

int
find_open (filenm, dirname)
     char *filenm, *dirname;
{
  /* FIX */
  char tmp[100];
  DIR *dirp;
#ifdef HAVE_DIRENT
  struct dirent *direntp;
#else
  struct direct *direntp;
#endif

  if ((dirp = opendir (dirname)) != 0)
  {
    while ((direntp = readdir (dirp)) != NULL)
    {
      if (!strcmp (direntp->d_name, filenm))
      {
	strcpy (tmp, dirname);
	strcat (tmp, PATH_DELIM);
	strcat (tmp, direntp->d_name);
	closedir (dirp);	/* just in case we error */
	run_program (tmp);
	return (1);
      }
    }
    closedir (dirp);
  }
  else
  {
    fprintf (stderr, "ERROR opening directory: %s\n", dirname);
  }
  return (0);			/* File not found */
}

/*
 * Test if filename is a valid rfile.
 * Must end in a `.r'
 */

int
rfile_test (filename)
     char *filename;
{
  int len;
  len = strlen (filename);

  /* Check the last two characters */
  if (filename[len - 1] == 'r' && filename[len - 2] == '.')
    return (1);

  return (0);
}

/*
 * Destroy a Path struct.
 */

void
path_Destroy (path)
     Path *path;
{
  int i;

  for (i = 0; i < path->ndir; i++)
    FREE (path->dira[i]);

  FREE (path->dira);
  path->ndir = 0;
  FREE (path);
}

/*
 * Simple character qsort.
 */

void
nsort (v, left, right)
     char *v[];
     int left, right;
{
  int i, last;

  if (left >= right)
    return;
  swap (v, left, (left + right) / 2);
  last = left;
  for (i = left + 1; i <= right; i++)
    if (strcmp (v[i], v[left]) < 0)
      swap (v, ++last, i);
  swap (v, left, last);
  nsort (v, left, last - 1);
  nsort (v, last + 1, right);
}

/*
 * Interchange v[i] and v[j]
 */

static void
swap (v, i, j)
     char *v[];
     int i, j;
{
  char *temp;

  temp = v[i];
  v[i] = v[j];
  v[j] = temp;
}

/*
 * Find the length of the longest string in the array.
 */

static int
nlength (carray, n)
     char *carray[];
     int n;
{
  int i, tmp, length;

  length = strlen (carray[0]);
  for (i = 1; i < n; i++)
  {
    tmp = strlen (carray[i]);
    if (tmp > length)
      length = tmp;
  }
  return (length);
}

/* **************************************************************
 * Help related functions
 * ************************************************************** */

/*
 * List contents of help directory,, and directories in
 * RLAB_SEARCH_PATH.
 */

#ifdef unix
void
help ()
{
  char **names;
  int i, length, ncol, nfile;
  DIR *dirp;
#ifdef HAVE_DIRENT
  struct dirent *direntp;
#else
  struct direct *direntp;
#endif
  Path *path;

  nfile = 0;

  if (pfile)
    rpclose (pfile);

  if ((pfile = popen (pager, "w")) == 0)
  {
    fprintf (stderr, "ERROR, could not open %s pager for write\n", pager);
    return;
  }

  /*
   * Print out the RLAB_HELP_DIR files 1st
   */

  if ((dirp = opendir (help_dir)) != 0)
  {
    /* Count ALL files */
    while ((direntp = readdir (dirp)) != NULL)
    {
      if (direntp->d_name[0] != '.')
	nfile++;
    }
    rewinddir (dirp);

    /* Create array of rfile names */
    names = (char **) MALLOC (nfile * sizeof (char *));
    i = 0;

    while ((direntp = readdir (dirp)) != NULL)
    {
      if (direntp->d_name[0] != '.')
      {
	names[i] = cpstr (direntp->d_name);
	i++;
      }
    }
    closedir (dirp);
  }
  else
  {
    fprintf (stderr, "ERROR opening directory: %s\n", help_dir);
    return;
  }

  /* Sort rfile names */
  nsort (names, 0, nfile - 1);

  length = nlength (names, nfile);
  ncol = TERM_WIDTH / (length + 2);

  /* Now print out rfile names */

  for (i = 1; i <= nfile; i++)
  {
    fprintf (pfile, "%*s ", length + 2, names[i - 1]);
    if ((i % ncol) == 0)
      fprintf (pfile, "\n");
    FREE (names[i - 1]);
  }

  fprintf (pfile, "\n");
  fflush (pfile);

  FREE (names);

  /*
   * Now do the rfiles in RLAB_LIB_DIR.
   */

  fprintf (pfile, "\n%s :\n", lib_dir);
  print_rfiles (lib_dir, pfile);

  /*
   * Now do the same for the directories in RLAB_SEARCH_PATH
   */

  /* Re-split the PATH, in case it has changed */
  if ((l_search_path = get_rlab_search_path ()) == 0)
    l_search_path = search_path;
  path = path_split (l_search_path);

  for (i = 0; i < path->ndir; i++)
  {
    fprintf (pfile, "%s :\n", path->dira[i]);
    print_rfiles (path->dira[i], pfile);
  }
  fflush (pfile);
  path_Destroy (path);

  rpclose (pfile);
  pfile = 0;
}
#endif /* unix */

#ifdef THINK_C
void
help ()
{
  char **names;
  int i, length, ncol, nfile;
  DIR *dirp;
#ifdef HAVE_DIRENT
  struct dirent *direntp;
#else
  struct direct *direntp;
#endif
  Path *path;

  nfile = 0;
  pfile = stdout;

  /*
   * Print out the RLAB_HELP_DIR files 1st
   */

  if ((dirp = opendir (help_dir)) != 0)
  {
    /* Count ALL files */
    while ((direntp = readdir (dirp)) != NULL)
    {
      if (direntp->d_name[0] != '.')
	nfile++;
    }
    rewinddir (dirp);

    /* Create array of rfile names */
    names = (char **) MALLOC (nfile * sizeof (char *));
    i = 0;

    while ((direntp = readdir (dirp)) != NULL)
    {
      if (direntp->d_name[0] != '.')
      {
	names[i] = cpstr (direntp->d_name);
	i++;
      }
    }
    closedir (dirp);
  }
  else
  {
    fprintf (stderr, "ERROR opening directory: %s\n", help_dir);
    return;
  }

  /* Sort rfile names */
  nsort (names, 0, nfile - 1);

  length = nlength (names, nfile);
  ncol = TERM_WIDTH / (length + 2);

  /* Now print out rfile names */
  line_counter = 0;
  show = 1;
  for (i = 1; i <= nfile; i++)
  {
    if (show)
      fprintf (stdout, "%*s ", length + 2, names[i - 1]);
    if ((i % ncol) == 0)
    {
      if (show)
	fprintf (stdout, "\n");
      line_counter++;
      show = page_break (stdout, line_counter);
    }
    FREE (names[i - 1]);
  }
  if (show)
    fprintf (stdout, "\n");
  fflush (stdout);
  FREE (names);

  /*
   * Now do the rfiles in RLAB_LIB_DIR.
   */

  show = more (stdout, "");
  if (show)
    fprintf (stdout, "\n%s :\n", lib_dir);
  line_counter = 3;
  print_rfiles (lib_dir, stdout);

  /*
   * Now do the same for the directories in RLAB_SEARCH_PATH
   */

  /* Re-split the PATH, in case it has changed */
  path = path_split (search_path);
  for (i = 0; i < path->ndir; i++)
  {
    show = more (stdout, "");
    if (show)
      fprintf (stdout, "\n%s :\n", path->dira[i]);
    line_counter = 3;
    print_rfiles (path->dira[i], stdout);
  }
  fflush (stdout);
  path_Destroy (path);
}

#endif /* THINK_C */

/*
 * Search the help directory, and then RLAB_SEARCH_PATH
 * for a file identified by `name'. Print out the
 * "help contents".
 */

void
help_name (name)
     char *name;
{
  char *tmp;
  int i, length;
  DIR *dirp;
#ifdef HAVE_DIRENT
  struct dirent *direntp;
#else
  struct direct *direntp;
#endif
  Path *path;

  /*
   * Search the default help directory 1st
   */

  /* Create array of rfiles names in this directory */
  if ((dirp = opendir (help_dir)) != 0)
  {
    while ((direntp = readdir (dirp)) != NULL)
    {
      if (!strcmp (name, direntp->d_name))
      {
	length = strlen (help_dir) + strlen (direntp->d_name) + 2;
	tmp = (char *) MALLOC (length * sizeof (char));
	strcpy (tmp, help_dir);
	strcat (tmp, PATH_DELIM);
	strcat (tmp, direntp->d_name);
	cat_help_file (tmp);
	closedir (dirp);
	FREE (tmp);
	FREE (name);
	return;
      }
    }
    closedir (dirp);
  }
  else
  {
    fprintf (stderr, "ERROR opening directory: %s\n", help_dir);
  }

  /*
   * Next, search the RLAB_LIB_DIR.
   */

  if ((tmp = find_full_filename (name, lib_dir)) != 0)
  {
    cat_help_file (tmp);
    FREE (name);
    FREE (tmp);
    return;
  }

  /*
   * Search RLAB_SEARCH_PATH
   */

  /* Re-split the PATH, in case it has changed */
  if ((l_search_path = get_rlab_search_path ()) == 0)
    l_search_path = search_path;
  path = path_split (l_search_path);

  for (i = 0; i < path->ndir; i++)
  {
    if ((tmp = find_full_filename (name, path->dira[i])) != 0)
    {
      cat_help_file (tmp);
      FREE (name);
      FREE (tmp);
      path_Destroy (path);
      return;
    }
  }
  fprintf (stderr, "ERROR, could not find/open: %s\n", name);
  FREE (name);
  path_Destroy (path);
}

/*
 * Cat a help file to stdout.
 */

static void
cat_help_file_unix (name)
     char *name;
{
#ifdef THINK_C
  my_pager (name);
#else
  char *sys_string;
  int length;

  /* Build system string */
  length = strlen (pager) + strlen (name) + 2;
  sys_string = (char *) MALLOC (length * sizeof (char));

  sprintf (sys_string, "%s %s", pager, name);
  system (sys_string);
  FREE (sys_string);
#endif /* THINK_C */
}

/*
 * A simple builtin cat for OSes that
 * cannot handle more than one process.
 * See The ANSI-C Programming Language.
 */

static void filecat _PROTO ((FILE * fin));

static void
cat_DOS (name)
     char *name;
{
  FILE *fp;
  if ((fp = fopen (name, "r")) == 0)
  {
    error_1 ("help: cannot open file %s for write\n", name);
  }
  else
  {
    filecat (fp);
    fclose (fp);
  }
}

static void
filecat (fn)
     FILE *fn;
{
  int c;
  while ((c = getc (fn)) != EOF)
  {
    putc (c, stdout);
  }
}

/*
 * Given a partial rfile name, return the full
 * pathname to that file.
 */

static char *
find_full_filename (rfilenm, dirname)
     char *rfilenm, *dirname;
{
  /* FIX */
  char tmp[100], *fullname;
  int length;
  DIR *dirp;
#ifdef HAVE_DIRENT
  struct dirent *direntp;
#else
  struct direct *direntp;
#endif

  strcpy (tmp, rfilenm);
  strcat (tmp, ".r");

  if ((dirp = opendir (dirname)) != 0)
  {
    while ((direntp = readdir (dirp)) != NULL)
    {
      if (!strcmp (direntp->d_name, tmp))
      {
	length = strlen (dirname) + strlen (direntp->d_name) + 2;
	fullname = (char *) MALLOC (length * sizeof (char));
	strcpy (fullname, dirname);
	strcat (fullname, PATH_DELIM);
	strcat (fullname, direntp->d_name);
	closedir (dirp);
	return (fullname);
      }
    }
    closedir (dirp);
  }
  else
  {
    fprintf (stderr, "ERROR opening directory: %s\n", dirname);
  }
  return (0);
}

/*
 * Check the global variable _rlab_search_path 
 * for a new path string.
 */

static char *
get_rlab_search_path ()
{
  ListNode *lpath;
  char *path;

  if ((lpath = btree_FindNode (get_symtab_ptr (), "_rlab_search_path")) == 0)
    return (0);

  if (e_type (lpath) == STRING)
  {
    path = cpstr (string_GetString (e_data (lpath)));
    return (path);
  }

  return (0);
}
