/* r_plot.c */

/*
 * This file contains the PLPLOT interface routines.
 */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1993, 1994  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

/*
 * The following dummy function is here just to
 * shut up some compilers that do not like to 
 * compile empty files.
 */

static int 
plot_garbage ()
{
  return (1);
}

#include "config.h"

#ifdef HAVE_RLAB_PLPLOT

#include "rlab.h"
#include "symbol.h"
#include "bltin.h"
#include "listnode.h"
#include "mem.h"
#include "r_string.h"
#include "matrix.h"
#include "print.h"
#include "util.h"

#include <stdio.h>

#define TARG_DESTROY(arg, targ)   if (targ.u.ent != arg.u.ent) \
                                    remove_tmp_destroy (targ.u.ent);

#define PL_DOUBLE 1
#include <plplot.h>
#include <pdf.h>
#include <plstream.h>

static int init_plot_device = 0;	/* Has the plot device been initialized */

static PLFLT **convert_matrix_to_array _PROTO ((Matrix * m));

/* **************************************************************
 * PLPLOT related functions and interface(s)
 * ************************************************************** */

/*
 * This function handles erros that occur in the plplot library.
 * We longjmp from here back to the prompt because the library
 * will exit otherwise.
 */

/* in main.c */
extern int dec_buff _PROTO ((void));
extern jmp_buf jmp[];

static int
rlab_plplot_exit (message)
     char *message;
{
  error_1 (message, 0);
  /* longjmp( jmp[dec_buff()], 1 ); */

  return (0);			/* shut up compiler */
}

/*
 * Make a hardcopy of the current plot-window.
 */

extern void plsfile _PROTO ((FILE * file));

void
_plot_plprint (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *fname, *device;
  FILE *sfile;
  PLINT cur_pls, new_pls;
  ListNode *S1, *S2;

  if (!(n_args == 1 || n_args == 2))
    error_1 ("_plprint: 1 or 2 argument(s) required", 0);

  S1 = bltin_get_string ("_plprint", d_arg, 1);
  S2 = 0;
  fname = string_GetString (e_data (S1));

  if (n_args == 1)
  {
    /* Default to BW Postscript */
    device = cpstr ("ps");
  }
  else
  {
    S2 = bltin_get_string ("_plprint", d_arg, 2);
    device = string_GetString (e_data (S2));
    if (!(!strcmp (device, "ps") ||
	  !strcmp (device, "psc") ||
	  !strcmp (device, "xfig") ||
	  !strcmp (device, "plmeta") ||
	  !strcmp (device, "ljii")))
    {
      remove_tmp_destroy (S1);
      remove_tmp_destroy (S2);
      error_1 ("plprint: invalid plot device: ", device);
    }
  }

  /*
   * Get the current stream number.
   */

  plgstrm (&cur_pls);

  /*
   * Create a new stream and make it the default.
   * The new stream number is returned (ipls).
   */

  plmkstrm (&new_pls);

  /* Open file for writes */

  sfile = get_file_ds (fname, "w", 0);

  /* Initialize stream */

  plsdev (device);
  plsfile (sfile);

  /* Copy the current stream to the newly-made stream */

  plcpstrm (cur_pls, 0);
  pladv (0);

  /* Remake current plot and then switch back to original stream */

  plreplot ();
  plflush ();
  plend1 ();
  plsstrm (cur_pls);

  close_file_ds (fname);
  remove_tmp_destroy (S1);
  if (S2)
    remove_tmp_destroy (S2);
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Re-plot the current stream
 */

void
_plot_plreplot (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  if (n_args != 0)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plreplot: no arguments allowed", 0);
  }

  plreplot ();

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Set the number of the current output stream
 */
void
_plot_plsstrm (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int n;

  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plsstrm: 1 argument required", 0);
  }

  n = (int) bltin_get_numeric_double ("_plot_plsstrm", d_arg, 1);
  plsstrm ((PLINT) n);
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plssub (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  double n1, n2;

  if (n_args != 2)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plssub: 2 arguments required", 0);
  }

  n1 = bltin_get_numeric_double ("_plot_plssub", d_arg, 1);
  n2 = bltin_get_numeric_double ("_plot_plssub", d_arg, 2);

  plssub (n1, n2);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plinit (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  if (n_args != 0)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plinit: no arguments allowed", 0);
  }

  plinit ();

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plstart (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *str;
  PLFLT n1, n2;
  PLStream *strptr;
  ListNode *S;

  if (n_args != 3)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plstart: 3 arguments required", 0);
  }

  S = bltin_get_string ("_plstart", d_arg, 1);
  n1 = (PLFLT) bltin_get_numeric_double ("_plstart", d_arg, 2);
  n2 = (PLFLT) bltin_get_numeric_double ("_plstart", d_arg, 3);
  str = string_GetString (e_data (S));

  if (!init_plot_device)
    plsexit (rlab_plplot_exit);

  plgpls (&strptr);
  strptr->server_nokill = 1;

  plstart (str, n1, n2);

  if (!init_plot_device)
    init_plot_device = 1;

  remove_tmp_destroy (S);
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plsfile (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *str;
  FILE *fn;
  ListNode *STR;

  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plsfile: 1 argument allowed", 0);
  }

  STR = bltin_get_string ("_plsfile", d_arg, 1);
  str = string_GetString (e_data (STR));

  if (!(fn = get_file_ds (str, "w", 0)))
  {
    warning_1 ("Cannot open ", str);
    remove_tmp_destroy (STR);
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }

  plsfile (fn);

  remove_tmp_destroy (STR);
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plenv (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLFLT xmin, xmax, ymin, ymax;
  PLINT just, axis;

  if (n_args != 6)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plenv: 6 arguments are required", 0);
  }

  xmin = (PLFLT) bltin_get_numeric_double ("_plenv", d_arg, 1);
  xmax = (PLFLT) bltin_get_numeric_double ("_plenv", d_arg, 2);
  ymin = (PLFLT) bltin_get_numeric_double ("_plenv", d_arg, 3);
  ymax = (PLFLT) bltin_get_numeric_double ("_plenv", d_arg, 4);
  just = (PLINT) bltin_get_numeric_double ("_plenv", d_arg, 5);
  axis = (PLINT) bltin_get_numeric_double ("_plenv", d_arg, 6);

  plenv (xmin, xmax, ymin, ymax, just, axis);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plline (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int npoints;
  ListNode *X, *Y;
  Matrix *x, *y;

  if (n_args != 3)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plline: 3 arguments required", 0);
  }

  npoints = (int) bltin_get_numeric_double ("_plline", d_arg, 1);
  X = bltin_get_numeric_matrix ("_plline", d_arg, 2);
  Y = bltin_get_numeric_matrix ("_plline", d_arg, 3);
  x = (Matrix *) e_data (X);
  y = (Matrix *) e_data (Y);

  plline (npoints, (PLFLT *) MDPTRr (x), (PLFLT *) MDPTRr (y));

  remove_tmp_destroy (X);
  remove_tmp_destroy (Y);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plline3 (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int npoints;
  ListNode *X, *Y, *Z;
  Matrix *x, *y, *z;

  if (n_args != 4)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plline3: 4 arguments required", 0);
  }

  npoints = (int) bltin_get_numeric_double ("_plline3", d_arg, 1);
  X = bltin_get_numeric_matrix ("_plline3", d_arg, 2);
  Y = bltin_get_numeric_matrix ("_plline3", d_arg, 3);
  Z = bltin_get_numeric_matrix ("_plline3", d_arg, 4);
  x = (Matrix *) e_data (X);
  y = (Matrix *) e_data (Y);
  z = (Matrix *) e_data (Z);

  plline3 (npoints, (PLFLT *) MDPTRr (x), 
	   (PLFLT *) MDPTRr (y), (PLFLT *) MDPTRr (z));

  remove_tmp_destroy (X);
  remove_tmp_destroy (Y);
  remove_tmp_destroy (Z);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plpoly3 (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int i, npoints;
  ListNode *X, *Y, *Z, *K, *tmp;
  Matrix *x, *y, *z, *k;
  int *ik;

  if (n_args != 5)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plpoly3: 5 arguments required", 0);
  }

  npoints = (int) bltin_get_numeric_double ("_plpoly3", d_arg, 1);
  X = bltin_get_numeric_matrix ("_plpoly3", d_arg, 2);
  Y = bltin_get_numeric_matrix ("_plpoly3", d_arg, 3);
  Z = bltin_get_numeric_matrix ("_plpoly3", d_arg, 4);
  K = bltin_get_numeric_matrix ("_plpoly3", d_arg, 5);
  x = (Matrix *) e_data (X);
  y = (Matrix *) e_data (Y);
  z = (Matrix *) e_data (Z);
  k = (Matrix *) e_data (K);

  /* Copy the draw/no-draw vector */
  tmp = install_tmp (D_VOID, 
		     ik = (PLINT *) MALLOC (sizeof (PLINT)*MNR (k)*MNC (k)),
		     free);
  for (i = 0; i < MNR (k) * MNC (k); i++)
    ik[i] = MATrv (ik, i);
  
  plpoly3 (npoints, (PLFLT *) MDPTRr (x), 
	   (PLFLT *) MDPTRr (y), (PLFLT *) MDPTRr (z), ik);

  FREE (ik);

  remove_tmp_destroy (X);
  remove_tmp_destroy (Y);
  remove_tmp_destroy (Z);
  remove_tmp_destroy (K);
  remove_tmp_destroy (tmp);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plend (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  if (n_args != 0)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plend: no arguments allowed", 0);
  }

  plend ();

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plend1 (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  if (n_args != 0)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plend1: no arguments allowed", 0);
  }

  plend1 ();

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_pllab (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *SX, *SY, *ST;
  char *sx, *sy, *st;

  if (n_args != 3)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_pllab: 3 arguments required", 0);
  }

  SX = bltin_get_string ("_pllab", d_arg, 1);
  SY = bltin_get_string ("_pllab", d_arg, 2);
  ST = bltin_get_string ("_pllab", d_arg, 3);

  sx = string_GetString (e_data (SX));
  sy = string_GetString (e_data (SY));
  st = string_GetString (e_data (ST));

  pllab (sx, sy, st);

  remove_tmp_destroy (SX);
  remove_tmp_destroy (SY);
  remove_tmp_destroy (ST);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Set the color for subsequent lines. You may specify which
 * colors are associated with which number by the plancol
 * routine if supported by the output driver.
 */

void
_plot_plcol (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT color;

  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plcol: 1 argument required", 0);
  }

  color = (PLINT) bltin_get_numeric_double ("_plcol", d_arg, 1);
  plcol (color);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Set the background color by 8 bit RGB value
 * Note: for some drivers this corresponds to a cmap 0 color
 */

void
_plot_plscolbg (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT r, g, b;

  if (n_args != 3)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plscolbg: 3 arguments required", 0);
  }

  r = (PLINT) bltin_get_numeric_double ("_plscolbg", d_arg, 1);
  g = (PLINT) bltin_get_numeric_double ("_plscolbg", d_arg, 2);
  b = (PLINT) bltin_get_numeric_double ("_plscolbg", d_arg, 3);

  plscolbg (r, g, b);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Set a given color from color map 0 by 8 bit RGB value
 * Does not result in any additional cells to be allocated.
 */

void
_plot_plscol0 (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT color, r, g, b;

  if (n_args != 4)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plscol0: 4 arguments required", 0);
  }

  color = (PLINT) bltin_get_numeric_double ("_plscol0", d_arg, 1);
  r = (PLINT) bltin_get_numeric_double ("_plscol0", d_arg, 2);
  g = (PLINT) bltin_get_numeric_double ("_plscol0", d_arg, 3);
  b = (PLINT) bltin_get_numeric_double ("_plscol0", d_arg, 4);

  plscol0 (color, r, g, b);
  
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Sets the line style according to one of eight predefined patters.
 */

void
_plot_pllsty (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT style;

  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_pllsty: 1 argument required", 0);
  }

  style = (PLINT) bltin_get_numeric_double ("_pllsty", d_arg, 1);
  pllsty (style);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plclr (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  if (n_args != 0)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plclr: no argument allowed", 0);
  }

  plclr ();

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plpoin (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT n, code;
  ListNode *X, *Y;

  if (n_args != 4)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plpoin: 4 arguments required", 0);
  }

  n = (PLINT) bltin_get_numeric_double ("_plpoin", d_arg, 1);
  X = bltin_get_numeric_matrix ("_plpoin", d_arg, 2);
  Y = bltin_get_numeric_matrix ("_plpoin", d_arg, 3);
  code = (PLINT) bltin_get_numeric_double ("_plpoin", d_arg, 4);

  plpoin (n, MDPTRr (e_data (X)), MDPTRr (e_data (Y)), code);

  remove_tmp_destroy (X);
  remove_tmp_destroy (Y);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plpoin3 (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT n, code;
  ListNode *X, *Y, *Z;

  if (n_args != 5)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plpoin3: 5 arguments required", 0);
  }

  n = (PLINT) bltin_get_numeric_double ("_plpoin3", d_arg, 1);
  X = bltin_get_numeric_matrix ("_plpoin3", d_arg, 2);
  Y = bltin_get_numeric_matrix ("_plpoin3", d_arg, 3);
  Z = bltin_get_numeric_matrix ("_plpoin3", d_arg, 4);
  code = (PLINT) bltin_get_numeric_double ("_plpoin3", d_arg, 5);

  plpoin3 (n, MDPTRr (e_data (X)), MDPTRr (e_data (Y)), 
	   MDPTRr (e_data (Z)), code);

  remove_tmp_destroy (X);
  remove_tmp_destroy (Y);
  remove_tmp_destroy (Z);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plhist (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT n, nbin, oldwin;
  PLFLT datmin, datmax;
  PLFLT *data;
  ListNode *X;

  if (n_args != 6)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plhist: 6 arguments required", 0);
  }

  n = (PLINT) bltin_get_numeric_double ("_plhist", d_arg, 1);
  X = bltin_get_numeric_matrix ("_plhist", d_arg, 2);
  data = (PLFLT *) MDPTRr (e_data (X));
  datmin = (PLFLT) bltin_get_numeric_double ("_plhist", d_arg, 3);
  datmax = (PLFLT) bltin_get_numeric_double ("_plhist", d_arg, 4);
  nbin = (PLINT) bltin_get_numeric_double ("_plhist", d_arg, 5);
  oldwin = (PLINT) bltin_get_numeric_double ("_plhist", d_arg, 6);

  plhist (n, data, datmin, datmax, nbin, oldwin);

  remove_tmp_destroy (X);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plspage (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT xp, yp, xleng, yleng, xoff, yoff;

  if (n_args != 6)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plspage: 6 arguments required", 0);
  }

  xp = (PLINT) bltin_get_numeric_double ("_plspage", d_arg, 1);
  yp = (PLINT) bltin_get_numeric_double ("_plspage", d_arg, 2);
  xleng = (PLINT) bltin_get_numeric_double ("_plspage", d_arg, 3);
  yleng = (PLINT) bltin_get_numeric_double ("_plspage", d_arg, 4);
  xoff = (PLINT) bltin_get_numeric_double ("_plspage", d_arg, 5);
  yoff = (PLINT) bltin_get_numeric_double ("_plspage", d_arg, 6);

  plspage (xp, yp, xleng, yleng, xoff, yoff);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Advance to the next subpage if sub=0, performing a page
 * advance if there are no remaining subpages on the current page. 
 * If subwindowing is not being used, pladv(0) will always advance 
 * the page. If sub>0, PLPLOT switches to the specified subpage. 
 * Note that this alows you to overwrite a plot on the specified 
 * subpage; if this is not what you intended, use plclr followed by 
 * plpage to first advance the page. This routine is called automatically
 * (with sub=0) by plenv, but if plenv is not used, pladv must be called 
 * after initializing PLPLOT but before defining the viewport.
 */

void
_plot_pladv (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT sub;

  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_pladv: 1 argument allowed", 0);
  }

  sub = (PLINT) bltin_get_numeric_double ("_pladv", d_arg, 1);
  pladv (sub);
  
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plgra (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  if (n_args != 0)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plgra: no arguments allowed", 0);
  }

  plgra ();

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_pltext (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  if (n_args != 0)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_pltext: no arguments allowed", 0);
  }

  pltext ();

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plflush (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  if (n_args != 0)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plflush: no arguments allowed", 0);
  }

  plflush ();

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plbox (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *xopt, *yopt;
  PLFLT xtick, ytick;
  PLINT nxsub, nysub;
  ListNode *XOPT, *YOPT;

  if (n_args != 6)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plbox: 6 arguments required", 0);
  }

  XOPT = bltin_get_string ("_plbox", d_arg, 1);
  YOPT = bltin_get_string ("_plbox", d_arg, 4);

  xopt = string_GetString (e_data (XOPT));
  yopt = string_GetString (e_data (YOPT));

  xtick = (PLFLT) bltin_get_numeric_double ("_plbox", d_arg, 2);
  ytick = (PLFLT) bltin_get_numeric_double ("_plbox", d_arg, 5);

  nxsub = (PLINT) bltin_get_numeric_double ("_plbox", d_arg, 3);
  nysub = (PLINT) bltin_get_numeric_double ("_plbox", d_arg, 6);

  plbox (xopt, xtick, nxsub, yopt, ytick, nysub);

  remove_tmp_destroy (XOPT);
  remove_tmp_destroy (YOPT);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Draw axes, numeric and text labels for a 
 * three-dimensional surface plot. 
 */

void
_plot_plbox3 (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *xopt, *yopt, *zopt;
  char *xlabel, *ylabel, *zlabel;
  PLFLT xtick, ytick, ztick;
  PLINT nxsub, nysub, nzsub;
  ListNode *XOPT, *YOPT, *ZOPT, *XLABEL, *YLABEL, *ZLABEL;

  if (n_args != 12)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plbox3: 12 arguments required", 0);
  }

  XOPT = bltin_get_string ("_plbox3", d_arg, 1);
  YOPT = bltin_get_string ("_plbox3", d_arg, 5);
  ZOPT = bltin_get_string ("_plbox3", d_arg, 9);

  xopt = string_GetString (e_data (XOPT));
  yopt = string_GetString (e_data (YOPT));
  zopt = string_GetString (e_data (ZOPT));

  XLABEL = bltin_get_string ("_plbox3", d_arg, 2);
  YLABEL = bltin_get_string ("_plbox3", d_arg, 6);
  ZLABEL = bltin_get_string ("_plbox3", d_arg, 10);

  xlabel = string_GetString (e_data (XLABEL));
  ylabel = string_GetString (e_data (YLABEL));
  zlabel = string_GetString (e_data (ZLABEL));

  xtick = (PLFLT) bltin_get_numeric_double ("_plbox3", d_arg, 3);
  ytick = (PLFLT) bltin_get_numeric_double ("_plbox3", d_arg, 7);
  ztick = (PLFLT) bltin_get_numeric_double ("_plbox3", d_arg, 11);

  nxsub = (PLINT) bltin_get_numeric_double ("_plbox3", d_arg, 4);
  nysub = (PLINT) bltin_get_numeric_double ("_plbox3", d_arg, 8);
  nzsub = (PLINT) bltin_get_numeric_double ("_plbox3", d_arg, 12);

  plbox3 (xopt, xlabel, xtick, nxsub,
	  yopt, ylabel, ytick, nysub,
	  zopt, zlabel, ztick, nzsub);

  remove_tmp_destroy (XOPT);
  remove_tmp_destroy (YOPT);
  remove_tmp_destroy (ZOPT);

  remove_tmp_destroy (XLABEL);
  remove_tmp_destroy (YLABEL);
  remove_tmp_destroy (ZLABEL);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plvsta (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  if (n_args != 0)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plvsta: no arguments allowed", 0);
  }

  plvsta ();

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plvasp (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLFLT aspect;

  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plvasp: 1 argument allowed", 0);
  }

  aspect = (PLFLT) bltin_get_numeric_double ("_plvasp", d_arg, 1);
  
  plvasp (aspect);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plwind (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLFLT xmin, xmax, ymin, ymax;

  if (n_args != 4)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plwind: 4 arguments required", 0);
  }

  xmin = (PLFLT) bltin_get_numeric_double ("_plwind", d_arg, 1);
  xmax = (PLFLT) bltin_get_numeric_double ("_plwind", d_arg, 2);
  ymin = (PLFLT) bltin_get_numeric_double ("_plwind", d_arg, 3);
  ymax = (PLFLT) bltin_get_numeric_double ("_plwind", d_arg, 4);

  plwind (xmin, xmax, ymin, ymax);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Create a three-dimensional surface plot...
 */

void
_plot_plot3d (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int i;
  PLFLT *x, *y, **zz;
  PLINT nx, ny, opt, side;
  ListNode *X, *Y, *Z;
  Matrix *z;

  if (n_args != 7)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plot3d: 7 arguments required", 0);
  }

  X = bltin_get_numeric_matrix ("_plot3d", d_arg, 1);
  Y = bltin_get_numeric_matrix ("_plot3d", d_arg, 2);
  Z = bltin_get_numeric_matrix ("_plot3d", d_arg, 3);

  x = (PLFLT *) MDPTRr (e_data (X));
  y = (PLFLT *) MDPTRr (e_data (Y));
  z = (Matrix *) e_data (Z);
  zz = convert_matrix_to_array (z);
  nx = (PLINT) bltin_get_numeric_double ("_plot3d", d_arg, 4);
  ny = (PLINT) bltin_get_numeric_double ("_plot3d", d_arg, 5);
  opt = (PLINT) bltin_get_numeric_double ("_plot3d", d_arg, 6);
  side = (PLINT) bltin_get_numeric_double ("_plot3d", d_arg, 7);

  plot3d (x, y, zz, nx, ny, opt, side);

  remove_tmp_destroy (X);
  remove_tmp_destroy (Y);
  remove_tmp_destroy (Z);

  /* Now free up the z-array */
  for (i = 0; i < MNR (z); i++)
    FREE (zz[i]);
  FREE (zz);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plmesh (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int i;
  PLFLT *x, *y, **zz;
  PLINT nx, ny, opt;
  ListNode *X, *Y, *Z;
  Matrix *z;

  if (n_args != 6)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plmesh: 6 arguments required", 0);
  }

  X = bltin_get_numeric_matrix ("_plmesh", d_arg, 1);
  Y = bltin_get_numeric_matrix ("_plmesh", d_arg, 2);
  Z = bltin_get_numeric_matrix ("_plmesh", d_arg, 3);

  x = (PLFLT *) MDPTRr (e_data (X));
  y = (PLFLT *) MDPTRr (e_data (Y));
  z = (Matrix *) e_data (Z);
  zz = convert_matrix_to_array (z);
  nx = (PLINT) bltin_get_numeric_double ("_plmesh", d_arg, 4);
  ny = (PLINT) bltin_get_numeric_double ("_plmesh", d_arg, 5);
  opt = (PLINT) bltin_get_numeric_double ("_plmesh", d_arg, 6);

  plmesh (x, y, zz, nx, ny, opt);

  remove_tmp_destroy (X);
  remove_tmp_destroy (Y);
  remove_tmp_destroy (Z);

  /* Now free up the z-array */
  for (i = 0; i < MNR (z); i++)
    FREE (zz[i]);
  FREE (zz);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plw3d (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLFLT basex, basey, height, xmin, xmax;
  PLFLT ymin, ymax, zmin, zmax, alt, az;

  if (n_args != 11)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plw3d: 11 arguments required", 0);
  }

  basex = (PLFLT) bltin_get_numeric_double ("_plw3d", d_arg, 1);
  basey = (PLFLT) bltin_get_numeric_double ("_plw3d", d_arg, 2);
  height = (PLFLT) bltin_get_numeric_double ("_plw3d", d_arg, 3);
  xmin = (PLFLT) bltin_get_numeric_double ("_plw3d", d_arg, 4);
  xmax = (PLFLT) bltin_get_numeric_double ("_plw3d", d_arg, 5);
  ymin = (PLFLT) bltin_get_numeric_double ("_plw3d", d_arg, 6);
  ymax = (PLFLT) bltin_get_numeric_double ("_plw3d", d_arg, 7);
  zmin = (PLFLT) bltin_get_numeric_double ("_plw3d", d_arg, 8);
  zmax = (PLFLT) bltin_get_numeric_double ("_plw3d", d_arg, 9);
  alt = (PLFLT) bltin_get_numeric_double ("_plw3d", d_arg, 10);
  az = (PLFLT) bltin_get_numeric_double ("_plw3d", d_arg, 11);

  plw3d (basex, basey, height, xmin, xmax,
	 ymin, ymax, zmin, zmax, alt, az);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Writes text at a specified position relative to the 
 * viewport boundaries. Text may be written inside or 
 * outside the viewport, but is clipped at the subpage 
 * boundaries. 
 */

void
_plot_plmtex (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *side, *text;
  ListNode *SIDE, *TEXT;
  PLFLT disp, pos, just;

  if (n_args != 5)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plmtex: 5 arguments required", 0);
  }

  SIDE = bltin_get_string ("_plmtex", d_arg, 1);
  TEXT = bltin_get_string ("_plmtex", d_arg, 5);

  side = string_GetString (e_data (SIDE));
  disp = (PLFLT) bltin_get_numeric_double ("_plmtex", d_arg, 2);
  pos = (PLFLT) bltin_get_numeric_double ("_plmtex", d_arg, 3);
  just = (PLFLT) bltin_get_numeric_double ("_plmtex", d_arg, 4);
  text = string_GetString (e_data (TEXT));

  plmtex (side, disp, pos, just, text);

  remove_tmp_destroy (SIDE);
  remove_tmp_destroy (TEXT);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Turn PLPLOT pause off (arg=1)
 * Turn PLPLOT pause on  (arg=0)
 */

void
_plot_plspause (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT pause;

  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plspause: 1 argument required", 0);
  }

  pause = (PLINT) bltin_get_numeric_double ("_plspause", d_arg, 1);

  plspause (pause);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Convert a RLaB matrix to a two-dimensional C array.
 */

static PLFLT **
convert_matrix_to_array (m)
     Matrix *m;
{
  PLFLT **a;
  int i, j, nrow, ncol;

  nrow = MNR (m);
  ncol = MNC (m);

  /* Create the new array */
  a = (PLFLT **) MALLOC (nrow * sizeof (PLFLT *));
  for (i = 0; i < nrow; i++)
    a[i] = (PLFLT *) MALLOC (ncol * sizeof (PLFLT));

  /* Now load it up */
  for (i = 0; i < nrow; i++)
    for (j = 0; j < ncol; j++)
      a[i][j] = MAT0 (m, i, j);

  return a;
}

/*
 * Specify the desired pen width. The width must be 
 * between 1 and a device dependent maximum value.
 */

void
_plot_plwid (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT width;

  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plwid: 1 argument required", 0);
  }

  width = (PLINT) bltin_get_numeric_double ("_plwid", d_arg, 1);

  plwid (width);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Writes text at a specified position and inclination within the viewport.
 * Text is clipped at the viewport boundaries. The reference point of a string
 * lies along a line passing through the string at half the height of a capital
 * letter. The position of the reference point along this line is determined by
 * just.
 */

void
_plot_plptex (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *TEXT;
  PLFLT x, y, dx, dy, just;
  char *text = 0;

  if (n_args != 6)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plptex: 6 arguments required", 0);
  }

  x = (PLFLT) bltin_get_numeric_double ("_plptex", d_arg, 1);
  y = (PLFLT) bltin_get_numeric_double ("_plptex", d_arg, 2);
  dx = (PLFLT) bltin_get_numeric_double ("_plptex", d_arg, 3);
  dy = (PLFLT) bltin_get_numeric_double ("_plptex", d_arg, 4);
  just = (PLFLT) bltin_get_numeric_double ("_plptex", d_arg, 5);
  TEXT = bltin_get_string ("_plptex", d_arg, 6);
  text = cpstr (string_GetString (e_data (TEXT)));

  plptex (x, y, dx, dy, just, text);

  remove_tmp_destroy (TEXT);
  FREE (text);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Set the character set to use for subsequent characer drawing.
 * May be called before initializing PLPLOT.
 * 0 Standard character set.
 * 1 Extended character set.
 */

void
_plot_plfontld (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT set;

  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plfontld: 1 argument required", 0);
  }

  set = (PLINT) bltin_get_numeric_double ("_plfontld", d_arg, 1);

  plfontld (set);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Sets the default characer font for subsequent character drawing.
 * Also affects symbols produced by plpoin.
 * 1 Normal font (simples and fastest)
 * 2 Roman font
 * 3 Italic font
 * 4 Script font
 */

void
_plot_plfont (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT font;

  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plfont: 1 argument required", 0);
  }

  font = (PLINT) bltin_get_numeric_double ("_plfont", d_arg, 1);

  plfont (font);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Set the current orientation. 
 * ori:  0 landscape
 * ori:  1 portrait
 * Not supported by all device drivers.
 */

void
_plot_plsori (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT ori;

  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plsori: 1 argument required", 0);
  }

  ori = (PLINT) bltin_get_numeric_double ("_plsori", d_arg, 1);

  plsori (ori);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Globally turn color output on/off
 */

void
_plot_plscolor (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT color;

  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plscolor: 1 argument required", 0);
  }

  color = (PLINT) bltin_get_numeric_double ("_plscolor", d_arg, 1);

  plscolor (color);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Contour plotting.
 */

/*
 * linear interpolation from singly dimensioned coord arrays.
 */

static int HasContour;

static void
plot_mapgrid (x, y, tx, ty, pltr_data)
     PLFLT x, y, *tx, *ty;
     PLPointer pltr_data;
{
  PLINT ul, ur, vl, vr;
  PLFLT du, dv;
  PLFLT xl, xr, yl, yr;

  PLcGrid *grid = (PLcGrid *) pltr_data;
  PLFLT *xg = grid->xg;
  PLFLT *yg = grid->yg;
  PLINT nx = grid->nx;
  PLINT ny = grid->ny;

  ul = (PLINT) x;
  ur = ul + 1;
  du = x - ul;

  vl = (PLINT) y;
  vr = vl + 1;
  dv = y - vl;

  if (x < 0 || x > nx - 1 || y < 0 || y > ny - 1)
  {
    error_1 ("_mapgrid: Invalid coordinates", 0);
  }

  xl = xg[ul];
  yl = yg[vl];

  if (ur == nx)
  {
    *tx = xl;
  }
  else
  {
    xr = xg[ur];
    *tx = xl * (1 - du) + xr * du;
  }
  if (vr == ny)
  {
    *ty = yl;
  }
  else
  {
    yr = yg[vr];
    *ty = yl * (1 - dv) + yr * dv;
  }
  HasContour = 1;
}

void
_plot_plcont (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int i;
  PLFLT *x, *y, **z, *clevel;
  PLINT nx, ny, nzx, nzy, kx, lx, ky, ly, nlevel;
  PLcGrid *pltr_data;
  ListNode *X, *Y, *CLEVEL, *ZM;
  Matrix *clevelm, *zm;

  if (n_args != 8)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plcont: 8 arguments required", 0);
  }

  X = bltin_get_numeric_matrix ("_plcont", d_arg, 1);
  Y = bltin_get_numeric_matrix ("_plcont", d_arg, 2);
  ZM = bltin_get_numeric_matrix ("_plcont", d_arg, 3);
  CLEVEL = bltin_get_numeric_matrix ("_plcont", d_arg, 8);

  x = (PLFLT *) MDPTRr (e_data (X));
  y = (PLFLT *) MDPTRr (e_data (Y));
  zm = (Matrix *) e_data (ZM);
  kx = (PLINT) bltin_get_numeric_double ("_plcont", d_arg, 4);
  lx = (PLINT) bltin_get_numeric_double ("_plcont", d_arg, 5);
  ky = (PLINT) bltin_get_numeric_double ("_plcont", d_arg, 6);
  ly = (PLINT) bltin_get_numeric_double ("_plcont", d_arg, 7);
  clevelm = (Matrix *) e_data (CLEVEL);

  z = convert_matrix_to_array (zm);
  nzx = (PLINT) MNR (zm);
  nzy = (PLINT) MNC (zm);

  clevel = (PLFLT *) MDPTRr (clevelm);
  nlevel = MNR (clevelm) * MNC (clevelm);

  pltr_data = (PLcGrid *) MALLOC (sizeof (PLcGrid));
  nx = (PLINT) MNR (e_data (X)) * MNC (e_data (X));
  ny = (PLINT) MNR (e_data (Y)) * MNC (e_data (Y));

  pltr_data->xg = x;
  pltr_data->nx = nx;
  pltr_data->yg = y;
  pltr_data->ny = ny;
  pltr_data->zg = 0;
  pltr_data->nz = 0;

  HasContour = 0;
  plcont (z, nzx, nzy, kx, lx, ky, ly, clevel,
	  nlevel, plot_mapgrid, pltr_data);

  remove_tmp_destroy (X);
  remove_tmp_destroy (Y);
  remove_tmp_destroy (ZM);
  remove_tmp_destroy (CLEVEL);

  /* Now free up the z-array */
  for (i = 0; i < MNR (zm); i++)
    FREE (z[i]);

  FREE (z);
  FREE (pltr_data);

  *return_ptr = (VPTR) scalar_Create ((double) HasContour);
  return;
}

void
_plot_plvpor (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLFLT xmin, xmax, ymin, ymax;

  if (n_args != 4)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plvpor: 4 arguments required", 0);
  }

  xmin = (PLFLT) bltin_get_numeric_double ("_plvpor", d_arg, 1);
  xmax = (PLFLT) bltin_get_numeric_double ("_plvpor", d_arg, 2);
  ymin = (PLFLT) bltin_get_numeric_double ("_plvpor", d_arg, 3);
  ymax = (PLFLT) bltin_get_numeric_double ("_plvpor", d_arg, 4);

  plvpor (xmin, xmax, ymin, ymax);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plvpas (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLFLT xmin, xmax, ymin, ymax, aspect;

  if (n_args != 5)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plvpas: 5 arguments required", 0);
  }

  xmin = (PLFLT) bltin_get_numeric_double ("_plvpas", d_arg, 1);
  xmax = (PLFLT) bltin_get_numeric_double ("_plvpas", d_arg, 2);
  ymin = (PLFLT) bltin_get_numeric_double ("_plvpas", d_arg, 3);
  ymax = (PLFLT) bltin_get_numeric_double ("_plvpas", d_arg, 4);
  aspect = (PLFLT) bltin_get_numeric_double ("_plvpas", d_arg, 5);

  plvpas (xmin, xmax, ymin, ymax, aspect);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

#if 0

/*
 *   for contour plot
 */

static int HasContour;

/*
 * linear interpolation from singly dimensioned coord arrays
 */

void
_plot_mapgrid (PLFLT x, PLFLT y, PLFLT * tx, PLFLT * ty, PLPointer pltr_data)
{
  PLINT ul, ur, vl, vr;
  PLFLT du, dv;
  PLFLT xl, xr, yl, yr;

  PLcGrid *grid = (PLcGrid *) pltr_data;
  PLFLT *xg = grid->xg;
  PLFLT *yg = grid->yg;
  PLINT nx = grid->nx;
  PLINT ny = grid->ny;

  ul = (PLINT) x;
  ur = ul + 1;
  du = x - ul;

  vl = (PLINT) y;
  vr = vl + 1;
  dv = y - vl;

  if (x < 0 || x > nx - 1 || y < 0 || y > ny - 1)
  {
    error_1 ("_mapgrid: Invalid coordinates", 0);
  }

  xl = xg[ul];
  yl = yg[vl];

  if (ur == nx)
  {
    *tx = xl;
  }
  else
  {
    xr = xg[ur];
    *tx = xl * (1 - du) + xr * du;
  }

  if (vr == ny)
  {
    *ty = yl;
  }
  else
  {
    yr = yg[vr];
    *ty = yl * (1 - dv) + yr * dv;
  }
  HasContour = 1;
}

/*
 * _plcont() contour plot,  t.s.yang  6/14/94  uc berkeley
 * 
 */

void
_plot_plcont (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int i, k;
  char annot[50];
  PLFLT xbeg, xend, ybeg, dy;
  PLFLT **z, *clevel;
  PLINT nx, ny, kx, lx, ky, ly, nlevel;
  ListNode *Z, *CLEVEL, *XG, *YG;
  Matrix *clevelm, *zm, *xg, *yg;
  PLcGrid pltr_data;

  if (n_args != 4)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plcont: 4 arguments required", 0);
  }

  Z = bltin_get_numeric_matrix ("_plcont", d_arg, 1);
  CLEVEL = bltin_get_numeric_matrix ("_plcont", d_arg, 2);
  XG = bltin_get_numeric_matrix ("_plcont", d_arg, 3);
  YG = bltin_get_numeric_matrix ("_plcont", d_arg, 4);

  zm = (Matrix *) e_data (Z);
  nx = (PLINT) MNR (zm);
  ny = (PLINT) MNC (zm);

  kx = 1;
  lx = nx;
  ky = 1;
  ly = ny;

  clevelm = (Matrix *) e_data (CLEVEL);

  z = convert_matrix_to_array (zm);
  clevel = (PLFLT *) MDPTRr (clevelm);
  nlevel = MNR (clevelm) * MNC (clevelm);

  /* grid arrays */
  xg = (Matrix *) e_data (XG);
  yg = (Matrix *) e_data (YG);
  pltr_data.xg = (PLFLT *) MDPTRr (xg);
  pltr_data.yg = (PLFLT *) MDPTRr (yg);
  pltr_data.nx = nx;
  pltr_data.ny = ny;

  /* location for legend */
  xbeg = (pltr_data.xg[nx - 1] - pltr_data.xg[0]) * 0.65 + pltr_data.xg[0];
  xend = xbeg + (pltr_data.xg[nx - 1] - pltr_data.xg[0]) / 8.0;
  ybeg = (pltr_data.yg[ny - 1] - pltr_data.yg[0]) * 11.3 / 12.0 + pltr_data.yg[0];
  dy = (pltr_data.yg[ny - 1] - pltr_data.yg[0]) / 20.0;

  k = 0;
  for (i = 0; i < nlevel; i++)
  {
    HasContour = 0;
    pllsty ((i % 8) + 1);
    plcol (i % 16);
    plcont (z, nx, ny, kx, lx, ky, ly,
	    &clevel[i], 1, _plot_mapgrid, (void *) &pltr_data);
    if (HasContour)
    {
      /* show value of contour on the graph */
      sprintf (annot, "%.2g", clevel[i]);
      pljoin (xbeg, ybeg - dy * k, xend, ybeg - dy * k);
      plptex (xend + dy / 4, ybeg - dy * k, 1.0, 0.0, 0.0, annot);
      k++;
    }
  }

  remove_tmp_destroy (Z);
  remove_tmp_destroy (CLEVEL);
  remove_tmp_destroy (XG);
  remove_tmp_destroy (YG);


  /* Now free up the z-array */
  for (i = 0; i < MNR (zm); i++)
    FREE (z[i]);
  FREE (z);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

#endif /* if 0 */

/*
 * _plerry() error bar plot,  t.s.yang  6/14/94  uc berkeley
 */

void
_plot_plerry (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int npoints;
  ListNode *X, *YMIN, *YMAX;
  Matrix *x, *ymin, *ymax;

  if (n_args != 4)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("_plerry: 4 arguments required", 0);
  }

  npoints = (int) bltin_get_numeric_double ("_plerry", d_arg, 1);
  X = bltin_get_numeric_matrix ("_plerry", d_arg, 2);
  YMIN = bltin_get_numeric_matrix ("_plerry", d_arg, 3);
  YMAX = bltin_get_numeric_matrix ("_plerry", d_arg, 4);

  x = (Matrix *) e_data (X);
  ymin = (Matrix *) e_data (YMIN);
  ymax = (Matrix *) e_data (YMAX);

  plerry (npoints, (PLFLT *) MDPTRr (x),
	  (PLFLT *) ymin->val.mr,
	  (PLFLT *) ymax->val.mr);

  remove_tmp_destroy (X);
  remove_tmp_destroy (YMIN);
  remove_tmp_destroy (YMAX);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

#endif /* HAVE_RLAB_PLPLOT */
