/* print.c */

/*
 * [fps]rintf() work-alike. Some code borrowed from mawk (GNU
 * copylefted) by Mike Brennan. I have modified this code to fit
 * RLaB, so any bugs are my fault (Ian Searle).
 */

/*  This file is a part of RLaB ("Our"-LaB)
    Copyright (C) 1992, 1994  Ian R. Searle

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    See the file ./COPYING
 ***********************************************************************/

#include "rlab.h"
#include "symbol.h"
#include "bltin.h"
#include "listnode.h"
#include "r_string.h"
#include "mem.h"
#include "util.h"
#include "scan.h"

#include <stdio.h>

#ifdef THINK_C
/* Mac lib conflict */
#define Open Openx
#endif

#ifndef FOPEN_MAX
#define FOPEN_MAX 20
#endif

#ifndef HAVE_FPRINTF_DEC
extern int fprintf ();
#endif

#ifndef HAVE_SPRINTF_DEC
extern int *sprintf ();
#endif

/* getline.c */
extern char scan_code[];

extern int rpclose _PROTO ((FILE *fp));

FILE *get_file_ds _PROTO ((char *name, char *mode, int buffsize));
int close_file_ds _PROTO ((char *name));

/*---------- types and defs for doing printf ------------*/
#define  PF_C		0
#define  PF_S		1
#define  PF_D		2	/* int conversion */
#define  PF_LD		3	/* long int */
#define  PF_F		4	/* float conversion */

#ifdef HAVE_PROTOTYPES
typedef int (*PRINTER) (VPTR, char *,...);
#else
typedef int (*PRINTER) ();
#endif

/* for switch on number of '*' and type */
#define  AST(num,type)  (5*(num)+(type))

static int do_printf _PROTO ((FILE * fp, char *format, int n_args,
			      int arg_cnt, Datum * d_arg, String * strv));

#define TARG_DESTROY(arg, targ)   if (targ.u.ent != arg.u.ent) \
                                    remove_tmp_destroy (targ.u.ent);

void
Open (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *mode, *name;
  int buffsize;
  FILE *f;
  ListNode *FN, *Mode;

  if (n_args < 2 || n_args > 3)
    error_1 ("open: requires 2 or 3 arguments", 0);

  /* arg MUST be a STRING */
  FN = bltin_get_string ("open", d_arg, 1);
  name = string_GetString (e_data (FN));
  Mode = bltin_get_string ("open", d_arg, 2);
  mode = string_GetString (e_data (Mode));
    
  if (n_args == 3)
  {
    /* BUFFER SIZE */
    buffsize = (int) bltin_get_numeric_double ("open", d_arg, 3);
    f = get_file_ds (name, mode, buffsize);
  }
  else
  {
    f = get_file_ds (name, mode, 0);
  }
  
  remove_tmp_destroy (FN);
  remove_tmp_destroy (Mode);

  if (f == 0)
    *return_ptr = (VPTR) scalar_Create (0.0);
  else
    *return_ptr = (VPTR) scalar_Create (1.0);
}

void
Close (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *F;

  if (n_args != 1)
    error_1 ("close: requires an arg", 0);

  F = bltin_get_string ("close", d_arg, 1);

  if (close_file_ds (string_GetString (e_data (F))))
    *return_ptr = (VPTR) scalar_Create (1.0);
  else
    *return_ptr = (VPTR) scalar_Create (0.0);

  remove_tmp_destroy (F);
}

/* Emulate C printf() */
void
Printf (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  FILE *fp;
  int arg_cnt, retval;
  char *format;
  ListNode *FORMAT;

  arg_cnt = 0;
  if (n_args == 0)
  {
    clean_bltin_args (d_arg, n_args);
    warning_1 ("printf:  requires at least 1 arg", 0);
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }

  /* 1st arg MUST be format */
  FORMAT = bltin_get_string ("printf", d_arg, ++arg_cnt);
  format = string_GetString (e_data (FORMAT));
  fp = stdout;

  retval = do_printf (fp, format, n_args, arg_cnt, d_arg, (String *) 0);
  *return_ptr = (VPTR) scalar_Create (retval);
  remove_tmp_destroy (FORMAT);
  {
    /* Skip the 1st arg. */
    int i;
    for (i = 1; i < n_args; i++)
    {
      if (d_arg[i].type == ENTITY)
	remove_tmp_destroy (d_arg[i].u.ent);
    }
  }
}

/* Emulate C fprintf() */
void
FPrintf (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  FILE *fp;
  char *filenm, *format;
  int arg_cnt, retval;
  ListNode *FN, *FORMAT;

  arg_cnt = 0;
  if (n_args < 2)
  {
    clean_bltin_args (d_arg, n_args);
    warning_1 ("fprintf: at least 2 args  required", 0);
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }

  /* 1st arg MUST be filename, 2nd format */
  FN = bltin_get_string ("fprintf", d_arg, ++arg_cnt);
  filenm = string_GetString (e_data (FN));
  FORMAT = bltin_get_string ("fprintf", d_arg, ++arg_cnt);
  format = string_GetString (e_data (FORMAT));

  if (!(fp = get_file_ds (filenm, "w", 0)))
  {
    warning_1 ("Cannot open ", filenm);
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }

  retval = do_printf (fp, format, n_args, arg_cnt, d_arg, (String *) 0);
  *return_ptr = (VPTR) scalar_Create ((double) retval);
  remove_tmp_destroy (FN);
  remove_tmp_destroy (FORMAT);
  {
    /* Skip the 1st ans 2nd args. */
    int i;
    for (i = 2; i < n_args; i++)
    {
      if (d_arg[i].type == ENTITY)
	remove_tmp_destroy (d_arg[i].u.ent);
    }
  }
  return;
}

/* 
 * Emulate C sprintf()
 */

void
SPrintf (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *format;
  Datum arg;
  int arg_cnt, retval;
  ListNode *FSTRING;
  String *strv;

  arg_cnt = 0;
  if (n_args < 2)
  {
    clean_bltin_args (d_arg, n_args);
    warning_1 ("sprintf: at least 2 arguments required", 0);
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }

  /* 1st arg MUST be variable, 2nd format */
  arg = d_arg[0];
  arg_cnt++;

  /*
   * Check the Datum type, if ENTITY force it to be a STRING,
   * if it is some sort of CONSTANT... error.
   */

  if (arg.type != ENTITY)
  {
    clean_bltin_args (d_arg, n_args);
    warning_1 ("sprintf() requires a variable as 1st arg", 0);
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }
  else
  {
    if (e_type (arg.u.ent) != STRING)
    {
      /* Special case, create a variable from an UNDEF */
      if (e_type (arg.u.ent) != UNDEF)
	listNode_DestroyDataOnly (arg.u.ent);
      strv = string_Create ((char *) 0);
      string_SetName (strv, cpstr (e_name (arg.u.ent)));
      listNode_AttachData (arg.u.ent, STRING, strv, string_Destroy);
    }
    else
      strv = (String *) e_data (arg.u.ent);
  }

  FSTRING = bltin_get_string ("sprintf", d_arg, ++arg_cnt);
  format = string_GetString (e_data (FSTRING));

  retval = do_printf ((FILE *) 0, format, n_args, arg_cnt, d_arg, strv);
  *return_ptr = (VPTR) scalar_Create ((double) retval);
  remove_tmp_destroy (FSTRING);
  {
    /* Skip the 1st arg. */
    int i;
    for (i = 2; i < n_args; i++)
    {
      if (d_arg[i].type == ENTITY)
	remove_tmp_destroy (d_arg[i].u.ent);
    }
  }
}

#define MAX_SPRINTF_SIZE  4096
char sprintf_buff[MAX_SPRINTF_SIZE];
char *sprintf_limit = sprintf_buff + MAX_SPRINTF_SIZE;

static int
do_printf (fp, format, n_args, arg_cnt, d_arg, strv)
     FILE *fp;
     char *format;
     int n_args, arg_cnt;
     Datum *d_arg;
     String *strv;
{
  char save;
  char *p;
  register char *q;
  char *target;
  int l_flag, h_flag;		/* seen %ld or %hd  */
  int ast_cnt;
  int ast[2];
  double dval = 0.0;
  int ival = 0;
  char sval[MAX_SPRINTF_SIZE];
  int num_conversion = 0;	/* for error messages */
  int pf_type = 0;		/* conversion type */
  PRINTER printer;		/* pts at fprintf() or sprintf() */
  int argcnt;
  int retval;			/* Attempt to return C-printf() return-val */

  Datum arg;

  argcnt = n_args - 1;		/* To count backward */
  q = format;
  retval = 0;

  if (fp == (FILE *) 0)		/* doing sprintf */
  {
    if (strv == 0)
      error_1 ("terrible error in [fs]printf()", 0);
    target = sprintf_buff;
    printer = (PRINTER) sprintf;
  }
  else
    /* doing printf */
  {
    target = (char *) fp;	/* will never change */
    printer = (PRINTER) fprintf;
  }

  /* Traverse format string, doing printf(). */
  while (1)
  {
    if (fp)			/* printf */
    {
      while (*q != '%')
      {
	if (*q == 0)
	{
	  fflush (fp);
	  return (retval);
	}
	else
	{
	  putc (*q, fp);
	  q++;
	  retval++;
	}
      }
    }
    else
      /* sprintf() */
    {
      while (*q != '%')
      {
	if (*q == 0)
	{
	  if (target > sprintf_limit)	/* damaged */
	    error_1 ("sprintf() problem, buff to small", 0);
	  else
	    /* really done */
	  {
	    int len = target - sprintf_buff;
	    string_memcpy (strv, sprintf_buff, len);
	    return (retval);
	  }
	}
	else
	{
	  *target++ = *q++;
	  retval++;
	}
      }
    }

    num_conversion++;

    if (*++q == '%')		/* %% */
    {
      if (fp)
      {
	putc (*q, fp);
	retval++;
      }
      else
      {
	*target++ = *q;
      }
      q++;
      continue;
    }

    /* 
     * We have found a conversion specifier, figure it out,
     * then print the data asociated with it.
     */

    if (argcnt <= 0)
    {
      (*printer) ((VPTR) target, "\n");
      fflush ((FILE *) target);
      warning_1 ("not enough arguments passed to printf()", 0);
      return (-1);
    }
    else
    {
      /* Get The data object from the arg-list */
      arg = bltin_get_arg ("printf", d_arg, ++arg_cnt);
      arg = convert_to_scalar (arg);
      argcnt--;
    }

    /* mark the '%' with p */
    p = q - 1;

    /* eat the flags */
    while (*q == '-' || *q == '+' || *q == ' ' ||
	   *q == '#' || *q == '0')
      q++;

    ast_cnt = 0;		/* asterisk count */
    if (*q == '*')
    {
      /* Use current arg as field width spec */
      ast[ast_cnt++] = (int)
	get_datum_value (arg, "printf(): * arg must be a scalar");
      q++;

      if (argcnt <= 0)
      {
	(*printer) ((VPTR) target, "\n");
	fflush ((FILE *) target);
	warning_1 ("not enough arguments passed to printf()", 0);
	return (-1);
      }
      else
      {
	/* Get next arg */
	arg = bltin_get_arg ("printf", d_arg, ++arg_cnt);
	arg = convert_to_scalar (arg);
	argcnt--;
      }
    }
    else
      while (scan_code[*(unsigned char *) q] == SC_DIGIT)
	q++;
    /* width is done */

    if (*q == '.')		/* have precision */
    {
      q++;
      if (*q == '*')
      {
	/* Use current arg as precision spec */
	ast[ast_cnt++] = (int)
	  get_datum_value (arg, "printf(): * arg must be a scalar");
	q++;

	if (argcnt <= 0)
	{
	  (*printer) ((VPTR) target, "\n");
	  fflush ((FILE *) target);
	  warning_1 ("not enough arguments passed to printf()", 0);
	  return (-1);
	}
	else
	{
	  /* Get next arg */
	  arg = bltin_get_arg ("printf", d_arg, ++arg_cnt);
	  arg = convert_to_scalar (arg);
	  argcnt--;
	}
      }
      else
	while (scan_code[*(unsigned char *) q] == SC_DIGIT)
	  q++;
    }

    if (argcnt < 0)
    {
      (*printer) ((VPTR) target, "\n");
      fflush ((FILE *) target);
      warning_1 ("not enough arguments passed to printf()", 0);
      return (-1);
    }

    l_flag = h_flag = 0;

    if (*q == 'l')
    {
      q++;
      l_flag = 1;
    }
    else if (*q == 'h')
    {
      q++;
      h_flag = 1;
    }

    /* Set pf_type and load val */
    switch (*q++)
    {
    case 's':
      if (l_flag + h_flag)
	error_1 ("printf(): bad conversion", format);

      if (e_type (arg.u.ent) == STRING)
      {
	/*
	 * This should be the only place we could overflow sval
	 */
	if (string_GetLength (e_data (arg.u.ent)) > MAX_SPRINTF_SIZE)
	  error_1 ("string too long! for printf()", 0);
	strcpy (sval, string_GetString (e_data (arg.u.ent)));
      }
      else if (e_type (arg.u.ent) == SCALAR)
	sprintf (sval, "%f", SVALr (e_data (arg.u.ent)));
      else if (e_type (arg.u.ent) == MATRIX)
	error_1 (e_name (arg.u.ent), "MATRIX invalid type for printf()");
      else if (e_type (arg.u.ent) == BTREE)
	error_1 (e_name (arg.u.ent), "LIST invalid type for printf()");

      pf_type = PF_S;
      break;
    case 'c':
      if (e_type (arg.u.ent) == SCALAR)
	ival = (int) SVALr (e_data (arg.u.ent));
      else if (e_type (arg.u.ent) == STRING)
	ival = (int) strtod (string_GetString (e_data (arg.u.ent)),
			     (char **) 0);
      else if (e_type (arg.u.ent) == MATRIX)
	error_1 (e_name (arg.u.ent), "MATRIX invalid type for printf()");
      else if (e_type (arg.u.ent) == BTREE)
	error_1 (e_name (arg.u.ent), "LIST invalid type for printf()");

      pf_type = PF_C;
      break;
    case 'd':
      dval = get_datum_value (arg, "printf(): arg must be a scalar");
      pf_type = PF_D;
      break;
    case 'o':
      error_1 ("printf(): \"o\" format not allowed", format);
      break;
    case 'x':
      error_1 ("printf(): \"x\" format not allowed", format);
      break;
    case 'X':
      error_1 ("printf(): \"X\" format not allowed", format);
      break;
    case 'i':
    case 'u':
      /* use strod() here */
      dval = get_datum_value (arg, "printf(): arg must be a scalar");
      pf_type = l_flag ? PF_LD : PF_D;
      break;
    case 'e':
    case 'g':
    case 'f':
    case 'E':
    case 'G':
      if (h_flag + l_flag)
	error_1 ("printf(): bad conversion", format);
      /* use strod() here */
      dval = get_datum_value (arg, "printf(): arg must be a scalar");
      pf_type = PF_F;
      break;

    default:
      error_1 ("printf(): bad conversion", format);
    }

    save = *q;
    *q = 0;

    /* ready to call printf() */
    /* 
     * target:   The output file (or variable for sprintf())
     * p:        the beginning of the format
     * ast:      array with asterisk values
     */
    switch (AST (ast_cnt, pf_type))
    {
    case AST (0, PF_C):
      retval += (*printer) ((VPTR) target, p, ival);
      break;

    case AST (1, PF_C):
      retval += (*printer) ((VPTR) target, p, ast[0], ival);
      break;

    case AST (2, PF_C):
      retval += (*printer) ((VPTR) target, p, ast[0], ast[1], ival);
      break;

    case AST (0, PF_S):
      retval += (*printer) ((VPTR) target, p, sval);
      break;

    case AST (1, PF_S):
      retval += (*printer) ((VPTR) target, p, ast[0], sval);
      break;

    case AST (2, PF_S):
      retval += (*printer) ((VPTR) target, p, ast[0], ast[1], sval);
      break;

    case AST (0, PF_D):
      retval += (*printer) ((VPTR) target, p, (int) dval);
      break;

    case AST (1, PF_D):
      retval += (*printer) ((VPTR) target, p, ast[0], (int) dval);
      break;

    case AST (2, PF_D):
      retval += (*printer) ((VPTR) target, p, ast[0], ast[1], (int) dval);
      break;

    case AST (0, PF_LD):
      retval += (*printer) ((VPTR) target, p, (long int) dval);
      break;

    case AST (1, PF_LD):
      retval += (*printer) ((VPTR) target, p, ast[0], (long int) dval);
      break;

    case AST (2, PF_LD):
      retval += (*printer) ((VPTR) target, p, ast[0], ast[1], (long int) dval);
      break;

    case AST (0, PF_F):
      retval += (*printer) ((VPTR) target, p, dval);
      break;

    case AST (1, PF_F):
      retval += (*printer) ((VPTR) target, p, ast[0], dval);
      break;

    case AST (2, PF_F):
      retval += (*printer) ((VPTR) target, p, ast[0], ast[1], dval);
      break;
    }
    if (fp == (FILE *) 0)
      while (*target)
	target++;
    *q = save;
  }
}

/* **************************************************************
 * Functions for handling files. Keep track of all the open files
 * with a list. Only close() when explicitly requested, or on exit.
 * ************************************************************** */

#include "mem.h"
#include "list.h"

/*
 * File list: Use a linked list to keep track of currently open
 * files. Identify Rfile structs by the name of the open file.
 */

static List *file_list;

/* RLaB File struct */
struct _r_file
{
  FILE *fileds;
  char *mode;
  int isfile;
  char *buffer;
};
typedef struct _r_file Rfile;

void
rfile_Destroy (rfile)
     Rfile *rfile;
{
  ASSERT (rfile);
  {
    if (rfile->isfile)
      fclose (rfile->fileds);
    else
      rpclose (rfile->fileds);
    
    FREE (rfile->mode);
    rfile->isfile = -1;
    FREE (rfile->buffer);
    FREE (rfile);
  }
}

FILE *
rfile_GetFileds (rfile)
     Rfile *rfile;
{
  return (rfile->fileds);
}

/* Initialize the file list */

void
init_file_list ()
{
  file_list = list_Create ();
}

void
destroy_file_list ()
{
  list_Destroy (file_list);
}

/*
 * Get the file descriptor asociated with the char string. This
 * function does all the work, searches the list, uses popen()
 * instead of fopen when necessary...
 */

FILE *
get_file_ds (name, mode, buffsize)
     char *name, *mode;
     int buffsize;
{
  int Pipe;
  FILE *fileds;
  ListNode *fnode;
  Rfile *new;

  /*
   * Check for stdout, stderr. We don't want to put them
   * on the file_list, since we don't want to close stdout
   * or stderr ever
   */
  if (!strcmp ("stdout", name))
    return (stdout);
  else if (!strcmp ("stderr", name))
    return (stderr);
  else if (!strcmp ("stdin", name))
    return (stdin);

  /* Check for pipe symbol '|' */
  if (*name == '|')
  {
    name++;
    Pipe = 1;
  }
  else
    Pipe = 0;

  /* Check list for previously opened occurence of file */
  if ((fnode = list_GetNodeByKey (file_list, name)))
  {
    return (rfile_GetFileds (e_data (fnode)));
  }
  else
  {
    /*
     * Make sure we don't try and open more files
     * than the system will allow
     */
    if (list_GetNumNodes (file_list) >= FOPEN_MAX - 3)
      error_1 ("exceeded system limit for # of open files", (char *) 0);

#ifndef THINK_C
    if (Pipe)
    {
      if ((fileds = popen (name, mode)) != 0)
      {
	/* Put new fileds on list */
	new = (Rfile *) MALLOC (sizeof (Rfile));
	new->fileds = fileds;
	new->mode = cpstr (mode);
	new->isfile = 0;
	new->buffer = 0;
	fnode = listNode_Create ();
	listNode_SetKey (fnode, cpstr (name));
	fnode = listNode_AttachData (fnode, 1, new, rfile_Destroy);
	list_PushNode (file_list, fnode);
      }
      else
	return (0);
    }
    else
    {
#endif
      if ((fileds = fopen (name, mode)) != 0)
      {
	/* Put new fileds on list */
	new = (Rfile *) MALLOC (sizeof (Rfile));
	new->fileds = fileds;
	new->mode = cpstr (mode);
	new->isfile = 1;
	if (buffsize != 0)
	{
	  new->buffer = (char *) MALLOC (sizeof (char) * buffsize);
	  if (setvbuf (fileds, new->buffer, _IOFBF, buffsize))
	  {
	    fprintf (stderr, "open: cannot create I/O buffer\n");
	    rfile_Destroy (new);
	    return (0);
	  }
	}
	else
	{
	  new->buffer = 0;
	}
	fnode = listNode_Create ();
	listNode_SetKey (fnode, cpstr (name));
	fnode = listNode_AttachData (fnode, 1, new, rfile_Destroy);
	list_PushNode (file_list, fnode);
      }
      else
	return (0);
#ifndef THINK_C	
    }
#endif
    return (fileds);
  }
}

/* Close the file asociated with the file name */

int
close_file_ds (name)
     char *name;
{
  ListNode *fnode;

  /* Check list */
  if (name == 0)
    return (0);
  if (*name == '|')
    name++;
  if ((fnode = list_GetNodeByKey (file_list, name)))
  {
    /* If on list, remove and  close */
    list_DetachNodeByAddr (file_list, fnode);
    listNode_Destroy (fnode);
    return (1);
  }
  else
    return (0);
}
