/* op_append.c */
/* Append related operations for the RLaB machine. */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "code.h"
#include "scalar.h"
#include "matrix.h"
#include "r_string.h"
#include "util.h"

/* **************************************************************
 * Create a vector from the command:
 *     expr ':' expr ':' expr
 *     start    end      increment
 *      d3       d2       d1         (if n = 3)
 * or   d2       d1                  (if n = 2)
 * Where the last expr is optional (default is 1)
 * ************************************************************** */
Datum
vector_create (n, d1, d2, d3)
     int n;
     Datum d1, d2, d3;
{
  double d1val, d2val, d3val;
  Datum new;
  Matrix *v = 0;

  switch (n)
  {
  case 2:
    d1val = get_datum_value (d1, "non-scalar illegal in i:j:k stmt");
    d2val = get_datum_value (d2, "non-scalar illegal in i:j:k stmt");
    v = matrix_CreateFill (d2val, d1val, (double) 1.0, 1);
    if (d1.type == ENTITY)
      remove_tmp_destroy (d1.u.ent);
    if (d2.type == ENTITY)
      remove_tmp_destroy (d2.u.ent);
    break;
  case 3:
    d1val = get_datum_value (d1, "non-scalar illegal in i:j:k stmt");
    d2val = get_datum_value (d2, "non-scalar illegal in i:j:k stmt");
    d3val = get_datum_value (d3, "non-scalar illegal in i:j:k stmt");
    v = matrix_CreateFill (d3val, d2val, d1val, 1);
    if (d1.type == ENTITY)
      remove_tmp_destroy (d1.u.ent);
    if (d2.type == ENTITY)
      remove_tmp_destroy (d2.u.ent);
    if (d3.type == ENTITY)
      remove_tmp_destroy (d3.u.ent);
    break;
  }

  /* Automatically demote to SCALAR if NC == 1 */
  if (MNC (v) == 1)
  {
    new.u.ent = install_tmp (SCALAR, scalar_Create (MAT (v, 1, 1)),
			     scalar_Destroy);
    new.type = ENTITY;
    matrix_Destroy (v);
  }
  else
  {
    new.u.ent = install_tmp (MATRIX, v, matrix_Destroy);
    new.type = ENTITY;
  }
  return (new);
}

Datum vector_ent_ent_append _PROTO ((ListNode * e1, ListNode * e2));
static Datum scalar_scalar_app _PROTO ((Scalar * s1, Scalar * s2));
static Datum scalar_matrix_app _PROTO ((Scalar * s, Matrix * m));
static Datum matrix_scalar_app _PROTO ((Matrix * m, Scalar * s));
static Datum matrix_matrix_app _PROTO ((Matrix * m1, Matrix * m2));
static Datum matrix_string_app _PROTO ((Matrix * m, String * s2));
static Datum string_matrix_app _PROTO ((String * s1, Matrix * m));
static Datum string_string_app _PROTO ((String * s1, String * s2));

/* **************************************************************
 * Create a MATRIX. To do this in the most general way we must
 * build the MATRIX by appending elements to it.
 * d1 , d2
 * d1 is the root element.
 * d2 is the appendee (element we are adding).
 * ************************************************************** */
Datum
vector_append (d1, d2)
     Datum d1, d2;
{
  Datum new;

  /* Convert all CONSTANT and iCONSTANT into entities in order
     to eliminate switch at this level */
  d1 = convert_to_scalar (d1);
  d2 = convert_to_scalar (d2);

  new = vector_ent_ent_append (d1.u.ent, d2.u.ent);
  remove_tmp_destroy (d1.u.ent);
  remove_tmp_destroy (d2.u.ent);
  return (new);
}

/* **************************************************************
 * Append an existing entity to another existing entity, return
 * the new entity in a new Datum. e1, e2
 * ************************************************************** */
Datum
vector_ent_ent_append (e1, e2)
     ListNode *e1, *e2;
{
  Datum new;

  switch (e_type (e1))
  {
  case SCALAR:
    switch (e_type (e2))
    {
    case SCALAR:		/* scalar, scalar */
      new = scalar_scalar_app (e_data (e1), e_data (e2));
      break;
    case MATRIX:		/* scalar, matrix */
      new = scalar_matrix_app (e_data (e1), e_data (e2));
      break;
    case STRING:
      error_2 (e_name (e1), e_name (e2), "cannot append STRING to numeric");
      break;
    default:
      error_2 (e_name (e1), e_name (e2), "wrong type(s) for append op");
      break;
    }
    break;
  case MATRIX:
    switch (e_type (e2))
    {
    case SCALAR:		/* matrix, scalar */
      new = matrix_scalar_app (e_data (e1), e_data (e2));
      break;
    case MATRIX:		/* matrix, matrix */
      new = matrix_matrix_app (e_data (e1), e_data (e2));
      break;
    case STRING:
      new = matrix_string_app (e_data (e1), e_data (e2));
      break;
    default:
      error_2 (e_name (e1), e_name (e2), "wrong type(s) for append op");
      break;
    }
    break;
  case STRING:
    switch (e_type (e2))
    {
    case STRING:
      new = string_string_app (e_data (e1), e_data (e2));
      break;
    case MATRIX:
      new = string_matrix_app (e_data (e1), e_data (e2));
      break;
    default:
      error_2 (e_name (e1), e_name (e2), "cannot append numeric to STRING");
      break;
    }
    break;
  default:
    error_1 (e_name (e1), "wrong type for append op");
    break;
  }
  return (new);
}

/* **************************************************************
 * Append functions for various combinations of entities.
 * ************************************************************** */
static Datum
scalar_scalar_app (s1, s2)
     Scalar *s1, *s2;
{
  Datum new;
  Matrix *v;

  if ((SVALi (s1) == 0.0) && (SVALi (s2) == 0.0))
  {
    v = matrix_Create (1, 2);
    MAT (v, 1, 1) = SVALr (s1);
    MAT (v, 1, 2) = SVALr (s2);
  }
  else
  {
    v = matrix_CreateC (1, 2);
    MATr (v, 1, 1) = SVALr (s1);
    MATi (v, 1, 1) = SVALi (s1);
    MATr (v, 1, 2) = SVALr (s2);
    MATi (v, 1, 2) = SVALi (s2);
  }
  new.u.ent = install_tmp (MATRIX, v, matrix_Destroy);
  new.type = ENTITY;
  return (new);
}

static Datum
scalar_matrix_app (s, m)
     Scalar *s;
     Matrix *m;
{
  int i;
  Datum new;
  Matrix *mnew;

  /* Error check 1st */
  if (MNR (m) > 1)
    error_1 (matrix_GetName (m),
	     "cannot append multi-row matrix to scalar");

  if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
  {
    /* Create the new matrix, and copy into it */
    mnew = matrix_Create (1, MNC (m) + 1);
    MAT (mnew, 1, 1) = SVALr (s);
    for (i = 2; i <= MNC (mnew); i++)
      MAT (mnew, 1, i) = MAT (m, 1, i - 1);
  }
  else
  {
    if (MTYPE (m) == COMPLEX)
    {
      /* Create new COMPLEX matrix and copy into it */
      mnew = matrix_CreateC (1, MNC (m) + 1);
      MATr (mnew, 1, 1) = SVALr (s);
      MATi (mnew, 1, 1) = SVALi (s);
      for (i = 2; i <= MNC (mnew); i++)
      {
	MATr (mnew, 1, i) = MATr (m, 1, i - 1);
	MATi (mnew, 1, i) = MATi (m, 1, i - 1);
      }
    }
    else
    {
      mnew = matrix_CreateC (1, MNC (m) + 1);
      MATr (mnew, 1, 1) = SVALr (s);
      MATi (mnew, 1, 1) = SVALi (s);
      for (i = 2; i <= MNC (mnew); i++)
      {
	MATr (mnew, 1, i) = MAT (m, 1, i - 1);
	MATi (mnew, 1, i) = 0.0;
      }
    }
  }
  new.u.ent = install_tmp (MATRIX, mnew, matrix_Destroy);
  new.type = ENTITY;
  return (new);
}

static Datum
matrix_scalar_app (m, s)
     Matrix *m;
     Scalar *s;
{
  int i, n_row, n_col;
  Datum new;
  Matrix *mnew;

  n_row = MNR (m);
  if (n_row > 1)
    error_1 (matrix_GetName (m), "must be row matrix for append");
  n_col = MNC (m);

  if (MTYPE (m) == REAL && SVALi (s) == 0.0)
  {
    mnew = matrix_Create (1, n_col + 1);
    for (i = 1; i <= n_col; i++)
      MAT (mnew, 1, i) = MAT (m, 1, i);
    MAT (mnew, 1, n_col + 1) = SVALr (s);
  }
  else
  {
    mnew = matrix_CreateC (1, n_col + 1);
    if (MTYPE (m) == REAL)
    {
      for (i = 1; i <= n_col; i++)
      {
	MATr (mnew, 1, i) = MAT (m, 1, i);
	MATi (mnew, 1, i) = 0.0;
      }
      MATr (mnew, 1, n_col + 1) = SVALr (s);
      MATi (mnew, 1, n_col + 1) = SVALi (s);
    }
    else
    {
      for (i = 1; i <= n_col; i++)
      {
	MATr (mnew, 1, i) = MATr (m, 1, i);
	MATi (mnew, 1, i) = MATi (m, 1, i);
      }
      MATr (mnew, 1, n_col + 1) = SVALr (s);
      MATi (mnew, 1, n_col + 1) = SVALi (s);
    }
  }
  new.u.ent = install_tmp (MATRIX, mnew, matrix_Destroy);
  new.type = ENTITY;
  return (new);
}

static Datum
matrix_matrix_app (m1, m2)
     Matrix *m1, *m2;
{
  int i, j, n_row, n_col, n_col_base;
  Datum new;
  Matrix *mnew = 0;

  n_col_base = MNC (m1);
  n_col = n_col_base + MNC (m2);
  n_row = MNR (m1);

  /*
   * Take care of special case NULL matrices.
   */
  if (MNR (m1) == 0 && MNC (m1) == 0)
  {
    mnew = matrix_Copy (m2);
    new.u.ent = install_tmp (MATRIX, mnew, matrix_Destroy);
    new.type = ENTITY;
    return (new);
  }
  if (MNR (m2) == 0 && MNC (m2) == 0)
  {
    mnew = matrix_Copy (m1);
    new.u.ent = install_tmp (MATRIX, mnew, matrix_Destroy);
    new.type = ENTITY;
    return (new);
  }

  if (n_row != MNR (m2))
    error_2 (matrix_GetName (m1), matrix_GetName (m2),
	     "row dimension mismatch for matrix append op");

  if (MTYPE (m1) == REAL && MTYPE (m2) == REAL)
  {
    mnew = matrix_Copy (m1);
    matrix_AppendColR (mnew, MNC (m2));
    for (i = 1; i <= n_row; i++)
      for (j = n_col_base + 1; j <= n_col; j++)
	MAT (mnew, i, j) = MAT (m2, i, (j - n_col_base));
  }
  else if (MTYPE (m1) == COMPLEX || MTYPE (m2) == COMPLEX)
  {
    /* One of them is COMPLEX */
    mnew = matrix_copy_complex (m1);
    matrix_AppendColC (mnew, MNC (m2));
    if (MTYPE (m2) == REAL)
    {
      for (i = 1; i <= n_row; i++)
	for (j = n_col_base + 1; j <= n_col; j++)
	{
	  MATr (mnew, i, j) = MAT (m2, i, (j - n_col_base));
	  MATi (mnew, i, j) = 0.0;
	}
    }
    else
    {
      for (i = 1; i <= n_row; i++)
	for (j = n_col_base + 1; j <= n_col; j++)
	{
	  MATr (mnew, i, j) = MATr (m2, i, (j - n_col_base));
	  MATi (mnew, i, j) = MATi (m2, i, (j - n_col_base));
	}
    }
  }
  else if (MTYPE (m1) == STRING && MTYPE (m2) == STRING)
  {
    mnew = matrix_Copy (m1);
    matrix_AppendColS (mnew, MNC (m2));
    for (i = 1; i <= n_row; i++)
      for (j = n_col_base + 1; j <= n_col; j++)
	MATs (mnew, i, j) = cpstr (MATs (m2, i, (j - n_col_base)));
  }
  else
  {
    error_2 (matrix_GetName (m1), matrix_GetName (m2),
	     "invlaid type for MATRIX, MATRIX append");
  }
  new.u.ent = install_tmp (MATRIX, mnew, matrix_Destroy);
  new.type = ENTITY;
  return (new);
}

static Datum
string_string_app (s1, s2)
     String *s1, *s2;
{
  Datum new;
  Matrix *m;

  m = matrix_CreateS (1, 2);
  MATsv (m, 0) = cpstr (string_GetString (s1));
  MATsv (m, 1) = cpstr (string_GetString (s2));
  new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
  new.type = ENTITY;
  return (new);
}

static Datum
matrix_string_app (m, s)
     Matrix *m;
     String *s;
{
  int i;
  Datum new;
  Matrix *mnew;

  if (MNR (m) == 0)
  {
    /* Special case, where lhs is empty */
    /* A potential leak ? */
    m = matrix_CreateS (0, 0);
  }
  if (MTYPE (m) != STRING)
    error_1 (matrix_GetName (m), "cannot append STRING to numeric MATRIX");

  if (MNR (m) > 1)
    error_2 (matrix_GetName (m), string_GetName (s),
	     "cannot append a single STRING to multi-row MATRIX");

  mnew = matrix_CreateS (1, MNC (m) + 1);
  for (i = 0; i < MNC (m); i++)
  {
    MATsv (mnew, i) = cpstr (MATsv (m, i));
  }
  MATsv (mnew, MNC (m)) = cpstr (string_GetString (s));

  new.u.ent = install_tmp (MATRIX, mnew, matrix_Destroy);
  new.type = ENTITY;
  return (new);
}

static Datum
string_matrix_app (s, m)
     String *s;
     Matrix *m;
{
  int i;
  Datum new;
  Matrix *mnew;

  if (MTYPE (m) != STRING)
    error_1 (matrix_GetName (m), "cannot append a MATRIX to a STRING");

  if (MNR (m) != 1)
    error_2 (matrix_GetName (m), string_GetName (s),
	     "cannot append a multi-row MATRIX to a single STRING");

  mnew = matrix_CreateS (1, MNC (m) + 1);
  MATsv (mnew, 0) = cpstr (string_GetString (s));
  for (i = 1; i <= MNC (m) + 1; i++)
    MATsv (mnew, i) = cpstr (MATsv (m, i));

  new.u.ent = install_tmp (MATRIX, mnew, matrix_Destroy);
  new.type = ENTITY;
  return (new);
}
