/* matrix.h */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1993  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#ifndef RLAB_MATRIX_H
#define RLAB_MATRIX_H

#include "rlab.h"
#include "complex.h"
#include "btree.h"
#include "scalar.h"

#include <stdio.h>
#include <math.h>

struct _matrix
{
  int type;			/* Structure type */
  char *name;			/* Object name */
  int nrow;			/* # of rows */
  int ncol;			/* # of columns */
  int dtype;			/* Element data type, REAL or COMPLEX */
  union mval
  {
    double *mr;			/* REAL data */
    Complex *mc;		/* COMPLEX data */
    char **ms;			/* STRING data */
  }
  val;
  Btree *list;
};

typedef struct _matrix Matrix;

extern Matrix *matrix_Create _PROTO ((int nr, int nc));
extern Matrix *matrix_CreateC _PROTO ((int nr, int nc));
extern Matrix *matrix_CreateS _PROTO ((int nr, int nc));
extern void matrix_Destroy _PROTO ((Matrix * matrix));
extern Matrix *matrix_Copy _PROTO ((Matrix * m_orig));
extern Matrix *matrix_Reshape _PROTO ((Matrix * m, int nrow, int ncol));

extern void matrix_Print _PROTO ((Matrix * matrix, FILE * stream));
extern void matrix_Write _PROTO ((Matrix * m, FILE * fn));
extern Matrix *matrix_Read _PROTO ((FILE * fn));

extern void matrix_Zero _PROTO ((Matrix * m));

extern void matrix_CheckIndices _PROTO ((Matrix * m, int row, int col));

extern int matrix_AppendColR _PROTO ((Matrix * matrix, int n_col));
extern int matrix_AppendColC _PROTO ((Matrix * matrix, int n_col));
extern int matrix_AppendColS _PROTO ((Matrix * matrix, int n_col));
extern int matrix_AppendRowR _PROTO ((Matrix * matrix, int n_row));
extern int matrix_AppendRowC _PROTO ((Matrix * matrix, int n_row));
extern int matrix_AppendRowS _PROTO ((Matrix * matrix, int n_row));
extern int matrix_Extend _PROTO ((Matrix * m, int n_row, int n_col));

extern Matrix *matrix_ExtractRowMatrix _PROTO ((Matrix * m, Matrix * v));
extern Matrix *matrix_ExtractColMatrix _PROTO ((Matrix * m, Matrix * v));
extern Matrix *matrix_ExtractColumn _PROTO ((Matrix * m, int c, Matrix * rv));
extern Matrix *matrix_ExtractRow _PROTO ((Matrix * m, int r, Matrix * cv));
extern Matrix *matrix_ExtractElement _PROTO ((Matrix * m, int r, int c));

extern Matrix *matrix_ExtractSubMatrix _PROTO ((Matrix * m, Matrix * v_row,
						Matrix * v_col));
extern Matrix *matrix_ExtractVector _PROTO ((Matrix * m, Matrix * mi));

extern int matrix_Assign _PROTO ((Matrix * m, Matrix * rv, Matrix * cv,
				  Matrix * rhs));
extern int matrix_Assign_el _PROTO ((Matrix * m, int ri, int ci, Matrix * rhs));
extern int matrix_AssignRow _PROTO ((Matrix * m, Matrix * cv, Matrix * rhs));
extern int matrix_AssignCol _PROTO ((Matrix * m, Matrix * rv, Matrix * rhs));
extern void matrix_AssignVector _PROTO ((Matrix * m, Matrix * ind,
					 Matrix * rhs));

extern void matrix_convert _PROTO ((Matrix * m));
extern Matrix *matrix_copy_complex _PROTO ((Matrix * m));

extern Matrix *matrix_CreateFill _PROTO ((double d1, double d2, double d3, int f));

extern int matrix_Truncate _PROTO ((Matrix * m, int nr, int nc));

#define matrix_screen_string(m)  if(MTYPE(m) == STRING) \
                                 error_1((((Matrix *)(m))->name), \
				 "STRING Matrix not allowed in NUMERIC context");

#define matrix_GetName(m)   (((Matrix *)(m))->name)
extern void matrix_SetName _PROTO ((Matrix * m, char *string));

#define matrix_GetNrow(m)  (((Matrix *)(m))->nrow)
#define matrix_GetNcol(m)  (((Matrix *)(m))->ncol)

/* Index matrices from 1 */
#define MAT(m,i,j)  (((Matrix *)(m))->val.mr[(j-1)*(((Matrix *)(m))->nrow)+(i-1)])
#define MATc(m,i,j) (((Matrix *)(m))->val.mc[(j-1)*(((Matrix *)(m))->nrow)+(i-1)])
#define MATr(m,i,j) (((Matrix *)(m))->val.mc[(j-1)*(((Matrix *)(m))->nrow)+(i-1)].r)
#define MATi(m,k,j) (((Matrix *)(m))->val.mc[(j-1)*(((Matrix *)(m))->nrow)+(k-1)].i)
#define MATs(m,k,j) (((Matrix *)(m))->val.ms[(j-1)*(((Matrix *)(m))->nrow)+(k-1)])

/* Index matrices from 0 */
#define MAT0(m,i,j)  (((Matrix *)(m))->val.mr[(j)*(((Matrix *)(m))->nrow)+(i)])
#define MATc0(m,i,j) (((Matrix *)(m))->val.mc[(j)*(((Matrix *)(m))->nrow)+(i)])
#define MATr0(m,i,j) (((Matrix *)(m))->val.mc[(j)*(((Matrix *)(m))->nrow)+(i)].r)
#define MATi0(m,k,j) (((Matrix *)(m))->val.mc[(j)*(((Matrix *)(m))->nrow)+(k)].i)
#define MATs0(m,k,j) (((Matrix *)(m))->val.ms[(j)*(((Matrix *)(m))->nrow)+(k)])

/* Allow programmers to index into matrix like a vector */
/* Real Matrix */
#define MATrv(m,i)    (((Matrix *)(m))->val.mr[i])
#define MATrv1(m,i)   (((Matrix *)(m))->val.mr[i-1])

/* Complex Matrix */
#define MATcv(m,i)    (((Matrix *)(m))->val.mc[i])
#define MATcv1(m,i)   (((Matrix *)(m))->val.mc[i-1])

/* Complex Matrix, real and imaginary parts */
#define MATcvr(m,j)   (((Matrix *)(m))->val.mc[j].r)
#define MATcvi(m,j)   (((Matrix *)(m))->val.mc[j].i)
#define MATcvr1(m,j)  (((Matrix *)(m))->val.mc[j-1].r)
#define MATcvi1(m,j)  (((Matrix *)(m))->val.mc[j-1].i)

/* String Matrix */
#define MATsv(m,j)     (((Matrix *)(m))->val.ms[j])
#define MATsv1(m,j)    (((Matrix *)(m))->val.ms[j-1])

#define MDPTRr(m)    (((Matrix *)(m))->val.mr)
#define MDPTRc(m)    (((Matrix *)(m))->val.mc)
#define MDPTRs(m)    (((Matrix *)(m))->val.ms)
#define MNR(m)       (((Matrix *)(m))->nrow)
#define MNC(m)       (((Matrix *)(m))->ncol)

#define MTYPE(m)     (((Matrix *)(m))->dtype)

#endif /* RLAB_MATRIX_H */
