/* matop1.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1994  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "matrix.h"
#include "matop1.h"
#include "matop2.h"
#include "r_string.h"
#include "bltin.h"
#include "mathl.h"
#include "util.h"
#include "mem.h"

#include "bl.h"
#include "fi.h"

#include <math.h>

static char null_str[] =
{'\0'};				/* Used with string comparisons */

/*
 * Define our own abs(). We need to do this, cause we need
 * and abs() that does double, as well as int.
 */
#define rabs(x) ((x) >= 0 ? (x) : -(x))

int do_cmplx _PROTO ((Matrix * m));

/* **************************************************************
 * Add 2 matrices
 * ************************************************************** */
Matrix *
matrix_Add (ma, mb)
     Matrix *ma, *mb;
{
  ASSERT (ma);
  ASSERT (mb);
  {
    register int i;
    Matrix *new = 0;

    /* check dimensions */
    if (MNR (ma) == MNR (mb) && MNC (ma) == MNC (mb))
    {
      switch (MTYPE (ma))
      {
      case REAL:
	switch (MTYPE (mb))
	{
	case REAL:		/* r + r */
	  new = matrix_Create (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	    new->val.mr[i] = ma->val.mr[i] + mb->val.mr[i];
	  break;
	case COMPLEX:		/* r + c */
	  new = matrix_CreateC (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	  {
	    new->val.mc[i].r = ma->val.mr[i] + mb->val.mc[i].r;
	    new->val.mc[i].i = mb->val.mc[i].i;
	  }
	  break;
	case STRING:
	  error_2 (matrix_GetName (ma), matrix_GetName (mb),
		   "cannot perform add-op with STRING MATRIX");
	  new = matrix_CreateS (MNR (ma), MNC (ma));
	  break;
	}
	break;
      case COMPLEX:
	switch (MTYPE (mb))
	{
	case REAL:		/* c + r */
	  new = matrix_CreateC (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	  {
	    new->val.mc[i].r = ma->val.mc[i].r + mb->val.mr[i];
	    new->val.mc[i].i = ma->val.mc[i].i;
	  }
	  break;
	case COMPLEX:		/* c + c */
	  new = matrix_CreateC (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	  {
	    new->val.mc[i].r = ma->val.mc[i].r + mb->val.mc[i].r;
	    new->val.mc[i].i = ma->val.mc[i].i + mb->val.mc[i].i;
	  }
	  break;
	case STRING:
	  error_2 (matrix_GetName (ma), matrix_GetName (mb),
		   "cannot perform add-op with STRING MATRIX");
	  break;
	}
	break;
      case STRING:
	if (MTYPE (mb) == STRING)
	{
	  new = matrix_CreateS (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	  {
	    FREE (new->val.ms[i]);
	    new->val.ms[i] = string_add (ma->val.ms[i], mb->val.ms[i]);
	  }
	}
	else
	  error_1 (matrix_GetName (mb),
		   "cannot add STRING and NUMERIC matrices");
	break;
      default:
	error_2 (matrix_GetName (ma), matrix_GetName (mb),
		 "type error adding A and B");
	break;
      }
    }
    else if ((MNR (ma) * MNC (ma)) == 1)
    {
      /* [A] is a 1-by-1 */
      Scalar *s;
      String *str;
      if (MTYPE (ma) == REAL)
      {
	s = scalar_Create (MAT (ma, 1, 1));
	new = matrix_scalar_add (mb, s);
	scalar_Destroy (s);
      }
      else if (MTYPE (ma) == COMPLEX)
      {
	s = scalar_CreateC (MATr (ma, 1, 1), MATi (ma, 1, 1));
	new = matrix_scalar_add (mb, s);
	scalar_Destroy (s);
      }
      else if (MTYPE (ma) == STRING)
      {
	str = string_Create (MATs (ma, 1, 1));
	new = matrix_string_add (mb, str, 2);
	string_Destroy (str);
      }
    }
    else if ((MNR (mb) * MNC (mb)) == 1)
    {
      /* [B] is a 1-by-1 */
      Scalar *s;
      String *str;
      if (MTYPE (mb) == REAL)
      {
	s = scalar_Create (MAT (mb, 1, 1));
	new = matrix_scalar_add (ma, s);
	scalar_Destroy (s);
      }
      else if (MTYPE (mb) == COMPLEX)
      {
	s = scalar_CreateC (MATr (mb, 1, 1), MATi (mb, 1, 1));
	new = matrix_scalar_add (ma, s);
	scalar_Destroy (s);
      }
      else if (MTYPE (mb) == STRING)
      {
	str = string_Create (MATs (mb, 1, 1));
	new = matrix_string_add (ma, str, 1);
	string_Destroy (str);
      }
    }
    else if (MNR (ma) == 0 || MNR (mb) == 0)
    {
      new = matrix_Create (0, 0);
    }
    else
    {
      error_2 (matrix_GetName (ma), matrix_GetName (mb),
	       "Dim must match for matrix add");
    }
    return (new);
  }
}

Matrix *
matrix_Sub (ma, mb)
     Matrix *ma, *mb;
{
  ASSERT (ma);
  ASSERT (mb);
  {
    register int i;
    Matrix *new = 0;

    /* check dimensions */
    if (MNR (ma) == MNR (mb) && MNC (ma) == MNC (mb))
    {
      switch (MTYPE (ma))
      {
      case REAL:
	switch (MTYPE (mb))
	{
	case REAL:		/* r - r */
	  new = matrix_Create (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	    new->val.mr[i] = ma->val.mr[i] - mb->val.mr[i];
	  break;
	case COMPLEX:		/* r - c */
	  new = matrix_CreateC (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	  {
	    new->val.mc[i].r = ma->val.mr[i] - mb->val.mc[i].r;
	    new->val.mc[i].i = -mb->val.mc[i].i;
	  }
	  break;
	case STRING:
	  error_2 (matrix_GetName (ma), matrix_GetName (mb),
		   "cannot perform add-op with STRING MATRIX");
	  break;
	}
	break;
      case COMPLEX:
	switch (MTYPE (mb))
	{
	case REAL:		/* c - r */
	  new = matrix_CreateC (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	  {
	    new->val.mc[i].r = ma->val.mc[i].r - mb->val.mr[i];
	    new->val.mc[i].i = ma->val.mc[i].i;
	  }
	  break;
	case COMPLEX:		/* c - c */
	  new = matrix_CreateC (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	  {
	    new->val.mc[i].r = ma->val.mc[i].r - mb->val.mc[i].r;
	    new->val.mc[i].i = ma->val.mc[i].i - mb->val.mc[i].i;
	  }
	  break;
	case STRING:
	  error_2 (matrix_GetName (ma), matrix_GetName (mb),
		   "cannot perform add-op with STRING MATRIX");
	  break;
	}
	break;
      case STRING:
	error_2 (matrix_GetName (ma), matrix_GetName (mb),
		 "cannot perform add-op with STRING MATRIX");
	break;
      }
    }
    else if ((MNR (ma) * MNC (ma)) == 1)
    {
      Scalar *s;
      if (MTYPE (ma) == REAL)
	s = scalar_Create (MAT (ma, 1, 1));
      else
	s = scalar_CreateC (MATr (ma, 1, 1), MATi (ma, 1, 1));
      new = matrix_scalar_sub2 (mb, s);
      scalar_Destroy (s);
    }
    else if ((MNR (mb) * MNC (mb)) == 1)
    {
      Scalar *s;
      if (MTYPE (mb) == REAL)
	s = scalar_Create (MAT (mb, 1, 1));
      else
	s = scalar_CreateC (MATr (mb, 1, 1), MATi (mb, 1, 1));
      new = matrix_scalar_sub1 (ma, s);
      scalar_Destroy (s);
    }
    else if (MNR (ma) == 0 || MNR (mb) == 0)
    {
      new = matrix_Create (0, 0);
    }
    else
    {
      error_2 (matrix_GetName (ma), matrix_GetName (mb),
	       "Dim. must match for matrix subtract");
    }
    return (new);
  }
}

Matrix *
matrix_El_Add (ma, mb)
     Matrix *ma, *mb;
{
  ASSERT (ma);
  ASSERT (mb);
  {
    register int i;
    Matrix *new = 0;

    /* check dimensions */
    if (MNR (ma) == MNR (mb) && MNC (ma) == MNC (mb))
    {
      switch (MTYPE (ma))
      {
      case REAL:
	switch (MTYPE (mb))
	{
	case REAL:		/* r + r */
	  new = matrix_Create (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	    new->val.mr[i] = ma->val.mr[i] + mb->val.mr[i];
	  break;
	case COMPLEX:		/* r + c */
	  new = matrix_CreateC (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	  {
	    new->val.mc[i].r = ma->val.mr[i] + mb->val.mc[i].r;
	    new->val.mc[i].i = mb->val.mc[i].i;
	  }
	  break;
	case STRING:
	  error_2 (matrix_GetName (ma), matrix_GetName (mb),
		   "cannot perform add-op with STRING MATRIX");
	  new = matrix_CreateS (MNR (ma), MNC (ma));
	  break;
	}
	break;
      case COMPLEX:
	switch (MTYPE (mb))
	{
	case REAL:		/* c + r */
	  new = matrix_CreateC (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	  {
	    new->val.mc[i].r = ma->val.mc[i].r + mb->val.mr[i];
	    new->val.mc[i].i = ma->val.mc[i].i;
	  }
	  break;
	case COMPLEX:		/* c + c */
	  new = matrix_CreateC (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	  {
	    new->val.mc[i].r = ma->val.mc[i].r + mb->val.mc[i].r;
	    new->val.mc[i].i = ma->val.mc[i].i + mb->val.mc[i].i;
	  }
	  break;
	case STRING:
	  error_2 (matrix_GetName (ma), matrix_GetName (mb),
		   "cannot perform add-op with STRING MATRIX");
	  break;
	}
	break;
      case STRING:
	if (MTYPE (mb) == STRING)
	{
	  new = matrix_CreateS (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	  {
	    FREE (new->val.ms[i]);
	    new->val.ms[i] = string_add (ma->val.ms[i], mb->val.ms[i]);
	  }
	}
	else
	  error_1 (matrix_GetName (mb),
		   "cannot add STRING and NUMERIC matrices");
	break;
      default:
	error_2 (matrix_GetName (ma), matrix_GetName (mb),
		 "type error adding A and B");
	break;
      }
    }
    else if ((MNR (ma) * MNC (ma)) == 1)
    {
      /* [A] is a 1-by-1 */
      Scalar *s;
      String *str;
      if (MTYPE (ma) == REAL)
      {
	s = scalar_Create (MAT (ma, 1, 1));
	new = matrix_scalar_add (mb, s);
	scalar_Destroy (s);
      }
      else if (MTYPE (ma) == COMPLEX)
      {
	s = scalar_CreateC (MATr (ma, 1, 1), MATi (ma, 1, 1));
	new = matrix_scalar_add (mb, s);
	scalar_Destroy (s);
      }
      else if (MTYPE (ma) == STRING)
      {
	str = string_Create (MATs (ma, 1, 1));
	new = matrix_string_add (mb, str, 2);
	string_Destroy (str);
      }
    }
    else if ((MNR (mb) * MNC (mb)) == 1)
    {
      /* [B] is a 1-by-1 */
      Scalar *s;
      String *str;
      if (MTYPE (mb) == REAL)
      {
	s = scalar_Create (MAT (mb, 1, 1));
	new = matrix_scalar_add (ma, s);
	scalar_Destroy (s);
      }
      else if (MTYPE (mb) == COMPLEX)
      {
	s = scalar_CreateC (MATr (mb, 1, 1), MATi (mb, 1, 1));
	new = matrix_scalar_add (ma, s);
	scalar_Destroy (s);
      }
      else if (MTYPE (mb) == STRING)
      {
	str = string_Create (MATs (mb, 1, 1));
	new = matrix_string_add (ma, str, 1);
	string_Destroy (str);
      }
    }
    else if (MNR (ma) == 0 || MNR (mb) == 0)
    {
      new = matrix_Create (0, 0);
    }
    /*
     * Handle special row/column conditions...
     */
    else if (MNR (ma) == MNR (mb) && MNC (ma) == 1)
    {
      new = matrix_VectorAddCol (mb, ma);
    }
    else if (MNR (mb) == MNR (ma) && MNC (mb) == 1)
    {
      new = matrix_VectorAddCol (ma, mb);
    }
    else if (MNC (ma) == MNC (mb) && MNR (ma) == 1)
    {
      new = matrix_VectorAddRow (mb, ma);
    }
    else if (MNC (mb) == MNC (ma) && MNR (mb) == 1)
    {
      new = matrix_VectorAddRow (ma, mb);
    }
    else
    {
      error_2 (matrix_GetName (ma), matrix_GetName (mb),
	       "Dim must match for matrix el-add");
    }
    return (new);
  }
}

Matrix *
matrix_El_Sub (ma, mb)
     Matrix *ma, *mb;
{
  ASSERT (ma);
  ASSERT (mb);
  {
    register int i;
    Matrix *new = 0;

    /* check dimensions */
    if (MNR (ma) == MNR (mb) && MNC (ma) == MNC (mb))
    {
      switch (MTYPE (ma))
      {
      case REAL:
	switch (MTYPE (mb))
	{
	case REAL:		/* r - r */
	  new = matrix_Create (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	    new->val.mr[i] = ma->val.mr[i] - mb->val.mr[i];
	  break;
	case COMPLEX:		/* r - c */
	  new = matrix_CreateC (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	  {
	    new->val.mc[i].r = ma->val.mr[i] - mb->val.mc[i].r;
	    new->val.mc[i].i = -mb->val.mc[i].i;
	  }
	  break;
	case STRING:
	  error_2 (matrix_GetName (ma), matrix_GetName (mb),
		   "cannot perform add-op with STRING MATRIX");
	  break;
	}
	break;
      case COMPLEX:
	switch (MTYPE (mb))
	{
	case REAL:		/* c - r */
	  new = matrix_CreateC (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	  {
	    new->val.mc[i].r = ma->val.mc[i].r - mb->val.mr[i];
	    new->val.mc[i].i = ma->val.mc[i].i;
	  }
	  break;
	case COMPLEX:		/* c - c */
	  new = matrix_CreateC (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	  {
	    new->val.mc[i].r = ma->val.mc[i].r - mb->val.mc[i].r;
	    new->val.mc[i].i = ma->val.mc[i].i - mb->val.mc[i].i;
	  }
	  break;
	case STRING:
	  error_2 (matrix_GetName (ma), matrix_GetName (mb),
		   "cannot perform add-op with STRING MATRIX");
	  break;
	}
	break;
      case STRING:
	error_2 (matrix_GetName (ma), matrix_GetName (mb),
		 "cannot perform add-op with STRING MATRIX");
	break;
      }
    }
    else if ((MNR (ma) * MNC (ma)) == 1)
    {
      Scalar *s;
      if (MTYPE (ma) == REAL)
	s = scalar_Create (MAT (ma, 1, 1));
      else
	s = scalar_CreateC (MATr (ma, 1, 1), MATi (ma, 1, 1));
      new = matrix_scalar_sub2 (mb, s);
      scalar_Destroy (s);
    }
    else if ((MNR (mb) * MNC (mb)) == 1)
    {
      Scalar *s;
      if (MTYPE (mb) == REAL)
	s = scalar_Create (MAT (mb, 1, 1));
      else
	s = scalar_CreateC (MATr (mb, 1, 1), MATi (mb, 1, 1));
      new = matrix_scalar_sub1 (ma, s);
      scalar_Destroy (s);
    }
    else if (MNR (ma) == 0 || MNR (mb) == 0)
    {
      new = matrix_Create (0, 0);
    }
    /*
     * Handle special row/column conditions...
     */
    else if (MNR (ma) == MNR (mb) && MNC (ma) == 1)
    {
      new = matrix_VectorSubCol2 (mb, ma);
    }
    else if (MNR (mb) == MNR (ma) && MNC (mb) == 1)
    {
      new = matrix_VectorSubCol1 (ma, mb);
    }
    else if (MNC (ma) == MNC (mb) && MNR (ma) == 1)
    {
      new = matrix_VectorSubRow2 (mb, ma);
    }
    else if (MNC (mb) == MNC (ma) && MNR (mb) == 1)
    {
      new = matrix_VectorSubRow1 (ma, mb);
    }
    else
    {
      error_2 (matrix_GetName (ma), matrix_GetName (mb),
	       "Dim. must match for matrix el-subtract");
    }
    return (new);
  }
}

/* **************************************************************
 * Perform [mc] = [ma][mb], [mc] is allocated and returned.
 * [ma] is MxN, [mb] is NxK, [mc] is MxK.
 *
 * For the Real-Real, and Complex-Complex multiplies, there are two
 * options. Either use the BLAS subroutines ( calls are commented
 * out), or use the C code (default). The C-code should be faster
 * unless you have a specially optimized set of BLAS subroutine.
 * ************************************************************** */

Matrix *
matrix_Multiply (ma, mb)
     Matrix *ma, *mb;
{
  Matrix *new = 0;

  switch (MTYPE (ma))
  {
  case REAL:
    switch (MTYPE (mb))
    {
    case REAL:			/* r * r */
      new = matrix_MultiplyRR (ma, mb);
      break;
    case COMPLEX:		/* r * c */
      new = matrix_MultiplyRC (ma, mb);
      break;
    case STRING:
      error_2 (matrix_GetName (ma), matrix_GetName (mb),
	       "cannot perform multiply-op with STRING MATRIX");
      break;
    }
    break;
  case COMPLEX:
    switch (MTYPE (mb))
    {
    case REAL:			/* c * r */
      new = matrix_MultiplyCR (ma, mb);
      break;
    case COMPLEX:		/* c * c */
      new = matrix_MultiplyCC (ma, mb);
      break;
    case STRING:
      error_2 (matrix_GetName (ma), matrix_GetName (mb),
	       "cannot perform multiply-op with STRING MATRIX");
      break;
    }
    break;
  case STRING:
    error_2 (matrix_GetName (ma), matrix_GetName (mb),
	     "cannot perform multiply-op with STRING MATRIX");
    break;
  }
  return (new);
}

extern void rmmpy _PROTO ((int m, int k, int n, double *A,
			   double *B, double *C));
extern void cmmpy _PROTO ((int m, int k, int n, Complex * A,
			   Complex * B, Complex * C));

extern void rcmmpy _PROTO ((int m, int k, int n, double *A,
			    Complex * B, Complex * C));

extern void crmmpy _PROTO ((int m, int k, int n, Complex * A,
			    double *B, Complex * C));

extern void rmmpyr _PROTO ((int M, int m, int k, int n, double *A,
			    double *B, double *C));
extern void cmmpyr _PROTO ((int M, int m, int k, int n, Complex * A,
			    Complex * B, Complex * C));

Matrix *
matrix_MultiplyRR (ma, mb)
     Matrix *ma, *mb;
{
  ASSERT (ma);
  ASSERT (mb);
  {
    static double alpha = 1.0;
    static double beta = 0.0;
    F_INT tra, trb;
    Matrix *mc = 0;


    /* Check [ma], [mb] dimensions */
    if (MNC (ma) != MNR (mb))
    {
      /*
       * Handle condition where one of the operands is a
       * scalar, or an empty matrix.
       */

      if (MNR (ma) == 1 && MNC (ma) == 1)
      {
	Scalar *stmp = scalar_Create (MAT (ma, 1, 1));
	mc = matrix_scalar_mul (mb, stmp);
	scalar_Destroy (stmp);
      }
      else if (MNR (mb) == 1 && MNC (mb) == 1)
      {
	Scalar *stmp = scalar_Create (MAT (mb, 1, 1));
	mc = matrix_scalar_mul (ma, stmp);
	scalar_Destroy (stmp);
      }
      else if (MNR (ma) == 0 || MNR (mb) == 0)
      {
	mc = matrix_Create (0, 0);
      }
      else
      {
	error_2 (ma->name, mb->name,
		 "dimensions must match for matrix multiply");
      }
    }
    else
    {
      F_INT m, n, k;
      mc = matrix_Create (MNR (ma), MNC (mb));
      tra = (F_INT) 'N';
      trb = (F_INT) 'N';
      m = (F_INT) MNR (ma);
      n = (F_INT) MNC (mb);
      k = (F_INT) MNC (ma);

      /* BLAS option: use `double alpha = 1.0, beta = 0.0' */
      RGEMM (&tra, &trb, &m, &n, &k, &alpha, MDPTRr (ma),
	     &m, MDPTRr (mb), &k, &beta, MDPTRr (mc), &m);

      /*
       * rmmpy (MNR (ma), MNC (ma), MNC (mb), MDPTRr (ma),
       *        MDPTRr (mb), MDPTRr (mc));
       */
    }
    return (mc);
  }
}

/*
 * [ma] is REAL, and [mb] is COMPLEX
 */

Matrix *
matrix_MultiplyRC (ma, mb)
     Matrix *ma, *mb;
{
  ASSERT (ma);
  ASSERT (mb);
  {
    Matrix *mc = 0;

    /* Check [ma], [mb] dimensions */
    if (MNC (ma) != MNR (mb))
    {
      /*
       * Handle condition where one of the operands is a
       * scalar, or an empty matrix.
       */

      if (MNR (ma) == 1 && MNC (ma) == 1)
      {
	Scalar *stmp = scalar_Create (MAT (ma, 1, 1));
	mc = matrix_scalar_mul (mb, stmp);
	scalar_Destroy (stmp);
      }
      else if (MNR (mb) == 1 && MNC (mb) == 1)
      {
	Scalar *stmp = scalar_CreateC (MATr (mb, 1, 1), MATi (mb, 1, 1));
	mc = matrix_scalar_mul (ma, stmp);
	scalar_Destroy (stmp);
      }
      else if (MNR (ma) == 0 || MNR (mb) == 0)
      {
	mc = matrix_Create (0, 0);
      }
      else
      {
	error_2 (ma->name, mb->name,
		 "dimensions must match for matrix multiply");
      }
    }
    else
    {
      mc = matrix_CreateC (MNR (ma), MNC (mb));
      rcmmpy (MNR (ma), MNC (ma), MNC (mb), MDPTRr (ma),
	      MDPTRc (mb), MDPTRc (mc));
    }
    return (mc);
  }
}

/*
 * [ma] is COMPLEX, and [mb] is REAL
 */

Matrix *
matrix_MultiplyCR (ma, mb)
     Matrix *ma, *mb;
{
  ASSERT (ma);
  ASSERT (mb);
  {
    Matrix *mc = 0;

    /* Check [ma], [mb] dimensions */
    if (MNC (ma) != MNR (mb))
    {
      /*
       * Handle condition where one of the operands is a
       * scalar, or an empty matrix.
       */

      if (MNR (ma) == 1 && MNC (ma) == 1)
      {
	Scalar *stmp = scalar_CreateC (MATr (ma, 1, 1), MATi (ma, 1, 1));
	mc = matrix_scalar_mul (mb, stmp);
	scalar_Destroy (stmp);
      }
      else if (MNR (mb) == 1 && MNC (mb) == 1)
      {
	Scalar *stmp = scalar_Create (MAT (mb, 1, 1));
	mc = matrix_scalar_mul (ma, stmp);
	scalar_Destroy (stmp);
      }
      else if (MNR (ma) == 0 || MNR (mb) == 0)
      {
	mc = matrix_Create (0, 0);
      }
      else
      {
	error_2 (ma->name, mb->name,
		 "dimensions must match for matrix multiply");
      }
    }
    else
    {
      mc = matrix_CreateC (MNR (ma), MNC (mb));
      crmmpy (MNR (ma), MNC (ma), MNC (mb), MDPTRc (ma),
	      MDPTRr (mb), MDPTRc (mc));
    }
    return (mc);
  }
}

/*
 * [ma] is COMPLEX, and [mb] is COMPLEX
 */

static Complex zalpha =
{1.0, 0.0};
static Complex zbeta =
{0.0, 0.0};

Matrix *
matrix_MultiplyCC (ma, mb)
     Matrix *ma, *mb;
{
  ASSERT (ma);
  ASSERT (mb);
  {
    F_INT tra, trb;
    Matrix *mc = 0;

    /* Check [ma], [mb] dimensions */
    if (MNC (ma) != MNR (mb))
    {
      /*
       * Handle condition where one of the operands is a
       * scalar, or an empty matrix.
       */

      if (MNR (ma) == 1 && MNC (ma) == 1)
      {
	Scalar *stmp = scalar_CreateC (MATr (ma, 1, 1), MATi (ma, 1, 1));
	mc = matrix_scalar_mul (mb, stmp);
	scalar_Destroy (stmp);
      }
      else if (MNR (mb) == 1 && MNC (mb) == 1)
      {
	Scalar *stmp = scalar_CreateC (MATr (mb, 1, 1), MATi (mb, 1, 1));
	mc = matrix_scalar_mul (ma, stmp);
	scalar_Destroy (stmp);
      }
      else if (MNR (ma) == 0 || MNR (mb) == 0)
      {
	mc = matrix_Create (0, 0);
      }
      else
      {
	error_2 (ma->name, mb->name,
		 "dimensions must match for matrix multiply");
      }
    }
    else
    {
      F_INT m, n, k;
      mc = matrix_CreateC (MNR (ma), MNC (mb));
      tra = (F_INT) 'N';
      trb = (F_INT) 'N';
      m = (F_INT) MNR (ma);
      n = (F_INT) MNC (mb);
      k = (F_INT) MNC (ma);

      /* BLAS option: Complex alpha = (1.0,0.0), beta = (0.0,0.0) */
      XGEMM (&tra, &trb, &m, &n, &k, &zalpha, MDPTRc (ma),
	     &m, MDPTRc (mb), &k, &zbeta, MDPTRc (mc), &m);

      /*
       * cmmpy (MNR (ma), MNC (ma), MNC (mb), MDPTRc (ma),
       *        MDPTRc (mb), MDPTRc (mc));
       */
    }
    return (mc);
  }
}

/* **************************************************************
 * Perform [mc] = [ma][mb], [mc] is allocated and returned.
 * [ma] is NxN, [mb] is NxN, [mc] is NxN.
 * THIS IS AN ELEMENT-BY-ELEMENT MULTIPLY.
 * ************************************************************** */
Matrix *
matrix_El_Mul (ma, mb)
     Matrix *ma, *mb;
{
  ASSERT (ma);
  ASSERT (mb);
  {
    register int i;
    Matrix *mc = 0;

    matrix_screen_string (ma);
    matrix_screen_string (mb);

    /* Check [ma], [mb] dimensions */
    if ((MNC (ma) == MNC (mb)) && (MNR (ma) == MNR (mb)))
    {
      switch (MTYPE (ma))
      {
      case REAL:
	switch (MTYPE (mb))
	{
	case REAL:		/* r * r */
	  mc = matrix_Create (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	    mc->val.mr[i] = ma->val.mr[i] * mb->val.mr[i];
	  break;
	case COMPLEX:		/* r * c */
	  mc = matrix_CreateC (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	    mc->val.mc[i] = complex_Multiply (ma->val.mr[i], 0.0,
					      mb->val.mc[i].r,
					      mb->val.mc[i].i);
	  break;
	}
	break;
      case COMPLEX:
	switch (MTYPE (mb))
	{
	case REAL:		/* c * r */
	  mc = matrix_CreateC (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	    mc->val.mc[i] = complex_Multiply (ma->val.mc[i].r,
					      ma->val.mc[i].i,
					      mb->val.mr[i], 0.0);
	  break;
	case COMPLEX:		/* c * c */
	  mc = matrix_CreateC (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	    mc->val.mc[i] = complex_Multiply (ma->val.mc[i].r,
					      ma->val.mc[i].i,
					      mb->val.mc[i].r,
					      mb->val.mc[i].i);
	  break;
	}
	break;
      }
    }
    else if ((MNR (ma) * MNC (ma)) == 1)
    {
      Scalar *s;
      if (MTYPE (ma) == REAL)
	s = scalar_Create (MAT (ma, 1, 1));
      else
	s = scalar_CreateC (MATr (ma, 1, 1), MATi (ma, 1, 1));
      mc = matrix_scalar_mul (mb, s);
      scalar_Destroy (s);
    }
    else if ((MNR (mb) * MNC (mb)) == 1)
    {
      Scalar *s;
      if (MTYPE (mb) == REAL)
	s = scalar_Create (MAT (mb, 1, 1));
      else
	s = scalar_CreateC (MATr (mb, 1, 1), MATi (mb, 1, 1));
      mc = matrix_scalar_mul (ma, s);
      scalar_Destroy (s);
    }
    else if (MNR (ma) == 0 || MNR (mb) == 0)
    {
      mc = matrix_Create (0, 0);
    }
    /*
     * Handle special row/column conditions...
     */
    else if (MNR (ma) == MNR (mb) && MNC (ma) == 1)
    {
      mc = matrix_VectorMulCol (mb, ma);
    }
    else if (MNR (mb) == MNR (ma) && MNC (mb) == 1)
    {
      mc = matrix_VectorMulCol (ma, mb);
    }
    else if (MNC (ma) == MNC (mb) && MNR (ma) == 1)
    {
      mc = matrix_VectorMulRow (mb, ma);
    }
    else if (MNC (mb) == MNC (ma) && MNR (mb) == 1)
    {
      mc = matrix_VectorMulRow (ma, mb);
    }
    else
    {
      error_2 (ma->name, mb->name,
	       "dimensions must match for element matrix multiply");
    }
    return (mc);
  }
}

/*
 * Element-by-element division of two matrices
 */
Matrix *
matrix_El_Div (ma, mb)
     Matrix *ma, *mb;
{
  ASSERT ((ma));
  ASSERT ((mb));
  {
    register int i;
    Matrix *mc = 0;

    matrix_screen_string (ma);
    matrix_screen_string (mb);

    /* Check [ma], [mb] dimensions */
    if ((MNC (ma) == MNC (mb)) && (MNR (ma) == MNR (mb)))
    {
      switch (MTYPE (ma))
      {
      case REAL:
	switch (MTYPE (mb))
	{
	case REAL:		/* r / r */
	  mc = matrix_Create (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	    mc->val.mr[i] = ma->val.mr[i] / mb->val.mr[i];
	  break;
	case COMPLEX:		/* r / c */
	  mc = matrix_CreateC (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	    mc->val.mc[i] = complex_div (ma->val.mr[i], 0.0,
					 mb->val.mc[i].r,
					 mb->val.mc[i].i);
	  break;
	}
	break;
      case COMPLEX:
	switch (MTYPE (mb))
	{
	case REAL:		/* c / r */
	  mc = matrix_CreateC (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	    mc->val.mc[i] = complex_div (ma->val.mc[i].r,
					 ma->val.mc[i].i,
					 mb->val.mr[i], 0.0);
	  break;
	case COMPLEX:		/* c / c */
	  mc = matrix_CreateC (MNR (ma), MNC (ma));
	  for (i = 0; i < MNR (ma) * MNC (ma); i++)
	    mc->val.mc[i] = complex_div (ma->val.mc[i].r,
					 ma->val.mc[i].i,
					 mb->val.mc[i].r,
					 mb->val.mc[i].i);
	  break;
	}
	break;
      }
    }
    else if ((MNR (ma) * MNC (ma)) == 1)
    {
      Scalar *s;
      if (MTYPE (ma) == REAL)
	s = scalar_Create (MAT (ma, 1, 1));
      else
	s = scalar_CreateC (MATr (ma, 1, 1), MATi (ma, 1, 1));
      mc = matrix_scalar_div2 (mb, s);
      scalar_Destroy (s);
    }
    else if ((MNR (mb) * MNC (mb)) == 1)
    {
      Scalar *s;
      if (MTYPE (mb) == REAL)
	s = scalar_Create (MAT (mb, 1, 1));
      else
	s = scalar_CreateC (MATr (mb, 1, 1), MATi (mb, 1, 1));
      mc = matrix_scalar_div1 (ma, s);
      scalar_Destroy (s);
    }
    else if (MNR (ma) == 0 || MNR (mb) == 0)
    {
      mc = matrix_Create (0, 0);
    }
    /*
     * Handle special row/column conditions...
     */
    else if (MNR (ma) == MNR (mb) && MNC (ma) == 1)
    {
      mc = matrix_VectorDivCol2 (mb, ma);
    }
    else if (MNR (mb) == MNR (ma) && MNC (mb) == 1)
    {
      mc = matrix_VectorDivCol1 (ma, mb);
    }
    else if (MNC (ma) == MNC (mb) && MNR (ma) == 1)
    {
      mc = matrix_VectorDivRow2 (mb, ma);
    }
    else if (MNC (mb) == MNC (ma) && MNR (mb) == 1)
    {
      mc = matrix_VectorDivRow1 (ma, mb);
    }
    else
    {
      error_2 (ma->name, mb->name,
	       "dimensions must match for element matrix divide");
    }
    return (mc);
  }
}

Matrix *
matrix_Negate (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i;
    Matrix *new = 0;

    if (MTYPE (m) == REAL)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = -m->val.mr[i];
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	new->val.mc[i].r = -m->val.mc[i].r;
	new->val.mc[i].i = -m->val.mc[i].i;
      }
    }
    else if (MTYPE (m) == STRING)
      error_1 (matrix_GetName (m), "cannot negate a STRING MATRIX");

    return (new);
  }
}

Matrix *
matrix_Abs (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i;
    Matrix *new;

    new = matrix_Create (MNR (m), MNC (m));
    if (MTYPE (m) == REAL)
    {
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = rabs (m->val.mr[i]);
    }
    else if (MTYPE (m) == COMPLEX)
    {
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = complex_Abs (m->val.mc[i]);
    }
    else
      matrix_screen_string (m);

    return (new);
  }
}

Matrix *
matrix_Log (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i;
    int dc = 0;
    Matrix *new = 0;

    if (MTYPE (m) == REAL && !(dc = do_cmplx (m)))
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = errcheck (log (m->val.mr[i]), "log");
    }
    else if (MTYPE (m) == REAL && dc)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mc[i] = complex_log (MATrv (m, i), 0.0);
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mc[i] = complex_Log (m->val.mc[i]);
    }
    else
      matrix_screen_string (m);

    return (new);
  }
}

#define log10e 0.43429448190325182765

Matrix *
matrix_Log10 (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i;
    int dc = 0;
    Complex ctmp;
    Matrix *new = 0;

    if (MTYPE (m) == REAL && !(dc = do_cmplx (m)))
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = errcheck (log10 (m->val.mr[i]), "log10");
    }
    else if (MTYPE (m) == REAL && dc)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	ctmp = complex_log (MATrv (m, i), 0.0);
	MATcv (new, i) = complex_Multiply (log10e, 0.0, ctmp.r, ctmp.i);
      }
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	ctmp = complex_Log (m->val.mc[i]);
	MATcv (new, i) = complex_Multiply (log10e, 0.0, ctmp.r, ctmp.i);
      }
    }
    else
      matrix_screen_string (m);

    return (new);
  }
}

Matrix *
matrix_Exp (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i;
    Matrix *new = 0;

    if (MTYPE (m) == REAL)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = errcheck (exp (m->val.mr[i]), "exp");
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mc[i] = complex_Exp (m->val.mc[i]);
    }
    else
      matrix_screen_string (m);

    return (new);
  }
}

Matrix *
matrix_Sqrt (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i;
    int dc = 0;
    Matrix *new = 0;

    if (MTYPE (m) == REAL && !(dc = do_cmplx (m)))
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = errcheck (sqrt (m->val.mr[i]), "sqrt");
    }
    else if (MTYPE (m) == REAL && dc)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mc[i] = complex_sqrt (MATrv (m, i), 0.0);
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mc[i] = complex_Sqrt (m->val.mc[i]);
    }
    else
      matrix_screen_string (m);

    return (new);
  }
}

/* **************************************************************
 * Create a new matrix by running the input through an int filter.
 * ************************************************************** */
Matrix *
matrix_Int (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i;
    Matrix *new = 0;

    if (MTYPE (m) == REAL)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = (double) ((int) m->val.mr[i]);
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	new->val.mc[i].r = (double) ((int) m->val.mc[i].r);
	new->val.mc[i].i = (double) ((int) m->val.mc[i].i);
      }
    }
    else
      matrix_screen_string (m);

    return (new);
  }
}

/* **************************************************************
 * Compute the transpose of a matrix, return a pointer to a new
 * matrix which contains the results.
 * ************************************************************** */
Matrix *
matrix_Transpose (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i, j;
    Matrix *new = 0;

    if (MTYPE (m) == REAL)
    {
      new = matrix_Create (MNC (m), MNR (m));
      for (i = 1; i <= MNR (m); i++)
	for (j = 1; j <= MNC (m); j++)
	  MAT (new, j, i) = MAT (m, i, j);
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (MNC (m), MNR (m));
      for (i = 1; i <= MNR (m); i++)
	for (j = 1; j <= MNC (m); j++)
	{
	  MATr (new, j, i) = MATr (m, i, j);
	  MATi (new, j, i) = -MATi (m, i, j);
	}
    }
    else if (MTYPE (m) == STRING)
    {
      new = matrix_CreateS (MNC (m), MNR (m));
      for (i = 1; i <= MNR (m); i++)
	for (j = 1; j <= MNC (m); j++)
	  MATs (new, j, i) = cpstr (MATs (m, i, j));
    }
    return (new);
  }
}

/*
 * Non-conjugate matrix transpose
 */

Matrix *
matrix_ElTranspose (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i, j;
    Matrix *new = 0;

    if (MTYPE (m) == REAL)
    {
      new = matrix_Create (MNC (m), MNR (m));
      for (i = 1; i <= MNR (m); i++)
	for (j = 1; j <= MNC (m); j++)
	  MAT (new, j, i) = MAT (m, i, j);
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (MNC (m), MNR (m));
      for (i = 1; i <= MNR (m); i++)
	for (j = 1; j <= MNC (m); j++)
	{
	  MATr (new, j, i) = MATr (m, i, j);
	  MATi (new, j, i) = MATi (m, i, j);
	}
    }
    else if (MTYPE (m) == STRING)
    {
      new = matrix_CreateS (MNC (m), MNR (m));
      for (i = 1; i <= MNR (m); i++)
	for (j = 1; j <= MNC (m); j++)
	  MATs (new, j, i) = cpstr (MATs (m, i, j));
    }
    return (new);
  }
}

/* **************************************************************
 * Create nrow by ncol matrix filled with random numbers.
 * ************************************************************** */

extern double rrand ();

Matrix *
matrix_Rand (nrow, ncol)
     int nrow, ncol;
{
  register int i;
  Matrix *new;

  new = matrix_Create (nrow, ncol);
  for (i = 0; i < nrow * ncol; i++)
    new->val.mr[i] = rrand ();

  return (new);
}

Matrix *
matrix_RandC (nrow, ncol)
     int nrow, ncol;
{
  register int i;
  Matrix *new;

  new = matrix_CreateC (nrow, ncol);
  for (i = 0; i < nrow * ncol; i++)
  {
    new->val.mc[i].r = (double) rrand ();
    new->val.mc[i].i = (double) rrand ();
  }
  return (new);
}

/*
 * Matrix_Any:
 * Return TRUE if ANY of the elements are non-zero
 */

Matrix *
matrix_Any (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i, j;
    Matrix *new;

    matrix_screen_string (m);

    if (MTYPE (m) == REAL)
    {
      if (MNR (m) == 1)		/* Vector operation */
      {
	new = matrix_Create (1, 1);
	MAT (new, 1, 1) = 0.0;
	for (i = 1; i <= MNC (m); i++)
	  if (MAT (m, 1, i) != 0.0)
	  {
	    MAT (new, 1, 1) = 1.0;
	    break;
	  }
      }
      else
	/* Matrix operation */
      {
	new = matrix_Create (1, MNC (m));
	for (i = 1; i <= MNC (m); i++)
	{
	  MAT (new, 1, i) = 0.0;
	  for (j = 1; j <= MNR (m); j++)
	  {
	    if (MAT (m, j, i) != 0.0)
	    {
	      MAT (new, 1, i) = 1.0;
	      break;
	    }
	  }
	}
      }
    }
    else
    {
      if (MNR (m) == 1)
      {
	new = matrix_Create (1, 1);
	MAT (new, 1, 1) = 0.0;
	for (i = 1; i <= MNC (m); i++)
	  if (MATr (m, 1, i) != 0.0 || MATi (m, 1, i) != 0.0)
	  {
	    MAT (new, 1, 1) = 1.0;
	    break;
	  }
      }
      else
      {
	new = matrix_Create (1, MNC (m));
	for (i = 1; i <= MNC (m); i++)
	{
	  MAT (new, 1, i) = 0.0;
	  for (j = 1; j <= MNR (m); j++)
	  {
	    if (MATr (m, j, i) != 0.0 || MATi (m, j, i) != 0.0)
	    {
	      MAT (new, 1, i) = 1.0;
	      break;
	    }
	  }
	}
      }
    }
    return (new);
  }
}

/*
 * Matrix_All:
 * Return TRUE if ALL of the elements are non-zero
 */

Matrix *
matrix_All (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int i, j;
    Matrix *new;

    matrix_screen_string (m);

    if (MTYPE (m) == REAL)
    {
      if (MNR (m) == 1)		/* Vector operation */
      {
	new = matrix_Create (1, 1);
	MAT (new, 1, 1) = 1.0;
	for (i = 1; i <= MNC (m); i++)
	  if (MAT (m, 1, i) == 0.0)
	  {
	    MAT (new, 1, 1) = 0.0;
	    break;
	  }
      }
      else
	/* Matrix operation */
      {
	new = matrix_Create (1, MNC (m));
	for (i = 1; i <= MNC (m); i++)
	{
	  MAT (new, 1, i) = 1.0;
	  for (j = 1; j <= MNR (m); j++)
	  {
	    if (MAT (m, j, i) == 0.0)
	    {
	      MAT (new, 1, i) = 0.0;
	      break;
	    }
	  }
	}
      }
    }
    else
      /* COMPLEX */
    {
      if (MNR (m) == 1)
      {
	new = matrix_Create (1, 1);
	MAT (new, 1, 1) = 1.0;
	for (i = 1; i <= MNC (m); i++)
	  if (MATr (m, 1, i) == 0.0 && MATi (m, 1, i) == 0.0)
	  {
	    MAT (new, 1, 1) = 0.0;
	    break;
	  }
      }
      else
      {
	new = matrix_Create (1, MNC (m));
	for (i = 1; i <= MNC (m); i++)
	{
	  MAT (new, 1, i) = 1.0;
	  for (j = 1; j <= MNR (m); j++)
	  {
	    if (MATr (m, j, i) == 0.0 && MATi (m, j, i) == 0.0)
	    {
	      MAT (new, 1, i) = 0.0;
	      break;
	    }
	  }
	}
      }
    }
    return (new);
  }
}

/* **************************************************************
 * Add a matrix and a scalar.
 * ************************************************************** */
Matrix *
matrix_scalar_add (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new = 0;

    matrix_screen_string (m);

    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* Most common operation */
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = m->val.mr[i] + SVALr (s);
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	new->val.mc[i].r = m->val.mc[i].r + SVALr (s);
	new->val.mc[i].i = m->val.mc[i].i + SVALi (s);
      }
    }
    else if ((MTYPE (m) == REAL) && (SVALi (s) != 0.0))
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	new->val.mc[i].r = m->val.mr[i] + SVALr (s);
	new->val.mc[i].i = SVALi (s);
      }
    }
    else
      error_1 ("invalid data types for matrix-scalar add-op",
	       (char *) 0);
    return (new);
  }
}

/* **************************************************************
 * Subtract a matrix and a scalar.
 * ************************************************************** */
Matrix *
matrix_scalar_sub1 (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new = 0;

    matrix_screen_string (m);

    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* Most common operation */
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = m->val.mr[i] - SVALr (s);
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	new->val.mc[i].r = m->val.mc[i].r - SVALr (s);
	new->val.mc[i].i = m->val.mc[i].i - SVALi (s);
      }
    }
    else if ((MTYPE (m) == REAL) && (SVALi (s) != 0.0))
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	new->val.mc[i].r = m->val.mr[i] - SVALr (s);
	new->val.mc[i].i = -SVALi (s);
      }
    }
    else
      error_1 (matrix_GetName (m),
	       "invalid data types for matrix-scalar subtract-op");
    return (new);
  }
}

Matrix *
matrix_scalar_sub2 (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new = 0;

    matrix_screen_string (m);

    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* Most common operation */
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = SVALr (s) - m->val.mr[i];
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	new->val.mc[i].r = SVALr (s) - m->val.mc[i].r;
	new->val.mc[i].i = SVALi (s) - m->val.mc[i].i;
      }
    }
    else if ((MTYPE (m) == REAL) && (SVALi (s) != 0.0))
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	new->val.mc[i].r = SVALr (s) - m->val.mr[i];
	new->val.mc[i].i = SVALi (s);
      }
    }
    else
      error_1 (matrix_GetName (m),
	       "invalid data types for matrix-scalar subtract-op");
    return (new);
  }
}

/* **************************************************************
 * Multiply a Matrix by a scalar.
 * ************************************************************** */
Matrix *
matrix_scalar_mul (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new = 0;

    matrix_screen_string (m);

    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* Most common operation */
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = m->val.mr[i] * SVALr (s);
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	new->val.mc[i] = complex_Multiply (m->val.mc[i].r, m->val.mc[i].i,
					   SVALr (s), SVALi (s));
      }
    }
    else if ((MTYPE (m) == REAL) && (SVALi (s) != 0.0))
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	new->val.mc[i] = complex_Multiply (m->val.mr[i], 0.0,
					   SVALr (s), SVALi (s));
      }
    }
    else
      error_2 (matrix_GetName (m), scalar_GetName (s),
	       "invalid data types for matrix-scalar multiply-op");
    return (new);
  }
}

/*
 * M / S
 */
Matrix *
matrix_scalar_div1 (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new = 0;

    matrix_screen_string (m);

    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* Most common operation */
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = m->val.mr[i] / SVALr (s);
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	new->val.mc[i] = complex_div (m->val.mc[i].r, m->val.mc[i].i,
				      SVALr (s), SVALi (s));
      }
    }
    else if ((MTYPE (m) == REAL) && (SVALi (s) != 0.0))
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	new->val.mc[i] = complex_div (m->val.mr[i], 0.0,
				      SVALr (s), SVALi (s));
      }
    }
    else
      error_2 (matrix_GetName (m), scalar_GetName (s),
	       "invalid data types for matrix-scalar divide-op");
    return (new);
  }
}

/*
 * S / M
 */
Matrix *
matrix_scalar_div2 (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new = 0;

    matrix_screen_string (m);

    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* Most common operation */
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = SVALr (s) / m->val.mr[i];
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	new->val.mc[i] = complex_div (SVALr (s), SVALi (s),
				      m->val.mc[i].r, m->val.mc[i].i);
      }
    }
    else if ((MTYPE (m) == REAL) && (SVALi (s) != 0.0))
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	new->val.mc[i] = complex_div (SVALr (s), SVALi (s),
				      m->val.mr[i], 0.0);
      }
    }
    else
      error_2 (matrix_GetName (m), scalar_GetName (s),
	       "invalid data types for scalar-matrix divide-op");
    return (new);
  }
}

/* **************************************************************
 * Perform an element-by element operation on a matrix. See the
 * comments for vector_ElOp().
 * ************************************************************** */
Matrix *
matrix_ElOp (m, fptr, fname)
     Matrix *m;
     double (*fptr) ();
     char *fname;
{
  ASSERT (m);
  {
    register int i;
    Matrix *new;

    if (MTYPE (m) == REAL)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = errcheck (fptr (m->val.mr[i]), fname);
    }
    else
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	new->val.mc[i].r = errcheck (fptr (m->val.mc[i].r), fname);
	new->val.mc[i].i = errcheck (fptr (m->val.mc[i].i), fname);
      }
    }
    return (new);
  }
}

/* **************************************************************
 * Below are the matrix-matrix, and matrix-scalar relational
 * and logical operations.
 * ************************************************************** */

/*
 * m1 == m2
 */

Matrix *
matrix_eq (m1, m2)
     Matrix *m1, *m2;
{
  ASSERT (m1);
  ASSERT (m2);
  {
    register int n;
    int nel;
    Matrix *new = 0;

    /* Special case, either m1, or m2 is a 1-by-1 */
    if (MNR (m1) == 1 && MNC (m1) == 1)
    {
      Scalar *stmp;
      String *strtmp;

      switch (MTYPE (m1))
      {
      case REAL:
	stmp = scalar_Create (MAT (m1, 1, 1));
	new = matrix_scalar_eq (m2, stmp);
	scalar_Destroy (stmp);
	break;
      case COMPLEX:
	stmp = scalar_CreateC (MATr (m1, 1, 1), MATi (m1, 1, 1));
	new = matrix_scalar_eq (m2, stmp);
	scalar_Destroy (stmp);
	break;
      case STRING:
	strtmp = string_Create (MATs (m1, 1, 1));
	new = matrix_string_eq (m2, strtmp);
	string_Destroy (strtmp);
	break;
      }
      return (new);
    }
    else if (MNR (m2) == 1 && MNC (m2) == 1)
    {
      Scalar *stmp;
      String *strtmp;

      switch (MTYPE (m2))
      {
      case REAL:
	stmp = scalar_Create (MAT (m2, 1, 1));
	new = matrix_scalar_eq (m1, stmp);
	scalar_Destroy (stmp);
	break;
      case COMPLEX:
	stmp = scalar_CreateC (MATr (m2, 1, 1), MATi (m2, 1, 1));
	new = matrix_scalar_eq (m1, stmp);
	scalar_Destroy (stmp);
	break;
      case STRING:
	strtmp = string_Create (MATs (m2, 1, 1));
	new = matrix_string_eq (m1, strtmp);
	string_Destroy (strtmp);
	break;
      }
      return (new);
    }

    /*
     * Back to regular programming.
     */

    if (MNR (m1) != MNR (m2) || MNC (m1) != MNC (m2))
      error_2 (matrix_GetName (m1), matrix_GetName (m2),
	       "dimensions must match for comparison");

    nel = MNR (m1) * MNC (m1);
    new = matrix_Create (MNR (m1), MNC (m1));
    if (MTYPE (m1) == REAL && MTYPE (m2) == REAL)
    {
      for (n = 0; n < nel; n++)
	new->val.mr[n] = (double) m1->val.mr[n] == m2->val.mr[n];
    }
    else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == COMPLEX)
    {
      for (n = 0; n < nel; n++)
	if ((m1->val.mc[n].r == m2->val.mc[n].r)
	    && (m1->val.mc[n].i == m2->val.mc[n].i))
	  new->val.mr[n] = 1.0;
	else
	  new->val.mr[n] = 0.0;
    }
    else if ((MTYPE (m1) == COMPLEX) && (MTYPE (m2) == REAL))
    {
      for (n = 0; n < nel; n++)
	if ((m1->val.mc[n].r == m2->val.mr[n]) && (m1->val.mc[n].i == 0.0))
	  new->val.mr[n] = 1.0;
	else
	  new->val.mr[n] = 0.0;
    }
    else if ((MTYPE (m1) == REAL) && (MTYPE (m2) == COMPLEX))
    {
      for (n = 0; n < nel; n++)
	if ((m1->val.mr[n] == m2->val.mc[n].r) && (m2->val.mc[n].i == 0.0))
	  new->val.mr[n] = 1.0;
	else
	  new->val.mr[n] = 0.0;
    }
    else if ((MTYPE (m1) == STRING) && (MTYPE (m2) == STRING))
    {
      for (n = 0; n < nel; n++)
	MATrv (new, n) =
	  (double) (strcmp (MATsv (m1, n), MATsv (m2, n)) == 0);
    }
    else
    {
      matrix_Zero (new);
    }
    return (new);
  }
}

/*
 * m1 != m2
 */
Matrix *
matrix_ne (m1, m2)
     Matrix *m1, *m2;
{
  ASSERT (m1);
  ASSERT (m2);
  {
    register int n;
    int nel;
    Matrix *new = 0;

    /* Special case, either m1, or m2 is a 1-by-1 */
    if (MNR (m1) == 1 && MNC (m1) == 1)
    {
      Scalar *stmp;
      String *strtmp;

      switch (MTYPE (m1))
      {
      case REAL:
	stmp = scalar_Create (MAT (m1, 1, 1));
	new = matrix_scalar_ne (m2, stmp);
	scalar_Destroy (stmp);
	break;
      case COMPLEX:
	stmp = scalar_CreateC (MATr (m1, 1, 1), MATi (m1, 1, 1));
	new = matrix_scalar_ne (m2, stmp);
	scalar_Destroy (stmp);
	break;
      case STRING:
	strtmp = string_Create (MATs (m1, 1, 1));
	new = matrix_string_ne (m2, strtmp);
	string_Destroy (strtmp);
	break;
      }
      return (new);
    }
    else if (MNR (m2) == 1 && MNC (m2) == 1)
    {
      Scalar *stmp;
      String *strtmp;

      switch (MTYPE (m2))
      {
      case REAL:
	stmp = scalar_Create (MAT (m2, 1, 1));
	new = matrix_scalar_ne (m1, stmp);
	scalar_Destroy (stmp);
	break;
      case COMPLEX:
	stmp = scalar_CreateC (MATr (m2, 1, 1), MATi (m2, 1, 1));
	new = matrix_scalar_ne (m1, stmp);
	scalar_Destroy (stmp);
	break;
      case STRING:
	strtmp = string_Create (MATs (m2, 1, 1));
	new = matrix_string_ne (m1, strtmp);
	string_Destroy (strtmp);
	break;
      }
      return (new);
    }

    /*
     * Back to regular programming.
     */

    if (MNR (m1) != MNR (m2) || MNC (m1) != MNC (m2))
      error_2 (matrix_GetName (m1), matrix_GetName (m2),
	       "dimensions must match for comparison");
    nel = MNR (m1) * MNC (m1);
    new = matrix_Create (MNR (m1), MNC (m1));
    if (MTYPE (m1) == REAL && MTYPE (m2) == REAL)
    {
      for (n = 0; n < nel; n++)
	new->val.mr[n] = (double) (m1->val.mr[n] != m2->val.mr[n]);
    }
    else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == COMPLEX)
    {
      for (n = 0; n < nel; n++)
	if ((m1->val.mc[n].r != m2->val.mc[n].r)
	    || (m1->val.mc[n].i != m2->val.mc[n].i))
	  new->val.mr[n] = 1.0;
	else
	  new->val.mr[n] = 0.0;
    }
    else if ((MTYPE (m1) == COMPLEX) && (MTYPE (m2) == REAL))
    {
      for (n = 0; n < nel; n++)
	if ((m1->val.mc[n].r != m2->val.mr[n]) || (m1->val.mc[n].i != 0.0))
	  new->val.mr[n] = 1.0;
	else
	  new->val.mr[n] = 0.0;
    }
    else if ((MTYPE (m1) == REAL) && (MTYPE (m2) == COMPLEX))
    {
      for (n = 0; n < nel; n++)
	if ((m1->val.mr[n] != m2->val.mc[n].r) || (m2->val.mc[n].i != 0.0))
	  new->val.mr[n] = 1.0;
	else
	  new->val.mr[n] = 0.0;
    }
    else if ((MTYPE (m1) == STRING) && (MTYPE (m2) == STRING))
    {
      for (n = 0; n < nel; n++)
	MATrv (new, n) =
	  (double) (strcmp (MATsv (m1, n), MATsv (m2, n)) != 0);
    }
    else
    {
      for (n = 0; n < nel; n++)
	MATrv (new, n) = 1.0;
    }
    return (new);
  }
}

/*
 * m1 <= m2
 */
Matrix *
matrix_le (m1, m2)
     Matrix *m1, *m2;
{
  ASSERT (m1);
  ASSERT (m2);
  {
    register int n;
    int nel;
    Matrix *new = 0;

    /* Special case, either m1, or m2 is a 1-by-1 */
    if (MNR (m1) == 1 && MNC (m1) == 1)
    {
      Scalar *stmp;
      String *strtmp;

      switch (MTYPE (m1))
      {
      case REAL:
	stmp = scalar_Create (MAT (m1, 1, 1));
	new = matrix_scalar_le1 (m2, stmp);
	scalar_Destroy (stmp);
	break;
      case COMPLEX:
	stmp = scalar_CreateC (MATr (m1, 1, 1), MATi (m1, 1, 1));
	new = matrix_scalar_le1 (m2, stmp);
	scalar_Destroy (stmp);
	break;
      case STRING:
	strtmp = string_Create (MATs (m1, 1, 1));
	new = matrix_string_le2 (m2, strtmp);
	string_Destroy (strtmp);
	break;
      }
      return (new);
    }
    else if (MNR (m2) == 1 && MNC (m2) == 1)
    {
      Scalar *stmp;
      String *strtmp;

      switch (MTYPE (m2))
      {
      case REAL:
	stmp = scalar_Create (MAT (m2, 1, 1));
	new = matrix_scalar_le2 (m1, stmp);
	scalar_Destroy (stmp);
	break;
      case COMPLEX:
	stmp = scalar_CreateC (MATr (m2, 1, 1), MATi (m2, 1, 1));
	new = matrix_scalar_le2 (m1, stmp);
	scalar_Destroy (stmp);
	break;
      case STRING:
	strtmp = string_Create (MATs (m2, 1, 1));
	new = matrix_string_le1 (m1, strtmp);
	string_Destroy (strtmp);
	break;
      }
      return (new);
    }

    /*
     * Back to regular programming.
     */

    if (MNR (m1) != MNR (m2) || MNC (m1) != MNC (m2))
      error_2 (matrix_GetName (m1), matrix_GetName (m2),
	       "dimensions must match for comparison");
    new = matrix_Create (MNR (m1), MNC (m1));
    nel = MNR (m1) * MNC (m1);

    if (MTYPE (m1) == REAL && MTYPE (m2) == REAL)
    {
      for (n = 0; n < nel; n++)
	new->val.mr[n] = (double) (m1->val.mr[n] <= m2->val.mr[n]);
    }
    else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == COMPLEX)
    {
      for (n = 0; n < nel; n++)
      {
	MATrv (new, n) = complex_le (MATcvr (m1, n), MATcvi (m1, n),
				     MATcvr (m2, n), MATcvi (m2, n));
      }
    }
    else if ((MTYPE (m1) == COMPLEX) && (MTYPE (m2) == REAL))
    {
      for (n = 0; n < nel; n++)
      {
	MATrv (new, n) = complex_le (MATcvr (m1, n), MATcvi (m1, n),
				     MATrv (m2, n), 0.0);
      }
    }
    else if ((MTYPE (m1) == REAL) && (MTYPE (m2) == COMPLEX))
    {
      for (n = 0; n < nel; n++)
      {
	MATrv (new, n) = complex_le (MATrv (m1, n), 0.0,
				     MATcvr (m2, n), MATcvi (m2, n));
      }
    }
    else if ((MTYPE (m1) == STRING) && (MTYPE (m2) == STRING))
    {
      for (n = 0; n < nel; n++)
	MATrv (new, n) =
	  (double) (strcmp (MATsv (m1, n), MATsv (m2, n)) <= 0);
    }
    else
    {
      matrix_Destroy (new);
      error_1 ("invalid data-types", (char *) 0);
    }
    return (new);
  }
}

/*
 * m1 >= m2
 */
Matrix *
matrix_ge (m1, m2)
     Matrix *m1, *m2;
{
  ASSERT (m1);
  ASSERT (m2);
  {
    register int n;
    int nel;
    Matrix *new = 0;

    /* Special case, either m1, or m2 is a 1-by-1 */
    if (MNR (m1) == 1 && MNC (m1) == 1)
    {
      Scalar *stmp;
      String *strtmp;

      switch (MTYPE (m1))
      {
      case REAL:
	stmp = scalar_Create (MAT (m1, 1, 1));
	new = matrix_scalar_ge1 (m2, stmp);
	scalar_Destroy (stmp);
	break;
      case COMPLEX:
	stmp = scalar_CreateC (MATr (m1, 1, 1), MATi (m1, 1, 1));
	new = matrix_scalar_ge1 (m2, stmp);
	scalar_Destroy (stmp);
	break;
      case STRING:
	strtmp = string_Create (MATs (m1, 1, 1));
	new = matrix_string_ge2 (m2, strtmp);
	string_Destroy (strtmp);
	break;
      }
      return (new);
    }
    else if (MNR (m2) == 1 && MNC (m2) == 1)
    {
      Scalar *stmp;
      String *strtmp;

      switch (MTYPE (m2))
      {
      case REAL:
	stmp = scalar_Create (MAT (m2, 1, 1));
	new = matrix_scalar_ge2 (m1, stmp);
	scalar_Destroy (stmp);
	break;
      case COMPLEX:
	stmp = scalar_CreateC (MATr (m2, 1, 1), MATi (m2, 1, 1));
	new = matrix_scalar_ge2 (m1, stmp);
	scalar_Destroy (stmp);
	break;
      case STRING:
	strtmp = string_Create (MATs (m2, 1, 1));
	new = matrix_string_ge1 (m1, strtmp);
	string_Destroy (strtmp);
	break;
      }
      return (new);
    }

    /*
     * Back to regular programming.
     */

    if (MNR (m1) != MNR (m2) || MNC (m1) != MNC (m2))
      error_2 (matrix_GetName (m1), matrix_GetName (m2),
	       "dimensions must match for comparison");
    new = matrix_Create (MNR (m1), MNC (m1));
    nel = MNR (m1) * MNC (m1);

    if (MTYPE (m1) == REAL && MTYPE (m2) == REAL)
    {
      for (n = 0; n < nel; n++)
	new->val.mr[n] = (double) (m1->val.mr[n] >= m2->val.mr[n]);
    }
    else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == COMPLEX)
    {
      for (n = 0; n < nel; n++)
      {
	MATrv (new, n) = complex_ge (MATcvr (m1, n), MATcvi (m1, n),
				     MATcvr (m2, n), MATcvi (m2, n));
      }
    }
    else if ((MTYPE (m1) == COMPLEX) && (MTYPE (m2) == REAL))
    {
      for (n = 0; n < nel; n++)
      {
	MATrv (new, n) = complex_ge (MATcvr (m1, n), MATcvi (m1, n),
				     MATrv (m2, n), 0.0);
      }
    }
    else if ((MTYPE (m1) == REAL) && (MTYPE (m2) == COMPLEX))
    {
      for (n = 0; n < nel; n++)
      {
	MATrv (new, n) = complex_ge (MATrv (m1, n), 0.0,
				     MATcvr (m2, n), MATcvi (m2, n));
      }
    }
    else if ((MTYPE (m1) == STRING) && (MTYPE (m2) == STRING))
    {
      for (n = 0; n < nel; n++)
	MATrv (new, n) =
	  (double) (strcmp (MATsv (m1, n), MATsv (m2, n)) >= 0);
    }
    else
    {
      matrix_Destroy (new);
      error_1 ("invalid data-types", (char *) 0);
    }
    return (new);
  }
}

/*
 * m1 < m2
 */
Matrix *
matrix_lt (m1, m2)
     Matrix *m1, *m2;
{
  ASSERT (m1);
  ASSERT (m2);
  {
    register int n;
    int nel;
    Matrix *new = 0;

    /* Special case, either m1, or m2 is a 1-by-1 */
    if (MNR (m1) == 1 && MNC (m1) == 1)
    {
      Scalar *stmp;
      String *strtmp;

      switch (MTYPE (m1))
      {
      case REAL:
	stmp = scalar_Create (MAT (m1, 1, 1));
	new = matrix_scalar_lt1 (m2, stmp);
	scalar_Destroy (stmp);
	break;
      case COMPLEX:
	stmp = scalar_CreateC (MATr (m1, 1, 1), MATi (m1, 1, 1));
	new = matrix_scalar_lt1 (m2, stmp);
	scalar_Destroy (stmp);
	break;
      case STRING:
	strtmp = string_Create (MATs (m1, 1, 1));
	new = matrix_string_lt2 (m2, strtmp);
	string_Destroy (strtmp);
	break;
      }
      return (new);
    }
    else if (MNR (m2) == 1 && MNC (m2) == 1)
    {
      Scalar *stmp;
      String *strtmp;

      switch (MTYPE (m2))
      {
      case REAL:
	stmp = scalar_Create (MAT (m2, 1, 1));
	new = matrix_scalar_lt2 (m1, stmp);
	scalar_Destroy (stmp);
	break;
      case COMPLEX:
	stmp = scalar_CreateC (MATr (m2, 1, 1), MATi (m2, 1, 1));
	new = matrix_scalar_lt2 (m1, stmp);
	scalar_Destroy (stmp);
	break;
      case STRING:
	strtmp = string_Create (MATs (m2, 1, 1));
	new = matrix_string_lt1 (m1, strtmp);
	string_Destroy (strtmp);
	break;
      }
      return (new);
    }

    /*
     * Back to regular programming.
     */

    if (MNR (m1) != MNR (m2) || MNC (m1) != MNC (m2))
      error_2 (matrix_GetName (m1), matrix_GetName (m2),
	       "dimensions must match for comparison");
    new = matrix_Create (MNR (m1), MNC (m1));
    nel = MNR (m1) * MNC (m1);

    if (MTYPE (m1) == REAL && MTYPE (m2) == REAL)
    {
      for (n = 0; n < nel; n++)
	new->val.mr[n] = (double) (m1->val.mr[n] < m2->val.mr[n]);
    }
    else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == COMPLEX)
    {
      for (n = 0; n < nel; n++)
      {
	MATrv (new, n) = complex_lt (MATcvr (m1, n), MATcvi (m1, n),
				     MATcvr (m2, n), MATcvi (m2, n));
      }
    }
    else if ((MTYPE (m1) == COMPLEX) && (MTYPE (m2) == REAL))
    {
      for (n = 0; n < nel; n++)
      {
	MATrv (new, n) = complex_lt (MATcvr (m1, n), MATcvi (m1, n),
				     MATrv (m2, n), 0.0);
      }
    }
    else if ((MTYPE (m1) == REAL) && (MTYPE (m2) == COMPLEX))
    {
      for (n = 0; n < nel; n++)
      {
	MATrv (new, n) = complex_lt (MATrv (m1, n), 0.0,
				     MATcvr (m2, n), MATcvi (m2, n));
      }
    }
    else if ((MTYPE (m1) == STRING) && (MTYPE (m2) == STRING))
    {
      for (n = 0; n < nel; n++)
	MATrv (new, n) =
	  (double) (strcmp (MATsv (m1, n), MATsv (m2, n)) < 0);
    }
    else
    {
      matrix_Destroy (new);
      error_1 ("invalid data-types", (char *) 0);
    }
    return (new);
  }
}

/*
 * m1 > m2
 */
Matrix *
matrix_gt (m1, m2)
     Matrix *m1, *m2;
{
  ASSERT (m1);
  ASSERT (m2);
  {
    register int n;
    int nel;
    Matrix *new = 0;

    /* Special case, either m1, or m2 is a 1-by-1 */
    if (MNR (m1) == 1 && MNC (m1) == 1)
    {
      Scalar *stmp;
      String *strtmp;

      switch (MTYPE (m1))
      {
      case REAL:
	stmp = scalar_Create (MAT (m1, 1, 1));
	new = matrix_scalar_gt1 (m2, stmp);
	scalar_Destroy (stmp);
	break;
      case COMPLEX:
	stmp = scalar_CreateC (MATr (m1, 1, 1), MATi (m1, 1, 1));
	new = matrix_scalar_gt1 (m2, stmp);
	scalar_Destroy (stmp);
	break;
      case STRING:
	strtmp = string_Create (MATs (m1, 1, 1));
	new = matrix_string_gt2 (m2, strtmp);
	string_Destroy (strtmp);
	break;
      }
      return (new);
    }
    else if (MNR (m2) == 1 && MNC (m2) == 1)
    {
      Scalar *stmp;
      String *strtmp;

      switch (MTYPE (m2))
      {
      case REAL:
	stmp = scalar_Create (MAT (m2, 1, 1));
	new = matrix_scalar_gt2 (m1, stmp);
	scalar_Destroy (stmp);
	break;
      case COMPLEX:
	stmp = scalar_CreateC (MATr (m2, 1, 1), MATi (m2, 1, 1));
	new = matrix_scalar_gt2 (m1, stmp);
	scalar_Destroy (stmp);
	break;
      case STRING:
	strtmp = string_Create (MATs (m2, 1, 1));
	new = matrix_string_gt1 (m1, strtmp);
	string_Destroy (strtmp);
	break;
      }
      return (new);
    }

    /*
     * Back to regular programming.
     */

    if (MNR (m1) != MNR (m2) || MNC (m1) != MNC (m2))
      error_2 (matrix_GetName (m1), matrix_GetName (m2),
	       "dimensions must match for comparison");
    new = matrix_Create (MNR (m1), MNC (m1));
    nel = MNR (m1) * MNC (m1);

    if (MTYPE (m1) == REAL && MTYPE (m2) == REAL)
    {
      for (n = 0; n < nel; n++)
	new->val.mr[n] = (double) (m1->val.mr[n] > m2->val.mr[n]);
    }
    else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == COMPLEX)
    {
      for (n = 0; n < nel; n++)
      {
	MATrv (new, n) = complex_gt (MATcvr (m1, n), MATcvi (m1, n),
				     MATcvr (m2, n), MATcvi (m2, n));
      }
    }
    else if ((MTYPE (m1) == COMPLEX) && (MTYPE (m2) == REAL))
    {
      for (n = 0; n < nel; n++)
      {
	MATrv (new, n) = complex_gt (MATcvr (m1, n), MATcvi (m1, n),
				     MATrv (m2, n), 0.0);
      }
    }
    else if ((MTYPE (m1) == REAL) && (MTYPE (m2) == COMPLEX))
    {
      for (n = 0; n < nel; n++)
      {
	MATrv (new, n) = complex_gt (MATrv (m1, n), 0.0,
				     MATcvr (m2, n), MATcvi (m2, n));
      }
    }
    else if ((MTYPE (m1) == STRING) && (MTYPE (m2) == STRING))
    {
      for (n = 0; n < nel; n++)
	MATrv (new, n) =
	  (double) (strcmp (MATsv (m1, n), MATsv (m2, n)) > 0);
    }
    else
    {
      matrix_Destroy (new);
      error_1 ("invalid data-types", (char *) 0);
    }
    return (new);
  }
}

/*
 * m1 && m2
 */
Matrix *
matrix_and (m1, m2)
     Matrix *m1, *m2;
{
  ASSERT (m1);
  ASSERT (m2);
  {
    register int n;
    int nel;
    Matrix *new;

    if (MNR (m1) != MNR (m2) || MNC (m1) != MNC (m2))
      error_2 (matrix_GetName (m1), matrix_GetName (m2),
	       "dimensions must match for comparison");
    new = matrix_Create (MNR (m1), MNC (m1));
    nel = MNR (m1) * MNC (m1);

    if (MTYPE (m1) == REAL && MTYPE (m2) == REAL)
    {
      for (n = 0; n < nel; n++)
	new->val.mr[n] = (double) (m1->val.mr[n] != 0.0
				   && m2->val.mr[n] != 0.0);
    }
    else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == COMPLEX)
    {
      for (n = 0; n < nel; n++)
      {
	MATrv (new, n) = complex_and (MATcvr (m1, n), MATcvi (m1, n),
				      MATcvr (m2, n), MATcvi (m2, n));
      }
    }
    else if ((MTYPE (m1) == COMPLEX) && (MTYPE (m2) == REAL))
    {
      for (n = 0; n < nel; n++)
      {
	MATrv (new, n) = complex_and (MATcvr (m1, n), MATcvi (m1, n),
				      MATrv (m2, n), 0.0);
      }
    }
    else if ((MTYPE (m1) == REAL) && (MTYPE (m2) == COMPLEX))
    {
      for (n = 0; n < nel; n++)
      {
	MATrv (new, n) = complex_and (MATrv (m1, n), 0.0,
				      MATcvr (m2, n), MATcvi (m2, n));
      }
    }
    else if ((MTYPE (m1) == STRING) && (MTYPE (m2) == STRING))
    {
      for (n = 0; n < nel; n++)
      {

	if (!strcmp (MATsv (m1, n), null_str) &&
	    !strcmp (MATsv (m2, n), null_str))
	  MATrv (new, n) = 1.0;
	else
	  MATrv (new, n) = 0.0;
      }
    }
    else
    {
      matrix_Destroy (new);
      error_1 ("invalid data-types", (char *) 0);
    }
    return (new);
  }
}

/*
 * m1 || m2
 */
Matrix *
matrix_or (m1, m2)
     Matrix *m1, *m2;
{
  ASSERT (m1);
  ASSERT (m2);
  {
    register int n;
    int nel;
    Matrix *new;

    if (MNR (m1) != MNR (m2) || MNC (m1) != MNC (m2))
      error_2 (matrix_GetName (m1), matrix_GetName (m2),
	       "dimensions must match for comparison");
    new = matrix_Create (MNR (m1), MNC (m1));
    nel = MNR (m1) * MNC (m1);

    if (MTYPE (m1) == REAL && MTYPE (m2) == REAL)
    {
      for (n = 0; n < nel; n++)
	new->val.mr[n] = (double) (m1->val.mr[n] != 0.0
				   || m2->val.mr[n] != 0.0);
    }
    else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == COMPLEX)
    {
      for (n = 0; n < nel; n++)
      {
	MATrv (new, n) = complex_or (MATcvr (m1, n), MATcvi (m1, n),
				     MATcvr (m2, n), MATcvi (m2, n));
      }
    }
    else if ((MTYPE (m1) == COMPLEX) && (MTYPE (m2) == REAL))
    {
      for (n = 0; n < nel; n++)
      {
	MATrv (new, n) = complex_or (MATcvr (m1, n), MATcvi (m1, n),
				     MATrv (m2, n), 0.0);
      }
    }
    else if ((MTYPE (m1) == REAL) && (MTYPE (m2) == COMPLEX))
    {
      for (n = 0; n < nel; n++)
      {
	MATrv (new, n) = complex_or (MATrv (m1, n), 0.0,
				     MATcvr (m2, n), MATcvi (m2, n));
      }
    }
    else if ((MTYPE (m1) == STRING) && (MTYPE (m2) == STRING))
    {
      for (n = 0; n < nel; n++)
      {

	if (!strcmp (MATsv (m1, n), null_str) ||
	    !strcmp (MATsv (m2, n), null_str))
	  MATrv (new, n) = 1.0;
	else
	  MATrv (new, n) = 0.0;
      }
    }
    else
    {
      matrix_Destroy (new);
      error_1 ("invalid data-types", (char *) 0);
    }
    return (new);
  }
}

/*
 * ! m
 */
Matrix *
matrix_not (m)
     Matrix *m;
{
  ASSERT (m);
  {
    register int n;
    int nel;
    Matrix *new;

    new = matrix_Create (MNR (m), MNC (m));
    nel = MNR (m) * MNC (m);

    if (MTYPE (m) == REAL)
    {
      for (n = 0; n < nel; n++)
	new->val.mr[n] = (double) (m->val.mr[n] == 0.0);
    }
    else if (MTYPE (m) == COMPLEX)
    {
      double r;
      for (n = 0; n < nel; n++)
      {
	r = sqrt (m->val.mc[n].r * m->val.mc[n].r
		  + m->val.mc[n].i * m->val.mc[n].i);
	new->val.mr[n] = (double) (r == 0.0);
      }
    }
    else if (MTYPE (m) == STRING)
    {
      for (n = 0; n < nel; n++)
      {
	if (strcmp (MATsv (m, n), null_str))
	  MATrv (new, n) = 0.0;
	else
	  MATrv (new, n) = 1.0;
      }
    }
    return (new);
  }
}

Matrix *
matrix_scalar_eq (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new;
    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* Most common operation */
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = (double) (MATrv (m, i) == SVALr (s));
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	if ((MATcvr (m, i) == SVALr (s)) && (MATcvi (m, i) == SVALi (s)))
	  MATrv (new, i) = 1.0;
	else
	  MATrv (new, i) = 0.0;
      }
    }
    else
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = 0.0;
    }
    return (new);
  }
}

Matrix *
matrix_scalar_ne (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new;
    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* Most common operation */
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = (double) (MATrv (m, i) != SVALr (s));
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	if ((MATcvr (m, i) != SVALr (s)) || (MATcvi (m, i) != SVALi (s)))
	  MATrv (new, i) = 1.0;
	else
	  MATrv (new, i) = 0.0;
      }
    }
    else
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = 1.0;
    }
    return (new);
  }
}

/*
 * SCALAR > MATRIX
 */
Matrix *
matrix_scalar_gt1 (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new = 0
     ;
    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* Most common operation */
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = (double) (SVALr (s) > MATrv (m, i));
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_gt (SVALr (s), SVALi (s),
				     MATcvr (m, i), MATcvi (m, i));
    }
    else if ((MTYPE (m) == REAL) && (SVALi (s) != 0.0))
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_gt (SVALr (s), SVALi (s),
				     MATrv (m, i), 0.0);
    }
    else
      error_1 ("invalid data types for matrix-scalar > op",
	       (char *) 0);
    return (new);
  }
}

/*
 * MATRIX > SCALAR
 */
Matrix *
matrix_scalar_gt2 (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new = 0;

    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* Most common operation */
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = (double) (MATrv (m, i) > SVALr (s));
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_gt (MATcvr (m, i), MATcvi (m, i),
				     SVALr (s), SVALi (s));

    }
    else if ((MTYPE (m) == REAL) && (SVALi (s) != 0.0))
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_gt (MATrv (m, i), 0.0,
				     SVALr (s), SVALi (s));

    }
    else
      error_1 ("invalid data types for matrix-scalar > op",
	       (char *) 0);
    return (new);
  }
}

/*
 * SCALAR < MATRIX
 */
Matrix *
matrix_scalar_lt1 (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new = 0;

    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* Most common operation */
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = (double) (SVALr (s) < MATrv (m, i));
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_lt (SVALr (s), SVALi (s),
				     MATcvr (m, i), MATcvi (m, i));
    }
    else if ((MTYPE (m) == REAL) && (SVALi (s) != 0.0))
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_lt (SVALr (s), SVALi (s),
				     MATrv (m, i), 0.0);
    }
    else
      error_1 ("invalid data types for matrix-scalar < op",
	       (char *) 0);
    return (new);
  }
}

/*
 * MATRIX < SCALAR
 */
Matrix *
matrix_scalar_lt2 (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new = 0;

    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* Most common operation */
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = (double) (MATrv (m, i) < SVALr (s));
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_lt (MATcvr (m, i), MATcvi (m, i),
				     SVALr (s), SVALi (s));

    }
    else if ((MTYPE (m) == REAL) && (SVALi (s) != 0.0))
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_lt (MATrv (m, i), 0.0,
				     SVALr (s), SVALi (s));

    }
    else
      error_1 ("invalid data types for matrix-scalar < op",
	       (char *) 0);
    return (new);
  }
}

/*
 * SCALAR >= MATRIX
 */
Matrix *
matrix_scalar_ge1 (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new = 0;

    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* Most common operation */
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = (double) (SVALr (s) >= MATrv (m, i));
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_ge (SVALr (s), SVALi (s),
				     MATcvr (m, i), MATcvi (m, i));
    }
    else if ((MTYPE (m) == REAL) && (SVALi (s) != 0.0))
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_ge (SVALr (s), SVALi (s),
				     MATrv (m, i), 0.0);
    }
    else
      error_1 ("invalid data types for matrix-scalar >= op",
	       (char *) 0);
    return (new);
  }
}

/*
 * MATRIX >= SCALAR
 */
Matrix *
matrix_scalar_ge2 (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new = 0;

    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* Most common operation */
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = (double) (MATrv (m, i) >= SVALr (s));
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_ge (MATcvr (m, i), MATcvi (m, i),
				     SVALr (s), SVALi (s));

    }
    else if ((MTYPE (m) == REAL) && (SVALi (s) != 0.0))
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_ge (MATrv (m, i), 0.0,
				     SVALr (s), SVALi (s));

    }
    else
      error_1 ("invalid data types for matrix-scalar >= op",
	       (char *) 0);
    return (new);
  }
}

/*
 * SCALAR <= MATRIX
 */
Matrix *
matrix_scalar_le1 (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new = 0;

    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* Most common operation */
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = (double) (SVALr (s) <= MATrv (m, i));
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_le (SVALr (s), SVALi (s),
				     MATcvr (m, i), MATcvi (m, i));
    }
    else if ((MTYPE (m) == REAL) && (SVALi (s) != 0.0))
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_le (SVALr (s), SVALi (s),
				     MATrv (m, i), 0.0);
    }
    else
      error_1 ("invalid data types for matrix-scalar <= op",
	       (char *) 0);
    return (new);
  }
}

/*
 * MATRIX <= SCALAR
 */
Matrix *
matrix_scalar_le2 (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new = 0;

    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* Most common operation */
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = (double) (MATrv (m, i) <= SVALr (s));
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_le (MATcvr (m, i), MATcvi (m, i),
				     SVALr (s), SVALi (s));

    }
    else if ((MTYPE (m) == REAL) && (SVALi (s) != 0.0))
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_le (MATrv (m, i), 0.0,
				     SVALr (s), SVALi (s));

    }
    else
      error_1 ("invalid data types for matrix-scalar <= op",
	       (char *) 0);
    return (new);
  }
}

Matrix *
matrix_scalar_and (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new = 0;

    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* Most common operation */
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = (double) ((MATrv (m, i) != 0.0) && (SVALr (s) != 0.0));
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_and (MATcvr (m, i), MATcvi (m, i),
				      SVALr (s), SVALi (s));

    }
    else if ((MTYPE (m) == REAL) && (SVALi (s) != 0.0))
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_and (MATrv (m, i), 0.0,
				      SVALr (s), SVALi (s));

    }
    else
      error_1 ("invalid data types for matrix-scalar  && op",
	       (char *) 0);
    return (new);
  }
}

Matrix *
matrix_scalar_or (m, s)
     Matrix *m;
     Scalar *s;
{
  ASSERT (m);
  ASSERT (s);
  {
    register int i;
    Matrix *new = 0;

    if ((MTYPE (m) == REAL) && (SVALi (s) == 0.0))
    {
      /* Most common operation */
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = (double) ((MATrv (m, i) != 0.0) || (SVALr (s) != 0.0));
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_or (MATcvr (m, i), MATcvi (m, i),
				     SVALr (s), SVALi (s));

    }
    else if ((MTYPE (m) == REAL) && (SVALi (s) != 0.0))
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	MATrv (new, i) = complex_or (MATrv (m, i), 0.0,
				     SVALr (s), SVALi (s));

    }
    else
      error_1 ("invalid data types for matrix-scalar || op",
	       (char *) 0);
    return (new);
  }
}

/* **************************************************************
 * Check the values of a MATRIX. We are looking to decide whether
 * or not we need to do a complex op on the matrix, or we can get
 * away with a real operation. Return TRUE (1) at the 1st sign of
 * an element < 0.0
 * ************************************************************** */
int
do_cmplx (m)
     Matrix *m;
{
  register int i, size;

  size = MNR (m) * MNC (m);
  for (i = 0; i < size; i++)
  {
    if (MATrv (m, i) < 0.0)
    {
      return (1);
    }
  }
  return (0);
}

int
do_cmplx_1 (m)
     Matrix *m;
{
  register int i, size;

  size = MNR (m) * MNC (m);
  for (i = 0; i < size; i++)
  {
    if (MATrv (m, i) >= 1.0 || MATrv (m, i) <= -1.)
    {
      return (1);
    }
  }
  return (0);
}
