/*
 * fi_2.c
 * Fortran Interfaces
 * Contains the computational interfaces to LAPACK.
 */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1994  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "mem.h"
#include "symbol.h"
#include "scalar.h"
#include "matrix.h"
#include "matop1.h"
#include "matop2.h"
#include "matop3.h"
#include "btree.h"
#include "util.h"

#include <math.h>

#include "lp.h"
#include "bl.h"
#include "fi.h"

/*
 * Pointers for temporary accounting.
 * Note that we CANNOT use these in a function, if that
 * function calls another function(s) that uses one of
 * the tmp? pointers.
 */
static ListNode *tmp1, *tmp2, *tmp3, *tmp4, *tmp5;

/* **************************************************************
 * Least Squares Solutions.
 * ************************************************************** */

static Matrix *matrix_LS_real _PROTO ((Matrix * a, Matrix * b));
static Matrix *matrix_LS_complex _PROTO ((Matrix * a, Matrix * b));

Matrix *
matrix_LS (a, b)
     Matrix *a, *b;
{
  ASSERT (a);
  ASSERT (b);
  {
    Matrix *m = 0;

    matrix_screen_string (a);
    matrix_screen_string (b);
    matrix_Detect_Inf (a);
    matrix_Detect_Nan (a);
    matrix_Detect_Inf (b);
    matrix_Detect_Nan (b);

    if (MTYPE (a) == REAL && MTYPE (b) == REAL)
      m = matrix_LS_real (a, b);
    else if (MTYPE (a) == COMPLEX || MTYPE (b) == COMPLEX)
      m = matrix_LS_complex (a, b);
    else
      error_2 (matrix_GetName (a), matrix_GetName (b),
	       "invalid matrix arguments to matrix_LS()");

    return (m);
  }
}

static Matrix *
matrix_LS_real (a, b)
     Matrix *a, *b;
{
  F_INT info, lda, ldb, lwork, m, n, nrhs, rank;
  double rcond, mtmp;
  Matrix *A, *B, *Btmp, *s, *work;

  m = MNR (a);
  n = MNC (a);
  nrhs = MNC (b);
  lda = m;
  ldb = max (m, n);
  rcond = -1.0;

  if (m >= n)
  {
    mtmp = max (2 * n, nrhs);
    lwork = 3 * n + max (mtmp, m);
  }
  else
  {
    mtmp = max (2 * m, nrhs);
    lwork = 3 * m + max (mtmp, n);
  }

  tmp1 = install_tmp (MATRIX, s = matrix_Create (1, min (m, n)),
		      matrix_Destroy);
  tmp2 = install_tmp (MATRIX, work = matrix_Create (1, lwork),
		      matrix_Destroy);
  tmp3 = install_tmp (MATRIX, A = matrix_Copy (a),
		      matrix_Destroy);
  tmp4 = install_tmp (MATRIX, B = matrix_Copy (b),
		      matrix_Destroy);

  /* Check to make sure B is O.K. */
  if (ldb > m)
    matrix_AppendRowR (B, ldb - m);

  signal (SIGINT, intcatch);
  RGELSS (&m, &n, &nrhs, MDPTRr (A), &lda, MDPTRr (B), &ldb, MDPTRr (s),
          &rcond, &rank, MDPTRr (work), &lwork, &info);
  signal (SIGINT, intcatch_wait);

  if ((int) info < 0)
  {
    fprintf (stderr, "ERROR: %ith argument to DGELSS is bad\n",
             (int) -info);
    error_1 ("illegal argument to LAPACK DGELSS()", (char *) 0);
  }
  else if ((int) info > 0)
    error_1 ("SVD algorithm failed to converge", (char *) 0);

  remove_tmp_destroy (tmp1);
  remove_tmp_destroy (tmp2);
  remove_tmp_destroy (tmp3);

  /* re-adjust B is necessary */
  if (m > n)
  {
    Btmp = matrix_ExtractColMatrix (B, matrix_CreateFill (1.0, (double) n, 1.0, 0));
    remove_tmp_destroy (tmp4);
    return (Btmp);
  }
  else
  {
    remove_tmp (tmp4);
    return (B);
  }
}

static Matrix *
matrix_LS_complex (a, b)
     Matrix *a, *b;
{
  F_INT info, lda, ldb, lwork, m, n, nrhs, rank;
  double rcond;
  Matrix *A, *B, *Btmp, *s, *rwork, *work;

  m = MNR (a);
  n = MNC (a);
  nrhs = MNC (b);
  lda = m;
  ldb = max (m, n);
  rcond = -1.0;

  if (m >= n)
    lwork = 2 * n + max (nrhs, m);
  else
    lwork = 2 * m + max (nrhs, n);

  tmp1 = install_tmp (MATRIX, s = matrix_Create (1, min (m, n)),
		      matrix_Destroy);
  tmp2 = install_tmp (MATRIX, work = matrix_CreateC (1, lwork),
		      matrix_Destroy);

  if (MTYPE (a) == REAL)
    tmp3 = install_tmp (MATRIX, A = matrix_copy_complex (a),
			matrix_Destroy);
  else
    tmp3 = install_tmp (MATRIX, A = matrix_Copy (a),
			matrix_Destroy);

  if (MTYPE (b) == REAL)
    tmp4 = install_tmp (MATRIX, B = matrix_copy_complex (b),
			matrix_Destroy);
  else
    tmp4 = install_tmp (MATRIX, B = matrix_Copy (b),
			matrix_Destroy);

  tmp5 = install_tmp (MATRIX, 
		      rwork = matrix_Create(1,max(5*min(m,n)-4,1)),
		      matrix_Destroy);

  /* Check to make sure B is O.K. */
  if (ldb > m)
    matrix_AppendRowC (B, ldb - m);

  signal (SIGINT, intcatch);
  XGELSS (&m, &n, &nrhs, MDPTRc (A), &lda, MDPTRc (B), &ldb, MDPTRr (s),
          &rcond, &rank, MDPTRc (work), &lwork, MDPTRr (rwork), &info);
  signal (SIGINT, intcatch_wait);

  if ((int) info < 0)
  {
    fprintf (stderr, "ERROR: %ith argument to ZGELSS is bad\n",
             (int) -info);
    error_1 ("illegal argument to LAPACK ZGELSS()", (char *) 0);
  }
  else if ((int) info > 0)
    error_1 ("SVD algorithm failed to converge", (char *) 0);

  remove_tmp_destroy (tmp1);
  remove_tmp_destroy (tmp2);
  remove_tmp_destroy (tmp3);
  remove_tmp_destroy (tmp5);

  /* re-adjust B is necessary */
  if (m > n)
  {
    Btmp = matrix_ExtractColMatrix (B, matrix_CreateFill (1.0, (double) n, 1.0, 0));
    remove_tmp_destroy (tmp4);
    return (Btmp);
  }
  else
  {
    remove_tmp (tmp4);
    return (B);
  }
}

/* **************************************************************
 * Factor a matrix
 * ************************************************************** */

static void matrix_Factor_Ge_Real _PROTO ((Matrix * m, Matrix ** lu,
					   Matrix ** pvt, double *cond));
static void matrix_Factor_Ge_Complex _PROTO ((Matrix * m, Matrix ** lu,
					      Matrix ** pvt, double *cond));
static void matrix_Factor_Sy_Real _PROTO ((Matrix * m, Matrix ** lu,
					   Matrix ** pvt, double *cond));
static void matrix_Factor_Sy_Complex _PROTO ((Matrix * m, Matrix ** lu,
					      Matrix ** pvt, double *cond));
extern double matrix_Norm _PROTO ((Matrix * m, char *type));

void
matrix_Factor_Ge (m, lu, pvt, cond)
     Matrix *m, **lu, **pvt;
     double *cond;
{
  ASSERT (m);
  {
    matrix_screen_string (m);
    matrix_Detect_Inf (m);
    matrix_Detect_Nan (m);

    if (MTYPE (m) == REAL)
      matrix_Factor_Ge_Real (m, lu, pvt, cond);
    else
      matrix_Factor_Ge_Complex (m, lu, pvt, cond);
  }
}

static void
matrix_Factor_Ge_Real (m, lu, pvt, cond)
     Matrix *m, **lu, **pvt;
     double *cond;
{
  ASSERT (m);
  {
    double anorm, rcond;
    int i;
    F_INT info, mm, n, lda, *ipiv, one, *iwork, lwork, norm;
    ListNode *tmp1, *tmp2, *tmp3, *tmp4, *tmp5;
    Matrix *tlu, *tpvt, *work;

    lwork = lda = mm = MNR (m);
    n = MNC (m);
    one = 1;
    norm = (F_INT) '1';

    if (mm != n)
      error_1 (matrix_GetName (m), "matrix must be square for factor()");

    tmp1 = install_tmp (D_VOID, ipiv = (F_INT *) MALLOC (sizeof (F_INT) * mm),
			free);
    tmp2 = install_tmp (MATRIX, tpvt = matrix_Create (1, mm),
			matrix_Destroy);
    tmp3 = install_tmp (MATRIX, tlu = matrix_Copy (m),
			matrix_Destroy);
    tmp4 = install_tmp (D_VOID, iwork = (F_INT *) MALLOC (sizeof (F_INT) * n),
			free);
    tmp5 = install_tmp (MATRIX, work = matrix_Create (1, 4 * lwork),
			matrix_Destroy);

    signal (SIGINT, intcatch);
    RGETRF (&mm, &n, MDPTRr (tlu), &lda, ipiv, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK DGETRF", 0);
    if ((int) info > 0)
      error_1 (matrix_GetName (m), "matrix is singular");

    /*
     * Check input for ill-conditioning.
     */

    anorm = matrix_Norm (m, "1");
    signal (SIGINT, intcatch);
    RGECON (&norm, &n, MDPTRr (tlu), &lda, &anorm, &rcond, MDPTRr (work),
            iwork, &info);
    signal (SIGINT, intcatch_wait);
    if (rcond <= DBL_EPSILON)
      warning_1 ("WARNING, ill-conditioned input to factor()", 0);

    /* Fill pvt */
    for (i = 0; i < mm; i++)
    {
      MAT0 (tpvt, 0, i) = ipiv[i];
    }

    *lu = tlu;
    *pvt = tpvt;
    *cond = rcond;

    /* Clean up */
    remove_tmp_destroy (tmp1);
    remove_tmp (tmp2);
    remove_tmp (tmp3);
    remove_tmp_destroy (tmp4);
    remove_tmp_destroy (tmp5);
  }
}

static void
matrix_Factor_Ge_Complex (m, lu, pvt, cond)
     Matrix *m, **lu, **pvt;
     double *cond;
{
  ASSERT (m);
  {
    double anorm, rcond, *rwork;
    int i;
    F_INT info, mm, n, lda, *ipiv, one, lwork, norm;
    ListNode *tmp1, *tmp2, *tmp3, *tmp4, *tmp5;
    Matrix *tlu, *tpvt, *work;

    lwork = lda = mm = MNR (m);
    n = MNC (m);
    one = 1;
    norm = (F_INT) '1';

    if (mm != n)
      error_1 (matrix_GetName (m), "matrix must be square for factor()");

    tmp1 = install_tmp (D_VOID, ipiv = (F_INT *) MALLOC (sizeof (F_INT) * mm),
			free);
    tmp2 = install_tmp (MATRIX, tpvt = matrix_Create (1, mm),
			matrix_Destroy);
    tmp3 = install_tmp (MATRIX, tlu = matrix_Copy (m),
			matrix_Destroy);
    tmp4 = install_tmp (D_VOID, 
			rwork = (double *) MALLOC(sizeof(double)*2*n),
			free);
    tmp5 = install_tmp (MATRIX, work = matrix_CreateC (1, 2 * lwork),
			matrix_Destroy);

    signal (SIGINT, intcatch);
    XGETRF (&mm, &n, MDPTRc (tlu), &lda, ipiv, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK ZGETRF", 0);
    if ((int) info > 0)
      error_1 (matrix_GetName (m), "matrix is singular");

    /*
     * Check input for ill-conditioning.
     */

    anorm = matrix_Norm (m, "1");
    signal (SIGINT, intcatch);
    XGECON (&norm, &n, MDPTRc (tlu), &lda, &anorm, &rcond, MDPTRc (work),
            rwork, &info);
    signal (SIGINT, intcatch_wait);
    if (rcond <= DBL_EPSILON)
      warning_1 ("WARNING, ill-conditioned input to factor()", 0);

    /* Fill pvt */
    for (i = 0; i < mm; i++)
    {
      MAT0 (tpvt, 0, i) = ipiv[i];
    }

    *lu = tlu;
    *pvt = tpvt;
    *cond = rcond;

    /* Clean up */
    remove_tmp_destroy (tmp1);
    remove_tmp (tmp2);
    remove_tmp (tmp3);
    remove_tmp_destroy (tmp4);
    remove_tmp_destroy (tmp5);
  }
}

void
matrix_Factor_Sy (m, ldl, pvt, cond)
     Matrix *m, **ldl, **pvt;
     double *cond;
{
  ASSERT (m);
  {
    matrix_screen_string (m);
    matrix_Detect_Inf (m);
    matrix_Detect_Nan (m);

    if (MTYPE (m) == REAL)
      matrix_Factor_Sy_Real (m, ldl, pvt, cond);
    else
      matrix_Factor_Sy_Complex (m, ldl, pvt, cond);
  }
}

static void
matrix_Factor_Sy_Real (m, ldl, pvt, cond)
     Matrix *m, **ldl, **pvt;
     double *cond;
{
  ASSERT (m);
  {
    double anorm, rcond;
    int i;
    F_INT info, mm, n, lda, *ipiv, one, *iwork, lwork, norm;
    F_INT uplo;
    ListNode *tmp1, *tmp2, *tmp3, *tmp4, *tmp5;
    Matrix *tldl, *tpvt, *work;

    uplo = (F_INT) 'L';
    lda = n = mm = MNR (m);
    one = 1;
    norm = (F_INT) '1';

    /*
     *Try and pick a good NB, without ILAENV.
     */
    
    if (n < 100)
      lwork = n;
    else
      lwork = 64 * n;
    
    if (MNR (m) != MNC (m))
      error_1 (matrix_GetName (m), "matrix must be square for factor()");

    tmp1 = install_tmp (D_VOID, ipiv = (F_INT *) MALLOC (sizeof (F_INT) * mm),
			free);
    tmp2 = install_tmp (MATRIX, tpvt = matrix_Create (1, mm),
			matrix_Destroy);
    tmp3 = install_tmp (MATRIX, tldl = matrix_Copy (m),
			matrix_Destroy);
    tmp4 = install_tmp (D_VOID, iwork = (F_INT *) MALLOC (sizeof (F_INT) * n),
			free);
    tmp5 = install_tmp (MATRIX, work = matrix_Create (1, lwork),
			matrix_Destroy);

    signal (SIGINT, intcatch);
    RSYTRF (&uplo, &n, MDPTRr (tldl), &lda, ipiv, MDPTRr (work), 
	    &lwork, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK DSYTRF", 0);
    if ((int) info > 0)
      error_1 (matrix_GetName (m), "matrix is singular");

    /*
     * Check input for ill-conditioning.
     */

    anorm = matrix_Norm (m, "1");
    remove_tmp_destroy (tmp5);
    tmp5 = install_tmp (MATRIX, work = matrix_Create (1, 2*n),
			matrix_Destroy);

    signal (SIGINT, intcatch);
    RSYCON (&uplo, &n, MDPTRr (tldl), &lda, ipiv, &anorm, &rcond, 
            MDPTRr (work), iwork, &info);
    signal (SIGINT, intcatch_wait);
    if (rcond <= DBL_EPSILON)
      warning_1 ("WARNING, ill-conditioned input to factor()", 0);

    /* Fill pvt */
    for (i = 0; i < mm; i++)
    {
      MAT0 (tpvt, 0, i) = ipiv[i];
    }

    *ldl = tldl;
    *pvt = tpvt;
    *cond = rcond;

    /* Clean up */
    remove_tmp_destroy (tmp1);
    remove_tmp (tmp2);
    remove_tmp (tmp3);
    remove_tmp_destroy (tmp4);
    remove_tmp_destroy (tmp5);
  }
}

static void
matrix_Factor_Sy_Complex (m, ldl, pvt, cond)
     Matrix *m, **ldl, **pvt;
     double *cond;
{
  ASSERT (m);
  {
    double anorm, rcond;
    int i;
    F_INT info, mm, n, lda, *ipiv, one, lwork, norm;
    F_INT uplo;
    ListNode *tmp1, *tmp2, *tmp3, *tmp5;
    Matrix *tldl, *tpvt, *work;

    lda = mm = MNR (m);
    n = MNC (m);
    one = 1;
    norm = (F_INT) '1';
    uplo = (F_INT) 'L';

    /*
     *Try and pick a good NB, without ILAENV.
     */
    
    if (n < 100)
      lwork = n;
    else
      lwork = 64 * n;
    
    if (mm != n)
      error_1 (matrix_GetName (m), "matrix must be square for factor()");

    tmp1 = install_tmp (D_VOID, ipiv = (F_INT *) MALLOC (sizeof (F_INT) * mm),
			free);
    tmp2 = install_tmp (MATRIX, tpvt = matrix_Create (1, mm),
			matrix_Destroy);
    tmp3 = install_tmp (MATRIX, tldl = matrix_Copy (m),
			matrix_Destroy);
    tmp5 = install_tmp (MATRIX, work = matrix_CreateC (1, lwork),
			matrix_Destroy);

    signal (SIGINT, intcatch);
    XHETRF (&uplo, &n, MDPTRc (tldl), &lda, ipiv, 
            MDPTRc (work), &lwork, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK ZHETRF", 0);
    if ((int) info > 0)
      error_1 (matrix_GetName (m), "matrix is singular");

    /*
     * Check input for ill-conditioning.
     */

    anorm = matrix_Norm (m, "1");
    remove_tmp_destroy (tmp5);
    tmp5 = install_tmp (MATRIX, work = matrix_CreateC (1, 2*n),
			matrix_Destroy);

    signal (SIGINT, intcatch);
    XHECON (&uplo, &n, MDPTRc (tldl), &lda, ipiv, &anorm, &rcond, 
            MDPTRc (work), &info);
    signal (SIGINT, intcatch_wait);
    if (rcond <= DBL_EPSILON)
      warning_1 ("WARNING, ill-conditioned input to factor()", 0);

    /* Fill pvt */
    for (i = 0; i < mm; i++)
    {
      MAT0 (tpvt, 0, i) = ipiv[i];
    }

    *ldl = tldl;
    *pvt = tpvt;
    *cond = rcond;

    /* Clean up */
    remove_tmp_destroy (tmp1);
    remove_tmp (tmp2);
    remove_tmp (tmp3);
    remove_tmp_destroy (tmp5);
  }
}

/* **************************************************************
 * Solution of linear equations (Ax = b) with an already factored
 * A, Usually via factor().
 * ************************************************************** */

Matrix *
matrix_Backsub (f, b)
     Btree *f;
     Matrix *b;
{
  ASSERT (f);
  ASSERT (b);
  {
    int i, ge, sy;
    F_INT *ipiv, lda, ldb, mm, n, nrhs, info;
    F_INT trans, uplo;
    ListNode *tmp, *tmp1, *tmp2, *tmp3 = 0;
    Matrix *B, *lu, *pvt;

    /*
     * Figure out what is in the list.
     * If there is an "lu" then use DGETRS.
     * If there is an "ldl" then use DSYTRS.
     */

    ge = sy = 0;   /* Initialize */
    lu = pvt = 0;  /* Initialize */

    if ((tmp = btree_FindNode (f, "lu")))
    {
      ge = 1; sy = 0;
      lu = (Matrix *) e_data (tmp);
      if (!(tmp = btree_FindNode (f, "pvt")))
	error_1 ("List must contain \"pvt\" for backsub()", 0);
      pvt = (Matrix *) e_data (tmp);
    }
    else if ((tmp = btree_FindNode (f, "ldl")))
    {
      ge = 0; sy = 1;
      lu = (Matrix *) e_data (tmp);
      if (!(tmp = btree_FindNode (f, "pvt")))
	error_1 ("List must contain \"pvt\" for backsub()", 0);
      pvt = (Matrix *) e_data (tmp);
    }
    else
      error_1 ("input List must contain \"lu\" or \"ldl\" for backsub()", 0);
    
    lda = mm = MNR (lu);
    n = MNC (lu);
    nrhs = MNC (b);
    ldb = MNR (b);
    trans = (F_INT) 'N';
    uplo = (F_INT) 'L';

    if (ldb != lda)
      error_1 (matrix_GetName (b), "b must have same number of rows as LHS");
    
    tmp1 = install_tmp (D_VOID, ipiv = (F_INT *) MALLOC (sizeof (F_INT) * mm),
			free);
    tmp2 = install_tmp (D_VOID, B = matrix_Copy (b), free);
    
    /* Fill ipiv */
    for (i = 0; i < mm; i++)
    {
      ipiv[i] = MAT0 (pvt, 0, i);
    }

    if (MTYPE (lu) == REAL && MTYPE (b) == REAL)
    {
      if (ge)
      {
	/* Call dgetrs */
	signal (SIGINT, intcatch);
	RGETRS (&trans, &n, &nrhs, MDPTRr (lu), &lda, ipiv, MDPTRr (B), &ldb,
		&info);
	signal (SIGINT, intcatch_wait);
	if ((int) info < 0)
	  error_1 ("Bad argument(s) to LAPACK DGETRF", 0);
      }
      else if (sy)
      {
	/* Call dsytrs */
	signal (SIGINT, intcatch);
	RSYTRS (&uplo, &n, &nrhs, MDPTRr (lu), &lda, ipiv, MDPTRr (B), &ldb,
		&info);
	signal (SIGINT, intcatch_wait);
	if ((int) info < 0)
	  error_1 ("Bad argument(s) to LAPACK DSYTRF", 0);
      }
      remove_tmp_destroy (tmp1);
      remove_tmp (tmp2);
    }
    else
    {
      /* Must make everything COMPLEX */
      if (MTYPE (lu) != COMPLEX)
	tmp3 = install_tmp (MATRIX, lu = matrix_copy_complex (lu),
			    matrix_Destroy);
      if (MTYPE (B) != COMPLEX)
      {
	remove_tmp_destroy (tmp2);
	tmp2 = install_tmp (MATRIX, B = matrix_copy_complex (b),
			    matrix_Destroy);
      }

      if (ge)
      {
	signal (SIGINT, intcatch);
	XGETRS (&trans, &n, &nrhs, MDPTRc (lu), &lda, ipiv, MDPTRc (B), &ldb,
		&info);
	signal (SIGINT, intcatch_wait);
	if ((int) info < 0)
	  error_1 ("Bad argument(s) to LAPACK ZGETRF", 0);
      }
      else if (sy)
      {
	signal (SIGINT, intcatch);
	XHETRS (&uplo, &n, &nrhs, MDPTRc (lu), &lda, ipiv, MDPTRc (B), &ldb,
		&info);
	signal (SIGINT, intcatch_wait);
	if ((int) info < 0)
	  error_1 ("Bad argument(s) to LAPACK ZHETRF", 0);
      }
      remove_tmp_destroy (tmp1);
      remove_tmp (tmp2);
      if (tmp3)
	remove_tmp_destroy (tmp3);
    }
    return (B);
  }
}

/* **************************************************************
 * Schur Decomposition
 * ************************************************************** */

static void matrix_Schur_Real _PROTO ((Matrix * m, Matrix ** t, Matrix ** z));
static void matrix_Schur_Complex _PROTO ((Matrix * m, Matrix ** t, Matrix ** z));

void
matrix_Schur (m, t, z)
     Matrix *m, **t, **z;
{
  ASSERT (m);
  {
    matrix_screen_string (m);
    matrix_Detect_Inf (m);
    matrix_Detect_Nan (m);

    if (MTYPE (m) == REAL)
      matrix_Schur_Real (m, t, z);
    else if (MTYPE (m) == COMPLEX)
      matrix_Schur_Complex (m, t, z);
  }
}

void (*rselect) ();		/* Dummy selction pointer */

static void
matrix_Schur_Real (m, t, z)
     Matrix *m, **t, **z;
{
  ASSERT (m);
  {
    int *bwork;
    F_INT info, lda, lwork, n, sdim, jobvs, sort;
    ListNode *tmp1, *tmp2, *tmp3, *tmp4, *tmp5;
    Matrix *a, *vs, *wr, *wi, *work;

    /* m must be n-by-n */
    if (MNR (m) != MNC (m))
      error_1 (matrix_GetName (m), "Must be N-by-N for schur()");

    n = (F_INT) MNR (m);
    lda = n;
    sdim = 0;
    lwork = max (1, 3 * n);
    bwork = (int *) MALLOC (2 * sizeof (int));	/* not used */
    jobvs = (F_INT) 'V';
    sort = (F_INT) 'N';

    tmp1 = install_tmp (MATRIX, a = matrix_Copy (m),
			matrix_Destroy);
    tmp2 = install_tmp (MATRIX, wr = matrix_Create (1, n),
			matrix_Destroy);
    tmp3 = install_tmp (MATRIX, wi = matrix_Create (1, n),
			matrix_Destroy);
    tmp4 = install_tmp (MATRIX, vs = matrix_Create (n, n),
			matrix_Destroy);
    tmp5 = install_tmp (MATRIX, work = matrix_Create (1, lwork),
			matrix_Destroy);

    signal (SIGINT, intcatch);
    RGEES (&jobvs, &sort, rselect, &n, MDPTRr (a), &lda, &sdim, MDPTRr (wr),
           MDPTRr (wi), MDPTRr (vs), &n, MDPTRr (work), &lwork, bwork,
           &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
    {
      error_1 ("illegal argument to DGEES", 0);
    }
    else if ((int) info > 0)
    {
      if ((int) info <= n)
      {
	fprintf (stderr, "schur (DGEES): Failed to compute all eigenvalues\n");
	fprintf (stderr, "               %i:%i converged eigenvalues\n",
                 (int) info, (int) n);
      }
      else if ((int) info == n + 1)
      {
	fprintf (stderr, "schur (DGEES): Eigenvalues could not be re-ordered\n");
	fprintf (stderr, "               This problem is very ill-conditioned\n");
	error_1 ("schur: cannot continue", 0);
      }
      else if ((int) info == n + 2)
      {
	fprintf (stderr, "schur (DGEES): Problems re-ordering\n");
	error_1 ("schur: cannot continue", 0);
      }
    }

    *t = a;
    *z = vs;

    remove_tmp (tmp1);
    remove_tmp_destroy (tmp2);
    remove_tmp_destroy (tmp3);
    remove_tmp (tmp4);
    remove_tmp_destroy (tmp5);
    FREE (bwork);
  }
}

static void
matrix_Schur_Complex (m, t, z)
     Matrix *m, **t, **z;
{
  ASSERT (m);
  {
    int *bwork;
    F_INT info, lda, lwork, n, sdim, jobvs, sort;
    ListNode *tmp1, *tmp2, *tmp3, *tmp4, *tmp5;
    Matrix *a, *vs, *w, *work, *rwork;

    /* m must be n-by-n */
    if (MNR (m) != MNC (m))
      error_1 (matrix_GetName (m), "Must be N-by-N for schur()");

    n = (F_INT) MNR (m);
    lda = n;
    sdim = 0;
    lwork = max (1, 3 * n);
    bwork = (int *) MALLOC (2 * sizeof (int));	/* not used */
    jobvs = (F_INT) 'V';
    sort = (F_INT) 'N';

    tmp1 = install_tmp(MATRIX, a = matrix_Copy (m),
		       matrix_Destroy);
    tmp2 = install_tmp(MATRIX, w = matrix_CreateC (1, n),
		       matrix_Destroy);
    tmp3 = install_tmp(MATRIX, rwork = matrix_Create (1, n),
		       matrix_Destroy);
    tmp4 = install_tmp(MATRIX, vs = matrix_CreateC (n, n),
		       matrix_Destroy);
    tmp5 = install_tmp(MATRIX, work = matrix_CreateC (1, lwork),
		       matrix_Destroy);

    signal (SIGINT, intcatch);
    XGEES (&jobvs, &sort, rselect, &n, MDPTRc (a), &lda, &sdim, MDPTRc (w),
           MDPTRc (vs), &n, MDPTRc (work), &lwork, MDPTRr (rwork),
           bwork, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
    {
      error_1 ("illegal argument to ZGEES", 0);
    }
    else if ((int) info > 0)
    {
      if ((int) info <= n)
      {
	fprintf (stderr, "schur (ZGEES): Failed to compute all eigenvalues\n");
	fprintf (stderr, "               %i:%i converged eigenvalues\n",
                 (int) info, (int) n);
      }
      else if ((int) info == n + 1)
      {
	fprintf (stderr, "schur (ZGEES): Eigenvalues could not be re-ordered\n");
	fprintf (stderr, "               This problem is very ill-conditioned\n");
	error_1 ("schur: cannot continue", 0);
      }
      else if ((int) info == n + 2)
      {
	fprintf (stderr, "schur (ZGEES): Problems re-ordering\n");
	error_1 ("schur: cannot continue", 0);
      }
    }

    *t = a;
    *z = vs;

    remove_tmp (tmp1);
    remove_tmp_destroy (tmp2);
    remove_tmp_destroy (tmp3);
    remove_tmp (tmp4);
    remove_tmp_destroy (tmp5);
    FREE (bwork);
  }
}

/* **************************************************************
 * Functions for Sylvester's equation, and Schur block moves.
 * ************************************************************** */

static void matrix_Sylv_Real _PROTO ((Matrix *a, Matrix *b, Matrix *c, 
				      Matrix **x));
static void matrix_Sylv_Complex _PROTO ((Matrix *a, Matrix *b, Matrix *c, 
					 Matrix **x));

void
matrix_Sylv (a, b, c, x)
     Matrix *a, *b, *c, **x;
{
  if (b == 0)
  {
    if (MTYPE (a) == COMPLEX || MTYPE (c) == COMPLEX)
      matrix_Sylv_Complex (a, b, c, x);
    else
      matrix_Sylv_Real (a, b, c, x);
  }
  else
  {
    if (MTYPE (a) == COMPLEX || MTYPE (b) == COMPLEX || MTYPE (c) == COMPLEX)
      matrix_Sylv_Complex (a, b, c, x);
    else
      matrix_Sylv_Real (a, b, c, x);
  }
}

static void
matrix_Sylv_Real (a, b, c, x)
     Matrix *a, *b, *c, **x;
{
  double scale;
  int lflag;
  F_INT info, isgn, m, n, lda, ldb, ldc;
  F_INT trana, tranb;
  Matrix *C;

  if (b == 0)
  {
    lflag = 1;    /* Lyapunov equation */
    isgn = 1;     /* A*X + X*A = -C */
    scale = 1.0;
    trana = (F_INT) 'N';
    tranb = (F_INT) 'T';

    /* Set & Check dimensions */
    m = lda = MNR (a);
    n = ldb = m;
    ldc = m;
    if (m != MNC (a))
      error_1 ("sylv: A.nr must equal A.nc", 0);
    if (m != MNR (c))
      error_1 ("sylv: A.nr must equal C.nr", 0);

    tmp1 = install_tmp(MATRIX, C = matrix_Negate (c), matrix_Destroy);

    RTRSYL (&trana, &tranb, &isgn, &m, &n, MDPTRr (a), &lda, 
	    MDPTRr (a), &ldb, MDPTRr (C), &ldc,  &scale,
	    &info);
  }
  else
  {
    lflag = 0;   /* Sylvester equation */
    isgn = 1;   /* A*X + X*B = C */
    scale = 1.0;

    /* Set & Check dimensions */
    m = lda = MNR (a);
    n = ldb = MNR (b);
    ldc = m;
    trana = (F_INT) 'N';
    tranb = (F_INT) 'N';

    if (m != MNC (a))
      error_1 ("sylv: A.nr must equal A.nc", 0);
    if (n != MNC (b))
      error_1 ("sylv: B.nr must equal B.nc", 0);
    if (m != MNR (c))
      error_1 ("sylv: A.nr must equal C.nr", 0);
    if (n != MNC (c))
      error_1 ("sylv: B.nr must equal C.nc", 0);

    tmp1 = install_tmp(MATRIX, C = matrix_Negate (c), matrix_Destroy);

    RTRSYL (&trana, &tranb, &isgn, &m, &n, MDPTRr (a), &lda, 
	    MDPTRr (b), &ldb, MDPTRr (C), &ldc,  &scale,
	    &info);
  }

  if ((int) info < 0)
  {
    fprintf (stderr, "sylv: %ith argument to DTRSYL illegal\n", (int) info);
    error_1 ("sylv: cannot continue", 0);
  }
  else if ((int) info == 1)
  {
    fprintf (stderr, "sylv: A and B have very close eigenvalues; perturbed values\n");
    fprintf (stderr, "      were used to solve the equation (but the matrices \n");
    fprintf (stderr, "      A and B are unchanged\n");
  }

  *x = C;

  remove_tmp (tmp1);
}

static void
matrix_Sylv_Complex (a, b, c, x)
     Matrix *a, *b, *c, **x;
{
  double scale;
  int lflag;
  F_INT info, isgn, m, n, lda, ldb, ldc;
  F_INT trana, tranb;
  ListNode *tmp1, *tmp2, *tmp3, *tmp4;
  Matrix *A, *B, *C;

  if (b == 0)
  {
    /*
     * Lyapunov equation
     */

    lflag = 1;
    isgn = 1;     /* A*X + X*A = -C */
    scale = 1.0;

    /* Set & Check dimensions */
    m = lda = MNR (a);
    n = ldb = m;
    ldc = m;
    trana = (F_INT) 'N';
    tranb = (F_INT) 'T';

    if (m != MNC (a))
      error_1 ("sylv: A.nr must equal A.nc", 0);
    if (m != MNR (c))
      error_1 ("sylv: A.nr must equal C.nr", 0);

    /*
     * Insure that all inputs are complex
     */

    tmp1 = install_tmp(MATRIX, C = matrix_Negate (c), matrix_Destroy);
    if (MTYPE (C) == REAL)
      tmp2 = install_tmp (MATRIX, C = matrix_copy_complex (C),
			  matrix_Destroy);
    else
      tmp2 = 0;
    if (MTYPE (a) == REAL)
      tmp3 = install_tmp (MATRIX, A = matrix_copy_complex (a),
			  matrix_Destroy);
    else
    {
      A = a;
      tmp3 = 0;
    }

    XTRSYL (&trana, &tranb, &isgn, &m, &n, MDPTRc (A), &lda, 
	    MDPTRc (A), &ldb, MDPTRc (C), &ldc,  &scale,
	    &info);

    if ((int) info < 0)
    {
      fprintf (stderr, "sylv: %ith argument to ZTRSYL illegal\n", (int) info);
      error_1 ("sylv: cannot continue", 0);
    }
    else if ((int) info == 1)
    {
      fprintf (stderr, "sylv: A and B have very close eigenvalues; perturbed values\n");
      fprintf (stderr, "      were used to solve the equation (but the matrices \n");
      fprintf (stderr, "      A and B are unchanged\n");
    }

    *x = C;

    if (tmp2 == 0)
      remove_tmp (tmp1);
    else
    {
      remove_tmp_destroy (tmp1);
      remove_tmp (tmp2);
    }
    if (tmp3 != 0)
      remove_tmp_destroy (tmp3);

    return;
  }
  else
  {
    /*
     * Sylvester equation
     */

    lflag = 0;
    isgn = 1;   /* A*X + X*B = C */
    scale = 1.0;

    /* Set & Check dimensions */
    m = lda = MNR (a);
    n = ldb = MNR (b);
    ldc = m;
    trana = (F_INT) 'N';
    tranb = (F_INT) 'N';

    if (m != MNC (a))
      error_1 ("sylv: A.nr must equal A.nc", 0);
    if (n != MNC (b))
      error_1 ("sylv: B.nr must equal B.nc", 0);
    if (m != MNR (c))
      error_1 ("sylv: A.nr must equal C.nr", 0);
    if (n != MNC (c))
      error_1 ("sylv: B.nr must equal C.nc", 0);

    /*
     * Insure that all inputs are complex
     */

    tmp1 = install_tmp (MATRIX, C = matrix_Negate (c),
			matrix_Destroy);
    if (MTYPE (C) == REAL)
      tmp2 = install_tmp (MATRIX, C = matrix_copy_complex (C),
			  matrix_Destroy);
    else
      tmp2 = 0;
    if (MTYPE (a) == REAL)
      tmp3 = install_tmp (MATRIX, A = matrix_copy_complex (a),
			  matrix_Destroy);
    else
    {
      A = a;
      tmp3 = 0;
    }
    if (MTYPE (b) == REAL)
      tmp4 = install_tmp (MATRIX, B = matrix_copy_complex (b),
			  matrix_Destroy);
    else
    {
      B = b;
      tmp4 = 0;
    }

    XTRSYL (&trana, &tranb, &isgn, &m, &n, MDPTRc (A), &lda, 
	    MDPTRc (B), &ldb, MDPTRc (C), &ldc,  &scale,
	    &info);

    if ((int) info < 0)
    {
      fprintf (stderr, "sylv: %ith argument to ZTRSYL illegal\n", (int) info);
      error_1 ("sylv: cannot continue", 0);
    }
    else if ((int) info == 1)
    {
      fprintf (stderr, "sylv: A and B have very close eigenvalues; perturbed values\n");
      fprintf (stderr, "      were used to solve the equation (but the matrices \n");
      fprintf (stderr, "      A and B are unchanged\n");
    }

    *x = C;

    if (tmp2 == 0)
      remove_tmp (tmp1);
    else
    {
      remove_tmp_destroy (tmp1);
      remove_tmp (tmp2);
    }
    if (tmp3 != 0)
      remove_tmp_destroy (tmp3);
    if (tmp4 != 0)
      remove_tmp_destroy (tmp4);

    return;
  }
}

/* **************************************************************
 * Re-Order the elements of a Schur decomposed system
 * ************************************************************** */

void
matrix_Schur_Reorder (t, z, ifst, ilst, T, Z)
     Matrix *t, *z, **T, **Z;
     int ifst, ilst;
{
  F_INT info, compq, n, ldq, ldt;
  F_INT IFST, ILST;
  Matrix *Tmp, *Zmp, *work;

  compq = (F_INT) 'V';
  n = (F_INT) MNR (t);
  tmp1 = install_tmp (MATRIX, Tmp = matrix_Copy (t), matrix_Destroy);
  ldt = (F_INT) MNR (t);
  tmp2 = install_tmp (MATRIX, Zmp = matrix_Copy (z), matrix_Destroy);
  ldq = (F_INT) MNR (z);
  IFST = (F_INT) ifst;
  ILST = (F_INT) ilst;

  if (MTYPE (Tmp) == REAL)
  {
    tmp3 = install_tmp (MATRIX, work = matrix_Create (1, n), 
			matrix_Destroy);
    
    RTREXC (&compq, &n, MDPTRr (Tmp), &ldt, MDPTRr (Zmp), &ldq, 
            &ifst, &ilst, MDPTRr (work), &info, 1);
    
    if ((int) info < 0) error_1 ("schord: illegal argument", 0);
    if ((int) info == 1) error_1 ("schord: two adjacent blocks too close", 0);

    *T = Tmp;
    *Z = Zmp;

    remove_tmp (tmp1);
    remove_tmp (tmp2);
    remove_tmp_destroy (tmp3);
  }
  else if (MTYPE (Tmp) == COMPLEX)
  {
    XTREXC (&compq, &n, MDPTRc (Tmp), &ldt, MDPTRc (Zmp), &ldq, 
            &ifst, &ilst, &info, 1);

    if ((int) info < 0) error_1 ("schord: illegal argument", 0);

    *T = Tmp;
    *Z = Zmp;

    remove_tmp (tmp1);
    remove_tmp (tmp2);
  }
}
