/* code.c */

/* code.c: This file contains the majority of the virtual machine.
   The main switch statement is here, as well as most of the
   op-code functions. Some of the class related op-code functions
   are off in files like op.c, op_matrix.c, etc. */

/* At this point in time code.c is a working implementation. It is
   NOT a fast implementation. There are many things that can, and
   will be done to make the interpreter faster once the language is
   finished and tested. 8/4/92 */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1993, 1994  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "code.h"
#include "rfile.h"
#include "list.h"
#include "btree.h"
#include "mem.h"
#include "y.tab.h"
#include "bltin.h"
#include "function.h"
#include "r_string.h"
#include "util.h"

#include <stdio.h>
#include <string.h>

#ifdef THINK_C
#define frame framex
#endif

struct _frame
{
  ListNode *sp;			/* symbol ptr for function */
  Inst *retpc;			/* ptr to machine position for function return */
  Datum *argn;			/* ptr to args on stack */
  int nargs;			/* # of args function is called with */
  List *arg_list;		/* list of user-function arguments */
  Datum *localn;		/* ptr to local vars on stack */
  int n_local;			/* number of local variables */
  int self;
};

typedef struct _frame Frame;

#define NFRAME 512		/* Fixed frame stack size */
static Frame *frame;		/* Frame stack */
static Frame *fp;		/* Frame pointer */

static int NSTACK = 1024;	/* Fixed stack size */
static Datum *stack;		/* the stack */
static Datum *stackp;		/* next free spot on stack */

static Program *program;	/* The current program for code generation */
static Inst *pc;		/* Program counter during execution */
static int progoff;		/* The current program offset, from beginning */

static int returning;		/* 1 if return stmt seen */
static int breaking = 0;
static int continuing = 0;

/* dissassem.c */
extern void diss_assemble _PROTO ((Inst * p, int progoff));

/* scan.l */
extern char *curr_file_name;
extern int lineno;		/* scan.l */
extern int loff;		/* rlab.y */
extern destroy_fstatic_tree _PROTO ((void));	/* rlab.y */

/* Protos for code.c functions */
void push _PROTO ((Datum d));
Datum pop _PROTO ((void));
void extern_push _PROTO ((Datum d));
Datum extern_pop _PROTO ((void));

void load_local_var _PROTO ((Function * f, int nargs));
void print _PROTO ((Datum d));
void inc _PROTO ((void));
void dec _PROTO ((void));
Datum assign _PROTO ((Datum d1, Datum d2));
void whilecode _PROTO ((void));
void forcode _PROTO ((void));
void ifcode _PROTO ((void));
void bltin _PROTO ((ListNode * sp, int narg, int popf));
void function_call _PROTO ((void));
void function_call_1 _PROTO ((void));
void function_call_self _PROTO ((void));
void userf _PROTO ((ListNode * sp, int nargs, int self));

void list_create _PROTO ((void));
void list_member _PROTO ((void));
void list_assign _PROTO ((void));
void list_el_create _PROTO ((void));

/* print.c */
extern void destroy_file_list _PROTO ((void));

/* relation.c */
extern Datum eq _PROTO ((Datum d1, Datum d2));
extern Datum ne _PROTO ((Datum d1, Datum d2));
extern Datum le _PROTO ((Datum d1, Datum d2));
extern Datum lt _PROTO ((Datum d1, Datum d2));
extern Datum ge _PROTO ((Datum d1, Datum d2));
extern Datum gt _PROTO ((Datum d1, Datum d2));
extern Datum and _PROTO ((Datum d1, Datum d2));
extern Datum or _PROTO ((Datum d1, Datum d2));
extern Datum not _PROTO ((Datum d));

/* print_code.c */
extern void print_code _PROTO ((Inst * p));

/* op.c */
extern Datum addition_op _PROTO ((Datum d1, Datum d2));
extern Datum subtraction_op _PROTO ((Datum d1, Datum d2));
extern Datum el_addition_op _PROTO ((Datum d1, Datum d2));
extern Datum el_subtraction_op _PROTO ((Datum d1, Datum d2));
extern Datum multiply_op _PROTO ((Datum d1, Datum d2));
extern Datum rdivide _PROTO ((Datum d1, Datum d2));
extern Datum el_rdivide _PROTO ((Datum d1, Datum d2));
extern Datum ldivide _PROTO ((Datum d1, Datum d2));
extern Datum el_ldivide _PROTO ((Datum d1, Datum d2));
extern Datum negate_op _PROTO ((Datum d));
extern Datum power_op _PROTO ((Datum d1, Datum d2));
extern Datum el_power _PROTO ((Datum d1, Datum d2));

/* op_vector.c */
extern Datum vector_create _PROTO ((int n, Datum d1, Datum d2, Datum d3));
extern Datum vector_append _PROTO ((Datum d1, Datum d2));

/* op_matrix.c */
extern Datum matrix_create _PROTO ((Datum d));
extern Datum matrix_stack _PROTO ((Datum d1, Datum d2));
extern Datum matrix_sub _PROTO ((int i_flag, Datum i1, Datum i2, Datum var));
extern Datum matrix_vector_sub _PROTO ((Datum d1, Datum d2));
extern Datum matrix_assign _PROTO ((int i_flag, Datum var, Datum i1,
				    Datum i2, Datum a));
extern Datum matrix_vector_assign _PROTO ((Datum m, Datum i1, Datum rhs));
extern Datum matrix_transpose _PROTO ((Datum d));
extern Datum matrix_el_transpose _PROTO ((Datum d));
extern Datum element_multiply _PROTO ((Datum d1, Datum d2));
extern Datum matrix_reshape_col _PROTO ((Datum d));

/* scan.l */
extern void scanner_cleanup _PROTO ((void));

/* **************************************************************
 * Miscellaneous functions to avoid using global data. Global data
 * is not that bad, but we try to avoid it here cause we might end
 * up using this file as part of a library.
 * ************************************************************** */

static int print_machine;	/* if TRUE print out machine contents */

void
set_print_machine (val)
     int val;
{
  print_machine = val;
}

static int line_nos;		/* if TRUE include line numbers in code */

void
set_line_nos (val)
     int val;
{
  line_nos = val;
}

static int use_pager;		/* if TRUE use popen () */

void
set_use_pager (val)
     int val;
{
  use_pager = val;
}

static char *pager;

void
set_code_pager (value)
     char *value;
{
  pager = cpstr (value);
}

int
get_progoff ()
{
  return progoff;
}

/* **************************************************************
 * Catch Ctrl-Cs
 * ************************************************************** */

static interrupt = 0;

void
intcatch (tmp)
     int tmp;
{
#ifdef __EMX__
  /* to re-enable signals under EMX */
  signal (SIGINT, SIG_ACK);
#endif

  signal (SIGINT, SIG_IGN);
  error_1 ("user-generated interrupt", 0);
}

void
intcatch_wait (tmp)
     int tmp;
{

#ifdef __EMX__
  /* to re-enable signals under EMX */
  signal (SIGINT, SIG_ACK);
#endif

  /*
   * Ignore interrupts until we finish with this one.
   */

  signal (SIGINT, SIG_IGN);

  /*
   * Set the flag so we stop next time we enter the
   * machine switch.
   */

  interrupt = 1;

  /*
   * Print something out to re-assure the user
   * that something is indeed happening.
   */

  fprintf (stderr, "caught user-generated interrupt, just a moment...\n");
  fflush (stderr);
}

/* **************************************************************
 * Initialization... This stuff is done this way so that machines
 * with small static memory sizes (PCs and MACs) can 
 * ************************************************************** */

/*
 * Initialize the Frame stack
 */

void
init_frame ()
{
  frame = (Frame *) MALLOC (sizeof (Frame) * NFRAME);
  fp = frame;
}

/*
 * Initialize the data stack.
 */

void
init_stack ()
{
  stack = (Datum *) MALLOC (sizeof (Datum) * NSTACK);
  stackp = stack;
}

/* **************************************************************
 * Functions that create and manipulate program arrays.
 * ************************************************************** */
Program *
program_Create (n)
     int n;
{
  Program *new;
  new = (Program *) MALLOC (sizeof (Program));

  new->ncode = n;
  new->prog = (Inst *) MALLOC (n * sizeof (Inst));
  new->progp = new->prog;
  new->off = 0;

  return (new);
}

void
program_Destroy (program)
     Program *program;
{
  program->ncode = 0;
  program->off = 0;
  program->progp = (Inst *) 0;
  FREE (program->prog);
  FREE (program);
}

/*
 * Grow the current program space.
 */

void
program_Grow ()
{
  int size;

  /* Increase machine size */
  size = sizeof (Inst) * program->ncode * 2;
  program->prog = (Inst *) REALLOC (program->prog, size);

  /* Increase ncode size to keep up with prog[] */
  program->ncode *= 2;

  /* Reset progp using old offset */
  program->progp = program->prog + progoff;
}

/*
 * Set program to point at the supplied program.
 */

void
program_Set (progptr)
     Program *progptr;
{
  program = progptr;
}

Program *
program_Get ()
{
  return (program);
}

/*
 * Get and restore, the program-counter (pc).
 * Add one so that when we restart, we are not
 * in the same place.
 */

Inst *
get_program_counter ()
{
  return (pc);
}

void
set_program_counter (program_counter)
     Inst *program_counter;
{
  pc = program_counter;
}

int
get_function_scope ()
{
  if ((fp - frame) > 0)
    return (LOCAL);
  else
    return (GLOBAL);
}

List *
get_function_arglist ()
{
  if ((fp - frame) > 0)
    return (function_GetArgPtr (e_data (fp->sp)));
  else
    return (0);
}

List *
get_function_locallist ()
{
  if ((fp - frame) > 0)
    return (function_GetLocalPtr (e_data (fp->sp)));
  else
    return (0);
}

List *
get_function_globallist ()
{
  if ((fp - frame) > 0)
    return (function_GetGlobalPtr (e_data (fp->sp)));
  else
    return (0);
}

void
reset_frame_ptr ()
{
  fp = frame;
}

Datum *
get_stackp ()
{
  return (stackp);
}

void
reset_stack_ptr ()
{
  stackp = stack;
}

void
set_stackp (new_stackp)
     Datum *new_stackp;
{
  stackp = new_stackp;
}

/* **************************************************************
 * Initialize the program for code generation.
 * ************************************************************** */
void
initcode ()
{
  /* Reset current program pointer */
  program->progp = program->prog;

  /* Reset parser/program offset */
  progoff = 0;

  /* Reset flow control indicators */
  returning = 0;
  breaking = 0;
  continuing = 0;

  if (line_nos)
  {
    /* Set the file name for the next section of code */
    (program->progp++)->op_code = OP_FILE_NAME;
    (program->progp++)->ptr = curr_file_name;
    (program->progp++)->op_code = OP_LINE_NO;
    (program->progp++)->op_code = lineno + loff;

    (program->progp)->op_code = STOP;
  }

  /* Compute the current code offset, for the parser */
  progoff = program->progp - program->prog;
}

/* **************************************************************
 * Push d (Datum) onto the stack. Update the stack pointer (stackp).
 * ************************************************************** */

void
push (d)
     Datum d;
{
  if (stackp >= &stack[NSTACK - 1])
  {
    error_1 ("push:", "stack overflow");
  }
  *stackp++ = d;
}

void
extern_push (d)
     Datum d;
{
  if (stackp >= &stack[NSTACK - 1])
  {
    error_1 ("push:", "stack overflow");
  }
  *stackp++ = d;
}

/* **************************************************************
 * Pop and return the top element from the stack.
 * ************************************************************** */

Datum
pop ()
{
  if (stackp <= stack)
    error_1 ("pop()", "stack underflow");
  return (*--stackp);
}

Datum
extern_pop ()
{
  if (stackp <= stack)
    error_1 ("pop()", "stack underflow");
  return (*--stackp);
}

/* **************************************************************
 * Install one instruction or operand. Put an instruction into the
 * next free spot. Return an offset from the base.
 * Becasue the machine instructions are a union, we have a "code()"
 * function for each type of instruction we want to install.
 * ************************************************************** */

int
code (f)
     int f;
{
  Inst *oprogp = program->progp;

  if (program->progp - program->prog >= program->ncode - 1)
  {
    program_Grow ();
    oprogp = program->progp;
  }

  (program->progp++)->op_code = f;
  progoff = program->progp - program->prog;

  /* Return the offset from prog, of the last instruction */
  return (oprogp - program->prog);
}

/* **************************************************************
 * Install a pointer to another op-code.
 * ************************************************************** */
int
codep (f)
     VPTR f;
{
  Inst *oprogp = program->progp;

  if (program->progp - program->prog >= program->ncode - 1)
  {
    program_Grow ();
    oprogp = program->progp;
  }

  (program->progp++)->ptr = f;
  progoff = program->progp - program->prog;
  return (oprogp - program->prog);
}

int
coded (f)
     double f;
{
  Inst *oprogp = program->progp;

  if (program->progp - program->prog >= program->ncode - 1)
  {
    program_Grow ();
    oprogp = program->progp;
  }

  (program->progp++)->d_val = f;
  progoff = program->progp - program->prog;
  return (oprogp - program->prog);
}

/* **************************************************************
 * Install op-codes (jump offsets) at special places in the 
 * program array.
 * ************************************************************** */

void
code_sp (offset, value)
     int offset, value;
{
  program->prog[offset].op_code = value;
}

/* **************************************************************
 * function_call() serves as the front-end for both user-functions
 * and built-in functions.
 * ************************************************************** */

void
function_call ()
{
  int nargs;
  Datum fst;
  ListNode *sp;

  /* Number of function arguments */
  nargs = (*pc).op_code;

  /* Entity holding user/builtin function */
  
  fst = pop ();
  sp = fst.u.ent;

  /*
   * Verify that we have either user-function or built-in
   */

  if (e_type (sp) != U_FUNCTION && e_type (sp) != BLTIN)
  {
    if (e_type (sp) == UNDEF)
      error_1 (e_name (sp), "UNDEFINED");
    else
      error_1 (e_name (sp), " is not a user or built-in function");
  }

  /*
   * Call bltin() if we have a built-in function
   */

  if (e_type (sp) == BLTIN)
  {
    /*
     * Call the built-in from here, all the 
     * right info is on the stack
     */

    bltin (sp, nargs, 0);
    pc++;
    return;
  }
  else
  {
    /*
     * Call the user-function from here, all the
     * right info is on the stack
     */
    userf (sp, nargs, 0);
    pc++;
    return;
  }
}

void
function_call_1 ()
{
  int nargs;
  ListNode *sp;

  /* Number of function arguments */
  nargs = (*pc).op_code;

  /* Entity holding user/builtin function */
  sp = (stackp - nargs - 1)->u.ent;

  /*
   * Verify that we have either user-function or built-in
   */

  if (e_type (sp) != U_FUNCTION && e_type (sp) != BLTIN)
  {
    if (e_type (sp) == UNDEF)
      error_1 (e_name (sp), "UNDEFINED");
    else
      error_1 (e_name (sp), " is not a user or built-in function");
  }

  /*
   * Call bltin() if we have a built-in function
   */

  if (e_type (sp) == BLTIN)
  {
    /*
     * Call the built-in from here, all the 
     * right info is on the stack
     */

    bltin (sp, nargs, 1);
    pc++;
    return;
  }
  else
  {
    /*
     * Call the user-function from here, all the
     * right info is on the stack
     */
    userf (sp, nargs, 1);
    pc++;
    return;
  }
}

void
userf (sp, nargs, self)
     ListNode *sp;
     int nargs, self;
{
  int i;
  char tag[5];
  Datum d;
  ListNode *lnode;
  List *fargs;

  /*
   * Check frame stack for overflow, then inc, and load.
   */
  if (fp >= &frame[NFRAME - 1])
    error_1 (listNode_GetKey (sp), "function call nested too deeply");

  fp++;
  fp->nargs = nargs;
  fp->sp = sp;
  sprintf (tag, "-%i", fp - frame);

  /*
   * Check the # of args on the stack vs. function requ.
   * If there are fewer args present than required, push UNDEFs
   * onto the stack
   */
  if (fp->nargs < function_GetNargs (e_data (sp)))
  {
    for (i = 0; i < function_GetNargs (e_data (sp)) - fp->nargs; i++)
    {
      /* Put an UNDEF arg on tmp-list */
      lnode = install_tmp (UNDEF, 0, 0);
      listNode_SetKey (lnode, cpstr (tag));
      d.u.ent = lnode;
      d.type = ENTITY;
      push (d);
    }
    fp->nargs = function_GetNargs (e_data (sp));
  }
  else if (fp->nargs > function_GetNargs (e_data (sp)))
  {
    /* Error and return */
    fp->sp = 0;
    fp->nargs = 0;
    fp--;
    error_1 (e_name (sp), "too many args to function");
  }

  fp->retpc = pc;		/* return pointer */
  fp->argn = stackp - 1;

  /*
   * Go through the function args, marking the constant
   * entities so that they aren't cleaned from the list.
   * Also change CONSTANTs into scalars.
   */
  
  fargs = function_GetArgPtr (e_data (sp));
  for (i = 1; i <= fp->nargs; i++)
  {
    ListNode *lfarg = list_GetNodeByPos (fargs, fp->nargs+1-i);
    char *name = lvar_GetName (e_data (lfarg));
      
    if (fp->argn[i - fp->nargs].type == ENTITY
	&& fp->argn[i - fp->nargs].u.ent->key == 0)
    {
      fp->argn[i - fp->nargs].u.ent->key = cpstr (tag);
      if (fp->argn[i - fp->nargs].u.ent->type == MATRIX)
	matrix_SetName (e_data (fp->argn[i - fp->nargs].u.ent), cpstr(name));
      else if (fp->argn[i - fp->nargs].u.ent->type == SCALAR)
	scalar_SetName (e_data (fp->argn[i - fp->nargs].u.ent), cpstr(name));
      else if (fp->argn[i - fp->nargs].u.ent->type == STRING)
	string_SetName (e_data (fp->argn[i - fp->nargs].u.ent), cpstr(name));
      else if (fp->argn[i - fp->nargs].u.ent->type == BTREE)
	btree_SetName (e_data (fp->argn[i - fp->nargs].u.ent), cpstr(name));
      else if (fp->argn[i - fp->nargs].u.ent->type == UNDEF)
	;
      else
	error_1 (e_name (sp), "invalid function argument");
    }
    else if (fp->argn[i - fp->nargs].type == CONSTANT)
    {
      fp->argn[i - fp->nargs].u.ent =
	install_tmp (SCALAR, scalar_Create (fp->argn[i - fp->nargs].u.val),
		     scalar_Destroy);
      fp->argn[i - fp->nargs].u.ent->key = cpstr (tag);
      fp->argn[i - fp->nargs].type = ENTITY;
      scalar_SetName (e_data (fp->argn[i - fp->nargs].u.ent), cpstr(name));
    }
    else if (fp->argn[i - fp->nargs].type == iCONSTANT)
    {
      fp->argn[i - fp->nargs].u.ent =
	install_tmp (SCALAR, scalar_CreateC (0.0, fp->argn[i - fp->nargs].u.val),
		     scalar_Destroy);
      fp->argn[i - fp->nargs].u.ent->key = cpstr (tag);
      fp->argn[i - fp->nargs].type = ENTITY;
      scalar_SetName (e_data (fp->argn[i - fp->nargs].u.ent), cpstr(name));
    }
  }

  /* set local variable data */
  if (function_HasLocalVar (e_data (fp->sp)))
  {
    fp->n_local = function_GetNlocal (e_data (fp->sp));
    load_local_var (e_data (fp->sp), nargs);
    fp->localn = stackp - 1;
  }
  else
  {
    /* 
     * we must be sure to do this to keep functions without
     * local variables from inadvertently inheriting the
     * wrong characteristics.
     */
    fp->n_local = 0;
    fp->localn = 0;
  }

  fp->self = self;
  /* Now execute the user-function */
  execute ((Inst *) function_GetCodePtr (listNode_GetData (sp)));
  returning = 0;

}

/*
 * Special, a function is calling itself.
 */
void
function_call_self ()
{
  int nargs;
  ListNode *sp;

  /* Entity holding user/builtin function (SELF) */
  sp = fp->sp;

  /* Number of args on the stack */
  nargs = (*pc).op_code;

  userf (sp, nargs, 0);
  pc++;
}

/* **************************************************************
 * Load a functions local variables onto the stack. All local 
 * variables are initialized to UNDEFINED entities, in order to be
 * consistent with global, and argument variable rules.
 * The name of the local var is installed in the new entity so the
 * user will have an easier time deciphering the output, AND the
 * entity will not get destroyed during operations. Put the
 * local var on the tmp-list so it will get destroyed later, when
 * program control returns to the prompt.
 * ************************************************************** */

static VPTR copy_arg _PROTO ((ListNode * cnode, char *name));

void
load_local_var (f, nargs)
     Function *f;
     int nargs;
{
  char *name;
  int i, n_var;
  Datum d;
  List *local_list;
  ListNode *cnode, *ent;
  VPTR copy;

  n_var = fp->n_local;
  local_list = function_GetLocalPtr (e_data (fp->sp));
  ent = list_GetLastNode (local_list);

  for (i = 0; i < n_var; i++)
  {
    /* Get local var name */
    name = cpstr (lvar_GetName (e_data (ent)));

    /*
     * Create entity to push on stack. Put the entity on
     * the temporary list in case of trouble later.
     * ALSO, check the arg-list for same name, if a match
     * occurs, COPY the arg (pass by value).
     */

    if ((cnode = list_GetNodeByKey (function_GetArgPtr (f), name)))
    {
      /* COPY */
      copy = copy_arg (cnode, name);

      if (copy)			/* handle UNDEFs */
	d.u.ent = install_tmp_switch ((int) *((int *) copy), copy);
      else
	d.u.ent = install_tmp (UNDEF, 0, 0);

      listNode_SetKey (d.u.ent, name);
      d.type = ENTITY;
    }
    else
    {
      if (!(strcmp ("nargs", name)))
      {
	/* Set local variable nargin (number of arguments). */
	d.u.ent = install_tmp (SCALAR, 
			       scalar_Create ((double) nargs),
			       scalar_Destroy);
      }
      else
      {
	d.u.ent = install_tmp (UNDEF, 0, 0);
      }
      listNode_SetKey (d.u.ent, name);
      d.type = ENTITY;
    }
    push (d);

    /* Get the next local var in the list */
    ent = listNode_GetPrevNode (ent);
  }
}

/* **************************************************************
 * Return from an RLaB User-Function
 * ************************************************************** */

void
function_return ()
{
  Datum ret_from_func _PROTO ((Datum d));

  Datum d;
  d = pop ();			/* pop function return value */
  d = ret_from_func (d);

  push (d);			/* put function return back on stack */
}

/* **************************************************************
 * Take care of removing arguments and local variables from the
 * stack now that the function is returning.
 * ************************************************************** */

Datum
ret_from_func (rd)
     Datum rd;
{
  int i;
  char tag[5];
  Datum tmp;

  if (rd.type == ENTITY)
  {
    /* we need to make sure we don't free() the return entity */
    if (e_name (rd.u.ent) != 0)
    {
      /* 
       * Zero out the name, IFF it is one of the local variables.
       * Local vars are already on the tmp-list. 
       */
      if (list_GetNodeByKey (function_GetLocalPtr (e_data (fp->sp)),
			     e_name (rd.u.ent)) != 0)
      {
	listNode_SetKey (rd.u.ent, 0);
	switch (e_type (rd.u.ent))
	{
	case SCALAR:
	  scalar_SetName (e_data (rd.u.ent), 0);
	  break;
	case MATRIX:
	  matrix_SetName (e_data (rd.u.ent), 0);
	  break;
	case BTREE:
	  btree_SetName (e_data (rd.u.ent), 0);
	  break;
	case STRING:
	  string_SetName (e_data (rd.u.ent), 0);
	  break;
	default:
	  break;
	}
      }
    }
  }

  /* Now pop all local-vars off of the stack */
  if (rd.type == ENTITY)
  {
    for (i = 0; i < fp->n_local; i++)
    {
      tmp = pop ();
      if (tmp.u.ent == rd.u.ent)
      {
	/*
	 * Return entity is a local variable, do nothing.
	 * We took care of it above.
	 */
	continue;
      }
      else
      {
	if (e_type (tmp.u.ent) == BTREE && e_name (rd.u.ent) != 0)
	{
	  ListNode *lnode, *nent;

	  /*
	   * Walk the tree checking for members that match rent.
	   * It is possible that we are trying to return part of
	   * a local list. If we are, then yank the returning 
	   * portion of the list from the local variable so that it
	   * does not get destroyed.
	   */

	  if ((lnode = btree_FindNodeR (e_data (tmp.u.ent), rd.u.ent)))
	  {
	    /* Put rent on the tmp-list */
	    nent = install_tmp_switch (e_type (rd.u.ent), e_data (rd.u.ent));

	    /* Fix the list member */
	    lnode->dtor = 0;
	    lnode->data = 0;

	    /* Fix the return datum */
	    rd.u.ent = nent;
	  }
	}
      }
      listNode_SetKey (tmp.u.ent, 0);
      remove_tmp_destroy (tmp.u.ent);
    }
  }
  else
  {
    for (i = 0; i < fp->n_local; i++)
    {
      tmp = pop ();
      listNode_SetKey (tmp.u.ent, 0);
      remove_tmp_destroy (tmp.u.ent);
    }
  }

  /* Now pop the args off of the stack */
  sprintf (tag, "-%i", fp - frame);
  for (i = 0; i < fp->nargs; i++)
  {
    tmp = pop ();
    if (tmp.type == ENTITY)
    {
      if (rd.u.ent != tmp.u.ent
	  && !strcmp (listNode_GetKey (tmp.u.ent), tag))
      {
	listNode_SetKey (tmp.u.ent, 0);
	remove_tmp_destroy (tmp.u.ent);
      }
    }
  }
  
  if (fp->self)
    pop ();

  pc = fp->retpc;

  --fp;
  returning = 1;
  return (rd);
}

/* **************************************************************
 * Default way to return from a function if there is no explicit
 * return statement. In this case we clean the args and local
 * variables off the stack. Push a 0 scalar onto the stack.
 * ************************************************************** */

void
function_default_return ()
{
  int i;
  char tag[5];
  Datum d;

  /* pop the local variables off the stack */
  for (i = 0; i < fp->n_local; i++)
  {
    d = pop ();
    listNode_SetKey (d.u.ent, 0);
    remove_tmp_destroy (d.u.ent);
  }

  /* pop the args off the stack */
  sprintf (tag, "-%i", fp - frame);
  for (i = 0; i < fp->nargs; i++)
  {
    d = pop ();
    if (d.type == ENTITY)
    {
      if (!strcmp (listNode_GetKey (d.u.ent), tag))
      {
	listNode_SetKey (d.u.ent, 0);
	remove_tmp_destroy (d.u.ent);
      }
    }
  }

  if (fp->self)
    pop ();

  pc = fp->retpc;

  --fp;
  returning = 1;

  /* 
   * Create a 0 scalar and push it on the stack so print()
   * will have something to do.
   */
  d.u.ent = install_tmp (SCALAR, scalar_Create (0.0), scalar_Destroy);
  d.type = ENTITY;
  push (d);
}

/* **************************************************************
 * Run the machine. p is the pointer to the machine code where
 * we will start execution.
 * ************************************************************** */

/*
 * Global var to handle return value from eval().
 */

static Datum eval_ret;

Datum
get_eval_ret ()
{
  return (eval_ret);
}

void
execute (p)
     Inst *p;
{
  char *sptr;
  int n, offset;
  Datum d, d1, d2, d3, new;
  String *str;

  pc = p;

  while (!returning)
  {
    if (interrupt)
    {
      signal (SIGINT, intcatch_wait);
      interrupt = 0;
      error_1 ("user-generated interrupt processed", 0);
    }

    switch ((*pc++).op_code)
    {
    case STOP:
      pc--;
      return;
      break;

    case OP_PUSH_VAR:
      new.u.ent = (ListNode *) ((*pc++).ptr);
      new.type = ENTITY;
      push (new);
      break;

    case OP_PUSH_ARG:
      offset = (*pc++).op_code;
      new = fp->argn[offset - fp->nargs];
      push (new);
      break;

    case OP_PUSH_LOCAL_VAR:
      offset = (*pc++).op_code;
      new = fp->localn[offset - fp->n_local];
      new.type = ENTITY;
      push (new);
      break;

    case OP_ADD:
      d2 = pop ();
      d1 = pop ();
      new = addition_op (d1, d2);
      push (new);
      break;

    case OP_SUB:
      d2 = pop ();
      d1 = pop ();
      new = subtraction_op (d1, d2);
      push (new);
      break;

    case OP_MUL:
      d2 = pop ();
      d1 = pop ();
      new = multiply_op (d1, d2);
      push (new);
      break;

    case OP_DIV:
      d2 = pop ();
      d1 = pop ();
      new = rdivide (d1, d2);
      push (new);
      break;

    case OP_LDIV:
      d2 = pop ();
      d1 = pop ();
      new = ldivide (d1, d2);
      push (new);
      break;

    case OP_NEGATE:
      d = pop ();
      new = negate_op (d);
      push (new);
      break;

    case OP_POWER:
      d2 = pop ();
      d1 = pop ();
      new = power_op (d1, d2);
      push (new);
      break;

    case OP_ASSIGN:
      d1 = pop ();
      d2 = pop ();
      new = assign (d2, d1);
      push (new);
      break;

    case OP_FOR:
      forcode ();
      break;

    case OP_EL_MUL:
      d2 = pop ();
      d1 = pop ();
      new = element_multiply (d1, d2);
      push (new);
      break;

    case OP_EL_DIV:
      d2 = pop ();
      d1 = pop ();
      new = el_rdivide (d1, d2);
      push (new);
      break;

    case OP_EL_LDIV:
      d2 = pop ();
      d1 = pop ();
      new = el_ldivide (d1, d2);
      push (new);
      break;

    case OP_EL_POWER:
      d2 = pop ();
      d1 = pop ();
      new = el_power (d1, d2);
      push (new);
      break;

    case OP_PUSH_CONSTANT:
      new.type = CONSTANT;
      new.u.val = (*pc++).d_val;
      push (new);
      break;

    case OP_PUSH_iCONSTANT:
      new.type = iCONSTANT;
      new.u.val = (*pc++).d_val;
      push (new);
      break;

    case OP_PRINT:
      d = pop ();
      print (d);
      break;

    case OP_GT:
      d2 = pop ();
      d1 = pop ();
      new = gt (d1, d2);
      push (new);
      break;

    case OP_LT:
      d2 = pop ();
      d1 = pop ();
      new = lt (d1, d2);
      push (new);
      break;

    case OP_EQ:
      d2 = pop ();
      d1 = pop ();
      new = eq (d1, d2);
      push (new);
      break;

    case OP_GE:
      d2 = pop ();
      d1 = pop ();
      new = ge (d1, d2);
      push (new);
      break;

    case OP_LE:
      d2 = pop ();
      d1 = pop ();
      new = le (d1, d2);
      push (new);
      break;

    case OP_NE:
      d2 = pop ();
      d1 = pop ();
      new = ne (d1, d2);
      push (new);
      break;

    case OP_AND:
      d2 = pop ();
      d1 = pop ();
      new = and (d1, d2);
      push (new);
      break;

    case OP_OR:
      d2 = pop ();
      d1 = pop ();
      new = or (d1, d2);
      push (new);
      break;

    case OP_NOT:
      d = pop ();
      new = not (d);
      push (new);
      break;

    case OP_IF:
      ifcode ();
      break;

    case OP_WHILE:
      whilecode ();
      break;

    case OP_SWAP:
      d2 = pop ();
      d1 = pop ();
      push (d2);
      push (d1);
      break;

    case OP_INC:
      inc ();
      break;

    case OP_DEC:
      dec ();
      break;

    case OP_POP:
      if (stackp <= stack)
      {
	error_1 ("pop()", "stack underflow");
      }
      --stackp;
      break;

    case OP_POP_CLEAN:
      if (stackp <= stack)
	error_1 ("pop()", "stack underflow");
      --stackp;
      if ((*stackp).type == ENTITY)
      {
	if ((*stackp).u.ent != 0)
	  remove_tmp_destroy ((*stackp).u.ent);
      }
      break;

    case OP_VECTOR_CREATE:
      n = (*pc++).op_code;	/* get create-code from machine */
      switch (n)
      {
      case 2:
	d1 = pop ();
	d2 = pop ();
	new = vector_create (2, d1, d2, d3);
	break;
      case 3:
	d1 = pop ();
	d2 = pop ();
	d3 = pop ();
	new = vector_create (3, d1, d2, d3);
	break;
      default:
	error_1 ("error in vector creation", 0);
	break;
      }
      push (new);
      break;

    case OP_VEC_APPEND:
      d2 = pop ();
      d1 = pop ();
      new = vector_append (d1, d2);
      push (new);
      break;

    case OP_MATRIX_VEC_SUB:
      d2 = pop ();
      d1 = pop ();
      new = matrix_vector_sub (d1, d2);
      push (new);
      break;

    case OP_MATRIX_VEC_ASSIGN:
      d3 = pop ();
      d2 = pop ();
      d1 = pop ();
      new = matrix_vector_assign (d1, d2, d3);
      push (new);
      break;

    case OP_MATRIX_CREATE:
      d = pop ();
      new = matrix_create (d);
      push (new);
      break;

    case OP_MATRIX_APPEND:
      d2 = pop ();
      d1 = pop ();
      new = matrix_stack (d1, d2);
      push (new);
      break;

    case OP_MATRIX_ASSIGN:
      n = (*pc++).op_code;	/* get key to # of indices on stack */
      switch (n)
      {
      case 1:			/* both row and column indices */
	d = pop ();
	d1 = pop ();
	d2 = pop ();
	d3 = pop ();
	new = matrix_assign (1, d3, d2, d1, d);
	break;
      case 2:			/* row index only, NYI */
	d = pop ();
	d1 = pop ();
	d2 = pop ();
	new = matrix_assign (2, d2, d1, d3, d);
	break;
      case 3:			/* column index only, NYI */
	d = pop ();
	d1 = pop ();
	d2 = pop ();
	new = matrix_assign (3, d2, d3, d1, d);
	break;
      }
      push (new);
      break;

    case OP_MATRIX_SUB:
      n = (*pc++).op_code;	/* get key to # of indices on stack */
      switch (n)
      {
      case 1:			/* both row and column indices */
	d1 = pop ();
	d2 = pop ();
	d3 = pop ();
	new = matrix_sub (1, d2, d1, d3);
	break;
      case 2:			/* row index only */
	d1 = pop ();
	d2 = pop ();
	new = matrix_sub (2, d1, d, d2);
	break;
      case 3:			/* column index only */
	d1 = pop ();
	d2 = pop ();
	new = matrix_sub (3, d, d1, d2);
	break;
      }
      push (new);
      break;

    case OP_LIST_CREATE:
      list_create ();
      break;

    case OP_LIST_MEMB:
      list_member ();
      break;

    case OP_LIST_ASSIGN:
      list_assign ();
      break;

    case OP_LIST_EL_CREATE:
      list_el_create ();
      break;

    case OP_FUNCTION_CALL:
      function_call ();
      break;

    case OP_FUNCTION_CALL_SELF:
      function_call_self ();
      break;

    case OP_FUNCTION_RETURN:
      function_return ();
      break;

    case OP_DEF_FUNC_RET:
      function_default_return ();
      break;

    case OP_TRANSPOSE:
      d = pop ();
      new = matrix_transpose (d);
      push (new);
      break;

    case OP_PUSH_STRING:
      sptr = cpstr ((*pc++).ptr);
      str = string_Create (sptr);
      new.type = ENTITY;
      new.u.ent = install_tmp (STRING, str, string_Destroy);
      push (new);
      break;

    case OP_BREAK:
      breaking = 1;
      break;

    case OP_CONTINUE:
      continuing = 1;
      break;

    case OP_QUIT:
      quit_code ();
      break;

    case OP_LINE_NO:
      /* No operation */
      pc++;			/* Inc program counter past line number */
      break;

    case OP_FILE_NAME:
      /* No operation */
      pc++;			/* Inc program counter past file name */
      break;

    case OP_JMP:
      pc = pc + (*pc).op_code;
      break;

    case OP_EMPTY_MATRIX_CREATE:
      new.u.ent = install_tmp (MATRIX, matrix_Create (0, 0), matrix_Destroy);
      new.type = ENTITY;
      push (new);
      break;

    case OP_MATRIX_COL:
      d = pop ();
      new = matrix_reshape_col (d);
      push (new);
      break;

    case OP_EL_TRANSPOSE:
      d = pop ();
      new = matrix_el_transpose (d);
      push (new);
      break;

    case OP_RFILE:
      rfile ();
      break;

    case OP_RFILE_NAME:
      rfile_load (cpstr ((*pc++).ptr));
      break;

    case OP_HELP:
      help ();
      break;

    case OP_HELP_NAME:
      help_name (cpstr ((*pc++).ptr));
      break;

    case OP_PUSH_UNDEF:
      new.u.ent = install_tmp (UNDEF, 0, 0);
      new.type = ENTITY;
      push (new);
      break;

    case OP_EL_ADD:
      d2 = pop ();
      d1 = pop ();
      new = el_addition_op (d1, d2);
      push (new);
      break;

    case OP_EL_SUB:
      d2 = pop ();
      d1 = pop ();
      new = el_subtraction_op (d1, d2);
      push (new);
      break;

    case OP_SAVE_EVAL:
      /*
       * pop() the return value and store it 
       * in a global var, Eval() will get it later.
       */
      eval_ret = pop ();
      break;

    case OP_FUNCTION_CALL_1:
      function_call_1 ();
      break;

    default:
      fprintf (stderr, "Invalid op-code: %d\n", (*--pc).op_code);
      fflush (stderr);
      error_1 ("execute:", "Invalid op-code");
      break;
    }
  }
}

/* **************************************************************
 * Increment a variable
 * ************************************************************** */
void
inc ()
{
  Datum d;
  d = pop ();

  switch (d.type)
  {
  case CONSTANT:
  case iCONSTANT:
    error_1 ("cannot increment a constant", 0);
    break;
  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      scalar_Inc (e_data (d.u.ent));
      break;
    default:
      error_1 ("inc()", "attempt to increment a non-scalar");
      break;
    }
    break;
  }
  push (d);
}

/* **************************************************************
 * Deccrement a variable
 * ************************************************************** */
void
dec ()
{
  Datum d;
  d = pop ();

  switch (d.type)
  {
  case CONSTANT:
    error_1 ("cannot decrement a constant", 0);
    break;
  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      scalar_Dec (e_data (d.u.ent));
      break;
    default:
      error_1 ("dec()", "attempt to deccrement a non-scalar");
      break;
    }
    break;
  }
  push (d);
}

/* **************************************************************
 * Assign the value of d2 to d1,  d1 = d2
 * d1 must be in the symbol table.
 * d2 must be an acceptable type.
 * ************************************************************** */
Datum
assign (d1, d2)
     Datum d1, d2;
{
  Btree *btree;
  Function *fun;
  Matrix *m;
  Scalar *s;
  String *string;
  Bltin *bltin = 0;

  /*
   * Check for bad LHS values.
   */

  if (d1.type == ENTITY)
  {
    if (e_type (d1.u.ent) == SCALAR || e_type (d1.u.ent) == MATRIX)
      ;
    else if (e_type (d1.u.ent) == BLTIN)
      error_1 ("cannot assign to a built-in function", 0);
    else if (e_type (d1.u.ent) == BTREE)
    {
      if (!strncmp (e_name (d1.u.ent), "$$", 2))
	error_1 ("cannot destroy global symbol table", 0);
    }
  }
  else if ((d1.type == CONSTANT) || (d1.type == iCONSTANT))
  {
    error_1 ("cannot assign a new value to a constant", 0);
  }

  /*
   * Now switch on the RHS value.
   */

  switch (d2.type)		/* RHS */
  {
  case CONSTANT:
    s = scalar_CreateC (d2.u.val, 0.0);
    scalar_SetName (s, cpstr (e_name (d1.u.ent)));
    listNode_DestroyDataOnly (d1.u.ent);
    listNode_AttachData (d1.u.ent, SCALAR, s, scalar_Destroy);
    break;

  case iCONSTANT:
    s = scalar_CreateC (0.0, d2.u.val);
    scalar_SetName (s, cpstr (e_name (d1.u.ent)));
    listNode_DestroyDataOnly (d1.u.ent);
    listNode_AttachData (d1.u.ent, SCALAR, s, scalar_Destroy);
    break;

  case ENTITY:
    switch (e_type (d2.u.ent))
    {
    case SCALAR:
      if (e_name (d2.u.ent) == 0)
	s = (Scalar *) remove_tmp (d2.u.ent);
      else
	s = scalar_Copy ((e_data (d2.u.ent)));

      scalar_SetName (s, cpstr (e_name (d1.u.ent)));
      listNode_DestroyDataOnly (d1.u.ent);
      listNode_AttachData (d1.u.ent, SCALAR, s, scalar_Destroy);
      break;

    case MATRIX:
      if (e_name (d2.u.ent) == 0)
	m = (Matrix *) remove_tmp (d2.u.ent);
      else
	m = matrix_Copy ((e_data (d2.u.ent)));

      matrix_SetName (m, cpstr (e_name (d1.u.ent)));
      listNode_DestroyDataOnly (d1.u.ent);
      listNode_AttachData (d1.u.ent, MATRIX, m, matrix_Destroy);
      break;

    case BTREE:
      if (e_name (d2.u.ent) == 0)
	btree = (Btree *) remove_tmp (d2.u.ent);
      else
      {
	if (!strncmp (e_name (d2.u.ent), "$$", 2))
	  error_1 ("Cannot copy the symbol table", 0);
	btree = btree_CopyTree (e_data (d2.u.ent));
      }
      btree_SetName (btree, cpstr (e_name (d1.u.ent)));
      listNode_DestroyDataOnly (d1.u.ent);
      listNode_AttachData (d1.u.ent, BTREE, btree, btree_Destroy);
      break;

    case STRING:
      if (e_name (d2.u.ent) == 0)
	string = (String *) remove_tmp (d2.u.ent);
      else
	string = string_Copy (e_data (d2.u.ent));

      string_SetName (string, cpstr (e_name (d1.u.ent)));
      listNode_DestroyDataOnly (d1.u.ent);
      listNode_AttachData (d1.u.ent, STRING, string, string_Destroy);
      break;

    case U_FUNCTION:
      if (e_name (d2.u.ent) == 0)
	fun = (Function *) remove_tmp (d2.u.ent);
      else
	fun = function_Copy (e_data (d2.u.ent));

      function_SetName (fun, cpstr (e_name (d1.u.ent)));
      listNode_DestroyDataOnly (d1.u.ent);
      listNode_AttachData (d1.u.ent, U_FUNCTION, fun, function_Destroy);
      break;

    case BLTIN:
      if (e_name (d2.u.ent) == 0)
	bltin = (Bltin *) remove_tmp (d2.u.ent);
      else
	error_1 (e_name (d2.u.ent),
		 "cannot assign existing builtin function");

      bltin->name = cpstr (e_name (d1.u.ent));
      listNode_DestroyDataOnly (d1.u.ent);
      listNode_AttachData (d1.u.ent, BLTIN, bltin, 0);
      break;

    case UNDEF:
      error_1 (e_name (d2.u.ent), "cannot assign undefined variable");
      break;

    default:
      error_1 (e_name (d2.u.ent), "invalid RHS type for assignment");
      break;
    }
    break;
  }
  return (d1);
}

/* **************************************************************
 * Pop top value from stack, and print it. If the object is a
 * matrix then page it to stdout.
 * ************************************************************** */

static FILE *pfile;
extern rpclose _PROTO ((FILE * fp));

void
print (d)
     Datum d;
{
  int fwidth, fprec, write_diary;
  FILE *diary_file_ptr = 0;

  fwidth = get_fwidth ();
  fprec = get_fprec ();

  write_diary = get_write_diary ();
  if (write_diary)
    diary_file_ptr = get_diary_file_ptr ();

  switch (d.type)
  {
  case CONSTANT:
    if (write_diary)
      fprintf (diary_file_ptr, "%*.*g\n", fwidth, fprec, d.u.val);
    fprintf (stdout, "%*.*g\n", fwidth, fprec, d.u.val);
    break;

  case iCONSTANT:
    if (write_diary)
      fprintf (diary_file_ptr, "%*.*gi\n", fwidth, fprec, d.u.val);
    fprintf (stdout, "%*.*gi\n", fwidth, fprec, d.u.val);
    break;

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      if (write_diary)
	scalar_Print (e_data (d.u.ent), diary_file_ptr);
      scalar_Print (e_data (d.u.ent), stdout);
      break;

    case MATRIX:
      if (use_pager)
      {
#ifdef HAVE_PIPE
	/*
	 * Pipe matrix output through the pager. Use a simple rule of
	 * thumb to avoid abusing popen/pclose.
	 */
	if (MNR (e_data (d.u.ent)) <= 8 && MNC (e_data (d.u.ent)) <= 5)
	{
	  if (write_diary)
	    matrix_Print (e_data (d.u.ent), diary_file_ptr);
	  matrix_Print (e_data (d.u.ent), stdout);
	}
	else
	{
	  if (pfile)
	    rpclose (pfile);
	  if ((pfile = popen (pager, "w")) != 0)
	  {
	    if (write_diary)
	      matrix_Print (e_data (d.u.ent), diary_file_ptr);
	    matrix_Print (e_data (d.u.ent), pfile);
	    rpclose (pfile);
	    pfile = 0;		/* normal finish */
	  }
	  else
	  {
	    fprintf (stderr, "ERROR, could not open %s pager for write\n", pager);
	    if (write_diary)
	      matrix_Print (e_data (d.u.ent), diary_file_ptr);
	    matrix_Print (e_data (d.u.ent), stdout);
	  }
	}
#else
	if (write_diary)
	  matrix_Print (e_data (d.u.ent), diary_file_ptr);
	matrix_Print (e_data (d.u.ent), stdout);
#endif /* HAVE_PIPE */

      }
      else
      {
	if (write_diary)
	  matrix_Print (e_data (d.u.ent), diary_file_ptr);
	matrix_Print (e_data (d.u.ent), stdout);
      }
      break;

    case BTREE:
      if (write_diary)
	btree_Print (e_data (d.u.ent), diary_file_ptr);
      btree_Print (e_data (d.u.ent), stdout);
      break;

    case STRING:
      if (write_diary)
	string_Print (e_data (d.u.ent), diary_file_ptr);

      string_Print (e_data (d.u.ent), stdout);
      break;

    case U_FUNCTION:
      if (print_machine)
	diss_assemble (function_GetCodePtr (e_data (d.u.ent)),
		       function_GetCodeSize (e_data (d.u.ent)));
      else
	fprintf (stderr, "\t<user-function>\n");
      break;

    case BLTIN:
      fprintf (stderr, "\t<builtin-function>\n");
      break;

    case UNDEF:
      if (write_diary)
	fprintf (diary_file_ptr, "\tUNDEFINED\n");
      fprintf (stdout, "\tUNDEFINED\n");
      break;

    default:
      error_1 ("Invalid entity type for print-op", 0);
      break;
    }
    break;
  }
  fflush (stdout);
}

/* **************************************************************
 * Evaluate built-in on top of stack. Builtins get their arguments
 * in an array of Datums. A single value is returned through the
 * argument list. The result is NEVER an entity, it is always a
 * data object (SCALAR, VECTOR ...).
 * ************************************************************** */

#define BLTIN_MAX_ARGS  32

void
bltin (sp, n_arg, popf)
     ListNode *sp;
     int n_arg, popf;
{
  int i;
  void (*built_in_func) ();
  Bltin *built_in;
  Datum d, d_args[BLTIN_MAX_ARGS];
  ListNode *ent;
  VPTR return_ptr;

  ent = sp;

  built_in = (Bltin *) listNode_GetData (ent);
  built_in_func = ((Bltin *) built_in)->func;

  /* pop args off stack and install in Datum array */
  if (n_arg > BLTIN_MAX_ARGS)
    error_1 (built_in->name, "Too many arguments to a built-in");
  for (i = n_arg - 1; i >= 0; i--)
    d_args[i] = pop ();

  if (popf)
    extern_pop();

  /* call built-in function */
  (*built_in_func) (&return_ptr, n_arg, d_args);

  /*
   * Check the return ptr, if it is a ListNode, then do NOT
   * install it on the temp list, just push it on the stack.
   */

  if ((int) *((int *) return_ptr) == LISTNODE)
  {
    /* Set the type correctly and push on the stack */
    int return_type;
    ListNode *ltmp;
    VPTR data;

    ltmp = (ListNode *) return_ptr;
    data = ((ListNode *) return_ptr)->data;
    return_type = (int) *((int *) data);
    ltmp->type = return_type;
    d.u.ent = (ListNode *) return_ptr;
    d.type = ENTITY;
    push (d);
  }
  else
  {
    /* The built-in function should ALWAYS return something */
    d.u.ent = install_tmp_switch ((int) *((int *) return_ptr), return_ptr);
    d.type = ENTITY;
    push (d);
  }
}

/* **************************************************************
 * Evaluate while construct
 * ************************************************************** */
void
whilecode ()
{
  double d_val = 0.0;
  Datum d;
  Inst *bodyptr, *endptr, *condptr;
  Matrix *m;

  /* Calc ptrs to various parts of stmt using offsets */
  bodyptr = (pc - 1) + (*pc).op_code;
  endptr = (pc - 1) + (*(pc + 1)).op_code;
  condptr = pc + 2;

  execute (condptr);		/* condition */
  d = pop ();

  /* Eval while argument */
  switch (d.type)
  {
  case CONSTANT:
  case iCONSTANT:
    d_val = d.u.val;
    break;
  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      d_val = SVALr (e_data (d.u.ent));
      break;
    case MATRIX:
      m = (Matrix *) e_data (d.u.ent);
      if (MNR (m) == 1 && MNC (m) == 1)
      {
	if (MTYPE (m) == REAL)
	  d_val = MAT (m, 1, 1);
	else if (MTYPE (m) == COMPLEX)
	  d_val = MATr (m, 1, 1);
	else
	  error_1 ("string matrix not allowed as while() argument", 0);
      }
      else if (MNR (m) == 0 && MNC (m) == 0)
      {
	d_val = 0.0;
      }
      else
	error_1 ("N > 1, matrix not allowed as while() argument", 0);
      break;
    default:
      error_1 ("Illegal object for while() argument", 0);
    }
    remove_tmp_destroy (d.u.ent);	/* Clean up garbage list */
    break;
  }

  while (d_val)
  {
    execute (bodyptr);		/* loop body */
    if (returning)
      break;
    if (breaking)
    {
      breaking = 0;
      break;
    }
    if (continuing)
      continuing = 0;

    execute (condptr);		/* condition */
    d = pop ();


    /* Eval while argument */
    switch (d.type)
    {
    case CONSTANT:
    case iCONSTANT:
      d_val = d.u.val;
      break;
    case ENTITY:
      switch (e_type (d.u.ent))
      {
      case SCALAR:
	d_val = SVALr (e_data (d.u.ent));
	break;
      case MATRIX:
	m = (Matrix *) e_data (d.u.ent);
	if (MNR (m) == 1 && MNC (m) == 1)
	{
	  if (MTYPE (m) == REAL)
	    d_val = MAT (m, 1, 1);
	  else if (MTYPE (m) == COMPLEX)
	    d_val = MATr (m, 1, 1);
	  else
	    error_1 ("string matrix not allowed as while() argument", 0);
	}
	else if (MNR (m) == 0 && MNC (m) == 0)
	{
	  d_val = 0.0;
	}
	else
	  error_1 ("N > 1, matrix not allowed as while() argument", 0);
	break;
      default:
	error_1 ("Illegal object for while() argument", 0);
      }
      remove_tmp_destroy (d.u.ent);	/* Clean up garbage list */
      break;
    }
  }

  if (!returning)
    pc = endptr;		/* next statement */
}

/* **************************************************************
 * Evaluate for construct.
 * *(savepc+1) == offset to body of loop
 * *(savepc+2) == offset end of loop
 * Form for loop, loop over for_loop_body length(vec_expr)
 * times. Set the value of var to the value in the ith
 * element of vec_expr.
 *
 * for ( var in vec_expr ) { body... }
 *
 * ************************************************************** */
void
forcode ()
{
  int i, limit;
  Datum d_var, d_vec_expr;
  ListNode *var, *vec_expr;
  Inst *bodyptr, *endptr, *condptr;

  limit = 0;
  vec_expr = 0;
  bodyptr = (pc - 1) + (*pc).op_code;
  endptr = (pc - 1) + (*(pc + 1)).op_code;
  condptr = pc + 2;

  /* execute code untill loop body */
  execute (condptr);

  /*
   * Then pop top two elements off of the stack.
   * These are the var, and the vec_expr
   * from:  for( var in vec_expr )
   */

  d_vec_expr = pop ();
  d_var = pop ();

  /* get var name, address */
  var = d_var.u.ent;

  /*
   * Get vector_expression entity.
   * Make sure it is an ENTITY.
   */

  if (d_vec_expr.type == ENTITY)
    vec_expr = d_vec_expr.u.ent;
  else if (d_vec_expr.type == CONSTANT)
  {
    vec_expr = install_tmp (SCALAR,
			    scalar_Create (d_vec_expr.u.val),
			    scalar_Destroy);
  }
  else if (d_vec_expr.type == iCONSTANT)
  {
    vec_expr = install_tmp (SCALAR,
			    scalar_CreateC (0.0, d_vec_expr.u.val),
			    scalar_Destroy);
  }

  /*
   * Now check the ENTITY for proper markings.
   */

  if (listNode_GetKey (vec_expr) != 0)
  {
    /*
     * Copy the vector. We have to do this in case the program
     * changes the vector in mid-loop.
     */
    ListNode *tmp = d_vec_expr.u.ent;
    if (e_type (tmp) == MATRIX)
    {
      vec_expr = install_tmp (MATRIX,
			      matrix_Copy (e_data (tmp)),
			      matrix_Destroy);
    }
    else if (e_type (tmp) == SCALAR)
    {
      vec_expr = install_tmp (SCALAR,
			      scalar_CreateC (SVALr (e_data (tmp)),
					      SVALi (e_data (tmp))),
			      scalar_Destroy);
    }
    else if (e_type (tmp) == STRING)
    {
      /* Make this into a 1-by-1 string matrix */
      Matrix *mtmp;
      mtmp = matrix_CreateS (1, 1);
      MATs (mtmp, 1, 1) = cpstr (string_GetString (e_data (tmp)));
      vec_expr = install_tmp (MATRIX, mtmp, matrix_Destroy);
    }
    else
      error_1 (e_name (tmp), "invalid type for-loop vector");
  }

  /*
   * Set the loop vector's name so it doesn't get cleaned.
   * Start the name with something users can't use.
   */

  listNode_SetKey (vec_expr, cpstr ("-LOOP-VECTOR"));

  /*
   * vec_expr can be a scalar or a vector.
   * Get the loop limit.
   */

  if (e_type (vec_expr) == MATRIX)
    limit = (int) (MNR (e_data (vec_expr)) * MNC (e_data (vec_expr)));
  else if (e_type (vec_expr) == SCALAR)
    limit = 1;
  else if (e_type (vec_expr) == STRING)
    limit = 1;
  else
    error_1 ("invalid vector expression used in for statement", 0);

  /* Check limit for no-execution condition */
  if (limit == 0)
  {
    pc = endptr;
    return;
  }

  /*
   * If the vector_expr evaluates to a SCALAR, then
   * execute the loop once with the var set to the
   * scalar value.
   */

  if (e_type (vec_expr) == SCALAR)
  {
    /* force var to be a scalar */
    if (e_type (var) != SCALAR)
    {
      listNode_DestroyDataOnly (var);
      listNode_AttachData (var, SCALAR, scalar_Create (0.0), scalar_Destroy);
      scalar_SetName (e_data (var), cpstr (e_name (var)));
    }

    for (i = limit; i <= limit; i++)
    {
      scalar_SetValC (e_data (var), SVALr (e_data (vec_expr)),
		      SVALi (e_data (vec_expr)));
      execute (bodyptr);	/* loop body */
      if (returning)
	break;
      if (breaking)
      {
	breaking = 0;
	break;
      }
      if (continuing)
	continuing = 0;
    }
  }
  else
  {
    /* Normal execution of for loop */

    if (MTYPE (e_data (vec_expr)) == REAL)
    {
      /* force var to be a scalar */
      if (e_type (var) != SCALAR)
      {
	listNode_DestroyDataOnly (var);
	listNode_AttachData (var, SCALAR, scalar_Create (0.0), scalar_Destroy);
	scalar_SetName (e_data (var), cpstr (e_name (var)));
      }
      for (i = 1; i <= limit; i++)
      {
	SVALr (e_data (var)) = MATrv (e_data (vec_expr), i - 1);

	execute (bodyptr);	/* loop body */
	if (returning)
	  break;
	if (breaking)
	{
	  breaking = 0;
	  break;
	}
	if (continuing)
	  continuing = 0;
      }
    }
    else if (MTYPE (e_data (vec_expr)) == COMPLEX)
    {
      /* force var to be a scalar */
      if (e_type (var) != SCALAR)
      {
	listNode_DestroyDataOnly (var);
	listNode_AttachData (var, SCALAR, scalar_Create (0.0), scalar_Destroy);
	scalar_SetName (e_data (var), cpstr (e_name (var)));
      }
      for (i = 1; i <= limit; i++)
      {
	SVALr (e_data (var)) = MATcvr (e_data (vec_expr), i - 1);
	SVALi (e_data (var)) = MATcvi (e_data (vec_expr), i - 1);

	execute (bodyptr);	/* loop body */
	if (returning)
	  break;
	if (breaking)
	{
	  breaking = 0;
	  break;
	}
	if (continuing)
	  continuing = 0;
      }
    }
    else if (MTYPE (e_data (vec_expr)) == STRING)
    {
      /* force var to be a string */
      if (e_type (var) != STRING)
      {
	listNode_DestroyDataOnly (var);
	listNode_AttachData (var, STRING, string_Create (0), string_Destroy);
	string_SetName (e_data (var), cpstr (e_name (var)));
      }
      for (i = 1; i <= limit; i++)
      {
	string_Set (e_data (var),
		    cpstr (MATsv (e_data (vec_expr), i - 1)));
	execute (bodyptr);	/* loop body */
	if (returning)
	  break;
	if (breaking)
	{
	  breaking = 0;
	  break;
	}
	if (continuing)
	  continuing = 0;
      }
    }
  }

  /* Now destroy the loop-vector */
  listNode_SetKey (vec_expr, 0);
  remove_tmp_destroy (vec_expr);

  if (!returning)
    pc = endptr;
}

/* **************************************************************
 * Evaluate if construct
 * ************************************************************** */
void
ifcode ()
{
  double d_val = 0.0;
  Datum d;
  Inst *bodyptr, *endptr, *condptr, *elseptr;
  Matrix *m;

  /* Set up ptrs and do a little error checking */
  if ((*pc).op_code == 0)
    error_1 ("Invalid ifcode offset", 0);
  bodyptr = (pc - 1) + (*pc).op_code;

  if ((*(pc + 1)).op_code)
    elseptr = (pc - 1) + (*(pc + 1)).op_code;
  else
    elseptr = (Inst *) 0;

  if ((*(pc + 2)).op_code == 0)
    error_1 ("Invalid ifcode offset", 0);
  endptr = (pc - 1) + (*(pc + 2)).op_code;
  condptr = pc + 3;

  /*Inst *savepc = pc; */

  execute (condptr);		/* condition */
  d = pop ();

  /*
   * We can only eval scalars and constants in a conditional stmt.
   */
  switch (d.type)
  {
  case CONSTANT:
  case iCONSTANT:
    d_val = d.u.val;
    break;
  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      d_val = SVALr (e_data (d.u.ent));
      break;
    case MATRIX:
      m = (Matrix *) e_data (d.u.ent);
      if (MNR (m) == 1 && MNC (m) == 1)
      {
	if (MTYPE (m) == REAL)
	  d_val = MAT (m, 1, 1);
	else if (MTYPE (m) == COMPLEX)
	  d_val = MATr (m, 1, 1);
	else
	  error_1 ("string matrix not allowed as if() argument", 0);
      }
      else if (MNR (m) == 0 && MNC (m) == 0)
      {
	d_val = 0.0;
      }
      else
	error_1 ("N > 1, matrix not allowed as if() argument", 0);
      break;
    default:
      error_1 ("Illegal object for if() argument", 0);
    }
    remove_tmp_destroy (d.u.ent);	/* Clean up garbage list */
    break;
  }

  if (d_val)
    execute (bodyptr);
  else if (elseptr)		/* else part */
    execute (elseptr);

  /* If we've seen any of the following leave the code pointer
     pointing to a STOP */
  if (!returning && !breaking && !continuing)
    pc = endptr;		/* next stmt */
}

/* **************************************************************
 * Create an list. The next instruction contains the number of
 * items on the stack that we need to install on the new list.
 * Either we are moving ptrs, as in: NAME '=' vec_expr, or we
 * are copying the data as in: vec_expr. There are three possibilities
 * for listNode keys...
 * "-"      Result of NAME '=' vec_expr: Use NAME as index
 * NULL ptr Result of vec_expr: copy and assign numerical index
 * NAME     Result of vec_expr: copy and assign numerical index
 * ************************************************************** */

void
list_create ()
{
  char jstr[20], *tmp;
  int i, j, n;
  Datum d, new_list;
  ListNode *lnode;

  new_list.u.ent = install_tmp (BTREE, btree_Create (), btree_Destroy);
  new_list.type = ENTITY;

  /* Get the # of items on the stack */
  n = (*pc++).op_code;

  j = n;
  /* Now pop each entity off the stack */
  for (i = 0; i < n; i++)
  {
    d = pop ();
    if (d.type == CONSTANT || d.type == iCONSTANT)
    {
      if (d.type == CONSTANT)
      {
	d.u.ent = install_tmp (SCALAR, scalar_CreateC (d.u.val, 0.0),
			       scalar_Destroy);
	d.type = ENTITY;
      }
      else if (d.type == iCONSTANT)
      {
	d.u.ent = install_tmp (SCALAR, scalar_CreateC (0.0, d.u.val),
			       scalar_Destroy);
	d.type = ENTITY;
      }
    }
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      if (e_name (d.u.ent) == 0 || strcmp ("-", e_name (d.u.ent)))
      {
	sprintf (jstr, "%d", j--);
	lnode = install (e_data (new_list.u.ent),
			 cpstr (jstr),
			 e_type (d.u.ent), scalar_Copy (e_data (d.u.ent)));
	scalar_SetName (e_data (lnode), cpstr (jstr));
      }
      else
      {
	tmp = scalar_GetName (e_data (d.u.ent));
	lnode = install (e_data (new_list.u.ent),
			 cpstr (tmp),
			 e_type (d.u.ent), e_data (d.u.ent));
	remove_tmp (d.u.ent);
	j--;
      }
      break;
    case MATRIX:
      if (e_name (d.u.ent) == 0 || strcmp ("-", e_name (d.u.ent)))
      {
	sprintf (jstr, "%d", j--);
	lnode = install (e_data (new_list.u.ent),
			 cpstr (jstr),
			 e_type (d.u.ent), matrix_Copy (e_data (d.u.ent)));
	matrix_SetName (e_data (lnode), cpstr (jstr));
      }
      else
      {
	tmp = matrix_GetName (e_data (d.u.ent));
	lnode = install (e_data (new_list.u.ent),
			 cpstr (tmp),
			 e_type (d.u.ent), e_data (d.u.ent));
	remove_tmp (d.u.ent);
	j--;
      }
      break;
    case STRING:
      if (e_name (d.u.ent) == 0 || strcmp ("-", e_name (d.u.ent)))
      {
	sprintf (jstr, "%d", j--);
	lnode = install (e_data (new_list.u.ent),
			 cpstr (jstr),
			 e_type (d.u.ent), string_Copy (e_data (d.u.ent)));
	string_SetName (e_data (lnode), cpstr (jstr));
      }
      else
      {
	tmp = string_GetName (e_data (d.u.ent));
	lnode = install (e_data (new_list.u.ent),
			 cpstr (tmp),
			 e_type (d.u.ent), e_data (d.u.ent));
	remove_tmp (d.u.ent);
	j--;
      }
      break;
    case BTREE:
      if (e_name (d.u.ent) == 0 || strcmp ("-", e_name (d.u.ent)))
      {
	sprintf (jstr, "%d", j--);
	lnode = install (e_data (new_list.u.ent),
			 cpstr (jstr),
			 e_type (d.u.ent), btree_CopyTree (e_data (d.u.ent)));
	btree_SetName (e_data (lnode), cpstr (jstr));
      }
      else
      {
	tmp = btree_GetName (e_data (d.u.ent));
	lnode = install (e_data (new_list.u.ent),
			 cpstr (tmp),
			 e_type (d.u.ent), e_data (d.u.ent));
	remove_tmp (d.u.ent);
	j--;
      }
      break;
    case U_FUNCTION:
      if (e_name (d.u.ent) == 0 || strcmp ("-", e_name (d.u.ent)))
      {
	sprintf (jstr, "%d", j--);
	lnode = install (e_data (new_list.u.ent),
			 cpstr (jstr),
			 e_type (d.u.ent), function_Copy (e_data (d.u.ent)));
	function_SetName (e_data (lnode), cpstr (jstr));
      }
      else
      {
	tmp = function_GetName (e_data (d.u.ent));
	lnode = install (e_data (new_list.u.ent),
			 cpstr (tmp),
			 e_type (d.u.ent), e_data (d.u.ent));
	remove_tmp (d.u.ent);
	j--;
      }
      break;
    case UNDEF:
      error_1 (e_name (d.u.ent), "UNDEFINED list member");
      break;
    default:
      error_1 ("something terribly wrong with list creation", 0);
      break;
    }
  }
  push (new_list);
}

/* **************************************************************
 * Resolve a list member reference.
 * code 1: The argument is an expr
 * code 2: The argument is a string (NAME)
 * Either way the stack must be popped to get the variable
 * id pointer, and maybe the vec_expr.
 * ************************************************************** */
void
list_member ()
{
  char name[100];
  int code;
  Datum new, vexpr, vid;

  code = (*pc++).op_code;

  if (code == 1)
    vexpr = pop ();
  vid = pop ();

  if (code == 1)
  {
    /*
     * An expr was used as index.
     * Evaluate the expr, and convert result to a string
     */
    switch (vexpr.type)
    {
    case CONSTANT:
      sprintf (name, "%.6g", vexpr.u.val);
      break;
    case iCONSTANT:
      sprintf (name, "%.6gi", vexpr.u.val);
      break;
    case ENTITY:
      switch (e_type (vexpr.u.ent))
      {
      case SCALAR:
	sprintf (name, "%.6g", SVALr (e_data (vexpr.u.ent)));
	break;
      case MATRIX:
	if (MTYPE (e_data (vexpr.u.ent)) == REAL)
	{
	  sprintf (name, "%.6g", MAT (e_data (vexpr.u.ent), 1, 1));
	}
	else if (MTYPE (e_data (vexpr.u.ent)) == COMPLEX)
	{
	  sprintf (name, "%.6g", MATr (e_data (vexpr.u.ent), 1, 1));
	}
	else if (MTYPE (e_data (vexpr.u.ent)) == STRING)
	{
	  strcpy (name, MATs (e_data (vexpr.u.ent), 1, 1));
	}
	break;
      case STRING:
	strcpy (name, string_GetString (e_data (vexpr.u.ent)));
	break;
      case UNDEF:
	error_1 (e_name (vexpr.u.ent), "index UNDEFINED");
	break;
      default:
	error_1 (e_name (vexpr.u.ent), "invalid type for list index");
	break;
      }
    }
  }
  else if (code == 2)
  {
    /* list.NAME was used to index. Use name to index into list */
    strcpy (name, (char *) (*pc++).ptr);
  }

  /*
   * Now check type of root entity and extract the proper
   * object. The root entity can be a normal object like
   * a matrix, in this case we look for certain predefined
   * object members.
   */

  switch (vid.type)
  {
  case CONSTANT:
  case iCONSTANT:
    if (!strcmp (name, "class"))
    {
      String *str = string_Create (cpstr ("num"));
      new.u.ent = install_tmp (STRING, str, string_Destroy);
    }
    else if (!strcmp (name, "n"))
    {
      int nc = 1;
      new.u.ent = install_tmp (SCALAR, scalar_Create ((double) nc),
			       scalar_Destroy);
    }
    else if (!strcmp (name, "nr"))
    {
      new.u.ent = install_tmp (SCALAR, scalar_Create (1.0),
			       scalar_Destroy);
    }
    else if (!strcmp (name, "nc"))
    {
      new.u.ent = install_tmp (SCALAR, scalar_Create (1.0),
			       scalar_Destroy);
    }
    else if (!strcmp (name, "type"))
    {
      if (vid.type == CONSTANT)
      {
	String *str = string_Create (cpstr ("real"));
	new.u.ent = install_tmp (STRING, str, string_Destroy);
      }
      else if (vid.type == iCONSTANT)
      {
	String *str = string_Create (cpstr ("complex"));
	new.u.ent = install_tmp (STRING, str, string_Destroy);
      }
    }
    else
      error_1 ("entity does not contain that attribute", 0);
    break;

  case ENTITY:
    switch (e_type (vid.u.ent))
    {
    case SCALAR:
      if (!strcmp (name, "nr"))
      {
	new.u.ent = install_tmp (SCALAR, scalar_Create (1.0),
				 scalar_Destroy);
      }
      else if (!strcmp (name, "nc"))
      {
	new.u.ent = install_tmp (SCALAR, scalar_Create (1.0),
				 scalar_Destroy);
      }
      else if (!strcmp (name, "n"))
      {
	int nc = 1;
	new.u.ent = install_tmp (SCALAR, scalar_Create ((double) nc),
				 scalar_Destroy);
      }
      else if (!strcmp (name, "class"))
      {
	String *str = string_Create (cpstr ("num"));
	new.u.ent = install_tmp (STRING, str, string_Destroy);
      }
      else if (!strcmp (name, "type"))
      {
	if (SVALi (e_data (vid.u.ent)) == 0.0)
	{
	  String *str = string_Create (cpstr ("real"));
	  new.u.ent = install_tmp (STRING, str, string_Destroy);
	}
	else
	{
	  String *str = string_Create (cpstr ("complex"));
	  new.u.ent = install_tmp (STRING, str, string_Destroy);
	}
      }
      else
	error_1 (e_name (vid.u.ent),
		 "Scalar entity does not contain that attribute");

      remove_tmp_destroy (vid.u.ent);
      break;

    case MATRIX:
      if (!strcmp (name, "nr"))
      {
	int nr = MNR (e_data (vid.u.ent));
	new.u.ent = install_tmp (SCALAR, scalar_Create ((double) nr),
				 scalar_Destroy);
      }
      else if (!strcmp (name, "nc"))
      {
	int nc = MNC (e_data (vid.u.ent));
	new.u.ent = install_tmp (SCALAR, scalar_Create ((double) nc),
				 scalar_Destroy);
      }
      else if (!strcmp (name, "n"))
      {
	int nc = MNR (e_data (vid.u.ent)) * MNC (e_data (vid.u.ent));
	new.u.ent = install_tmp (SCALAR, scalar_Create ((double) nc),
				 scalar_Destroy);
      }
      else if (!strcmp (name, "class"))
      {
	if (MTYPE (e_data (vid.u.ent)) != STRING)
	{
	  String *str = string_Create (cpstr ("num"));
	  new.u.ent = install_tmp (STRING, str, string_Destroy);
	}
	else
	{
	  String *str = string_Create (cpstr ("string"));
	  new.u.ent = install_tmp (STRING, str, string_Destroy);
	}
      }
      else if (!strcmp (name, "type"))
      {
	if (MTYPE (e_data (vid.u.ent)) == REAL)
	{
	  String *str = string_Create (cpstr ("real"));
	  new.u.ent = install_tmp (STRING, str, string_Destroy);
	}
	else if (MTYPE (e_data (vid.u.ent)) == COMPLEX)
	{
	  String *str = string_Create (cpstr ("complex"));
	  new.u.ent = install_tmp (STRING, str, string_Destroy);
	}
	else if (MTYPE (e_data (vid.u.ent)) == STRING)
	{
	  String *str = string_Create (cpstr (""));
	  new.u.ent = install_tmp (STRING, str, string_Destroy);
	}
      }
      else
	error_1 (e_name (vid.u.ent),
		 "Matrix entity does not contain that attribute");

      remove_tmp_destroy (vid.u.ent);
      break;

    case STRING:
      if (!strcmp (name, "l"))
      {
	int l = string_GetLength (e_data (vid.u.ent));
	new.u.ent = install_tmp (SCALAR, scalar_Create ((double) l),
				 scalar_Destroy);
      }
      else if (!strcmp (name, "class"))
      {
	String *str = string_Create (cpstr ("string"));
	new.u.ent = install_tmp (STRING, str, string_Destroy);
      }
      else if (!strcmp (name, "type"))
      {
	String *str = string_Create (cpstr ("string"));
	new.u.ent = install_tmp (STRING, str, string_Destroy);
      }
      else if (!strcmp (name, "nr"))
      {
	int nr = 1;
	new.u.ent = install_tmp (SCALAR, scalar_Create ((double) nr),
				 scalar_Destroy);
      }
      else if (!strcmp (name, "nc"))
      {
	int nc = 1;
	new.u.ent = install_tmp (SCALAR, scalar_Create ((double) nc),
				 scalar_Destroy);
      }
      else if (!strcmp (name, "n"))
      {
	int nc = 1;
	new.u.ent = install_tmp (SCALAR, scalar_Create ((double) nc),
				 scalar_Destroy);
      }
      else
	error_1 (e_name (vid.u.ent), "entity does not contain that attribute");

      remove_tmp_destroy (vid.u.ent);
      break;

    case BTREE:
      if ((new.u.ent = btree_FindNode (e_data (vid.u.ent), name)) == 0)
      {
	/* Create an UNDEF entity */
	new.u.ent = listNode_Create ();
	listNode_SetKey (new.u.ent, cpstr (name));
	listNode_SetType (new.u.ent, UNDEF);
	btree_AddNode (e_data (vid.u.ent), new.u.ent);
      }
      else
      {

	/*
	 * We found it, now clean up the parent list if possible.
	 */

	if (e_name (vid.u.ent) == 0)
	{
	  ListNode *ltmp = listNode_Create ();
	  ltmp = install_tmp (e_type (new.u.ent), e_data (new.u.ent), 
			      e_dtor (new.u.ent));
	  e_data (new.u.ent) = 0;
	  e_dtor (new.u.ent) = 0;
	  remove_tmp_destroy (vid.u.ent);

	  new.u.ent = ltmp;
	  new.type = ENTITY;
	}
      }
      break;

    case U_FUNCTION:
      if (!strcmp (name, "class"))
      {
	String *str = string_Create (cpstr ("function"));
	new.u.ent = install_tmp (STRING, str, string_Destroy);
      }
      else if (!strcmp (name, "type"))
      {
	String *str = string_Create (cpstr ("user"));
	new.u.ent = install_tmp (STRING, str, string_Destroy);
      }
      else
	error_1 (e_name (vid.u.ent),
		 "User-function does not contain that attribute");

      remove_tmp_destroy (vid.u.ent);
      break;

    case BLTIN:
      if (!strcmp (name, "class"))
      {
	String *str = string_Create (cpstr ("function"));
	new.u.ent = install_tmp (STRING, str, string_Destroy);
      }
      else if (!strcmp (name, "type"))
      {
	String *str = string_Create (cpstr ("builtin"));
	new.u.ent = install_tmp (STRING, str, string_Destroy);
      }
      else
	error_1 (e_name (vid.u.ent), "entity does not contain that attribute");

      remove_tmp_destroy (vid.u.ent);
      break;

    case UNDEF:
      error_1 (e_name (vid.u.ent), "list root UNDEFINED");
      break;
    }
    break;

  default:
    error_1 (e_name (vid.u.ent), "invalid type for list member reference");
    break;
  }

  new.type = ENTITY;
  push (new);
}

void
list_assign ()
{
  char name[100], *sname;
  int code;
  Datum d, new, vexpr, vid_d;
  Btree *btree;
  ListNode *new_node;

  code = (*pc++).op_code;	/* Flag that tells us to use an
				   expression index, or a NAME
				   index */

  d = pop ();			/* The RHS */
  if (code == 1)
    vexpr = pop ();		/* The index */
  vid_d = pop ();		/* List variable */

  /*
   * If vid_d is not a list already, create the list.
   * But, do not overwrite the old object, yet.
   */
  if (e_type (vid_d.u.ent) != BTREE)
  {
    btree = btree_Create ();
    btree_SetName (btree, cpstr (e_name (vid_d.u.ent)));
  }
  else
  {
    btree = (Btree *) e_data (vid_d.u.ent);
  }

  if (code == 1)
  {
    /*
     * A vec_expr was used as index.
     * Evaluate the vexpr, and convert result to a string
     */
    switch (vexpr.type)
    {
    case CONSTANT:
      sprintf (name, "%.6g", vexpr.u.val);
      break;
    case iCONSTANT:
      sprintf (name, "%.6gi", vexpr.u.val);
      break;
    case ENTITY:
      switch (e_type (vexpr.u.ent))
      {
      case SCALAR:
	sprintf (name, "%.6g", SVALr (e_data (vexpr.u.ent)));
	break;
      case MATRIX:
	if (MTYPE (e_data (vexpr.u.ent)) == REAL)
	{
	  sprintf (name, "%.6g", MAT (e_data (vexpr.u.ent), 1, 1));
	}
	else if (MTYPE (e_data (vexpr.u.ent)) == COMPLEX)
	{
	  sprintf (name, "%.6g", MATr (e_data (vexpr.u.ent), 1, 1));
	}
	else if (MTYPE (e_data (vexpr.u.ent)) == STRING)
	{
	  strcpy (name, MATs (e_data (vexpr.u.ent), 1, 1));
	}
	break;
      case STRING:
	sname = string_GetString (e_data (vexpr.u.ent));
	strcpy (name, sname);
	break;
      case UNDEF:
	error_1 (e_name (vexpr.u.ent), "list index UNDEFINED");
	break;
      default:
	error_1 (e_name (vexpr.u.ent), "invalid type for list index");
	break;
      }
      remove_tmp_destroy (vexpr.u.ent);		/* Clean up garbage list */
    }
  }
  else if (code == 2)
  {
    /* list.NAME was used to index list */
    sname = (char *) (*pc++).ptr;
    strcpy (name, sname);
  }

  if ((new.u.ent = btree_FindNode (btree, name)) == 0)
  {
    new_node = listNode_Create ();
    listNode_AttachData (new_node, UNDEF, 0, 0);
    listNode_SetKey (new_node, cpstr (name));
    btree_AddNode (btree, new_node);
    new.u.ent = new_node;
  }
  new.type = ENTITY;

  /* Now do the assign */
  new = assign (new, d);

  /* Now overwrite the old LHS if it was NOT a LIST */

  if (e_type (vid_d.u.ent) != BTREE)
  {
    listNode_DestroyDataOnly (vid_d.u.ent);
    listNode_AttachData (vid_d.u.ent, BTREE, btree, btree_Destroy);
  }

  /* Push the LHS back on the stack */
  push (vid_d);
}


/* **************************************************************
 * Given: NAME '=' vec_expr
 * Copy the vec_expr into a ListNode with name NAME, and push
 * the ListNode back on the stack to be inserted into a LIST later.
 * ************************************************************** */
void
list_el_create ()
{
  char *name;
  Datum rhs;
  ListNode *lnode;

  lnode = 0;

  /* Get the list-var-name from the machine */
  name = cpstr ((*pc++).ptr);

  /* Pop the rhs from the stack */
  rhs = pop ();

  /* Create (copy) the new object */
  switch (rhs.type)
  {
  case CONSTANT:
    lnode = install_tmp (SCALAR, scalar_Create (rhs.u.val), scalar_Destroy);
    scalar_SetName (e_data (lnode), name);
    break;
  case iCONSTANT:
    lnode = install_tmp (SCALAR, scalar_CreateC (0.0, rhs.u.val),
			 scalar_Destroy);
    scalar_SetName (e_data (lnode), name);
    break;
  case ENTITY:
    switch (e_type (rhs.u.ent))
    {
    case SCALAR:
      lnode = install_tmp (SCALAR, scalar_Copy (e_data (rhs.u.ent)),
			   scalar_Destroy);
      scalar_SetName (e_data (lnode), name);
      break;
    case MATRIX:
      lnode = install_tmp (MATRIX, matrix_Copy (e_data (rhs.u.ent)),
			   matrix_Destroy);
      matrix_SetName (e_data (lnode), name);
      break;
    case STRING:
      lnode = install_tmp (STRING, string_Copy (e_data (rhs.u.ent)),
			   string_Destroy);
      string_SetName (e_data (lnode), name);
      break;
    case BTREE:
      lnode = install_tmp (BTREE, btree_CopyTree (e_data (rhs.u.ent)),
			   btree_Destroy);
      btree_SetName (e_data (lnode), name);
      break;
    case U_FUNCTION:
      lnode = install_tmp (U_FUNCTION, function_Copy (e_data (rhs.u.ent)),
			   function_Destroy);
      string_SetName (e_data (lnode), name);
      break;
    case UNDEF:
      error_1 (e_name (rhs.u.ent), "cannot create UNDEFINED list member");
      break;
    default:
      error_1 (e_name (rhs.u.ent), "invalid type for LIST");
    }
    /* Try and clean-up the rhs ENTITY if possible */
    remove_tmp_destroy (rhs.u.ent);
    break;
  }
  /*
   * Give the new element a UNIQUE name, so we can identify
   * it in list_create().
   */
  listNode_SetKey (lnode, cpstr ("-"));

  /* Push the new object on the stack */
  rhs.u.ent = lnode;
  rhs.type = ENTITY;
  push (rhs);
}

/* **************************************************************
 * Find the line number ascociated with the current error. Back
 * up from the current program instruction untill we find a
 * OP_LINE_NO.
 * ************************************************************** */
int
find_lineno ()
{
  Inst *p;

  p = pc;
  if (!p)
    return (1);			/* This is very rare */
  while ((*p).op_code != OP_LINE_NO)
  {
    if ((*p).op_code == OP_FILE_NAME)
      return 1;
    p--;
  }
  return ((*(p + 1)).op_code - loff);
}

char *
find_file_name ()
{
  char *fptr;

  if ((fp - frame) == 0)
  {
    fptr = (char *) program->prog[1].ptr;
  }
  else
  {
    fptr = (char *) ((Function *) ((ListNode *) fp->sp)->data)->code[1].ptr;
  }
  return (fptr);
}

/* **************************************************************
 * We leave RLaB via this indirect route in order to make `quit'
 * an executable statement. Also we try and clean up memory
 * in order to make debugging memory leaks easier.
 * ************************************************************** */
void
quit_code ()
{
  FREE (program->prog);
  FREE (frame);
  FREE (stack);

  delete_symbol_table ();
  scanner_cleanup ();
  destroy_file_list ();
  destroy_fstatic_tree ();

#ifdef MALLOC_DUMP
  malloc_dump (2);
#endif

  exit (1);
}

/*
 * Copy an argument off of the stack
 */

static VPTR
copy_arg (cnode, name)
     ListNode *cnode;
     char *name;
{
  int offset;
  Datum cvar;
  ListNode *copy;
  Scalar *s;
  String *str;
  Matrix *m;
  Btree *btr;
  Bltin *bltptr;
  Function *fptr;

  /*
   * Find the arg on the stack.
   * We have the node, get the offset,
   * and find the pos on the stack.
   */

  offset = lvar_GetOffset (e_data (cnode));
  cvar = fp->argn[offset - fp->nargs];

  copy = listNode_Create ();

  switch (e_type (cvar.u.ent))
  {
  case SCALAR:
    s = scalar_Copy (e_data (cvar.u.ent));
    scalar_SetName (s, name);
    return ((VPTR) s);
    break;
  case MATRIX:
    m = matrix_Copy (e_data (cvar.u.ent));
    matrix_SetName (m, name);
    return ((VPTR) m);
    break;
  case STRING:
    str = string_Copy (e_data (cvar.u.ent));
    string_SetName (str, name);
    return ((VPTR) str);
    break;
  case BTREE:
    btr = btree_CopyTree (e_data (cvar.u.ent));
    btree_SetName (btr, name);
    return ((VPTR) btr);
    break;
  case UNDEF:
    return (0);
  case U_FUNCTION:
    fptr = function_Copy (e_data (cvar.u.ent));
    function_SetName (fptr, name);
    return ((VPTR) fptr);
    break;
  case BLTIN:
    bltptr = (Bltin *) e_data (cvar.u.ent);
    return ((VPTR) bltptr);
    break;
  default:
    return (0);
    break;
  }
  return (0);
}
