/* bltin2.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1994  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "symbol.h"
#include "mem.h"
#include "list.h"
#include "btree.h"
#include "bltin.h"
#include "scop1.h"
#include "matop1.h"
#include "matop2.h"
#include "r_string.h"
#include "util.h"
#include "mathl.h"
#include "function.h"
#include "print.h"

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#ifdef THINK_C
char *getpref (char *);
#define getenv(env_name) getpref(env_name)
#endif

/* scan.l */
extern int new_file _PROTO ((char *file_name));

#define rabs(x) ((x) >= 0 ? (x) : -(x))

/*
 * Generate a NaN or an Inf.
 */

static const unsigned char __nan[8] = r__nan_bytes;
#define	R_NAN	(*(const double *) __nan)

static const unsigned char __inf_val[8] = r__inf_val_bytes;
#define	R_INF   (*(const double *) __inf_val)

/* **************************************************************
 * Set/Reset the current directory.
 * ************************************************************** */

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

void
CD (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  static char *cd_string;
  ListNode *DIR;

  /* Check n_args */
  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("cd: requires 1 argument", 0);
  }

  /* Copy the argument string into env_string */
  DIR = bltin_get_string ("cd", d_arg, 1);
  
  if (cd_string != 0)
    FREE (cd_string);
  cd_string = cpstr (string_GetString (e_data (DIR)));
  remove_tmp_destroy (DIR);

  if (chdir (cd_string))
  {
    switch (errno)
    {
    case EACCES:
      fprintf (stderr, "Search permission to: %sdenied\n", cd_string);
      errno = 0;
      break;

    case ENOTDIR:
      fprintf (stderr, "Part of the path is not a directory: %s\n", cd_string);
      errno = 0;
      break;

    case ENOENT:
      fprintf (stderr, "Part of the path does not exist: %s\n", cd_string);
      errno = 0;
      break;

    default:
      error_1 ("error during call to cd()", 0);
    }
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }
  *return_ptr = (VPTR) scalar_Create (1.0);
}

void
Exist (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;

  /* Check n_args */
  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("exist: requires 1 argument", 0);
  }

  arg = d_arg[0];

  switch (arg.type)
  {
  case CONSTANT:
  case iCONSTANT:
    error_1 ("invalid argument to exist", 0);
  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case UNDEF:
      *return_ptr = (VPTR) scalar_Create (0.0);
      break;

    default:
      *return_ptr = (VPTR) scalar_Create (1.0);
      break;
    }
    remove_tmp_destroy (arg.u.ent);
  }
  return;
}

static char *endptr;

void
Strtod (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int i, nel;
  ListNode *M;
  Matrix *m, *new = 0;

  /* Check n_args */
  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("strtod: requires 1 argument", 0);
  }

  M = bltin_get_matrix ("strtod", d_arg, 1);
  m = (Matrix *) e_data (M);

  switch (MTYPE (m))
  {
  case REAL:
  case COMPLEX:
    error_1 ("strtod: input must be string", matrix_GetName (m));
    break;

  case STRING:
    new = matrix_Create (MNR (m), MNC (m));
    nel = MNR (m) * MNC (m);
    for (i = 0; i < nel; i++)
    {
      MATrv (new, i) = strtod (MATsv (m, i), &endptr);
      if (endptr == MATsv (m, i) && MATrv (new, i) == 0.0)
      {
	/* strtod could not recognize the string as a number */
	MATrv (new, i) = R_NAN;
      }
    }
    break;

  default:
    error_1 ("strtod: invalid matrix type", 0);
    break;
  }

  remove_tmp_destroy (M);
  *return_ptr = (VPTR) new;
  return;
}

void
System (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  double retval;
  ListNode *S;
  char *s;

  /* Check n_args */
  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("system: 1 argument allowed", 0);
  }

  S = bltin_get_string ("system", d_arg, 1);
  s = string_GetString (e_data (S));

  retval = (double) system (s);
  *return_ptr = (VPTR) scalar_Create (retval);
  remove_tmp_destroy (S);
}

/* **************************************************************
 * Return the type of an object, REAL or COMPLEX.
 * ************************************************************** */
void
Type (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;
  Matrix *m;
  Scalar *s;
  String *str;

  /* Check n_args */
  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("type: 1 argument allowed", 0);
  }

  arg = bltin_get_arg ("type", d_arg, 1);

  switch (arg.type)
  {
  case CONSTANT:
    *return_ptr = (VPTR) string_Create (cpstr ("real"));
    break;

  case iCONSTANT:
    *return_ptr = (VPTR) string_Create (cpstr ("complex"));
    break;

  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      s = (Scalar *) e_data (arg.u.ent);
      if (SVALi (s) == 0.0)
	*return_ptr = (VPTR) string_Create (cpstr ("real"));
      else
	*return_ptr = (VPTR) string_Create (cpstr ("complex"));
      break;

    case MATRIX:
      m = (Matrix *) e_data (arg.u.ent);
      if (MTYPE (m) == REAL)
	*return_ptr = (VPTR) string_Create (cpstr ("real"));
      else if (MTYPE (m) == COMPLEX)
	*return_ptr = (VPTR) string_Create (cpstr ("complex"));
      else if (MTYPE (m) == STRING)
	*return_ptr = (VPTR) string_Create (cpstr ("string"));
      break;

    case STRING:
      str = (String *) e_data (arg.u.ent);
      *return_ptr = (VPTR) string_Create (cpstr ("string"));
      break;

    case BLTIN:
      str = (String *) e_data (arg.u.ent);
      *return_ptr = (VPTR) string_Create (cpstr ("builtin"));
      break;

    case U_FUNCTION:
      str = (String *) e_data (arg.u.ent);
      *return_ptr = (VPTR) string_Create (cpstr ("user"));
      break;

    case BTREE:
      {
	/* Try and report the contents of the type element */
	ListNode *type;

	if ((type = btree_FindNode (e_data (arg.u.ent), "type")))
	{
	  if (e_type (type) == STRING)
	  {
	    *return_ptr = (VPTR)
	      string_Create (cpstr (string_GetString (e_data (type))));
	  }
	  else
	    *return_ptr = (VPTR) string_Create (cpstr (""));
	}
	else
	{
	  *return_ptr = (VPTR) string_Create (cpstr (""));
	}
	break;
      }
    default:
      error_1 (e_name (arg.u.ent), "invalid type for type()");
      break;
    }
    remove_tmp_destroy (arg.u.ent);
    break;
  }
}

/* **************************************************************
 * Return the size of an object in bytes.
 * ************************************************************** */

size_t ent_sizeof _PROTO ((ListNode * arg));
extern size_t btree_sizeof _PROTO ((Btree * btree));

void
Sizeof (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;
  int size = 0;

  /* Check n_args */
  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("size: 1 argument required", 0);
  }

  arg = bltin_get_arg ("size", d_arg, 1);

  switch (arg.type)
  {
  case CONSTANT:
    size = sizeof (double);
    break;
  case iCONSTANT:
    size = sizeof (Complex);
    break;
  case ENTITY:
    size = ent_sizeof (arg.u.ent);
    remove_tmp_destroy (arg.u.ent);
    break;
  default:
    error_1 ("size: invalid type", 0);
  }

  *return_ptr = (VPTR) scalar_Create ((double) size);
  return;
}

size_t
ent_sizeof (arg)
     ListNode *arg;
{
  Matrix *m;
  int i;
  size_t size = 0;

  switch (e_type (arg))
  {
  case SCALAR:
    size = sizeof (Complex);
    break;

  case MATRIX:
    m = (Matrix *) e_data (arg);
    switch (MTYPE (m))
    {
    case REAL:
      size = MNR (m) * MNC (m) * sizeof (double);
      break;
    case COMPLEX:
      size = MNR (m) * MNC (m) * sizeof (Complex);
      break;
    case STRING:
      size = MNR (m) * MNC (m) * sizeof (char *);
      for (i = 0; i < MNR (m) * MNC (m); i++)
	size += strlen (MATsv (m, i)) * sizeof (char);
      break;
    }
    break;

  case STRING:
    size = sizeof (char) * string_GetLength (e_data (arg));
    break;

  case BLTIN:
    size = 0;
    break;

  case U_FUNCTION:
    size = function_GetCodeSize (e_data (arg)) * sizeof (Inst);
    break;

  case BTREE:
    size += btree_sizeof (e_data (arg));
    break;

  case UNDEF:
    size = 0;
    break;

  default:
    error_1 (e_name (arg), "invalid type for size");
    break;
  }
  return size;
}

void
Garbage (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  garbage_print ();
  *return_ptr = (VPTR) scalar_Create (0.0);
}

/* **************************************************************
 * Time related functions. Original versions contributed
 * by T. L. Kay. Modified to use time() instead of gettimeofday(),
 * since time() seems more reliable/portable, Ian Searle.
 * ************************************************************** */

/*
 * Start the timer.
 */

#ifdef HAVE_TIME_H
#include <time.h>
#endif

static time_t tictime;

void
Tic (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  /* Check n_args */
  if (n_args != 0)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("tic: no arguments allowed", 0);
  }

  tictime = time (0);
  if (tictime == -1)
  {
    errno = 0;
    error_1 ("ERROR in system time() call", (char *) 0);
  }
  *return_ptr = (VPTR) scalar_Create (0.);
  return;
}

/*
 * Report the elapsed time, in seconds,
 * since last call to tic().
 */

void
Toc (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  time_t toctime;
  double elapsed;

  /* Check n_args */
  if (n_args != 0)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("toc: no arguments allowed", 0);
  }

  if (tictime)
  {
    toctime = time (0);
    if (toctime == -1)
    {
      errno = 0;
      error_1 ("ERROR in system time() call", (char *) 0);
    }
#ifdef HAVE_DIFFTIME
    elapsed = difftime (toctime, tictime);
#else
    elapsed = toctime - tictime;
#endif
    *return_ptr = (VPTR) scalar_Create (elapsed);
  }
  else
    error_1 ("must call tic() 1st", (char *) 0);

  return;
}

/*
 * Return a matrix containing the indices of the
 * non-zero elements of the input matrix.
 */

void
Find (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *M;
  Matrix *mfind;

  /* Check n_args */
  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("find: 1 arg allowed", 0);
  }

  M = bltin_get_numeric_matrix ("find", d_arg, 1);
  mfind = matrix_Find (e_data (M));
  remove_tmp_destroy (M);
  *return_ptr = (VPTR) mfind;
}

void
Inf (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  /* Check n_args */
  if (n_args != 0)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("inf: no arguments allowed", 0);
  }

  *return_ptr = (VPTR) scalar_Create (R_INF);
}

/*
 * Generate a NaN.
 */

void
Nan (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  /* Check n_args */
  if (n_args != 0)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("nan: no arguments allowed", 0);
  }

  *return_ptr = (VPTR) scalar_Create (R_NAN);
}

/* **************************************************************
 * RLaB interface to system getenv() function.
 * ************************************************************** */

void
Getenv (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *retval;
  ListNode *E;
  String *str, *sr;

  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("getenv: requires 1 argument", 0);
  }

  E = bltin_get_string ("getenv", d_arg, 1);
  str = (String *) e_data (E);

  retval = getenv (string_GetString (str));

  sr = string_Create (cpstr (retval));

  *return_ptr = (VPTR) sr;
  remove_tmp_destroy (E);
  return;
}

#ifdef HAVE_PUTENV

/* **************************************************************
 * RLaB interface to system putenv() function.
 * ************************************************************** */

static char *putenv_string = 0;

void
Putenv (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int retval;
  ListNode *P;
  String *str;

  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("putenv: requires 1 argument", 0);
  }

  P = bltin_get_string ("putenv", d_arg, 1);
  str = (String *) e_data (P);

  if (putenv_string == 0)
    FREE (putenv_string);

  putenv_string = cpstr (string_GetString (str));

  retval = putenv (putenv_string);

  *return_ptr = (VPTR) scalar_Create ((double) retval);
  remove_tmp_destroy (P);
  return;
}
#endif  /* HAVE_PUTENV */

/* **************************************************************
 * RLaB interface to ANSI-C function tmpnam().
 * ************************************************************** */

void
Tmpnam (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *fn;
  String *retf;

  if (n_args != 0)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("tmpnam: no arguments allowed", 0);
  }

  fn = tmpnam (0);
  retf = string_Create (cpstr (fn));

  *return_ptr = (VPTR) retf;
  return;
}

void
Sign (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *M;
  Matrix *msign;

  /* Check n_args */
  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("sign: 1 argument allowed", 0);
  }

  M = bltin_get_numeric_matrix ("sign", d_arg, 1);
  msign = matrix_Sign (e_data (M));
  remove_tmp_destroy (M);
  *return_ptr = (VPTR) msign;
}
