/* 
 * bi_io.c: Binary file I/O 
 */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1994  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "symbol.h"
#include "mem.h"
#include "list.h"
#include "btree.h"
#include "bltin.h"
#include "scop1.h"
#include "matrix.h"
#include "matop1.h"
#include "matop2.h"
#include "r_string.h"
#include "util.h"
#include "mathl.h"
#include "print.h"
#include "function.h"

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#ifdef HAVE_UNISTD_H
#include <sys/types.h>
#include <unistd.h>
#endif

#ifndef HAVE_FREAD_DEC
extern size_t fread ();
#endif

/*
 * Set some fixed sizes.
 * Try to do this in a way that will work on
 * older C compilers.
 */

#if SIZEOF_LONG_INT == 4
typedef long int FOURB;
#else
#if SIZEOF_INT == 4
typedef int FOURB;
#else
#if SIZEOF_SHORT_INT == 4
typedef short int FOURB;
#endif  /* SHORT_INT */
#endif  /* INT */
#endif  /* LONG_INT */

typedef size_t (*FREADER) ();

size_t fread_swap _PROTO ((VPTR ptr, size_t size, size_t nitems, 
			   FILE *stream));

FREADER freadptr;        /* Use this to point one or another fread variants */

#define TARG_DESTROY(arg, targ)   if (targ.u.ent != arg.u.ent) \
                                    remove_tmp_destroy (targ.u.ent);

#define FREAD(ptr, size, n, fn) \
        if ((*freadptr) (ptr, size, n, fn) != n) \
          if (!ferr_check ("readb", fn)) { return (0); }

/*
 * Set a component of type that determines whether this
 * is a big or little endian word machine.
 */
 
#ifdef WORDS_BIGENDIAN
static FOURB word = 1000;
#else
static FOURB word = 0;
#endif

static int btree_WriteB _PROTO ((Btree *btree, FILE *fn));
static Btree * btree_ReadB _PROTO ((FILE *fn));

static int matrix_WriteB _PROTO ((Matrix *m, FILE *fn));
static Matrix * matrix_ReadB _PROTO ((FILE *fn, int type, int order));

static int scalar_WriteB _PROTO ((Scalar *s, FILE *fn));
static Scalar * scalar_ReadB _PROTO ((FILE *fn));

static int string_WriteB _PROTO ((String *str, FILE *fn));
static String * string_ReadB _PROTO ((FILE *fn));

static int ferr_check _PROTO ((char *name, FILE *fn));
static int fread_check _PROTO ((FILE *fn));

static void reverse_word _PROTO ((VPTR word_ptr, int nbytes));

/*
 * Integers used for entity identification
 */

static FOURB SM = 5;     /* String Matrix */
static FOURB BT = 6;     /* Binary Tree */
static FOURB SC = 7;     /* SCalar */
static FOURB ST = 8;     /* STring */
static FOURB MT = 0;     /* MaTrix */

/*
 * Binary write
 */

void
WriteB (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int i, retval = 0;
  char *string;
  FILE *fn;
  ListNode *FN, *E;

  FN = bltin_get_string ("writeb", d_arg, 1);
  string = string_GetString (e_data (FN));

  if ((fn = get_file_ds (string, "wb", 0)) == 0)
  {
    warning_1 (string, "cannot open for write");
    remove_tmp_destroy (FN);
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }

  /* Get argument(s) to write */
  for (i = 2; i <= n_args; i++)
  {
    E = bltin_get_entity ("writeb", d_arg, i);
    switch (e_type (E))
    {
    case BTREE:
      if ((retval = btree_WriteB (e_data (E), fn)) == 0)
	fprintf (stderr, "ERROR, writeb: data file may be corrupt\n");
      break;

    case BLTIN:
      warning_1 ("writeb: cannot write BLTIN-FUNCTION to file", 0);
      break;

    case U_FUNCTION:
      warning_1 ("writeb: cannot write USER-FUNCTION to file", 0);
      break;

    case UNDEF:
      warning_1 ("writeb: cannot write UNDEFINED to file", 0);
      break;

    case MATRIX:
      if ((retval = matrix_WriteB (e_data (E), fn)) == 0)
	fprintf (stderr, "ERROR, writeb: data file may be corrupt\n");
      break;
    }
    remove_tmp_destroy (E);
  }
  *return_ptr = (VPTR) scalar_Create ((double) retval);
}

void
ReadB (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *name, *string;
  Datum arg2;
  size_t stat;
  FILE *fn;
  Btree *btree, *symtab;
  ListNode *lnode, *FN;
  Matrix *m;
  Scalar *s;
  String *str;

  FOURB type, tmp;

  /* Check n_args */
  if (n_args > 2)
    error_1 ("readb: wrong number of input arguments", 0);

  /* get file name from argument list, always the 1st argument */
  FN = bltin_get_string ("readb", d_arg, 1);
  string = string_GetString (e_data (FN));

  if (n_args == 1)
  {
    /* Use the global symbol table */
    symtab = get_symtab_ptr ();
  }
  else
  {
    /* Use a user-supplied list */
    arg2 = d_arg[1];
    
    /* Check to see if variable exists, etc... */
    if (arg2.type != ENTITY)
      error_1 ("readb: 2nd argument must be a variable", 0);
    if (e_type (arg2.u.ent) != UNDEF)
      listNode_DestroyDataOnly (arg2.u.ent);
    symtab = btree_Create ();
    btree_SetName (symtab, cpstr (e_name (arg2.u.ent)));
    listNode_AttachData (arg2.u.ent, BTREE, symtab, btree_Destroy);
  }
    
  if ((fn = get_file_ds (string, "rb", 0)) == 0)
  {
    warning_1 (string, "cannot open for readb");
    *return_ptr = (VPTR) scalar_Create (0.0);
    remove_tmp_destroy (FN);
    return;
  }

  while ((stat = fread (&type, sizeof (FOURB), 1, fn)) == 1)
  {
    if (type < 0 || type > 9999)
    {
      /* try flipping word to see if we can fix it? */
      reverse_word (&type, sizeof (FOURB));
    }
  
    /* Filter out the MATLAB stuff we cannot (don't want to) read */
    if ((type/1000) > 1)
      error_1 ("readb: cannot read this binary format, see `help readb'", 0);
    if (((type/10) % 10)  != 0)
      error_1 ("readb: cannot read this binary format, see `help readb'", 0);

    /* Determine which fread to use */
    if (word == 1000 && (type/1000 == 1))
      freadptr = (FREADER) fread;
    else if (word == 0 && (type/1000 == 0))
      freadptr = (FREADER) fread;
    else
      freadptr = (FREADER) fread_swap;

    tmp = type;
    if ((tmp % 10) == SC)
    {
      if ((s = scalar_ReadB (fn)) == 0)
	error_1 ("readb: cannot read scalar", 0);
      name = scalar_GetName (s);
      if ((lnode = btree_FindNode (symtab, name)) != 0)
      {
	listNode_DestroyDataOnly (lnode);
	listNode_AttachData (lnode, SCALAR, s, scalar_Destroy);
      }
      else
	install (symtab, cpstr (name), SCALAR, s);
    }
    else if ((tmp % 10) == 0 || (tmp % 10) == SM)
    {
      if ((m = matrix_ReadB (fn, type, type/1000)) == 0)
	error_1 ("readb: cannot read matrix", 0);
      name = matrix_GetName (m);
      if ((lnode = btree_FindNode (symtab, name)) != 0)
      {
	listNode_DestroyDataOnly (lnode);
	listNode_AttachData (lnode, MATRIX, m, matrix_Destroy);
      }
      else
	install (symtab, cpstr (name), MATRIX, m);
    }
    else if ((tmp % 10) == ST)
    {
      if ((str = string_ReadB (fn)) == 0)
	error_1 ("readb: cannot read string", 0);
      name = string_GetName (str);
      if ((lnode = btree_FindNode (symtab, name)) != 0)
      {
	listNode_DestroyDataOnly (lnode);
	listNode_AttachData (lnode, STRING, str, string_Destroy);
      }
      else
	install (symtab, cpstr (name), STRING, str);
    }
    else if ((tmp % 10) == BT)
    {
      if ((btree = btree_ReadB (fn)) == 0)
	error_1 ("readb: cannot read list", 0);
      name = btree_GetName (btree);
      if ((lnode = btree_FindNode (symtab, name)) != 0)
      {
	listNode_DestroyDataOnly (lnode);
	listNode_AttachData (lnode, BTREE, btree, btree_Destroy);
      }
      else
	install (symtab, cpstr (name), BTREE, btree);
    }
    else
      error_1 ("readb: unknown data type ?", 0);
  }

  if (feof (fn))
  {
    close_file_ds (string);
    *return_ptr = (VPTR) scalar_Create (1.0);
  }
  else
  {
    warning_1 ("readb: abnormal exit, did not encounter EOF", 0);
    *return_ptr = (VPTR) scalar_Create (0.0);
  }
  remove_tmp_destroy (FN);
  return;
}

/*
 * Like readb(), but reads in one object at a time.
 */

void
GetB (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *name, *string;
  FOURB type, tmp;
  size_t stat;
  Datum arg, arg2;
  FILE *fn;
  Btree *btree, *symtab;
  ListNode *lnode, *FN;
  Matrix *m;
  Scalar *s;
  String *str;

  /* Check n_args */
  if (n_args > 2)
    error_1 ("getb: wrong number of input arguments", 0);

  FN = bltin_get_string ("getb", d_arg, 1);
  string = string_GetString (e_data (FN));

  if (n_args == 1)
  {
    /* Use the global symbol table */
    symtab = get_symtab_ptr ();
  }
  else
  {
    /* Use a user-supplied list */
    arg2 = d_arg[1];

    /* Check to see if variable exists, etc... */
    if (arg.type != ENTITY)
      error_1 ("getb: 2nd argument must be a variable", 0);
    if (e_type (arg2.u.ent) != UNDEF)
      listNode_DestroyDataOnly (arg2.u.ent);
    symtab = btree_Create ();
    btree_SetName (symtab, cpstr (e_name (arg2.u.ent)));
    listNode_AttachData (arg2.u.ent, BTREE, symtab, btree_Destroy);
  }

  if ((fn = get_file_ds (string, "rb", 0)) == 0)
  {
    warning_1 (string, "cannot open for getb");
    *return_ptr = (VPTR) scalar_Create (0.0);
    remove_tmp_destroy (FN);
    return;
  }

  stat = fread (&type, sizeof (FOURB), 1, fn);
  if (stat != 1)
  {
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }

  if (type < 0 || type > 9999)
  {
    /* try flipping word to see if we can fix it? */
    reverse_word (&type, sizeof (FOURB));
  }
  
  /* Filter out the MATLAB stuff we cannot (don't want to) read */
  if ((type/1000) > 1)
    error_1 ("readb: cannot read this binary format, see `help readb'", 0);
  if (((type/10) % 10)  != 0)
    error_1 ("readb: cannot read this binary format, see `help readb'", 0);
  
  /* Determine which fread to use */
  if (word == 1000 && (type/1000 == 1))
    freadptr = (FREADER) fread;
  else if (word == 0 && (type/1000 == 0))
    freadptr = (FREADER) fread;
  else
    freadptr = (FREADER) fread_swap;
  
  tmp = type;
  if ((tmp % 10) == SC)
  {
    if ((s = scalar_ReadB (fn)) == 0)
      error_1 ("readb: cannot read scalar", 0);
    name = scalar_GetName (s);
    if ((lnode = btree_FindNode (symtab, name)) != 0)
    {
      listNode_DestroyDataOnly (lnode);
      listNode_AttachData (lnode, SCALAR, s, scalar_Destroy);
    }
    else
      install (symtab, cpstr (name), SCALAR, s);
  }
  else if ((tmp % 10) == 0 || (tmp % 10) == SM)
  {
    if ((m = matrix_ReadB (fn, type, type/1000)) == 0)
      error_1 ("readb: cannot read matrix", 0);
    name = matrix_GetName (m);
    if ((lnode = btree_FindNode (symtab, name)) != 0)
    {
      listNode_DestroyDataOnly (lnode);
      listNode_AttachData (lnode, MATRIX, m, matrix_Destroy);
    }
    else
      install (symtab, cpstr (name), MATRIX, m);  
  }
  else if ((tmp % 10) == ST)
  {
    if ((str = string_ReadB (fn)) == 0)
      error_1 ("readb: cannot read string", 0);
    name = string_GetName (str);
    if ((lnode = btree_FindNode (symtab, name)) != 0)
    {
      listNode_DestroyDataOnly (lnode);
      listNode_AttachData (lnode, STRING, str, string_Destroy);
    }
    else
      install (symtab, cpstr (name), STRING, str);
  }
  else if ((tmp % 10) == BT)
  {
    if ((btree = btree_ReadB (fn)) == 0)
      error_1 ("readb: cannot read list", 0);
    name = btree_GetName (btree);
    if ((lnode = btree_FindNode (symtab, name)) != 0)
    {
      listNode_DestroyDataOnly (lnode);
      listNode_AttachData (lnode, BTREE, btree, btree_Destroy);
    }
    else
      install (symtab, cpstr (name), BTREE, btree);
  }
  else
  {
    error_1 ("getb: unknown data type ?", 0);
  }
  *return_ptr = (VPTR) scalar_Create (1.0);
  remove_tmp_destroy (FN);
  return;
}

/***********************
 * Supporting Routines *
 ***********************/

/**********************
 * Write out Matrices *
 * type:
 * REAL    0000
 * COMPLEX 0000
 * STRING  0050
 * BTREE   0060 used by other functions...
 * SCALAR  0070
 * STRING  0080
 **********************/

static int
matrix_WriteB (m, fn)
     Matrix *m;
     FILE *fn;
{
  int i;
  FOURB imagf, nlen, type;
  FOURB nrow, ncol;

  /* Check matrix name */
  if (m->name == 0)
  {
    fprintf (stderr, "writeb: cannot write data without a name\n");
    return (0);
  }

  /* Write header info */

  /* Figure out 1st word */
  if (MTYPE (m) == REAL)
  {
    type = word + MT;
    imagf = 0;
  }
  else if (MTYPE (m) == COMPLEX)
  {
    type = word + MT;
    imagf = 1;
  }
  else if (MTYPE (m) == STRING)
  {
    type = word + SM;
    imagf = 0;
  }

  nrow = (FOURB) m->nrow;
  ncol = (FOURB) m->ncol;

  fwrite (&type, sizeof (FOURB), 1, fn);
  fwrite (&nrow, sizeof (FOURB), 1, fn);
  fwrite (&ncol, sizeof (FOURB), 1, fn);
  fwrite (&imagf, sizeof (FOURB), 1, fn);
  nlen = (FOURB) strlen (m->name) + 1;
  fwrite (&nlen, sizeof (FOURB), 1, fn);
  fwrite (m->name, sizeof (char), nlen, fn);
  
  if (MTYPE (m) == REAL)
  {
    fwrite (m->val.mr, sizeof (double), MNR (m) * MNC (m), fn);
  }
  else if (MTYPE (m) == COMPLEX)
  {
    /* We must write out the real and complex parts separately */
    for (i = 0; i < MNR (m) * MNC (m); i++)
      fwrite (&(m->val.mc[i].r), sizeof (double), 1, fn);
    for (i = 0; i < MNR (m) * MNC (m); i++)
      fwrite (&(m->val.mc[i].i), sizeof (double), 1, fn);
  }
  else if (MTYPE (m) == STRING)
  {
    for (i = 0; i < MNR (m) * MNC (m); i++)
    {
      nlen = (FOURB) strlen (m->val.ms[i]) + 1;
      fwrite (&nlen, sizeof (FOURB), 1, fn);
      fwrite (m->val.ms[i], sizeof (char), nlen, fn);
    }
  }

  if (ferror (fn))
    warning_1 ("writeb: error occurred during write", 0);
  if (feof (fn))
    warning_1 ("writeb: EOF occurred during write", 0);
      
  return (1);
}

/*****************
 * Read a Matrix *
 *****************/

static Matrix *
matrix_ReadB (fn, type, order)
     FILE *fn;
     int type;
     int order;
{
  char *name;
  int i;
  FOURB nlen, imagf, nrow, ncol, tmp;
  Matrix *m = 0;

  /* Read rest of header info */
  FREAD (&nrow, sizeof (FOURB), 1, fn);
  FREAD (&ncol, sizeof (FOURB), 1, fn);
  FREAD (&imagf, sizeof (FOURB), 1, fn);
  FREAD (&nlen, sizeof (FOURB), 1, fn);

  name = (char *) MALLOC (sizeof (char) * nlen);
  FREAD (name, sizeof (char), nlen, fn);

  /* Now create matrix */

  tmp = type;
  if ((tmp % 10) == 0)
  {
    if (imagf == 0)
    {
      m = matrix_Create (nrow, ncol);
      FREAD (m->val.mr, sizeof (double), nrow*ncol, fn);
    }
    else if (imagf == 1)
    {
      m = matrix_CreateC (nrow, ncol);
      for (i = 0; i < nrow * ncol; i++)
	FREAD (&(m->val.mc[i].r), sizeof (double), 1, fn);
      for (i = 0; i < nrow * ncol; i++)
	FREAD (&(m->val.mc[i].i), sizeof (double), 1, fn);
    }
  }
  else if ((tmp % 10) == SM)
  {
    m = matrix_CreateS (nrow, ncol);
    for (i = 0; i < nrow*ncol; i++)
    {
      FREAD (&nlen, sizeof (FOURB), 1, fn);
      m->val.ms[i] = (char *) MALLOC (sizeof (char) * nlen);
      FREAD (m->val.ms[i], sizeof (char), nlen, fn);
    }
  }

  m->name = name;
  return (m);
}

/**************************
 * Write out Binary Trees *
 **************************/

static void btree_writeb_nodes _PROTO ((ListNode *, FILE *));

static int
btree_WriteB (btree, fn)
     Btree *btree;
     FILE *fn;
{
  char *name;
  FOURB num_nodes, type;
  FOURB nlen;

  type = word + BT;

  if (btree_GetName (btree) == 0)
  {
    fprintf (stderr, "writeb: cannot write data without a name\n");
    return (0);
  }
    
  /* Write out the header information */
  fwrite (&type, sizeof (FOURB), 1, fn);

  /* Number of nodes */
  num_nodes = btree_GetNumNodes (btree);
  fwrite (&num_nodes, sizeof (FOURB), 1, fn);

  /* Tree Name */
  name = btree_GetName (btree);
  nlen = strlen (name) + 1;
  fwrite (&nlen, sizeof (FOURB), 1, fn);
  fwrite (name, sizeof (char), nlen, fn);

  /* Now write out contents */
  btree_writeb_nodes (btree->root_node, fn);

  return (1);
}

static void
btree_writeb_nodes (node, fn)
     ListNode *node;
     FILE *fn;
{
  int retval;

  if (node != 0)
  {
    btree_writeb_nodes (node->prev, fn);
    switch (e_type (node))
    {
    case SCALAR:
      if ((retval = scalar_WriteB (e_data (node), fn)) == 0)
	fprintf (stderr, "ERROR, writeb: output file is corrupt");	
      break;
    case MATRIX:
      if ((retval = matrix_WriteB (e_data (node), fn)) == 0)
	fprintf (stderr, "ERROR, writeb: output file is corrupt");
      break;
    case BTREE:
      if ((retval = btree_WriteB (e_data (node), fn)) == 0)
	fprintf (stderr, "ERROR, writeb: output file is corrupt");
      break;
    case STRING:
      if ((retval = string_WriteB (e_data (node), fn)) == 0)
	fprintf (stderr, "ERROR, writeb: output file is corrupt");
      break;
    case UNDEF:
    case U_FUNCTION:
    case BLTIN:
      break;
    default:
      fprintf (stderr, "ERROR, writeb: Object type: %i\n", e_type (node));
      warning_1 ("Invalid object for btree_write()", 0);
      break;
    }
    btree_writeb_nodes (node->next, fn);
  }
}

/**********************
 * Read a Binary Tree *
 **********************/

static Btree *
btree_ReadB (fn)
     FILE *fn;
{
  int i;
  FOURB num_nodes, nlen, tmp, type;
  char *name;
  Btree *newlist;
  Btree *btree;
  Scalar *s;
  Matrix *m;
  String *str;

  FREAD (&num_nodes, sizeof (FOURB), 1, fn);
  
  FREAD (&nlen, sizeof (FOURB), 1, fn);
  name = (char *) MALLOC (sizeof (char) * nlen);
  FREAD (name, sizeof (char), nlen, fn);

  /* Create a list and start reding object(s) from fn */
  newlist = btree_Create ();
  btree_SetName (newlist, name);

  for (i = 0; i < num_nodes; i++)
  {
    FREAD (&type, sizeof (FOURB), 1, fn);
    tmp = type;

    if ((tmp % 10) == SC)
    {
      if ((s = scalar_ReadB (fn)) == 0)
	error_1 ("readb: cannot read list-scalar", 0);
      name = scalar_GetName (s);
      install (newlist, cpstr (name), SCALAR, s);
    }
    else if ((tmp % 10) == 0 || (tmp % 10) == SM)
    {
      if ((m = matrix_ReadB (fn, type, type/1000)) == 0)
	error_1 ("readb: cannot read list-matrix", 0);
      name = matrix_GetName (m);
      install (newlist, cpstr (name), MATRIX, m);
    }
    else if ((tmp % 10) == ST)
    {
      if ((str = string_ReadB (fn)) == 0)
	error_1 ("readb: cannot read list-string", 0);
      name = string_GetName (str);
      install (newlist, cpstr (name), STRING, str);
    }
    else if ((tmp % 10) == BT)
    {
      if ((btree = btree_ReadB (fn)) == 0)
	error_1 ("readb: cannot read list-list", 0);
      name = string_GetName (btree);
      install (newlist, cpstr (name), BTREE, btree);
    }
    else
      error_1 ("readb: unknown type ?", 0);
  }
  return (newlist);
}

/******************
 * Write a Scalar *
 ******************/

static int 
scalar_WriteB (s, fn)
     Scalar *s;
     FILE *fn;
{
  FOURB type, nlen;

  type = word + SC;

  if (scalar_GetName (s) == 0)
  {
    fprintf (stderr, "writeb: cannot write data without a name\n");
    return (0);
  }

  /* Write header info */
  fwrite (&type, sizeof (FOURB), 1, fn);
  nlen = strlen (scalar_GetName (s)) + 1;
  fwrite (&nlen, sizeof (FOURB), 1, fn);
  fwrite (scalar_GetName (s), sizeof (char), nlen, fn);

  fwrite (&(s->val), sizeof (Complex), 1, fn);

  return (1);
}

/*****************
 * Read a Scalar *
 *****************/

static Scalar * 
scalar_ReadB (fn)
     FILE *fn;
{
  char *name;
  FOURB nlen;
  Scalar *s;

  s = scalar_Create (0.0);

  /* Read header info */
  FREAD(&nlen, sizeof (FOURB), 1, fn);
  name = (char *) MALLOC (sizeof (char) * nlen);
  FREAD(name, sizeof (char), nlen, fn);
  scalar_SetName (s, name);
  FREAD(&(s->val), sizeof (Complex), 1, fn);

  return (s);
}

static int 
string_WriteB (str, fn)
     String *str;
     FILE *fn;
{
  FOURB type, nlen;

  type = word + ST;

  if (string_GetName (str) == 0)
  {
    fprintf (stderr, "writeb: cannot write data without a name\n");
    return (0);
  }

  /* Write header info */
  fwrite (&type, sizeof (FOURB), 1, fn);

  nlen = strlen (string_GetName (str)) + 1;
  fwrite (&nlen, sizeof (FOURB), 1, fn);
  fwrite (string_GetName (str), sizeof (char), nlen, fn);

  nlen = strlen (string_GetString (str)) + 1;
  fwrite (&nlen, sizeof (FOURB), 1, fn);
  fwrite (str->string, sizeof (char), nlen, fn);

  return (1);
}

static String * 
string_ReadB (fn)
     FILE *fn;
{
  char *name, *string;
  FOURB nlen;
  String *str;

  /* Read header info */
  FREAD (&nlen, sizeof (FOURB), 1, fn);
  name = (char *) MALLOC (sizeof (char) * nlen);
  FREAD (name , sizeof (char), nlen, fn);

  FREAD (&nlen, sizeof (FOURB), 1, fn);
  string = (char *) MALLOC (sizeof (char) * nlen);
  FREAD (string, sizeof (char), nlen, fn);
  
  str = string_Create (string);
  string_SetName (str, name);

  return (str);
}

/***************************
 * Check for errors or EOF *
 ***************************/

static int
ferr_check (name, fn)
     char *name;
     FILE *fn;
{
  if (ferror (fn))
  {
    warning_1 (name, ": error occurred during I/O");
    return 0;
  }
  if (feof (fn))
  {
    warning_1 (name, ": EOF occurred during I/O");
    return 0;
  }
  return 1;
}

static int
fread_check (fn)
     FILE *fn;
{
  if (ferror (fn))
    return 0;
  if (feof (fn))
    return 0;

  return 1;
}

/*************************************
 * Reverse the byte order of a word. *
 *************************************/

static void
reverse_word (word_ptr, nbytes)
     void * word_ptr;
     int nbytes;
{
  char rword[32], *tmp;
  int i;

  memcpy (rword, word_ptr, nbytes);
  tmp = word_ptr;
  for (i = 0; i < nbytes; i++)
  {
    tmp[i] = rword[(nbytes - 1) - i];
  }
  
  return;
}

/*********************************************
 * A cover for fread() that swaps the bytes  *
 * to perform big/little endian translation. *
 * ONLY works for elemental data, i.e. ints  *
 * doubles, etc... not strucures.            *
 *********************************************/

size_t
fread_swap (ptr, size, nitems, stream)
     VPTR ptr;
     size_t size;
     size_t nitems;
     FILE *stream;
{
  char *fptr, *tmp;
  int i;
  size_t stat;

  /* Point to beginning of array */
  tmp = (char *) ptr;

  for (i = 0; i < nitems; i++)
  {
    /*
     * Inc tmp as we go along reading and swapping
     * as we go.
     */

    fptr = tmp + i*size;
    stat = fread (fptr, size, 1, stream);

    if (stat != 1)    /* Early return */
      return (stat);

    reverse_word (fptr, size);
  }
  return (nitems);
}

/* **************************************************************
 * RLaB interface to fread()
 * var = fread ( FILE, nitems, type, swap_flag )
 *
 * var:    The return value, a matrix.
 *
 * FILE:   File descriptor string ("stdin", "stdout", etc...).
 * nitems: Number of items to read.
 * type:   "char", "short int", "int", "float", "double"
 * swap_flag: (optional) swap bytes if TRUE (1).
 * ************************************************************** */

#define FREADERR(ptr, size, n, fn) \
      if ((*freadptr) (ptr, size, n, fn) != n) \
        if (!fread_check (fn)) { error_1 ("fread: error during read", 0); }

void
Fread (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  unsigned char uchar_tmp;
  char char_tmp;
  short int sint_tmp;
  unsigned int uint_tmp;
  int int_tmp;
  float float_tmp;
  double double_tmp;

  char *fnstring, *type;
  int i, nitems, swapf;
  FILE *fn;
  Matrix *m;
  ListNode *FN, *TYPE;

  /* Check n_args */
  if (n_args < 3)
    error_1 ("fread: wrong number of input arguments", 0);
  
  FN = bltin_get_string ("fread", d_arg, 1);
  fnstring = string_GetString (e_data (FN));

  nitems = (int) bltin_get_numeric_double ("fread", d_arg, 2);

  TYPE = bltin_get_string ("fread", d_arg, 3);
  type = string_GetString (e_data (TYPE));

  if (n_args == 4)
  {
    swapf = (int) bltin_get_numeric_double ("fread", d_arg, 4);
  }
  else
  {
    swapf = 0;
  }

  if (swapf)
    freadptr = (FREADER) fread_swap;
  else
    freadptr = (FREADER) fread;

  if ((fn = get_file_ds (fnstring, "rb", 0)) == 0)
  {
    warning_1 (fnstring, "cannot open for fread");
    *return_ptr = (VPTR) scalar_Create (0.0);
    remove_tmp_destroy (FN);
    remove_tmp_destroy (TYPE);
    return;
  }

  m = matrix_Create (nitems, 1);
  
  if (!strcmp (type, "char"))
  {
    for (i = 0; i < nitems; i++)
    {
      FREADERR (&char_tmp, sizeof (char), 1, fn);
      MATrv (m, i) = (double) char_tmp;
    }
  }
  else if (!strcmp (type, "unsigned char"))
  {
    for (i = 0; i < nitems; i++)
    {
      FREADERR (&uchar_tmp, sizeof (unsigned char), 1, fn);
      MATrv (m, i) = (double) uchar_tmp;
    }
  }
  else if (!strcmp (type, "short int"))
  {
    for (i = 0; i < nitems; i++)
    {
      FREADERR (&sint_tmp, sizeof (short int), 1, fn);
      MATrv (m, i) = (double) sint_tmp;
    }
  }
  else if (!strcmp (type, "unsigned int"))
  {
    for (i = 0; i < nitems; i++)
    {
      FREADERR (&uint_tmp, sizeof (unsigned int), 1, fn);
      MATrv (m, i) = (double) uint_tmp;
    }
  }
  else if (!strcmp (type, "int"))
  {
    for (i = 0; i < nitems; i++)
    {
      FREADERR (&int_tmp, sizeof (int), 1, fn);
      MATrv (m, i) = (double) int_tmp;
    }
  }
  else if (!strcmp (type, "float"))
  {
    for (i = 0; i < nitems; i++)
    {
      FREADERR (&float_tmp, sizeof (float), 1, fn);
      MATrv (m, i) = (double) float_tmp;
    }
  }
  else if (!strcmp (type, "double"))
  {
    for (i = 0; i < nitems; i++)
    {
      FREADERR (&double_tmp, sizeof (double), 1, fn);
      MATrv (m, i) = (double) double_tmp;
    }
  }
  else
  {
    error_1 ("fread: invalid TYPE argument", 0);
  }

  *return_ptr = (VPTR) m;
  remove_tmp_destroy (FN);
  remove_tmp_destroy (TYPE);
  return;
}


/* **************************************************************
 * RLaB interface to fseek.
 *
 * fseek ( FILENAME, OFFSET, ORIGIN )
 * FILENAME:   string
 * OFFSET:     byte (character) offset from position defined
 *             by ORIGIN.
 * ORIGIN:     "SEEK_SET"  (beginning (default))
 *             "SEEK_CUR"  (current position)
 *             "SEEK_END"  (end of file)
 * ************************************************************** */

void
Fseek (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *fnstring, *origin;
  long int offset;
  int seek_val = 0;
  FILE *fn;
  ListNode *FN, *ORIGIN;

  ORIGIN = 0;

  /* Check n_args */
  if (n_args < 2 || n_args > 3)
    error_1 ("fseek: 2 or 3 arguments allowed", 0);
  
  FN = bltin_get_string ("fseek", d_arg, 1);
  fnstring = string_GetString (e_data (FN));
  offset = (long int) bltin_get_numeric_double ("fseek", d_arg, 2);

  if (n_args == 3)
  {
    ORIGIN = bltin_get_string ("fseek", d_arg, 3);
    origin = string_GetString (e_data (ORIGIN));
    if (!strcmp (origin, "SEEK_SET"))
      seek_val = SEEK_SET;
    else if (!strcmp (origin, "SEEK_CUR"))
      seek_val = SEEK_CUR;
    else if (!strcmp (origin, "SEEK_END"))
      seek_val = SEEK_END;
    else
    {
      remove_tmp_destroy (FN);
      remove_tmp_destroy (ORIGIN);
      error_1 ("fseek: invalid ORIGIN argument", 0);
    }
  }
  else
  {
    seek_val = SEEK_SET;
  }
  
  if ((fn = get_file_ds (fnstring, "rb", 0)) == 0)
  {
    warning_1 (fnstring, "cannot open for fseek");
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }

  fseek (fn, offset, seek_val);
  
  *return_ptr = (VPTR) scalar_Create (1.0);
  remove_tmp_destroy (FN);
  if (n_args == 3)
    remove_tmp_destroy (ORIGIN);
  return;
}
