/* r_plot.c */

/*
 * This file contains the PLPLOT interface routines.
 */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1993  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

/*
 * The following dummy function is here just to
 * shut up some compilers that do not like to 
 * compile empty files.
 */

static int plot_garbage () { return (1); } 

#include "config.h"

#ifdef HAVE_RLAB_PLPLOT

#include "rlab.h"
#include "code.h"
#include "bltin.h"
#include "listnode.h"
#include "mem.h"
#include "r_string.h"
#include "matrix.h"
#include "util.h"

#include <stdio.h>

#define TARG_DESTROY(arg, targ)   if (targ.u.ent != arg.u.ent) \
                                    remove_tmp_destroy (targ.u.ent);

#define PL_DOUBLE 1
#include <plplot.h>

static int init_plot_device = 0;     /* Has the plot device been initialized */

static PLFLT ** convert_matrix_to_array _PROTO ((Matrix *m));

/* **************************************************************
 * PLPLOT related functions and interface(s)
 * ************************************************************** */

/*
 * This function handles erros that occur in the plplot library.
 * We longjmp from here back to the prompt because the library
 * will exit otherwise.
 */

/* in main.c */
extern int dec_buff _PROTO ((void));
extern jmp_buf jmp[];


static int
rlab_plplot_exit (message)
     char *message;
{
  error_1 (message, 0);
  /* longjmp( jmp[dec_buff()], 1 ); */

  return (0);   /* shut up compiler */
}

/*
 * Make a hardcopy of the current plot-window.
 */

extern FILE *get_file_ds _PROTO ((char *name, char *mode, int buffsize));
extern int close_file_ds _PROTO ((char *name));
extern void plsfile _PROTO ((FILE *file));

void
_plot_plprint (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *fname, *device;
  Datum arg1, arg2;
  FILE *sfile;
  PLINT cur_pls, new_pls;

  if (!(n_args == 1 || n_args == 2))
    error_1 ("_plprint: 1 or 2 argument(s) required", 0);
  
  arg1 = get_bltin_arg ("_plprint", d_arg, 1, STRING);
  fname = string_GetString (e_data (arg1.u.ent));

  if (n_args == 1)
  {
    /* Default to BW Postscript */
    device = cpstr ("ps");
  }
  else 
  {
    arg2 = get_bltin_arg ("_plprint", d_arg, 2, STRING);
    device = string_GetString (e_data (arg2.u.ent));
    if (!(!strcmp (device, "ps") ||
	!strcmp (device, "psc") ||
	!strcmp (device, "xfig") ||
	!strcmp (device, "plmeta") ||
	!strcmp (device, "ljii")))
      error_1 ("plprint: invalid plot device: ", device);	
  }

  /*
   * Get the current stream number.
   */

  plgstrm (&cur_pls);

  /*
   * Create a new stream and make it the default.
   * The new stream number is returned (ipls).
   */

  plmkstrm(&new_pls);

  /* Open file for writes */
    
  sfile = get_file_ds (fname, "w", 0);

  /* Initialize stream */
  
  plsdev (device);
  plsfile (sfile);

  /* Copy the current stream to the newly-made stream */

  plcpstrm (cur_pls, 0);
  pladv (0);
  
  /* Remake current plot and then switch back to original stream */
  
  plreplot ();
  plflush ();
  plend1 ();
  plsstrm (cur_pls);

  close_file_ds (fname);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Re-plot the current stream
 */

void
_plot_plreplot (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  if (n_args != 0)
    error_1 ("_plreplot: no arguments allowed", 0);
  
  plreplot ();

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Set the number of the current output stream
 */
void
_plot_plsstrm (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg1, targ1;
  double n;

  if (n_args != 1)
    error_1 ("_plsstrm: 1 argument required", 0);
  
  arg1 = get_bltin_arg ("_plsstrm", d_arg, 1, NUM);
 
  targ1 = convert_to_matrix (arg1);
  n = (double) MAT (e_data (targ1.u.ent), 1, 1);

  plsstrm ((PLINT) n);

  TARG_DESTROY (arg1, targ1);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plssub (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg1, targ1, arg2, targ2;
  double n1, n2;

  if (n_args != 2)
    error_1 ("_plssub: 2 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plssub", d_arg, 1, NUM);
  arg2 = get_bltin_arg ("_plssub", d_arg, 2, NUM);
 
  targ1 = convert_to_matrix (arg1);
  targ2 = convert_to_matrix (arg2);

  n1 = MAT (e_data (targ1.u.ent), 1, 1);
  n2 = MAT (e_data (targ2.u.ent), 1, 1);

  plssub (n1, n2);

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plinit (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  if (n_args != 0)
    error_1 ("_plinit: no arguments allowed", 0);
  
  plinit ();

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plstart (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg1, arg2, targ2, arg3, targ3;
  char *str;
  double n1, n2;

  if (n_args != 3)
    error_1 ("_plstart: 3 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plstart", d_arg, 1, STRING);
  arg2 = get_bltin_arg ("_plstart", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("_plstart", d_arg, 3, NUM);
 
  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);

  str = string_GetString (e_data (arg1.u.ent));
  n1 = MAT (e_data (targ2.u.ent), 1, 1);
  n2 = MAT (e_data (targ3.u.ent), 1, 1);

  if (!init_plot_device)
    plsexit (rlab_plplot_exit);

  plstart (str, n1, n2);

  if (!init_plot_device)
    init_plot_device = 1;

  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plenv (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg1, arg2, arg3, arg4, arg5, arg6;
  Datum targ1, targ2, targ3, targ4, targ5, targ6;
  PLFLT xmin, xmax, ymin, ymax;
  PLINT just, axis;

  if (n_args != 6)
    error_1 ("_plenv: 6 arguments are required", 0);
  
  arg1 = get_bltin_arg ("_plenv", d_arg, 1, NUM);
  arg2 = get_bltin_arg ("_plenv", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("_plenv", d_arg, 3, NUM);
  arg4 = get_bltin_arg ("_plenv", d_arg, 4, NUM);
  arg5 = get_bltin_arg ("_plenv", d_arg, 5, NUM);
  arg6 = get_bltin_arg ("_plenv", d_arg, 6, NUM);
 
  targ1 = convert_to_matrix (arg1);
  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);
  targ4 = convert_to_matrix (arg4);
  targ5 = convert_to_matrix (arg5);
  targ6 = convert_to_matrix (arg6);

  xmin = (PLFLT) MAT (e_data (targ1.u.ent), 1, 1);
  xmax = (PLFLT) MAT (e_data (targ2.u.ent), 1, 1);
  ymin = (PLFLT) MAT (e_data (targ3.u.ent), 1, 1);
  ymax = (PLFLT) MAT (e_data (targ4.u.ent), 1, 1);
  just = (PLINT) MAT (e_data (targ5.u.ent), 1, 1);
  axis = (PLINT) MAT (e_data (targ6.u.ent), 1, 1);

  plenv (xmin, xmax, ymin, ymax, just, axis);

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  TARG_DESTROY (arg4, targ4);
  TARG_DESTROY (arg5, targ5);
  TARG_DESTROY (arg6, targ6);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plline (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int npoints;
  Datum arg1, arg2, arg3;
  Datum targ1, targ2, targ3;
  Matrix *x, *y;

  if (n_args != 3)
    error_1 ("_plline: 3 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plline", d_arg, 1, NUM);
  arg2 = get_bltin_arg ("_plline", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("_plline", d_arg, 3, NUM);
 
  targ1 = convert_to_matrix (arg1);
  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);

  npoints = (int) MAT (e_data (targ1.u.ent), 1, 1);
  x = (Matrix *) e_data (targ2.u.ent);
  y = (Matrix *) e_data (targ3.u.ent);

  plline (npoints, (PLFLT *) MDPTRr (x), (PLFLT *) y->val.mr);

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plend (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  if (n_args != 0)
    error_1 ("_plend: no arguments allowed", 0);
  
  plend ();

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plend1 (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  if (n_args != 0)
    error_1 ("_plend1: no arguments allowed", 0);
  
  plend1 ();

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_pllab (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg1, arg2, arg3;
  char *sx, *sy, *st;

  if (n_args != 3)
    error_1 ("_pllab: 3 arguments required", 0);
  
  arg1 = get_bltin_arg ("_pllab", d_arg, 1, STRING);
  arg2 = get_bltin_arg ("_pllab", d_arg, 2, STRING);
  arg3 = get_bltin_arg ("_pllab", d_arg, 3, STRING);
 
  sx = string_GetString (e_data (arg1.u.ent));
  sy = string_GetString (e_data (arg2.u.ent));
  st = string_GetString (e_data (arg3.u.ent));

  pllab (sx, sy, st);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Set the color for subsequent lines. You may specify which
 * colors are associated with which number by the plancol
 * routine if supported by the output driver.
 */

void
_plot_plcol (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg1, targ1;
  PLINT color;

  if (n_args != 1)
    error_1 ("_plcol: 1 argument required", 0);
  
  arg1 = get_bltin_arg ("_plcol", d_arg, 1, NUM);
  targ1 = convert_to_matrix (arg1);
 
  color = (PLINT) MAT (e_data (targ1.u.ent), 1, 1);
  plcol (color);

  TARG_DESTROY (arg1, targ1);
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Set the background color by 8 bit RGB value
 * Note: for some drivers this corresponds to a cmap 0 color
 */

void
_plot_plscolbg (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg1, targ1, arg2, targ2, arg3, targ3;
  PLINT r, g, b;

  if (n_args != 3)
    error_1 ("_plscolbg: 3 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plscolbg", d_arg, 1, NUM);
  arg2 = get_bltin_arg ("_plscolbg", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("_plscolbg", d_arg, 3, NUM);

  targ1 = convert_to_matrix (arg1);
  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);
 
  r = (PLINT) MAT (e_data (targ1.u.ent), 1, 1);
  g = (PLINT) MAT (e_data (targ2.u.ent), 1, 1);
  b = (PLINT) MAT (e_data (targ3.u.ent), 1, 1);

  plscolbg (r, g, b);

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Set a given color from color map 0 by 8 bit RGB value
 * Does not result in any additional cells to be allocated.
 */
 
void
_plot_plscol0 (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg1, targ1, arg2, targ2, arg3, targ3;
  Datum arg4, targ4;
  PLINT color, r, g, b;

  if (n_args != 4)
    error_1 ("_plscol0: 4 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plscol0", d_arg, 1, NUM);
  arg2 = get_bltin_arg ("_plscol0", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("_plscol0", d_arg, 3, NUM);
  arg4 = get_bltin_arg ("_plscol0", d_arg, 4, NUM);

  targ1 = convert_to_matrix (arg1);
  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);
  targ4 = convert_to_matrix (arg4);
 
  color = (PLINT) MAT (e_data (targ1.u.ent), 1, 1);
  r = (PLINT) MAT (e_data (targ2.u.ent), 1, 1);
  g = (PLINT) MAT (e_data (targ3.u.ent), 1, 1);
  b = (PLINT) MAT (e_data (targ4.u.ent), 1, 1);

  plscol0 (color, r, g, b);

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  TARG_DESTROY (arg4, targ4);
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Sets the line style according to one of eight predefined patters.
 */

void
_plot_pllsty (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg1, targ1;
  PLINT style;

  if (n_args != 1)
    error_1 ("_pllsty: 1 argument required", 0);
  
  arg1 = get_bltin_arg ("_pllsty", d_arg, 1, NUM);
  targ1 = convert_to_matrix (arg1);
 
  style = (PLINT) MAT (e_data (targ1.u.ent), 1, 1);
  pllsty (style);

  TARG_DESTROY (arg1, targ1);
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plclr (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  if (n_args != 0)
    error_1 ("_plclr: no argument allowed", 0);
  
  plclr ();

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plpoin (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT n, code;
  Datum arg1, arg2, arg3, arg4;
  Datum targ1, targ2, targ3, targ4;
  
  if (n_args != 4)
    error_1 ("_plpoin: 4 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plpoin", d_arg, 1, NUM);
  arg2 = get_bltin_arg ("_plpoin", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("_plpoin", d_arg, 3, NUM);
  arg4 = get_bltin_arg ("_plpoin", d_arg, 4, NUM);
 
  targ1 = convert_to_matrix (arg1);
  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);
  targ4 = convert_to_matrix (arg4);

  n = (PLINT) MAT (e_data (targ1.u.ent), 1, 1);
  code = (PLINT) MAT (e_data (targ4.u.ent), 1, 1);

  plpoin (n, MDPTRr (e_data (targ2.u.ent)),
	  MDPTRr (e_data (targ3.u.ent)),  code);

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  TARG_DESTROY (arg4, targ4);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plhist (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT n, nbin, oldwin;
  PLFLT datmin, datmax;
  PLFLT *data;
  Datum arg1, arg2, arg3, arg4, arg5, arg6;
  Datum targ1, targ2, targ3, targ4, targ5, targ6;
  
  if (n_args != 6)
    error_1 ("_plhist: 6 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plhist", d_arg, 1, NUM);
  arg2 = get_bltin_arg ("_plhist", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("_plhist", d_arg, 3, NUM);
  arg4 = get_bltin_arg ("_plhist", d_arg, 4, NUM);
  arg5 = get_bltin_arg ("_plhist", d_arg, 5, NUM);
  arg6 = get_bltin_arg ("_plhist", d_arg, 6, NUM);
 
  targ1 = convert_to_matrix (arg1);
  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);
  targ4 = convert_to_matrix (arg4);
  targ5 = convert_to_matrix (arg5);
  targ6 = convert_to_matrix (arg6);

  n = (PLINT) MAT (e_data (targ1.u.ent), 1, 1);
  data = (PLFLT *) MDPTRr (e_data (targ2.u.ent));
  datmin = (PLFLT) MAT (e_data (targ3.u.ent), 1, 1);
  datmax = (PLFLT) MAT (e_data (targ4.u.ent), 1, 1);
  nbin = (PLINT) MAT (e_data (targ5.u.ent), 1, 1);
  oldwin = (PLINT) MAT (e_data (targ6.u.ent), 1, 1);

  plhist (n, data, datmin, datmax, nbin, oldwin);

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  TARG_DESTROY (arg4, targ4);
  TARG_DESTROY (arg5, targ5);
  TARG_DESTROY (arg6, targ6);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plspage (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT xp, yp, xleng, yleng, xoff, yoff;
  Datum arg1, arg2, arg3, arg4, arg5, arg6;
  Datum targ1, targ2, targ3, targ4, targ5, targ6;
  
  if (n_args != 6)
    error_1 ("_plspage: 6 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plspage", d_arg, 1, NUM);
  arg2 = get_bltin_arg ("_plspage", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("_plspage", d_arg, 3, NUM);
  arg4 = get_bltin_arg ("_plspage", d_arg, 4, NUM);
  arg5 = get_bltin_arg ("_plspage", d_arg, 5, NUM);
  arg6 = get_bltin_arg ("_plspage", d_arg, 6, NUM);

  targ1 = convert_to_matrix (arg1);
  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);
  targ4 = convert_to_matrix (arg4);
  targ5 = convert_to_matrix (arg5);
  targ6 = convert_to_matrix (arg6);

  xp = (PLINT) MAT (e_data (targ1.u.ent), 1, 1);
  yp = (PLINT) MAT (e_data (targ2.u.ent), 1, 1);
  xleng = (PLINT) MAT (e_data (targ3.u.ent), 1, 1);
  yleng = (PLINT) MAT (e_data (targ4.u.ent), 1, 1);
  xoff = (PLINT) MAT (e_data (targ5.u.ent), 1, 1);
  yoff = (PLINT) MAT (e_data (targ6.u.ent), 1, 1);

  plspage (xp, yp, xleng, yleng, xoff, yoff);

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  TARG_DESTROY (arg4, targ4);
  TARG_DESTROY (arg5, targ5);
  TARG_DESTROY (arg6, targ6);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Advance to the next subpage if sub=0, performing a page
 * advance if there are no remaining subpages on the current page. 
 * If subwindowing is not being used, pladv(0) will always advance 
 * the page. If sub>0, PLPLOT switches to the specified subpage. 
 * Note that this alows you to overwrite a plot on the specified 
 * subpage; if this is not what you intended, use plclr followed by 
 * plpage to first advance the page. This routine is called automatically
 * (with sub=0) by plenv, but if plenv is not used, pladv must be called 
 * after initializing PLPLOT but before defining the viewport.
 */

void
_plot_pladv (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT sub;
  Datum arg1, targ1;
  
  if (n_args != 1)
    error_1 ("_pladv: 1 argument allowed", 0);
  
  arg1 = get_bltin_arg ("_pladv", d_arg, 1, NUM);
  targ1 = convert_to_matrix (arg1);

  sub = (PLINT) MAT (e_data (targ1.u.ent), 1, 1);
  pladv (sub);

  TARG_DESTROY (arg1, targ1);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plgra (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  if (n_args != 0)
    error_1 ("_plgra: no arguments allowed", 0);
  
  plgra ();

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_pltext (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  if (n_args != 0)
    error_1 ("_pltext: no arguments allowed", 0);
  
  pltext ();

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plflush (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  if (n_args != 0)
    error_1 ("_plflush: no arguments allowed", 0);
  
  plflush ();

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plbox (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *xopt, *yopt;
  PLFLT xtick, ytick;
  PLINT nxsub, nysub;
  Datum arg1, arg2, arg3, arg4, arg5, arg6;
  Datum targ2, targ3, targ5, targ6;
  
  if (n_args != 6)
    error_1 ("_plbox: 6 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plbox", d_arg, 1, STRING);
  arg2 = get_bltin_arg ("_plbox", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("_plbox", d_arg, 3, NUM);
  arg4 = get_bltin_arg ("_plbox", d_arg, 4, STRING);
  arg5 = get_bltin_arg ("_plbox", d_arg, 5, NUM);
  arg6 = get_bltin_arg ("_plbox", d_arg, 6, NUM);

  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);
  targ5 = convert_to_matrix (arg5);
  targ6 = convert_to_matrix (arg6);

  xopt = string_GetString (e_data (arg1.u.ent));
  yopt = string_GetString (e_data (arg4.u.ent));

  xtick = (PLFLT) MAT (e_data (targ2.u.ent), 1, 1);
  ytick = (PLFLT) MAT (e_data (targ5.u.ent), 1, 1);

  nxsub = (PLINT) MAT (e_data (targ3.u.ent), 1, 1);
  nysub = (PLINT) MAT (e_data (targ6.u.ent), 1, 1);

  plbox (xopt, xtick, nxsub, yopt, ytick, nysub);

  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  TARG_DESTROY (arg5, targ5);
  TARG_DESTROY (arg6, targ6);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Draw axes, numeric and text labels for a 
 * three-dimensional surface plot. 
 */

void
_plot_plbox3 (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *xopt, *yopt, *zopt;
  char *xlabel, *ylabel, *zlabel;
  PLFLT xtick, ytick, ztick;
  PLINT nxsub, nysub, nzsub;
  Datum arg1, arg2, arg3, arg4, arg5, arg6;
  Datum arg7, arg8, arg9, arg10, arg11, arg12;
  Datum targ3, targ4;
  Datum targ7, targ8, targ11, targ12;
  
  if (n_args != 12)
    error_1 ("_plbox3: 12 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plbox3", d_arg, 1, STRING);   /* xopt */
  arg2 = get_bltin_arg ("_plbox3", d_arg, 2, STRING);   /* xlabel */
  arg3 = get_bltin_arg ("_plbox3", d_arg, 3, NUM);      /* xtick */
  arg4 = get_bltin_arg ("_plbox3", d_arg, 4, NUM);      /* nxsub */
  arg5 = get_bltin_arg ("_plbox3", d_arg, 5, STRING);   /* yopt */
  arg6 = get_bltin_arg ("_plbox3", d_arg, 6, STRING);   /* ylabel */
  arg7 = get_bltin_arg ("_plbox3", d_arg, 7, NUM);      /* ytick */
  arg8 = get_bltin_arg ("_plbox3", d_arg, 8, NUM);      /* nysub */
  arg9 = get_bltin_arg ("_plbox3", d_arg, 9, STRING);   /* zopt */
  arg10 = get_bltin_arg ("_plbox3", d_arg, 10, STRING); /* zlabel */
  arg11 = get_bltin_arg ("_plbox3", d_arg, 11, NUM);    /* ztick */
  arg12 = get_bltin_arg ("_plbox3", d_arg, 12, NUM);    /* nzsub */

  targ3 = convert_to_matrix (arg3);
  targ4 = convert_to_matrix (arg4);
  targ7 = convert_to_matrix (arg7);
  targ8 = convert_to_matrix (arg8);
  targ11 = convert_to_matrix (arg11);
  targ12 = convert_to_matrix (arg12);

  xopt = string_GetString (e_data (arg1.u.ent));
  yopt = string_GetString (e_data (arg5.u.ent));
  zopt = string_GetString (e_data (arg9.u.ent));

  xlabel = string_GetString (e_data (arg2.u.ent));
  ylabel = string_GetString (e_data (arg6.u.ent));
  zlabel = string_GetString (e_data (arg10.u.ent));

  xtick = (PLFLT) MAT (e_data (targ3.u.ent), 1, 1);
  ytick = (PLFLT) MAT (e_data (targ7.u.ent), 1, 1);
  ztick = (PLFLT) MAT (e_data (targ11.u.ent), 1, 1);
  
  nxsub = (PLINT) MAT (e_data (targ4.u.ent), 1, 1);
  nysub = (PLINT) MAT (e_data (targ8.u.ent), 1, 1);
  nzsub = (PLINT) MAT (e_data (targ12.u.ent), 1, 1);

  plbox3 (xopt, xlabel, xtick, nxsub,
	  yopt, ylabel, ytick, nysub,
	  zopt, zlabel, ztick, nzsub);

  TARG_DESTROY (arg3, targ3);
  TARG_DESTROY (arg4, targ4);
  TARG_DESTROY (arg7, targ7);
  TARG_DESTROY (arg8, targ8);
  TARG_DESTROY (arg11, targ11);
  TARG_DESTROY (arg12, targ12);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plvsta (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  if (n_args != 0)
    error_1 ("_plvsta: no arguments allowed", 0);
  
  plvsta ();

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plwind (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLFLT xmin, xmax, ymin, ymax;
  Datum arg1, arg2, arg3, arg4;
  Datum targ1, targ2, targ3, targ4;
  
  if (n_args != 4)
    error_1 ("_plwind: 4 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plbox", d_arg, 1, NUM);
  arg2 = get_bltin_arg ("_plbox", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("_plbox", d_arg, 3, NUM);
  arg4 = get_bltin_arg ("_plbox", d_arg, 4, NUM);

  targ1 = convert_to_matrix (arg1);
  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);
  targ4 = convert_to_matrix (arg4);

  xmin = (PLFLT) MAT (e_data (targ1.u.ent), 1, 1);
  xmax = (PLFLT) MAT (e_data (targ2.u.ent), 1, 1);
  ymin = (PLFLT) MAT (e_data (targ3.u.ent), 1, 1);
  ymax = (PLFLT) MAT (e_data (targ4.u.ent), 1, 1);

  plwind (xmin, xmax, ymin, ymax);

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  TARG_DESTROY (arg4, targ4);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Create a three-dimensional surface plot...
 */

void
_plot_plot3d (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int i;
  PLFLT *x, *y, **zz;
  PLINT nx, ny, opt, side;
  Datum arg1, arg2, arg3, arg4, arg5, arg6, arg7;
  Datum targ1, targ2, targ3, targ4, targ5, targ6, targ7;
  Matrix *z;
  
  if (n_args != 7)
    error_1 ("_plot3d: 7 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plot3d", d_arg, 1, NUM);
  arg2 = get_bltin_arg ("_plot3d", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("_plot3d", d_arg, 3, NUM);
  arg4 = get_bltin_arg ("_plot3d", d_arg, 4, NUM);
  arg5 = get_bltin_arg ("_plot3d", d_arg, 5, NUM);
  arg6 = get_bltin_arg ("_plot3d", d_arg, 6, NUM);
  arg7 = get_bltin_arg ("_plot3d", d_arg, 7, NUM);

  targ1 = convert_to_matrix (arg1);
  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);
  targ4 = convert_to_matrix (arg4);
  targ5 = convert_to_matrix (arg5);
  targ6 = convert_to_matrix (arg6);
  targ7 = convert_to_matrix (arg7);

  x = (PLFLT *) MDPTRr (e_data (targ1.u.ent));
  y = (PLFLT *) MDPTRr (e_data (targ2.u.ent));
  z = (Matrix *) e_data (targ3.u.ent);
  zz = convert_matrix_to_array (z);
  nx = (PLINT) MAT (e_data (targ4.u.ent), 1, 1);
  ny = (PLINT) MAT (e_data (targ5.u.ent), 1, 1);
  opt = (PLINT) MAT (e_data (targ6.u.ent), 1, 1);
  side = (PLINT) MAT (e_data (targ7.u.ent), 1, 1);

  plot3d (x, y, zz, nx, ny, opt, side);

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  TARG_DESTROY (arg4, targ4);
  TARG_DESTROY (arg5, targ5);
  TARG_DESTROY (arg6, targ6);
  TARG_DESTROY (arg7, targ7);

  /* Now free up the z-array */
  for (i = 0; i < MNR (z); i++)
    FREE (zz[i]);
  FREE (zz);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plmesh (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int i;
  PLFLT *x, *y, **zz;
  PLINT nx, ny, opt;
  Datum arg1, arg2, arg3, arg4, arg5, arg6;
  Datum targ1, targ2, targ3, targ4, targ5, targ6;
  Matrix *z;
  
  if (n_args != 6)
    error_1 ("_plmesh: 6 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plmesh", d_arg, 1, NUM);
  arg2 = get_bltin_arg ("_plmesh", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("_plmesh", d_arg, 3, NUM);
  arg4 = get_bltin_arg ("_plmesh", d_arg, 4, NUM);
  arg5 = get_bltin_arg ("_plmesh", d_arg, 5, NUM);
  arg6 = get_bltin_arg ("_plmesh", d_arg, 6, NUM);

  targ1 = convert_to_matrix (arg1);
  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);
  targ4 = convert_to_matrix (arg4);
  targ5 = convert_to_matrix (arg5);
  targ6 = convert_to_matrix (arg6);

  x = (PLFLT *) MDPTRr (e_data (targ1.u.ent));
  y = (PLFLT *) MDPTRr (e_data (targ2.u.ent));
  z = (Matrix *) e_data (targ3.u.ent);
  zz = convert_matrix_to_array (z);
  nx = (PLINT) MAT (e_data (targ4.u.ent), 1, 1);
  ny = (PLINT) MAT (e_data (targ5.u.ent), 1, 1);
  opt = (PLINT) MAT (e_data (targ6.u.ent), 1, 1);

  plmesh (x, y, zz, nx, ny, opt);

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  TARG_DESTROY (arg4, targ4);
  TARG_DESTROY (arg5, targ5);
  TARG_DESTROY (arg6, targ6);

  /* Now free up the z-array */
  for (i = 0; i < MNR (z); i++)
    FREE (zz[i]);
  FREE (zz);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
_plot_plw3d (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLFLT basex, basey, height, xmin, xmax;
  PLFLT ymin, ymax, zmin, zmax, alt, az;
  Datum arg1, arg2, arg3, arg4, arg5;
  Datum arg6, arg7, arg8, arg9, arg10, arg11;
  Datum targ1, targ2, targ3, targ4, targ5;
  Datum targ6, targ7, targ8, targ9, targ10, targ11;
  
  if (n_args != 11)
    error_1 ("_plw3d: 11 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plw3d", d_arg, 1, NUM);
  arg2 = get_bltin_arg ("_plw3d", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("_plw3d", d_arg, 3, NUM);
  arg4 = get_bltin_arg ("_plw3d", d_arg, 4, NUM);
  arg5 = get_bltin_arg ("_plw3d", d_arg, 5, NUM);
  arg6 = get_bltin_arg ("_plw3d", d_arg, 6, NUM);
  arg7 = get_bltin_arg ("_plw3d", d_arg, 7, NUM);
  arg8 = get_bltin_arg ("_plw3d", d_arg, 8, NUM);
  arg9 = get_bltin_arg ("_plw3d", d_arg, 9, NUM);
  arg10 = get_bltin_arg ("_plw3d", d_arg, 10, NUM);
  arg11 = get_bltin_arg ("_plw3d", d_arg, 11, NUM);

  targ1 = convert_to_matrix (arg1);
  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);
  targ4 = convert_to_matrix (arg4);
  targ5 = convert_to_matrix (arg5);
  targ6 = convert_to_matrix (arg6);
  targ7 = convert_to_matrix (arg7);
  targ8 = convert_to_matrix (arg8);
  targ9 = convert_to_matrix (arg9);
  targ10 = convert_to_matrix (arg10);
  targ11 = convert_to_matrix (arg11);

  basex = (PLFLT) MAT (e_data (targ1.u.ent), 1, 1);
  basey = (PLFLT) MAT (e_data (targ2.u.ent), 1, 1);
  height = (PLFLT) MAT (e_data (targ3.u.ent), 1, 1);
  xmin = (PLFLT) MAT (e_data (targ4.u.ent), 1, 1);
  xmax = (PLFLT) MAT (e_data (targ5.u.ent), 1, 1);
  ymin = (PLFLT) MAT (e_data (targ6.u.ent), 1, 1);
  ymax = (PLFLT) MAT (e_data (targ7.u.ent), 1, 1);
  zmin = (PLFLT) MAT (e_data (targ8.u.ent), 1, 1);
  zmax = (PLFLT) MAT (e_data (targ9.u.ent), 1, 1);
  alt =  (PLFLT) MAT (e_data (targ10.u.ent), 1, 1);
  az =  (PLFLT) MAT (e_data (targ11.u.ent), 1, 1);

  plw3d (basex, basey, height, xmin, xmax,
	 ymin, ymax, zmin, zmax, alt, az);

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  TARG_DESTROY (arg4, targ4);
  TARG_DESTROY (arg5, targ5);
  TARG_DESTROY (arg6, targ6);
  TARG_DESTROY (arg7, targ7);
  TARG_DESTROY (arg8, targ8);
  TARG_DESTROY (arg9, targ9);
  TARG_DESTROY (arg10, targ10);
  TARG_DESTROY (arg11, targ11);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Writes text at a specified position relative to the 
 * viewport boundaries. Text may be written inside or 
 * outside the viewport, but is clipped at the subpage 
 * boundaries. 
 */

void
_plot_plmtex (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *side, *text;
  PLFLT disp, pos, just;
  Datum arg1, arg2, arg3, arg4, arg5;
  Datum targ2, targ3, targ4;
  
  if (n_args != 5)
    error_1 ("_plmtex: 5 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plmtex", d_arg, 1, STRING);
  arg2 = get_bltin_arg ("_plmtex", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("_plmtex", d_arg, 3, NUM);
  arg4 = get_bltin_arg ("_plmtex", d_arg, 4, NUM);
  arg5 = get_bltin_arg ("_plmtex", d_arg, 5, STRING);

  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);
  targ4 = convert_to_matrix (arg4);

  side = string_GetString (e_data (arg1.u.ent));
  disp = (PLFLT) MAT (e_data (targ2.u.ent), 1, 1);
  pos = (PLFLT) MAT (e_data (targ3.u.ent), 1, 1);
  just = (PLFLT) MAT (e_data (targ4.u.ent), 1, 1);
  text = string_GetString (e_data (arg5.u.ent));

  plmtex (side, disp, pos, just, text);

  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  TARG_DESTROY (arg4, targ4);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Turn PLPLOT pause off (arg=1)
 * Turn PLPLOT pause on  (arg=0)
 */

void
_plot_plspause (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT pause;
  Datum arg, targ;
  
  if (n_args != 1)
    error_1 ("_plspause: 1 argument required", 0);
  
  arg = get_bltin_arg ("_plspause", d_arg, 1, NUM);
  targ = convert_to_matrix (arg);

  pause = (PLINT) MAT (e_data (targ.u.ent), 1, 1);

  plspause (pause);

  TARG_DESTROY (arg, targ);
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Convert a RLaB matrix to a two-dimensional C array.
 */

static PLFLT **
convert_matrix_to_array (m)
     Matrix *m;
{
  PLFLT **a;
  int i, j, nrow, ncol;

  nrow = MNR (m);
  ncol = MNC (m);

  /* Create the new array */
  a = (PLFLT **) MALLOC (nrow*sizeof (PLFLT *));
  for (i = 0; i < nrow; i++)
    a[i] = (PLFLT *) MALLOC (ncol*sizeof (PLFLT));

  /* Now load it up */
  for (i = 0; i < nrow; i++)
    for (j = 0; j < ncol; j++)
      a[i][j] = MAT0 (m, i, j);

  return a;
}

/*
 * Specify the desired pen width. The width must be 
 * between 1 and a device dependent maximum value.
 */

void
_plot_plwid (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT width;
  Datum arg, targ;

  if (n_args != 1)
    error_1 ("_plwid: 1 argument required", 0);
  
  arg = get_bltin_arg ("_plwid", d_arg, 1, NUM);
  targ = convert_to_matrix (arg);

  width = (PLINT) MAT (e_data (targ.u.ent), 1, 1);

  plwid (width);
  
  TARG_DESTROY (arg, targ);
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Writes text at a specified position and inclination within the viewport.
 * Text is clipped at the viewport boundaries. The reference point of a string
 * lies along a line passing through the string at half the height of a capital
 * letter. The position of the reference point along this line is determined by
 * just.
 */

void
_plot_plptex (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLFLT x, y, dx, dy, just;
  char *text = 0;
  Datum arg1, arg2, arg3, arg4, arg5, arg6;
  Datum targ1, targ2, targ3, targ4, targ5;

  if (n_args != 6)
    error_1 ("_plptex: 6 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plptex", d_arg, 1, NUM);
  targ1 = convert_to_matrix (arg1);
  x = (PLFLT) MAT (e_data (targ1.u.ent), 1, 1);

  arg2 = get_bltin_arg ("_plptex", d_arg, 2, NUM);
  targ2 = convert_to_matrix (arg2);
  y = (PLFLT) MAT (e_data (targ2.u.ent), 1, 1);

  arg3 = get_bltin_arg ("_plptex", d_arg, 3, NUM);
  targ3 = convert_to_matrix (arg3);
  dx = (PLFLT) MAT (e_data (targ3.u.ent), 1, 1);

  arg4 = get_bltin_arg ("_plptex", d_arg, 4, NUM);
  targ4 = convert_to_matrix (arg4);
  dy = (PLFLT) MAT (e_data (targ4.u.ent), 1, 1);

  arg5 = get_bltin_arg ("_plptex", d_arg, 5, NUM);
  targ5 = convert_to_matrix (arg5);
  just = (PLFLT) MAT (e_data (targ5.u.ent), 1, 1);

  arg6 = get_bltin_arg ("_plptex", d_arg, 6, 0);
  if (arg6.type != ENTITY)
    error_1 ("_plptex: last argument must be class STRING", 0);
  if (e_type (arg6.u.ent) == STRING)
    text = cpstr (string_GetString (e_data (arg6.u.ent)));
  else if (e_type (arg6.u.ent) == MATRIX)
  {
    Matrix *m;
    m = (Matrix *) e_data (arg6.u.ent);
    if (MTYPE (m) != STRING)
      error_1 ("_plptex: last argument must be class STRING", 0);
    text = cpstr (MATs (m, 1, 1));
  }
  else
    error_1 ("_plptex: last argument must be class STRING", 0);
  
  plptex (x, y, dx, dy, just, text);

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  TARG_DESTROY (arg4, targ4);
  TARG_DESTROY (arg5, targ5);
  FREE (text);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Set the character set to use for subsequent characer drawing.
 * May be called before initializing PLPLOT.
 * 0 Standard character set.
 * 1 Extended character set.
 */

void
_plot_plfontld (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT set;
  Datum arg, targ;

  if (n_args != 1)
    error_1 ("_plfontld: 1 argument required", 0);
  
  arg = get_bltin_arg ("_plfontld", d_arg, 1, NUM);
  targ = convert_to_matrix (arg);

  set = (PLINT) MAT (e_data (targ.u.ent), 1, 1);

  plfontld (set);
  
  TARG_DESTROY (arg, targ);
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Sets the default characer font for subsequent character drawing.
 * Also affects symbols produced by plpoin.
 * 1 Normal font (simples and fastest)
 * 2 Roman font
 * 3 Italic font
 * 4 Script font
 */

void
_plot_plfont (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT font;
  Datum arg, targ;

  if (n_args != 1)
    error_1 ("_plfont: 1 argument required", 0);
  
  arg = get_bltin_arg ("_plfont", d_arg, 1, NUM);
  targ = convert_to_matrix (arg);

  font = (PLINT) MAT (e_data (targ.u.ent), 1, 1);

  plfont (font);
  
  TARG_DESTROY (arg, targ);
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Set the current orientation. 
 * ori:  0 landscape
 * ori:  1 portrait
 * Not supported by all device drivers.
 */

void
_plot_plsori (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT ori;
  Datum arg, targ;

  if (n_args != 1)
    error_1 ("_plsori: 1 argument required", 0);
  
  arg = get_bltin_arg ("_plsori", d_arg, 1, NUM);
  targ = convert_to_matrix (arg);

  ori = (PLINT) MAT (e_data (targ.u.ent), 1, 1);

  plsori (ori);
  
  TARG_DESTROY (arg, targ);
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Globally turn color output on/off
 */

void
_plot_plscolor (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLINT color;
  Datum arg, targ;

  if (n_args != 1)
    error_1 ("_plscolor: 1 argument required", 0);
  
  arg = get_bltin_arg ("_plscolor", d_arg, 1, NUM);
  targ = convert_to_matrix (arg);

  color = (PLINT) MAT (e_data (targ.u.ent), 1, 1);

  plscolor (color);
  
  TARG_DESTROY (arg, targ);
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/*
 * Contour plotting.
 */

/*
 * linear interpolation from singly dimensioned coord arrays.
 */

static int HasContour;

static void
plot_mapgrid(x, y, tx, ty, pltr_data)
     PLFLT x, y, *tx, *ty;
     PLPointer pltr_data;
{
  PLINT ul, ur, vl, vr;
  PLFLT du, dv;
  PLFLT xl, xr, yl, yr;
  
  PLcGrid *grid = (PLcGrid *) pltr_data;
  PLFLT *xg = grid->xg;
  PLFLT *yg = grid->yg;
  PLINT nx = grid->nx;
  PLINT ny = grid->ny;
  
  ul = (PLINT) x;
  ur = ul + 1;
  du = x - ul;
  
  vl = (PLINT) y;
  vr = vl + 1;
  dv = y - vl;
  
  if (x < 0 || x > nx - 1 || y < 0 || y > ny - 1) 
  {
    error_1("_mapgrid: Invalid coordinates",0);
  }
  
  xl = xg[ul];
  yl = yg[vl];
  
  if (ur == nx) 
  {
    *tx = xl;
  }
  else 
  {
    xr = xg[ur];
    *tx = xl * (1 - du) + xr * du;
  }
  if (vr == ny) 
  {
    *ty = yl;
  }
  else 
  {
    yr = yg[vr];
    *ty = yl * (1 - dv) + yr * dv;
  }
  HasContour = 1;
}

void
_plot_plcont (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int i;
  PLFLT *x, *y, **z, *clevel;
  PLINT nx, ny, nzx, nzy, kx, lx, ky, ly, nlevel;
  PLcGrid *pltr_data;
  Datum arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8;
  Datum targ1, targ2, targ3, targ4, targ5, targ6, targ7, targ8;
  Matrix *clevelm, *zm;
  
  if (n_args != 8)
    error_1 ("_plcont: 8 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plcont", d_arg, 1, NUM);
  arg2 = get_bltin_arg ("_plcont", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("_plcont", d_arg, 3, NUM);
  arg4 = get_bltin_arg ("_plcont", d_arg, 4, NUM);
  arg5 = get_bltin_arg ("_plcont", d_arg, 5, NUM);
  arg6 = get_bltin_arg ("_plcont", d_arg, 6, NUM);
  arg7 = get_bltin_arg ("_plcont", d_arg, 7, NUM);
  arg8 = get_bltin_arg ("_plcont", d_arg, 8, NUM);

  targ1 = convert_to_matrix (arg1);
  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);
  targ4 = convert_to_matrix (arg4);
  targ5 = convert_to_matrix (arg5);
  targ6 = convert_to_matrix (arg6);
  targ7 = convert_to_matrix (arg7);
  targ8 = convert_to_matrix (arg8);

  x = (PLFLT *) MDPTRr (e_data (targ1.u.ent));
  y = (PLFLT *) MDPTRr (e_data (targ2.u.ent));
  zm = (Matrix *) e_data (targ3.u.ent);
  kx = (PLINT) MAT (e_data (targ4.u.ent), 1, 1);
  lx = (PLINT) MAT (e_data (targ5.u.ent), 1, 1);
  ky = (PLINT) MAT (e_data (targ6.u.ent), 1, 1);
  ly = (PLINT) MAT (e_data (targ7.u.ent), 1, 1);
  clevelm = (Matrix *) e_data (targ8.u.ent);

  z = convert_matrix_to_array (zm);
  nzx = (PLINT) MNR (zm);
  nzy = (PLINT) MNC (zm);

  clevel = (PLFLT *) MDPTRr (clevelm);
  nlevel = MNR (clevelm) * MNC (clevelm);

  pltr_data = (PLcGrid *) MALLOC (sizeof (PLcGrid));
  nx = (PLINT) MNR (e_data (targ1.u.ent)) * MNC (e_data (targ1.u.ent));
  ny = (PLINT) MNR (e_data (targ2.u.ent)) * MNC (e_data (targ2.u.ent));

  pltr_data->xg = x;
  pltr_data->nx = nx;
  pltr_data->yg = y;   
  pltr_data->ny = ny;
  pltr_data->zg = 0;
  pltr_data->nz = 0;

  HasContour = 0;
  plcont (z, nzx, nzy, kx, lx, ky, ly, clevel, 
	  nlevel, plot_mapgrid, pltr_data);

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  TARG_DESTROY (arg4, targ4);
  TARG_DESTROY (arg5, targ5);
  TARG_DESTROY (arg6, targ6);
  TARG_DESTROY (arg7, targ7);
  TARG_DESTROY (arg8, targ8);
  
  /* Now free up the z-array */
  for (i = 0; i < MNR (zm); i++)
    FREE (z[i]);
  FREE (z);
  
  FREE (pltr_data);

  *return_ptr = (VPTR) scalar_Create ((double) HasContour);
  return;
}

void
_plot_plvpor (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  PLFLT xmin, xmax, ymin, ymax;
  Datum arg1, arg2, arg3, arg4;
  Datum targ1, targ2, targ3, targ4;
  
  if (n_args != 4)
    error_1 ("_plvpor: 4 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plvpor", d_arg, 1, NUM);
  arg2 = get_bltin_arg ("_plvpor", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("_plvpor", d_arg, 3, NUM);
  arg4 = get_bltin_arg ("_plvpor", d_arg, 4, NUM);

  targ1 = convert_to_matrix (arg1);
  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);
  targ4 = convert_to_matrix (arg4);

  xmin = (PLFLT) MAT (e_data (targ1.u.ent), 1, 1);
  xmax = (PLFLT) MAT (e_data (targ2.u.ent), 1, 1);
  ymin = (PLFLT) MAT (e_data (targ3.u.ent), 1, 1);
  ymax = (PLFLT) MAT (e_data (targ4.u.ent), 1, 1);

  plvpor (xmin, xmax, ymin, ymax);

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  TARG_DESTROY (arg4, targ4);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

#if 0

/*
 *   for contour plot
 */

static int  HasContour;

/*
 * linear interpolation from singly dimensioned coord arrays
 */

void
_plot_mapgrid(PLFLT x, PLFLT y, PLFLT *tx, PLFLT *ty, PLPointer pltr_data)
{
  PLINT ul, ur, vl, vr;
  PLFLT du, dv;
  PLFLT xl, xr, yl, yr;
  
  PLcGrid *grid = (PLcGrid *) pltr_data;
  PLFLT *xg = grid->xg;
  PLFLT *yg = grid->yg;
  PLINT nx = grid->nx;
  PLINT ny = grid->ny;
  
  ul = (PLINT) x;
  ur = ul + 1;
  du = x - ul;
  
  vl = (PLINT) y;
  vr = vl + 1;
  dv = y - vl;

  if (x < 0 || x > nx - 1 || y < 0 || y > ny - 1) 
  {
    error_1("_mapgrid: Invalid coordinates",0);
  }

  xl = xg[ul];
  yl = yg[vl];

  if (ur == nx) 
  {
    *tx = xl;
  } 
  else 
  {
    xr = xg[ur];
    *tx = xl * (1 - du) + xr * du;
  }
  
  if (vr == ny) 
  {
    *ty = yl;
  } 
  else 
  {
    yr = yg[vr];
    *ty = yl * (1 - dv) + yr * dv;
  }
  HasContour = 1;
}

/*
 * _plcont() contour plot,  t.s.yang  6/14/94  uc berkeley
 * 
 */

void
_plot_plcont (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int     i, k;
  char    annot[50];
  PLFLT   xbeg, xend, ybeg, dy;
  PLFLT   **z, *clevel;
  PLINT   nx, ny, kx, lx, ky, ly, nlevel;
  Datum   arg1, arg2, arg3, arg4;
  Datum   targ1, targ2, targ3, targ4;
  Matrix  *clevelm, *zm, *xg, *yg;
  PLcGrid pltr_data;
  
  if (n_args != 4)
    error_1 ("_plcont: 4 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plcont", d_arg, 1, NUM);  /* z      */
  arg2 = get_bltin_arg ("_plcont", d_arg, 2, NUM);  /* clevel */
  arg3 = get_bltin_arg ("_plcont", d_arg, 3, NUM);  /* x grid */
  arg4 = get_bltin_arg ("_plcont", d_arg, 4, NUM);  /* y grid */

  targ1 = convert_to_matrix (arg1);
  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);
  targ4 = convert_to_matrix (arg4);

  zm = (Matrix *) e_data (targ1.u.ent);
  nx = (PLINT) MNR (zm);
  ny = (PLINT) MNC (zm);

  kx = 1;
  lx = nx;
  ky = 1;
  ly = ny;

  clevelm = (Matrix *) e_data (targ2.u.ent);

  z = convert_matrix_to_array (zm);
  clevel = (PLFLT *) MDPTRr (clevelm);
  nlevel = MNR (clevelm) * MNC (clevelm);

  /* grid arrays */
  xg = (Matrix *) e_data (targ3.u.ent);
  yg = (Matrix *) e_data (targ4.u.ent);
  pltr_data.xg = (PLFLT *) MDPTRr (xg);
  pltr_data.yg = (PLFLT *) MDPTRr (yg);
  pltr_data.nx = nx;
  pltr_data.ny = ny;

  /* location for legend */
  xbeg  = (pltr_data.xg[nx-1]-pltr_data.xg[0])*0.65 + pltr_data.xg[0];
  xend  = xbeg + (pltr_data.xg[nx-1]-pltr_data.xg[0])/8.0;
  ybeg  = (pltr_data.yg[ny-1]-pltr_data.yg[0])*11.3/12.0 + pltr_data.yg[0];
  dy    = (pltr_data.yg[ny-1]-pltr_data.yg[0])/20.0;
  
  k = 0;
  for (i = 0; i < nlevel; i++) 
  {
    HasContour = 0;
    pllsty((i%8)+1);
    plcol(i%16);        
    plcont(z, nx, ny, kx, lx, ky, ly, 
	   &clevel[i], 1, _plot_mapgrid, (void *) &pltr_data);  
    if (HasContour) 
    {
      /* show value of contour on the graph */       
      sprintf(annot,"%.2g",clevel[i]);   	       
      pljoin(xbeg,ybeg-dy*k,xend,ybeg-dy*k);      
      plptex(xend+dy/4,ybeg-dy*k,1.0,0.0,0.0,annot);
      k++;     
    }   
  }

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  TARG_DESTROY (arg4, targ4);

  /* Now free up the z-array */
  for (i = 0; i < MNR (zm); i++)
    FREE (z[i]);
  FREE (z);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

#endif  /* if 0 */

/*
 * _plerry() error bar plot,  t.s.yang  6/14/94  uc berkeley
 * 
 */
void
_plot_plerry (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int npoints;
  Datum arg1, arg2, arg3, arg4;
  Datum targ1, targ2, targ3, targ4;
  Matrix *x, *ymin, *ymax;

  if (n_args != 4)
    error_1 ("_plerry: 4 arguments required", 0);
  
  arg1 = get_bltin_arg ("_plerry", d_arg, 1, NUM);
  arg2 = get_bltin_arg ("_plerry", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("_plerry", d_arg, 3, NUM);
  arg4 = get_bltin_arg ("_plerry", d_arg, 4, NUM);
 
  targ1 = convert_to_matrix (arg1);
  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);
  targ4 = convert_to_matrix (arg4);

  npoints = (int) MAT (e_data (targ1.u.ent), 1, 1);
  x = (Matrix *) e_data (targ2.u.ent);
  ymin = (Matrix *) e_data (targ3.u.ent);
  ymax = (Matrix *) e_data (targ4.u.ent);

  plerry (npoints, (PLFLT *) MDPTRr (x),
	  (PLFLT *) ymin->val.mr, 
	  (PLFLT *) ymax->val.mr);

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  TARG_DESTROY (arg4, targ4);

  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

#endif  /* HAVE_RLAB_PLPLOT */
