/* math_2.c 
 * Miscellaneous math functions for RLaB */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1993  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "bltin.h"
#include "code.h"
#include "symbol.h"
#include "util.h"
#include "scop1.h"
#include "matop1.h"
#include "matop2.h"
#include "matop3.h"
#include "btree.h"
#include "listnode.h"
#include "r_string.h"
#include "fi_1.h"
#include "mathl.h"

#include <math.h>

#define TARG_DESTROY(arg, targ)   if (targ.u.ent != arg.u.ent) \
                                    remove_tmp_destroy (targ.u.ent);

extern int matrix_is_symm _PROTO ((Matrix *m));

/* **************************************************************
 * Compute condition number of a general matrix.
 * ************************************************************** */
void
Rcond (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg, targ;
  double d_tmp;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to cond()", 0);

  /* get arg from list */
  arg = get_bltin_arg ("rcond", d_arg, 1, NUM);
  targ = convert_to_matrix (arg);

  d_tmp = matrix_Rcond (e_data (targ.u.ent));
  *return_ptr = (VPTR) scalar_Create (d_tmp);

  TARG_DESTROY (arg, targ);
  return;
}

/* **************************************************************
 * Compute the determinant of a matrix
 * ************************************************************** */
void
Det (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg, targ;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to det()", 0);

  /* get arg from list */
  arg = get_bltin_arg ("det()", d_arg, 1, NUM);

  targ = convert_to_matrix (arg);
  *return_ptr = (VPTR) matrix_Det (e_data (targ.u.ent));

  TARG_DESTROY (arg, targ);
  return;
}

/* **************************************************************
 * Compute the singular value decomposition of a matrix.
 * ************************************************************** */
void
Svd (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *str;
  int aflag;
  Datum arg, arg2, targ;
  Btree *rlist;
  Matrix *rsv, *lsv, *sigma;

  /* Check n_args */
  if (n_args < 1 || n_args > 2)
    error_1 ("Wrong number of args to svd()", 0);

  /* get arg from list */
  arg = get_bltin_arg ("svd()", d_arg, 1, NUM);
  targ = convert_to_matrix (arg);
  aflag = 2;			/* default */

  if (n_args == 2)
  {
    arg2 = get_bltin_arg ("svd()", d_arg, 2, STRING);
    str = string_GetString (e_data (arg2.u.ent));
    if (!strncmp ("S", str, 1))
      aflag = 2;
    else if (!strncmp ("s", str, 1))
      aflag = 2;
    else if (!strncmp ("A", str, 1))
      aflag = 1;
    else if (!strncmp ("a", str, 1))
      aflag = 1;
    else if (!strncmp ("N", str, 1))
      aflag = 3;
    else if (!strncmp ("n", str, 1))
      aflag = 3;
    else
      error_1 ("Invalid 2nd argument to svd()", 0);
  }

  matrix_Svd (e_data (targ.u.ent), &rsv, &lsv, &sigma, aflag);
  rlist = btree_Create ();
  install (rlist, cpstr ("vt"), MATRIX, rsv);
  install (rlist, cpstr ("u"), MATRIX, lsv);
  install (rlist, cpstr ("sigma"), MATRIX, sigma);
  matrix_SetName (rsv, cpstr ("vt"));
  matrix_SetName (lsv, cpstr ("u"));
  matrix_SetName (sigma, cpstr ("sigma"));

  TARG_DESTROY (arg, targ);
  *return_ptr = (VPTR) rlist;
}

/* **************************************************************
 * Compute the Eigenvalues of [A] (standard eigenvalue problem)
 * or [A], [B] (generalized eigenvalue problem). This function checks
 * for symmetry before deciding which solver to use.
 * ************************************************************** */
void
Eig (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int sol = 0;
  Datum arg1, arg2, targ1, targ2;
  Btree *rlist;
  Matrix *vec, *val, *lvec;

  /* Check n_args */
  if ((n_args != 1) && (n_args != 2))
    error_1 ("eig: Wrong number of arguments", 0);

  if (n_args == 1)
  {
    arg1 = get_bltin_arg ("eig()", d_arg, 1, NUM);
    targ1 = convert_to_matrix (arg1);
    sol = 1;        /* Standard problem, rlab checks symmetry */
  }
  else if (n_args == 2)
  {
    arg1 = get_bltin_arg ("eig()", d_arg, 1, NUM);
    targ1 = convert_to_matrix (arg1);
    arg2 = get_bltin_arg ("eig()", d_arg, 2, NUM);
    targ2 = convert_to_matrix (arg2);
    sol = 2;       /* Generalized problem, rlab checks symmetry */
  }
  else
    error_1 ("eig: invalid number of arguments", 0);

  /* Create list for returning results */
  rlist = btree_Create ();

  if (sol == 1)
  {
    /* Standard problem, check symmetry */
    if (matrix_is_symm (e_data (targ1.u.ent)))
    {
      /* Standard Eigenvalue Problem, Symmetric */
      matrix_Eig_SEP (e_data (targ1.u.ent), &val, &vec);
    }
    else
    {
      /* Standard Eigenvalue Problem, Non-Symmetric */
      matrix_Eig_NEP (e_data (targ1.u.ent), &val, &vec, &lvec, 0);
    }
    install (rlist, cpstr ("val"), MATRIX, val);
    install (rlist, cpstr ("vec"), MATRIX, vec);
    matrix_SetName (val, cpstr ("val"));
    matrix_SetName (vec, cpstr ("vec"));
    TARG_DESTROY (arg1, targ1);
  }
  else if (sol == 2)
  {
    /* Generalized problem, check symmetry */
    if (matrix_is_symm (e_data (targ1.u.ent)))
    {
      matrix_Eig_GSEP (e_data (targ1.u.ent), e_data (targ2.u.ent),
		       &val, &vec);
    }
    else
    {
      matrix_Eig_GNEP (e_data (targ1.u.ent), e_data (targ2.u.ent),
		       &val, &vec);
    }
    install (rlist, cpstr ("val"), MATRIX, val);
    install (rlist, cpstr ("vec"), MATRIX, vec);
    matrix_SetName (val, cpstr ("val"));
    matrix_SetName (vec, cpstr ("vec"));
    TARG_DESTROY (arg1, targ1);
    TARG_DESTROY (arg2, targ2);
  }
  *return_ptr = (VPTR) rlist;
}

/* **************************************************************
 * Compute various norms of matrices.
 * ************************************************************** */
void
Norm (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *s;
  double d, p;
  Datum arg1, arg2, targ1, targ2;
  Matrix *m = 0;

  /* Check n_args */
  if ((n_args < 1) || (n_args > 2))
    error_1 ("Wrong number of args to norm()", 0);

  /*
   * norm() takes one arg, and defaults to 1-norm.
   * Otherwise norm() takes two args. The first is always
   * the object, the second (optional) arg denotes the type
   * of norm. "1" = 1-norm, "i" = infinity norm,
   * "f" = frobenius, "m" = max(abs([a]))
   */

  arg1 = get_bltin_arg ("norm", d_arg, 1, NUM);
  if (n_args == 2)
    arg2 = get_bltin_arg ("norm", d_arg, 2, 0);

  targ1 = convert_to_matrix (arg1);

  if (n_args == 1)
  {
    d = matrix_Norm (e_data (targ1.u.ent), "1");
    *return_ptr = (VPTR) scalar_Create (d);
  }
  else
  {
    if (arg2.type == CONSTANT 
	|| (arg2.type == ENTITY 
	    && (e_type (arg2.u.ent) == SCALAR 
		|| e_type (arg2.u.ent) == MATRIX)))
    {
      /* Compute a P-Norm */
      targ2 = convert_to_matrix (arg2);
      p = (double) MAT (e_data (targ2.u.ent), 1, 1);
      TARG_DESTROY (arg2, targ2);

      m = (Matrix *) e_data (targ1.u.ent);      
      if (MNR (m) != 1 && MNC (m) != 1)
	error_1 ("norm: cannot compute P-norm of a matrix", 0);
      d = matrix_PNorm (m, p);
      *return_ptr = (VPTR) scalar_Create (d);
    }
    else
    {
      s = string_GetString (e_data (arg2.u.ent));
      d = matrix_Norm (e_data (targ1.u.ent), s);
      *return_ptr = (VPTR) scalar_Create (d);
    }
  }
  TARG_DESTROY (arg1, targ1);
  return;
}

/* **************************************************************
 * Compute the Cholesky factorization.
 * ************************************************************** */
void
Chol (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg, targ;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to chol()", 0);

  /* get arg from list */
  arg = get_bltin_arg ("chol()", d_arg, 1, NUM);
  targ = convert_to_matrix (arg);

  *return_ptr = (VPTR) matrix_Chol (e_data (targ.u.ent));

  TARG_DESTROY (arg, targ);

  return;
}

/* **************************************************************
 * Compute the QR decomposition of the input matrix.
 * ************************************************************** */
void
QR (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int pflag;
  Btree *btree;
  Datum arg, arg2, targ;
  Matrix *q, *r, *p;
  char *str;

  /* Check n_args */
  if (n_args != 1 && n_args != 2)
    error_1 ("Wrong number of args to qr()", 0);
  pflag = 0;

  /* get arg from list */
  arg = get_bltin_arg ("qr()", d_arg, 1, NUM);
  targ = convert_to_matrix (arg);

  if (n_args == 2)
  {
    arg2 = get_bltin_arg ("qr()", d_arg, 2, STRING);
    str = string_GetString (e_data (arg2.u.ent));
    if (strcmp ("p", str) && strcmp ("P", str))
      error_1 ("qr: 2nd arg must be \"p\"", 0);
    pflag = 1;
  }

  btree = btree_Create ();
  if (pflag == 0)
  {
    matrix_Qr (e_data (targ.u.ent), &q, &r);
    install (btree, cpstr ("q"), MATRIX, q);
    install (btree, cpstr ("r"), MATRIX, r);
    matrix_SetName (q, cpstr ("q"));
    matrix_SetName (r, cpstr ("r"));
  }
  else
  {
    matrix_QrP (e_data (targ.u.ent), &q, &r, &p);
    install (btree, cpstr ("q"), MATRIX, q);
    install (btree, cpstr ("r"), MATRIX, r);
    install (btree, cpstr ("p"), MATRIX, p);
    matrix_SetName (q, cpstr ("q"));
    matrix_SetName (r, cpstr ("r"));
    matrix_SetName (p, cpstr ("p"));
  }
  TARG_DESTROY (arg, targ);
  *return_ptr = (VPTR) btree;
}

/* **************************************************************
 * Compute the Hessenberg form of the input matrix.
 * ************************************************************** */
void
Hess (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Btree *btree;
  Datum arg, targ;
  Matrix *p, *h;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to hess()", 0);

  /* get arg from list */
  arg = get_bltin_arg ("hess()", d_arg, 1, NUM);

  targ = convert_to_matrix (arg);
  matrix_Hess (e_data (targ.u.ent), &p, &h);
  btree = btree_Create ();
  install (btree, cpstr ("p"), MATRIX, p);
  install (btree, cpstr ("h"), MATRIX, h);
  matrix_SetName (p, cpstr ("p"));
  matrix_SetName (h, cpstr ("h"));

  *return_ptr = (VPTR) btree;
  TARG_DESTROY (arg, targ);
}

/* **************************************************************
 * Balance a matrix.
 * ************************************************************** */
void
Balance (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg, targ;
  Btree *btree;
  Matrix *Ab, *t;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to balance()", 0);

  /* get arg from list */
  arg = get_bltin_arg ("balance()", d_arg, 1, NUM);

  targ = convert_to_matrix (arg);
  matrix_Balance (e_data (targ.u.ent), &Ab, &t);

  btree = btree_Create ();
  install (btree, cpstr ("ab"), MATRIX, Ab);
  install (btree, cpstr ("t"), MATRIX, t);
  matrix_SetName (Ab, cpstr ("Ab"));
  matrix_SetName (t, cpstr ("t"));
  *return_ptr = (VPTR) btree;

  TARG_DESTROY (arg, targ);
  return;
}

/* **************************************************************
 * max function
 * ************************************************************** */
void
Max (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg, targ, arg2, targ2;
  
  /* Check n_args */
  if (n_args == 1)
  {
    arg = get_bltin_arg ("max", d_arg, 1, NUM);
    targ = convert_to_matrix (arg);
    *return_ptr = (VPTR) matrix_Max (e_data (targ.u.ent));
    TARG_DESTROY (arg, targ);
    return;
  }
  else if (n_args == 2)
  {
    arg = get_bltin_arg ("max", d_arg, 1, NUM);
    targ = convert_to_matrix (arg);
    arg2 = get_bltin_arg ("max", d_arg, 2, NUM);
    targ2 = convert_to_matrix (arg2);

    *return_ptr = (VPTR) matrix_2_max (e_data (targ.u.ent), e_data (targ2.u.ent));
    
    TARG_DESTROY (arg, targ);
    TARG_DESTROY (arg2, targ2);
    return;
  }
  else
    error_1 ("Wrong number of args to max()", 0);
}

/* **************************************************************
 * max function that returns the corresponding index value.
 * ************************************************************** */
void
MaxI (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to maxi()", 0);

  arg = get_bltin_arg ("maxi", d_arg, 1, NUM);
  switch (arg.type)
  {
  case CONSTANT:
    *return_ptr = (VPTR) scalar_Create (1.0);
    break;
  case iCONSTANT:
    *return_ptr = (VPTR) scalar_Create (1.0);
    break;
  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      *return_ptr = (VPTR) scalar_Create (1.0);
      break;
    case MATRIX:
      *return_ptr = (VPTR) matrix_Maxi (e_data (arg.u.ent));
      break;
    default:
      error_1 (e_name (arg.u.ent), "invalid argument type for maxi()");
      break;
    }
    break;
  }
}

/* **************************************************************
 * min function
 * ************************************************************** */
void
Min (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg, targ, arg2, targ2;
  
  /* Check n_args */
  if (n_args == 1)
  {
    arg = get_bltin_arg ("min", d_arg, 1, NUM);
    targ = convert_to_matrix (arg);
    *return_ptr = (VPTR) matrix_Min (e_data (targ.u.ent));
    TARG_DESTROY (arg, targ);
    return;
  }
  else if (n_args == 2)
  {
    arg = get_bltin_arg ("min", d_arg, 1, NUM);
    targ = convert_to_matrix (arg);
    arg2 = get_bltin_arg ("min", d_arg, 2, NUM);
    targ2 = convert_to_matrix (arg2);

    *return_ptr = (VPTR) matrix_2_min (e_data (targ.u.ent), e_data (targ2.u.ent));
    
    TARG_DESTROY (arg, targ);
    TARG_DESTROY (arg2, targ2);
    return;
  }
  else
    error_1 ("Wrong number of args to min()", 0);
}

/* **************************************************************
 * min function that returns the corresponding index value.
 * ************************************************************** */
void
MinI (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to mini()", 0);

  arg = get_bltin_arg ("mini", d_arg, 1, NUM);
  switch (arg.type)
  {
  case CONSTANT:
    *return_ptr = (VPTR) scalar_Create (1.0);
    break;
  case iCONSTANT:
    *return_ptr = (VPTR) scalar_Create (1.0);
    break;
  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      *return_ptr = (VPTR) scalar_Create (1.0);
      break;
    case MATRIX:
      *return_ptr = (VPTR) matrix_Mini (e_data (arg.u.ent));
      break;
    default:
      error_1 (e_name (arg.u.ent), "invalid argument type for mini()");
      break;
    }
    break;
  }
}

/* **************************************************************
 * Vector sort function. Sort the input vector. Return the sorted
 * vector, and a vector of the sorted indices.
 * ************************************************************** */

static void r_qsort _PROTO ((double *v, int left, int right, double *ind));
static void csort _PROTO ((char *v[], int left, int right, double *ind));

void
Sort (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int i, j, n;
  Btree *btree;
  Datum arg;
  Matrix *sind, *m, *mcopy;

  sind = 0; mcopy = 0;    /* Initialize */

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to sort()", 0);

  arg = get_bltin_arg ("sort", d_arg, 1, 0);

  switch (arg.type)
  {
  case CONSTANT:
    sind = matrix_Create (1, 1);
    mcopy = matrix_Create (1, 1);
    MAT (sind, 1, 1) = 1.0;
    MAT (mcopy, 1, 1) = arg.u.val;
    break;
  case iCONSTANT:
    sind = matrix_Create (1, 1);
    mcopy = matrix_CreateC (1, 1);
    MAT (sind, 1, 1) = 1.0;
    MATr (mcopy, 1, 1) = 0.0;
    MATi (mcopy, 1, 1) = arg.u.val;
    break;
  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      if (SVALi (e_data (arg.u.ent)) == 0.0)
      {
	sind = matrix_Create (1, 1);
	mcopy = matrix_Create (1, 1);
	MAT (sind, 1, 1) = 1.0;
	MAT (mcopy, 1, 1) = SVALr (e_data (arg.u.ent));
      }
      else
      {
	sind = matrix_Create (1, 1);
	mcopy = matrix_CreateC (1, 1);
	MAT (sind, 1, 1) = 1.0;
	MATr (mcopy, 1, 1) = SVALr (e_data (arg.u.ent));
	MATi (mcopy, 1, 1) = SVALi (e_data (arg.u.ent));
      }
      break;

    case STRING:
      sind = matrix_Create (1, 1);
      mcopy = matrix_CreateS (1, 1);
      MAT (sind, 1, 1) = 1.0;
      MATs (mcopy, 1, 1) = cpstr (string_GetString (e_data (arg.u.ent)));
      break; 

    case MATRIX:
      m = (Matrix *) e_data (arg.u.ent);
      switch (MTYPE (m))
      {
      case REAL:
	if (MNR (m) == 1 || MNC (m) == 1)
	{
	  /* Vector sort */
	  n = max (MNR (m), MNC (m));
	  sind = matrix_CreateFill (1.0, (double) n, 1.0, 0);
	  mcopy = matrix_Copy (m);
	  r_qsort ((double *) MDPTRr (mcopy), 0, n - 1,
		   (double *) MDPTRr (sind));
	}
	else
	{
	  /* Matrix sort (column-wise) */
	  n = MNR (m);
	  sind = matrix_CreateFillSind (MNR (m), MNC (m));
	  mcopy = matrix_Copy (m);
	  for (i = 0; i < MNC (m); i++)
	    r_qsort ((double *) (mcopy->val.mr + (i * n)), 0, n - 1,
		     (double *) (sind->val.mr + (i * n)));
	}
	break;
      case COMPLEX:
	if (MNR (m) == 1 || MNC (m) == 1)
	{
	  int size = MNR (m) * MNC (m);
	  n = max (MNR (m), MNC (m));
	  sind = matrix_CreateFill (1.0, (double) n, 1.0, 0);
	  mcopy = matrix_Abs (m);
	  r_qsort ((double *) MDPTRr (mcopy), 0, n - 1,
		   (double *) MDPTRr (sind));

	  /* Now sort [m] according to [sind] */
	  matrix_Destroy (mcopy);
	  mcopy = matrix_CreateC (MNR (m), MNC (m));
	  for (i = 1; i <= size; i++)
	  {
	    MATcvr1 (mcopy, i) = MATcvr1 (m, ((int) MATrv1 (sind, i)));
	    MATcvi1 (mcopy, i) = MATcvi1 (m, ((int) MATrv1 (sind, i)));
	  }
	}
	else
	{
	  /* Matrix sort (column-wise) */
	  n = MNR (m);
	  sind = matrix_CreateFillSind (MNR (m), MNC (m));
	  mcopy = matrix_Abs (m);
	  for (i = 0; i < MNC (m); i++)
	    r_qsort ((double *) (mcopy->val.mr + (i * n)), 0, n - 1,
		     (double *) (sind->val.mr + (i * n)));

	  /* Now sort [m] according to [sind] */
	  matrix_Destroy (mcopy);
	  mcopy = matrix_CreateC (MNR (m), MNC (m));
	  for (i = 1; i <= MNC (m); i++)
	    for (j = 1; j <= MNR (m); j++)
	    {
	      MATr (mcopy, j, i) = MATr (m, ((int) MAT (sind, j, i)), i);
	      MATi (mcopy, j, i) = MATi (m, ((int) MAT (sind, j, i)), i);
	    }
	}
	break;
      case STRING:
	if (MNR (m) == 1 || MNC (m) == 1)
	{
	  /* Vector sort */
	  n = max (MNR (m), MNC (m));
	  sind = matrix_CreateFill (1.0, (double) n, 1.0, 0);
	  mcopy = matrix_Copy (m);
	  csort ((char **) MDPTRs (mcopy), 0, n - 1, (double *) MDPTRr (sind));
	}
	else
	{
	  /* Matrix sort (column-wise) */
	  n = MNR (m);
	  sind = matrix_CreateFillSind (MNR (m), MNC (m));
	  mcopy = matrix_Copy (m);
	  for (i = 0; i < MNC (m); i++)
	    csort ((char **) (mcopy->val.ms + (i * n)), 0, n - 1,
		   (double *) (sind->val.mr + (i * n)));
	}
	break;
      }
      break;
    default:
      error_1 (e_name (arg.u.ent), "invalid type for sort()");
    }
  }
  btree = btree_Create ();
  install (btree, cpstr ("val"), MATRIX, mcopy);
  install (btree, cpstr ("ind"), MATRIX, sind);
  *return_ptr = (VPTR) btree;
  return;
}

/* **************************************************************
 * Sort() support functions (they do the real work).
 * ************************************************************** */

static void qswap _PROTO ((double *v, int i, int j, double *ind));

static void
r_qsort (v, left, right, ind)
     double *v, *ind;
     int left, right;
{
  int i, last;

  if (left >= right)		/* Do nothing if array contains */
    return;			/* fewer than two elements */

  qswap (v, left, (left + right) / 2, ind);	/* Move partitiion element */
  last = left;			/* to v[0] */

  for (i = left + 1; i <= right; i++)	/* Partition */
    if (v[i] < v[left])
      qswap (v, ++last, i, ind);

  qswap (v, left, last, ind);	/* Restore partition element */
  r_qsort (v, left, last - 1, ind);
  r_qsort (v, last + 1, right, ind);
}

static void
qswap (v, i, j, ind)
     double *v, *ind;
     int i, j;
{
  double tmp;

  tmp = v[i];			/* swap values */
  v[i] = v[j];
  v[j] = tmp;

  tmp = ind[i];			/* swap indices */
  ind[i] = ind[j];
  ind[j] = tmp;
}

/*
 * Simple character qsort.
 */

static void cswap _PROTO ((char *v[], int i, int j, double *ind));

static void
csort (v, left, right, ind)
     char *v[];
     int left, right;
     double *ind;
{
  int i, last;

  if (left >= right)
    return;
  cswap (v, left, (left + right) / 2, ind);
  last = left;
  for (i = left + 1; i <= right; i++)
    if (strcmp (v[i], v[left]) < 0)
      cswap (v, ++last, i, ind);
  cswap (v, left, last, ind);
  csort (v, left, last - 1, ind);
  csort (v, last + 1, right, ind);
}

/*
 * Interchange v[i] and v[j]
 */

static void
cswap (v, i, j, ind)
     char *v[];
     int i, j;
     double *ind;
{
  char *temp;
  double tmp;

  temp = v[i];
  v[i] = v[j];
  v[j] = temp;

  tmp = ind[i];
  ind[i] = ind[j];
  ind[j] = tmp;
}
