/* list.c */

/*  This file is a part of RLaB ("Our"-LaB)
    Copyright (C) 1992  Ian R. Searle

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    See the file ./COPYING
 ***********************************************************************/

#include "rlab.h"
#include "list.h"
#include "mem.h"

#include <stdio.h>
#include <string.h>

static int list_setNumNodes _PROTO ((List * list, int num));
static int list_setFirstNode _PROTO ((List * list, ListNode * first));
static int list_setLastNode _PROTO ((List * list, ListNode * last));

/* ************************************************************************
 * To dynamically allocate and initialize an instance of a List.
 * ************************************************************************ */
List *
list_Create ()
{
  List *newlist;
  newlist = (List *) MALLOC (sizeof (List));
  list_Initialize (newlist);
  return (newlist);
}

/* ************************************************************************
 * Free all of the dynamically allocated memory associated with
 * a List.  If the List has any ListNode's on it, then
 * list_DestroyAllNodes() is called.
 * ************************************************************************ */
void
list_Destroy (list)
     List *list;
{
  ASSERT (list);
  {
    if (list_GetNumNodes (list) > 0)
      list_DestroyAllNodes (list);
    list->name = 0;
    list->type = 0;
    FREE (list);
  }
}

/* ************************************************************************
 * Free all memory associated with all of the ListNode's that
 * currently reside on a List.  The difference between
 * this function and list_Destroy() is that the List itself
 * is not deallocated. Since we're destroying all the list members
 * lets try and be fast.
 * ************************************************************************ */
void
list_DestroyAllNodes (list)
     List *list;
{
  ASSERT (list);
  {
    ListNode *lnode, *nextNode;
    int i, numNodes;

    numNodes = list_GetNumNodes (list);
    lnode = list_GetFirstNode (list);

    for (i = 1; i <= numNodes; i++)
    {
      nextNode = listNode_GetNextNode (lnode);
      listNode_Destroy (lnode);
      lnode = nextNode;
    }

    list_setNumNodes (list, 0);
    list_setFirstNode (list, 0);
    list_setLastNode (list, 0);
  }
}

/* ************************************************************************
 *  Destroy all the nodes on the list, but not the node's data.
 *  There are situations where we want the ListNode's destroyed,
 *  but not the data associated with the ListNode's.
 *  Note that this function is identical to list_DestroyAllNodes(),
 *  except listNode_DestroyNodeOnly() is called in liue of
 *  listNode_Destroy().
 *  Note that this function does not destroy the List itself.
 *  In order to do so, one must call list_Destroy(), after
 *  calling this function.
 * ************************************************************************ */
int
list_DestroyNodesOnly (list)
     List *list;
{
  ASSERT (list);
  {
    int numNodes;

    while ((numNodes = list_GetNumNodes (list)))
    {
      /* detach the last node on the list until there are no more nodes */
      ListNode *nextNode = list_DetachNodeByPos (list, numNodes);

      if (nextNode)
	listNode_DestroyNodeOnly (nextNode);
    }
    return (1);
  }
}

/*
 * Clean the nodes without keys off of a list.
 */

void
list_GarbageCollect (list)
     List *list;
{
  ASSERT (list);
  {
    ListNode *lnode, *last, *next;

    lnode = list_GetFirstNode (list);
    last = list_GetLastNode (list);

    while (lnode != last)
    {
      next = lnode->next;
      if (lnode->key == 0)
	list_DestroyNodeByAddr (list, lnode);
      lnode = next;
    }
  }
}

/* ************************************************************************
 * Destroy the ListNode that is at position pos in a List.
 * First the list-node is detached, then destroyed, and then
 * the list count is updated.
 * ************************************************************************ */
int
list_DestroyNodeByPos (list, pos)
     List *list;
     int pos;
{
  ASSERT (list);
  {
    if (list_IsPosValid (list, pos))
    {
      ListNode *node2Destroy = list_DetachNodeByPos (list, pos);
      if (node2Destroy)
      {
	listNode_Destroy (node2Destroy);
	return (1);
      }
    }
    return (0);
  }
}

/* ************************************************************************
 * Given a pointer to a list-node, detach and destroy the
 * list-node if it currently resides on the list.
 * ************************************************************************ */
int
list_DestroyNodeByAddr (list, node2Destroy)
     List *list;
     ListNode *node2Destroy;
{
  ASSERT (list);
  ASSERT (node2Destroy);
  {
    ListNode *destroyMe = list_DetachNodeByAddr (list, node2Destroy);

    if (destroyMe)
    {
      listNode_Destroy (destroyMe);
      return (1);
    }
  }
  return (0);
}

/* **************************************************************
 * Destory a node on a list, but do NOT destroy the data.
 * ************************************************************** */
int
list_DestroyNodeOnlyByAddr (list, lnode)
     List *list;
     ListNode *lnode;
{
  ASSERT (list);
  ASSERT (lnode);
  {
    ListNode *destroyMe = list_DetachNodeByAddr (list, lnode);

    if (destroyMe)
    {
      listNode_Destroy (destroyMe);
      return (1);
    }
  }
  return (0);
}

/* ************************************************************************
 * PURPOSE: Given a List, a ListNode, and a position to insert the
 *	    ListNode in the List, insert the ListNode into the List.
 *
 *	    It may seem rather odd that this function returns a pointer
 *	    which was passed into it (the ListNode pointer).  Just
 *	    because the caller passed the pointer in, does not mean
 *	    that the caller has the pointer's value stored away, so
 *	    we return it just in case.
 *
 *          This function gets two pointers first thing:  A pointer to
 *	    the node that will be behind the nodeToInsert, and a
 *	    pointer to the node that will be ahead of the nodeToInsert.
 *
 *	    If there is a nodeBehind, then we know that there are some
 *	    ListNode's currently on the list, and also that we
 *	    are inserting at a position other than 1.  Therefore, we
 *	    insert the node ahead of the nodeBehind.
 *	    If there is a nodeBehind but not a nodeAhead, then that
 *          means that nodeToInsert is the last node in the list,
 *	    and therefore we call list_setLastNode() to take care of it.
 *
 *          If there is no nodeBehind, then we are either inserting
 *          at pos 1, or there are no ListNode's currently on the
 *          list.  So, if:
 *
 *	    nodeAhead is not NULL:  pos must be 1.
 *	    nodeAhead is NULL:      There are no nodes on the list.
 *
 *	    If nodeAhead is not NULL, then we attach nodeToInsert
 *          behind nodeAhead.
 *          If nodeAhead is NULL, then we set the last node to
 *          nodeToInsert (because there were no nodes on the list,
 *	    nodeToInsert becomes the first and also the last).
 *
 *          Regardless of nodeAhead, if nodeBehind is NULL, we
 *          set the first node equal to nodeToInsert.
 *
 *          Finally, we set the number of nodes in the list equal
 *  	    to 1 more than the current number of nodes.
 * ************************************************************************ */
ListNode *
list_InsertNode (list, nodeToInsert, pos)
     List *list;
     ListNode *nodeToInsert;
     int pos;
{
  ASSERT (list);
  ASSERT (nodeToInsert);
  ASSERT (pos > 0 && pos <= (list_GetNumNodes (list) + 1));
  ASSERT (!listNode_IsNodeAttached (nodeToInsert));
  {
    ListNode *nodeBehind = list_GetNodeByPos (list, pos - 1);
    ListNode *nodeAhead = list_GetNodeByPos (list, pos);

    if (nodeBehind)
    {
      /* There is at least 1 node on list go ahead an stuff node in place */
      listNode_AttachAhead (nodeToInsert, nodeBehind);
      if (!nodeAhead)
	list_setLastNode (list, nodeToInsert);
    }
    else
    {
      /* We are either inserting at pos = 1, or there are no nodes on list */
      if (nodeAhead)
      {
	listNode_AttachBehind (nodeToInsert, nodeAhead);
      }
      else
      {
	list_setLastNode (list, nodeToInsert);
	list_setFirstNode (list, nodeToInsert);
      }
    }
    list_setNumNodes (list, list_GetNumNodes (list) + 1);
    return (nodeToInsert);
  }
}

/* ************************************************************************
 * Search the list for the node having a key that matches the searchkey
 * Return a pointer to the ListNode. If the node with the matching key is
 * NOT found return a NULL pointer.
 * ************************************************************************ */
ListNode *
list_GetNodeByKey (list, key)
     List *list;
     char *key;
{
  ASSERT (list);
  ASSERT (key);
  {
    ListNode *lnode = list_GetFirstNode (list);

    while (lnode)
    {
      if ((strcmp (key, listNode_GetKey (lnode))) == 0)
	return (lnode);

      lnode = listNode_GetNextNode (lnode);
    }
    return (0);
  }
}

/* ************************************************************************
 * PURPOSE:     This function inserts a ListNode onto a List at the end
 *		of the List.  Note that list_EnqueueNode() and
 *		list_DequeueNode() may be used to emulate a queue with a
 *		linked list.
 * ************************************************************************ */
ListNode *
list_EnqueueNode (list, nodeToInsert)
     List *list;
     ListNode *nodeToInsert;
{
  ASSERT (list);
  ASSERT (nodeToInsert);
  return list_InsertNode (list, nodeToInsert, list_GetNumNodes (list) + 1);
}

/* ************************************************************************
 * PURPOSE: "Dequeue" a ListNode from a List.  list_EnqueueNode() and
 *	    list_DequeueNode() may be used together to make a List
 *	    emulate a queue.  list_EnqueueNode() adds a ListNode
 *	    to the end of the List, and list_DequeueNode() detachs
 *	    the first ListNode in the List.
 *
 * ************************************************************************ */
ListNode *
list_DequeueNode (list)
     List *list;
{
  ASSERT (list);
  return list_DetachNodeByPos (list, 1);
}

/* ************************************************************************
 * Push a ListNode onto the front of a List. Do Not use list_InsertNode,
 * we want this to be faster, so we are more specific.
 * ************************************************************************ */
ListNode *
list_PushNode (list, pnode)
     List *list;
     ListNode *pnode;
{
  ASSERT (list);
  {
    if (list->numNodes == 0)
    {
      list->firstNode = list->lastNode = pnode;
      pnode->next = 0;
    }
    else
    {
      ListNode *lnode = list->firstNode;
      pnode->next = lnode;
      lnode->prev = pnode;
      list->firstNode = pnode;
    }
    pnode->prev = 0;
    list->numNodes++;
    return (pnode);
  }
}

/* ************************************************************************
 * PURPOSE: Pop a ListNode from a List.  Popping a ListNode, means
 *	    detaching the last ListNode in the List.
 * ************************************************************************ */
ListNode *
list_PopNode (list)
     List *list;
{
  ASSERT (list);
  return (list_DetachNodeByPos (list, list_GetNumNodes (list)));
}

/* ************************************************************************
 * Detach a ListNode from a List, given the ListNode's
 * position in the List.  If the position is not valid,
 * NULL is returned.
 * ************************************************************************ */
ListNode *
list_DetachNodeByPos (list, pos)
     List *list;
     int pos;
{
  ASSERT (list);
  {
    ListNode *detachedNode = NULL;
    int numNodesOnList = list_GetNumNodes (list);

    if (list_IsPosValid (list, pos))
    {
      if ((detachedNode = list_GetNodeByPos (list, pos)))
      {
	if (pos == 1)
	  list_setFirstNode (list, list_GetNodeByPos (list, 2));

	if (pos == numNodesOnList)
	  list_setLastNode (list,
			    list_GetNodeByPos (list, numNodesOnList - 1));

	listNode_Detach (detachedNode);
	list_setNumNodes (list, list_GetNumNodes (list) - 1);
      }
      return (detachedNode);
    }
    return (NULL);
  }
}

/* ************************************************************************
 * Given a pointer to a ListNode, detach the ListNode from
 * the List if the ListNode currently resides in the list.
 * Return a the pointer to the detached node, NULL if failure.
 * ************************************************************************ */
ListNode *
list_DetachNodeByAddr (list, lnode)
     List *list;
     ListNode *lnode;
{
  ASSERT (list);
  ASSERT (lnode);
  {
    int n = list_GetNumNodes (list);

    if (lnode == list_GetFirstNode (list))
    {
      list_setFirstNode (list, listNode_GetNextNode (lnode));
      if (lnode == list_GetLastNode (list))
	list_setLastNode (list, listNode_GetPrevNode (lnode));
      listNode_Detach (lnode);
      list->numNodes = n - 1;
      return (lnode);
    }
    else if (lnode == list_GetLastNode (list))
    {
      list_setLastNode (list, listNode_GetPrevNode (lnode));
      listNode_Detach (lnode);
      list->numNodes = n - 1;
      return (lnode);
    }
    else
    {
      listNode_Detach (lnode);
      list->numNodes = n - 1;
      return (lnode);
    }
  }
}

/* ************************************************************************
 * Return a pointer to the ListNode that resides at the specified position
 * in the list. Return a NULL if the position does not exist.
 * ************************************************************************ */
ListNode *
list_GetNodeByPos (list, pos)
     List *list;
     int pos;
{
  ASSERT (list);
  ASSERT (pos >= 0);
  {
    ListNode *nextNode;
    int i;

    if (!list_IsPosValid (list, pos))
      return (0);

    nextNode = list_GetFirstNode (list);
    for (i = 1; i != pos; i++)
    {
      nextNode = listNode_GetNextNode (nextNode);
    }
    return (nextNode);
  }
}

/* ************************************************************************
 * PURPOSE: Return the data contained in the ListNode at position
 *	    pos in a List.  Notice that, unlike list_PopNodeData()
 *	    and list_DequeueNodeData(), this function does not detach
 *	    the ListNode and free it.  The List is untouched
 *	    by this operation.
 *
 * Get the pointer to the list-node's data. Let list_GetNodeByPos do the
 * error checking.
 * ************************************************************************ */
VPTR
list_GetNodeDataByPos (list, pos)
     List *list;
     int pos;
{
  ListNode *listNode;
  VPTR data;

  if (list_IsPosValid (list, pos))
  {
    listNode = list_GetNodeByPos (list, pos);
    return (data = listNode_GetData (listNode));
  }
  return (0);
}

/* ***********************************************************************
 * PURPOSE:
 *          Print the contents of a List to a file.
 * *********************************************************************** */
void
list_PrintToFile (list, stream)
     List *list;
     FILE *stream;
{
  ASSERT (list);
  ASSERT (stream);
  {
    ListNode *nextNode;
    int j, pos;

    pos = 1;
    while (pos <= list_GetNumNodes (list))
    {
      fprintf (stream, "   ");
      for (j = 0; j < 5; j++)
      {
	if ((nextNode = list_GetNodeByPos (list, pos++)))
	  fprintf (stream, "%-15s", listNode_GetKey (nextNode));
      }
      fprintf (stream, "\n");
    }
  }
}

/* ************************************************************************
 * Return a pointer to the last node in a _list struct.
 * ************************************************************************ */
ListNode *
list_GetLastNode (list)
     List *list;
{
  return list->lastNode;
}

/* ************************************************************************
 * Return a pointer to the first node in a List struct.
 * ************************************************************************ */
ListNode *
list_GetFirstNode (list)
     List *list;
{
  return list->firstNode;
}

/* ************************************************************************
 * PURPOSE:     Using private list-class functions,
 *              set all members a List to initial values.
 * ************************************************************************ */
int
list_Initialize (list)
     List *list;
{
  ASSERT (list);
  {
    list_setNumNodes (list, 0);
    list_setFirstNode (list, NULL);
    list_setLastNode (list, NULL);
    list->type = LIST;
    list->name = 0;
  }
  return (1);
}

/* ************************************************************************
 * Set the number of nodes in a List struct.
 * ************************************************************************ */
static int
list_setNumNodes (list, numNodes)
     List *list;
     int numNodes;
{
  ASSERT (list);
  ASSERT (numNodes >= 0);
  list->numNodes = numNodes;
  return (1);
}

/* ************************************************************************
 * Set the firstNode pointer in a List.
 * ************************************************************************ */
static int
list_setFirstNode (list, firstNode)
     List *list;
     ListNode *firstNode;
{
  ASSERT (list);
  list->firstNode = firstNode;
  return (1);
}

/* ************************************************************************
 * Set the lastNode pointer in a List.
 * ************************************************************************ */
static int
list_setLastNode (list, lastNode)
     List *list;
     ListNode *lastNode;
{
  ASSERT (list);
  list->lastNode = lastNode;
  return (1);
}

/* ************************************************************************
 * Check to see if a given position is valid for a given List.
 * ************************************************************************ */
int
list_IsPosValid (list, pos)
     List *list;
     int pos;
{
  return (pos > 0 && pos <= list_GetNumNodes (list));
}
