/* complex.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

/*
 * The functions with a capitolized name ( _Log ) take Complex as args.
 * They in turn call the lower level complex functions which take
 * doubles as args. At this point it appears that I will not use the
 * top-level functions. They will stay awhile, and if I don't use them
 * they will dissappear.
 */

#include "complex.h"
#include "bltin.h"
#include "mathl.h"

#include <math.h>

Complex
complex_Multiply (r1, i1, r2, i2)
     double r1, i1, r2, i2;
{
  Complex c;
  c.r = r1 * r2 - i1 * i2;
  c.i = i1 * r2 + r1 * i2;
  return (c);
}

Complex
complex_Div (a, b)
     Complex a, b;
{
  return (complex_div (a.r, a.i, b.r, b.i));
}

Complex
complex_div (ar, ai, br, bi)
     double ar, ai, br, bi;
{

#ifdef THINK_C
  long double abr, abi;
  long double ratio, den;
#else
  double abr, abi;
  double ratio, den;
#endif

  Complex c;

  if ((abr = br) < 0.0)
    abr = -abr;
  if ((abi = bi) < 0.0)
    abi = -abi;
  if (abr <= abi)
  {
    if (abi == 0)
      error_1 ("complex division by zero", (char *) 0);
    ratio = br / bi;
    den = bi * (1.0 + ratio * ratio);
    c.r = (ar * ratio + ai) / den;
    c.i = (ai * ratio - ar) / den;
  }
  else
  {
    ratio = bi / br;
    den = br * (1.0 + ratio * ratio);
    c.r = (ar + ai * ratio) / den;
    c.i = (ai - ar * ratio) / den;
  }
  return (c);
}

Complex
complex_pow (ar, ai, br, bi)
     double ar, ai, br, bi;
{
  double logr, logi, x, y;
  Complex r;

  logr = log (complex_abs (ar, ai));
  logi = errcheck (atan2 (ai, ar), "complex_Pow");

  x = errcheck (exp (logr * br - logi * bi), "complex_Pow");
  y = logr * bi + logi * br;

  r.r = x * errcheck (cos (y), "complex_Pow");
  r.i = x * errcheck (sin (y), "complex_Pow");

  return (r);
}

Complex
complex_Pow (a, b)
     Complex a, b;
{
  return (complex_pow (a.r, a.i, b.r, b.i));
}

Complex
complex_Conj (r, i)
     double r, i;
{
  Complex c;
  c.r = r;
  c.i = -i;
  return (c);
}

double
complex_abs (real, imag)
     double real, imag;
{
  double temp;

  if (real < 0.0)
    real = -real;
  if (imag < 0.0)
    imag = -imag;
  if (imag > real)
  {
    /* swap places */
    temp = real;
    real = imag;
    imag = temp;
  }
  if ((real + imag) == real)
    return (real);
  temp = imag / real;
  temp = real * errcheck (sqrt (1.0 + temp * temp), "cabs");
  return (temp);
}

double
complex_Abs (c)
     Complex c;
{
  return (complex_abs (c.r, c.i));
}

Complex
complex_sin (zr, zi)
     double zr, zi;
{
  Complex r;
  r.r = sin (zr) * cosh (zi);
  r.i = cos (zr) * sinh (zi);
  return (r);
}

Complex
complex_Sin (z)
     Complex z;
{
  return (complex_sin (z.r, z.i));
}

/*
 * Complex ArcSin()
 * in RLaB: -1i .* log (1i .* x + sqrt (1 - x.^2)));
 */

Complex
complex_Asin (z)
     Complex z;
{
  return (complex_asin (z.r, z.i));
}

  
Complex
complex_asin (zr, zi)
     double zr, zi;
{
  Complex tmp, tmp1;

  tmp = complex_Multiply (zr, zi, zr, zi);
  tmp.r = 1.0 - tmp.r;
  tmp.i = -tmp.i;
  tmp = complex_Sqrt (tmp);
  tmp1 = complex_Multiply (0.0, 1.0, zr, zi);
  tmp.r = tmp.r + tmp1.r;
  tmp.i = tmp.i + tmp1.i;
  tmp = complex_Log (tmp);
  tmp = complex_Multiply (0.0, -1.0, tmp.r, tmp.i);

  return (tmp);
}

Complex
complex_cos (zr, zi)
     double zr, zi;
{
  Complex r;

  r.r = cos (zr) * cosh (zi);
  r.i = -sin (zr) * sinh (zi);
  return (r);
}

Complex
complex_Cos (z)
     Complex z;
{
  return (complex_cos (z.r, z.i));
}

/*
 * Complex ArcCos()
 * in RLaB: -1i .* log (x + 1i .* sqrt (1 - x.^2)));
 */

Complex
complex_Acos (z)
     Complex z;
{
  return (complex_acos (z.r, z.i));
}

Complex
complex_acos (zr, zi)
     double zr, zi;
{
  Complex tmp;

  tmp = complex_Multiply (zr, zi, zr, zi);
  tmp.r = 1.0 - tmp.r;
  tmp.i = -tmp.i;
  tmp = complex_Sqrt (tmp);
  tmp = complex_Multiply (0.0, 1.0, tmp.r, tmp.i);
  tmp.r = zr + tmp.r;
  tmp.i = zi + tmp.i;
  tmp = complex_Log (tmp);
  tmp = complex_Multiply (0.0, -1.0, tmp.r, tmp.i);

  return (tmp);
}

Complex
complex_tan (zr, zi)
     double zr, zi;
{
  return (complex_Div (complex_sin (zr, zi), complex_cos (zr, zi)));
}

Complex
complex_Tan (z)
     Complex z;
{
  return (complex_Div (complex_Sin (z), complex_Cos (z)));
}

/*
 * Complex ArcTan()
 * In RLaB: (1i/2) .* log ((1i + x)./(1i - x));
 */

Complex
complex_Atan (z)
     Complex z;
{
  Complex tmp, tmp1, tmp2;

  tmp1.r = z.r;
  tmp1.i = 1.0 + z.i;
  tmp2.r = -z.r;
  tmp2.i = 1.0 - z.i;
  tmp = complex_Div (tmp1, tmp2);
  tmp = complex_Log (tmp);
  tmp = complex_Multiply (0.0, 0.5, tmp.r, tmp.i);

  return (tmp);
}

Complex
complex_log (zr, zi)
     double zr, zi;
{
  Complex r;

  r.i = atan2 (zi, zr);
  r.r = log (complex_abs (zr, zi));
  return (r);
}

Complex
complex_Log (z)
     Complex z;
{
  return (complex_log (z.r, z.i));
}

Complex
complex_sqrt (zr, zi)
     double zr, zi;
{
  double mag, t;
  Complex r;

  if ((mag = complex_abs (zr, zi)) == 0.0)
    r.r = r.i = 0.0;
  else if (zr > 0.0)
  {
    r.r = t = sqrt (0.5 * (mag + zr));
    t = zi / t;
    r.i = 0.5 * t;
  }
  else
  {
    t = sqrt (0.5 * (mag - zr));
    if (zi < 0.0)
      t = -t;
    r.i = t;
    t = zi / t;
    r.r = 0.5 * t;
  }
  return (r);
}

Complex
complex_Sqrt (z)
     Complex z;
{
  return (complex_sqrt (z.r, z.i));
}

Complex
complex_exp (zr, zi)
     double zr, zi;
{
  double expx;
  Complex r;

  expx = exp (zr);
  r.r = expx * cos (zi);
  r.i = expx * sin (zi);
  return (r);
}

Complex
complex_Exp (z)
     Complex z;
{
  return (complex_exp (z.r, z.i));
}

Complex
complex_mod (xr, xi, yr, yi)
     double xr, xi, yr, yi;
{
  Complex quotient, retval, tmp;

  quotient = complex_div (xr, xi, yr, yi);
  quotient.r = (int) quotient.r;
  quotient.i = (int) quotient.i;
  tmp = complex_Multiply (yr, yi, quotient.r, quotient.i);
  retval.r = xr - tmp.r;
  retval.i = xi - tmp.i;
  return (retval);
}

Complex
complex_Mod (c1, c2)
     Complex c1, c2;
{
  return (complex_mod (c1.r, c1.i, c2.r, c2.i));
}

/* **************************************************************
 * Complex relational functions. Compare the magnitudes of
 * each complex number.
 * ************************************************************** */

double
complex_le (r1, i1, r2, i2)
     double r1, i1, r2, i2;
{
  return ((double) ((r1*r1+i1*i1) <= (r2*r2+i2*i2)));
}

double
complex_lt (r1, i1, r2, i2)
     double r1, i1, r2, i2;
{
  return ((double) ((r1*r1+i1*i1) < (r2*r2+i2*i2)));
}

double
complex_ge (r1, i1, r2, i2)
     double r1, i1, r2, i2;
{
  return ((double) ((r1*r1+i1*i1) >= (r2*r2+i2*i2)));
}

double
complex_gt (r1, i1, r2, i2)
     double r1, i1, r2, i2;
{
  return ((double) ((r1*r1+i1*i1) > (r2*r2+i2*i2)));
}

double
complex_or (r1, i1, r2, i2)
     double r1, i1, r2, i2;
{
  return ((double) ((r1 != 0.0) || (r2 != 0.0) || (i1 != 0.0) ||  (i2 != 0.0)));
}

double
complex_and (r1, i1, r2, i2)
     double r1, i1, r2, i2;
{
  return ((double) ((r1 != 0.0) && (r2 != 0.0) && (i1 != 0.0) && (i2 != 0.0)));
}
