/* bltin1.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1994  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "code.h"
#include "symbol.h"
#include "mem.h"
#include "list.h"
#include "btree.h"
#include "bltin.h"
#include "scop1.h"
#include "matop1.h"
#include "matop2.h"
#include "r_string.h"
#include "util.h"
#include "mathl.h"
#include "function.h"

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

/* scan.l */
extern int new_file _PROTO ((char *file_name));

/* print.c */
extern FILE *get_file_ds _PROTO ((char *name, char *mode, int buffsize));
extern int close_file_ds _PROTO ((char *name));

#define rabs(x) ((x) >= 0 ? (x) : -(x))

#define TARG_DESTROY(arg, targ)   if (targ.u.ent != arg.u.ent) \
                                    remove_tmp_destroy (targ.u.ent);

/* **************************************************************
 * Return a list of the current built-in and user functions in the
 * symbol-table.
 * ************************************************************** */
void
What (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;
  Matrix *m = 0;

  if (n_args == 0)
    m = (Matrix *) print_function_list (0);
  else if (n_args == 1)
  {
    arg = get_bltin_arg ("what", d_arg, 1, BTREE);
    m = (Matrix *) print_function_list (e_data (arg.u.ent));
  }
  else
    error_1 ("what() only takes 0 or 1 arg", (char *) 0);

  *return_ptr = (VPTR) m;
}

/* **************************************************************
 * Return a list of current objects in the symbol table. NFY
 * ************************************************************** */
void
Who (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;
  Matrix *m = 0;

  if (n_args == 0)
    m = (Matrix *) print_object_list (0);
  else if (n_args == 1)
  {
    arg = get_bltin_arg ("who", d_arg, 1, BTREE);
    m = (Matrix *) print_object_list (e_data (arg.u.ent));
  }
  else
    error_1 ("who() only takes 0 or 1 arg", (char *) 0);

  *return_ptr = (VPTR) m;
}

void
Members (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;
  Matrix *m = 0;

  if (n_args == 1)
  {
    arg = get_bltin_arg ("members", d_arg, 1, BTREE);
    m = (Matrix *) btree_members ((e_data (arg.u.ent)));
  }
  else
    error_1 ("members() only takes 1 arg", (char *) 0);

  *return_ptr = (VPTR) m;
}

/* **************************************************************
 * Load an RLaB file.
 * ************************************************************** */

extern int run_program _PROTO ((char *ptr));

void
Load (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *string;
  Datum arg;

  /* Check n_args */
  if (n_args != 1)
  {
    warning_1 ("Wrong number of args to load()", 0);
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }

  /* get arg from list */
  arg = get_bltin_arg ("load", d_arg, 1, STRING);
  string = string_GetString (e_data (arg.u.ent));

  /* Run rlab on the file */
  run_program (string);

  close_file_ds (string);

  *return_ptr = (VPTR) scalar_Create (1.0);
}

/* **************************************************************
 * Return an object filled with zeros.
 * ************************************************************** */
void
Zero (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  double d[2];
  int i;
  Datum arg[2];
  Matrix *m = 0;

  if (n_args == 1)
  {
    arg[0] = get_bltin_arg ("zeros", d_arg, 1, MATRIX);
    if (MNR (e_data (arg[0].u.ent)) * MNC (e_data (arg[0].u.ent)) != 2)
      error_1 ("must supply a matrix with n = 2 to zeros()", 0);
    matrix_screen_string (e_data (arg[0].u.ent));
    m = matrix_Create ((int) MATrv (e_data (arg[0].u.ent), 0),
		       (int) MATrv (e_data (arg[0].u.ent), 1));
  }
  else if (n_args == 2)
  {
    for (i = 0; i <= 1; i++)
    {
      arg[i] = get_bltin_arg ("zeros", d_arg, i + 1, 0);
      switch (arg[i].type)
      {
      case CONSTANT:
	d[i] = arg[i].u.val;
	break;
      case iCONSTANT:
	d[i] = 0.0;
	break;
      case ENTITY:
	switch (e_type (arg[i].u.ent))
	{
	case SCALAR:
	  d[i] = SVALr (e_data (arg[i].u.ent));
	  break;
	case MATRIX:
	  if (MNR (e_data (arg[i].u.ent)) == 1 &&
	      MNC (e_data (arg[i].u.ent)) == 1)
	  {
	    if (MTYPE (e_data (arg[i].u.ent)) == REAL)
	      d[i] = MAT (e_data (arg[i].u.ent), 1, 1);
	    else if (MTYPE (e_data (arg[i].u.ent)) == COMPLEX)
	      d[i] = MATr (e_data (arg[i].u.ent), 1, 1);
	    else
	      error_1 (e_name (arg[i].u.ent),
		       "invalid matrix type for zeros()");
	  }
	  else
	    error_1 (e_name (arg[i].u.ent), "Must be a 1x1 matrix");
	  break;
	default:
	  error_1 (e_name (arg[i].u.ent), "invalid type for zeros()");
	}
	break;
      default:
	error_1 (e_name (arg[i].u.ent), "invalid type for zeros()");
      }
    }
    m = matrix_Create ((int) d[0], (int) d[1]);
  }
  else
    error_1 ("zeros() requires 1 or 2 args", 0);

  matrix_Zero (m);
  *return_ptr = (VPTR) m;
}

void
rOnes (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  double d[2];
  int i, size;
  Datum arg[2];
  Matrix *m = 0;

  if (n_args == 1)
  {
    arg[0] = get_bltin_arg ("ones", d_arg, 1, MATRIX);
    if (MNR (e_data (arg[0].u.ent)) * MNC (e_data (arg[0].u.ent)) != 2)
      error_1 ("must supply a matrix with n = 2 to ones()", 0);
    matrix_screen_string (e_data (arg[0].u.ent));
    m = matrix_Create ((int) MATrv (e_data (arg[0].u.ent), 0),
		       (int) MATrv (e_data (arg[0].u.ent), 1));
  }
  else if (n_args == 2)
  {
    for (i = 0; i <= 1; i++)
    {
      arg[i] = get_bltin_arg ("ones", d_arg, i + 1, 0);
      switch (arg[i].type)
      {
      case CONSTANT:
	d[i] = arg[i].u.val;
	break;
      case iCONSTANT:
	d[i] = 0.0;
	break;
      case ENTITY:
	switch (e_type (arg[i].u.ent))
	{
	case SCALAR:
	  d[i] = SVALr (e_data (arg[i].u.ent));
	  break;
	case MATRIX:
	  if (MNR (e_data (arg[i].u.ent)) == 1 &&
	      MNC (e_data (arg[i].u.ent)) == 1)
	  {
	    if (MTYPE (e_data (arg[i].u.ent)) == REAL)
	      d[i] = MAT (e_data (arg[i].u.ent), 1, 1);
	    else if (MTYPE (e_data (arg[i].u.ent)) == COMPLEX)
	      d[i] = MATr (e_data (arg[i].u.ent), 1, 1);
	    else
	      error_1 (e_name (arg[i].u.ent),
		       "invalid matrix type for ones()");
	  }
	  else
	    error_1 (e_name (arg[i].u.ent), "Must be a 1x1 matrix");
	  break;
	default:
	  error_1 (e_name (arg[i].u.ent), "invalid type for ones()");
	}
	break;
      default:
	error_1 (e_name (arg[i].u.ent), "invalid type for ones()");
      }
    }
    m = matrix_Create ((int) d[0], (int) d[1]);
  }
  else
    error_1 ("ones() requires 1 or 2 args", 0);

  /*
   * Now fill the matrix with 1s
   */

  size = MNR (m) * MNC (m);
  for (i = 0; i < size; i++)
  {
    MATrv (m, i) = 1.0;
  }
  *return_ptr = (VPTR) m;
}

void
Diag (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int i, k, smax, smin, size;
  Datum arg1, arg2;
  Matrix *m, *marg;

  m = 0;   /* Initialize */

  if (n_args != 1 && n_args != 2)
    error_1 ("Wrong number of args to diag()", 0);

  /* get arg(s) from list */
  arg1 = get_bltin_arg ("diag", d_arg, 1, 0);
  arg1 = convert_to_matrix (arg1);
  if (n_args == 1)
  {
    k = 0;
  }
  else
  {
    arg2 = get_bltin_arg ("diag", d_arg, 2, 0);
    arg2 = convert_to_matrix (arg2);
    matrix_screen_string (e_data (arg2.u.ent));
    k = MAT (e_data (arg2.u.ent), 1, 1);
  }
  marg = (Matrix *) e_data (arg1.u.ent);
  matrix_screen_string (marg);

  if (MNR (marg) == 1 || MNC (marg) == 1)
  {
    /* Create a diagonal matrix */
    size = max (MNR (marg), MNC (marg)) + rabs (k);
    if (MTYPE (marg) == REAL)
    {
      m = matrix_Create (size, size);
      matrix_Zero (m);
      if (k < 0)
      {
	for (i = 1 - k; i <= size; i++)
	  MAT (m, i, (i + k)) = MATrv1 (marg, (i + k));
      }
      else
      {
	for (i = 1; i <= size - k; i++)
	  MAT (m, i, (i + k)) = MATrv1 (marg, i);
      }
    }
    else if (MTYPE (marg) == COMPLEX)
    {
      m = matrix_CreateC (size, size);
      matrix_Zero (m);
      if (k < 0)
      {
	for (i = 1 - k; i <= size; i++)
	{
	  MATr (m, i, (i + k)) = MATcvr1 (marg, (i + k));
	  MATi (m, i, (i + k)) = MATcvi1 (marg, (i + k));
	}
      }
      else
      {
	for (i = 1; i <= size - k; i++)
	{
	  MATr (m, i, (i + k)) = MATcvr1 (marg, i);
	  MATi (m, i, (i + k)) = MATcvi1 (marg, i);
	}
      }
    }
  }
  else
  {
    /* Extract the diagonal elements */
    smin = min (MNR (marg), MNC (marg));
    smax = max (MNR (marg), MNC (marg));

    if (k >= 0)
    {
      if (MNR (marg) >= MNC (marg))
	size = smin - k;
      else
	size = smin - max (0, k - (MNC (marg) - MNR (marg)));
    }
    else
    {
      if (MNR (marg) >= MNC (marg))
	size = smin - max (0, -k - (MNR (marg) - MNC (marg)));
      else
	size = smin + k;
    }
    if (size <= 0)
      size = 0;

    if (MTYPE (marg) == REAL)
    {
      m = matrix_Create (size, 1);
      matrix_Zero (m);
      if (k >= 0)
      {
	for (i = 1; i <= size; i++)
	  MAT (m, i, 1) = MAT (marg, i, i + k);
      }
      else
      {
	for (i = 1; i <= size; i++)
	  MAT (m, i, 1) = MAT (marg, i - k, i);
      }
    }
    else if (MTYPE (marg) == COMPLEX)
    {
      m = matrix_CreateC (size, 1);
      matrix_Zero (m);
      if (k >= 0)
      {
	for (i = 1; i <= size; i++)
	{
	  MATr (m, i, 1) = MATr (marg, i, i + k);
	  MATi (m, i, 1) = MATi (marg, i, i + k);
	}
      }
      else
      {
	for (i = 1; i <= size; i++)
	{
	  MATr (m, i, 1) = MATr (marg, i - k, i);
	  MATi (m, i, 1) = MATi (marg, i - k, i);
	}
      }
    }
  }
  *return_ptr = (VPTR) m;
}

/* **************************************************************
 * Return the size of an entity.
 * ************************************************************** */
void
Size (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;
  Matrix *m;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to size()", (char *) 0);

  /* get arg from list */
  arg = get_bltin_arg ("size", d_arg, 1, 0);

  switch (arg.type)
  {
  case CONSTANT:
  case iCONSTANT:
    m = matrix_Create (1, 2);
    MAT (m, 1, 1) = 1.0;
    MAT (m, 1, 2) = 1.0;
    *return_ptr = (VPTR) m;
    break;

  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
    case STRING:
      m = matrix_Create (1, 2);
      MAT (m, 1, 1) = 1.0;
      MAT (m, 1, 2) = 1.0;
      *return_ptr = (VPTR) m;
      break;

    case MATRIX:
      m = matrix_Create (1, 2);
      MAT (m, 1, 1) = (double) MNR (e_data (arg.u.ent));
      MAT (m, 1, 2) = (double) MNC (e_data (arg.u.ent));
      *return_ptr = (VPTR) m;
      break;

    case BTREE:
      *return_ptr = (VPTR)
	scalar_Create ((double) btree_GetRealNumNodes (e_data (arg.u.ent)));
      break;

    default:
      error_1 (e_name (arg.u.ent), "invalid type for size()");
      break;
    }
  }
}

/* **************************************************************
 * Return the length of an entity.
 * ************************************************************** */
void

#ifdef THINK_C
Lengthx (return_ptr, n_args, d_arg)
#else
Length (return_ptr, n_args, d_arg)
#endif

     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to Length()", (char *) 0);

  /* get arg from list */
  arg = get_bltin_arg ("length", d_arg, 1, 0);

  switch (arg.type)
  {
  case CONSTANT:
  case iCONSTANT:
    *return_ptr = (VPTR) scalar_Create (1.0);
    break;

  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      *return_ptr = (VPTR) scalar_Create (1.0);
      break;

    case MATRIX:
      if (MNR (e_data (arg.u.ent)) >= MNC (e_data (arg.u.ent)))
	*return_ptr = (VPTR) scalar_Create ((double) MNR (e_data (arg.u.ent)));
      else
	*return_ptr = (VPTR) scalar_Create ((double) MNC (e_data (arg.u.ent)));
      break;

    case STRING:
      *return_ptr = (VPTR)
	scalar_Create ((double) string_GetLength (e_data (arg.u.ent)));
      break;

    case BTREE:
      *return_ptr = (VPTR)
	scalar_Create ((double) btree_GetRealNumNodes (e_data (arg.u.ent)));
      break;

    default:
      error_1 (e_name (arg.u.ent), "invalid type for length()");
      break;
    }
  }
}

/* **************************************************************
 * Return the "class" of an object as a character string.
 * ************************************************************** */
void
Class (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *str_tmp;
  Datum arg;

  str_tmp = 0;   /* Initialize */

  if (n_args != 1)
    error_1 ("class() takes a single argument", (char *) 0);

  arg = get_bltin_arg ("class", d_arg, 1, 0);

  switch (arg.type)
  {
  case CONSTANT:
    str_tmp = cpstr ("num");
    break;

  case iCONSTANT:
    str_tmp = cpstr ("num");
    break;

  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      str_tmp = cpstr ("num");
      break;

    case MATRIX:
      if (MTYPE (e_data (arg.u.ent)) != STRING)
	str_tmp = cpstr ("num");
      else
	str_tmp = cpstr ("string");
      break;

    case BTREE:
      str_tmp = cpstr ("list");

      break;
    case STRING:
      str_tmp = cpstr ("string");
      break;

    case BLTIN:
      str_tmp = cpstr ("function");
      break;

    case U_FUNCTION:
      str_tmp = cpstr ("function");
      break;

    default:
      error_1 ("invalid type for class()", 0);
      break;
    }
    break;

  default:
    error_1 ("invalid type for class()", 0);
    break;
  }
  /* Do NOT free() str_tmp, it's value s used by string_Create() */
  *return_ptr = (VPTR) string_Create (str_tmp);
}

/* **************************************************************
 * Change the numerical output format.
 * format()    should print the current values of the formats
 *
 * ************************************************************** */
void
Format (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg1, arg2;

  if (n_args > 2)
    error_1 ("format() takes two args at most", (char *) 0);

  if (n_args == 1)
  {
    /* Set format from user input */
    arg1 = get_bltin_arg ("format", d_arg, 1, SCALAR);
    arg1 = convert_to_scalar (arg1);
    fprec = (int) SVALr (e_data (arg1.u.ent));
  }
  else if (n_args == 2)
  {
    arg1 = get_bltin_arg ("format", d_arg, 1, SCALAR);
    arg2 = get_bltin_arg ("format", d_arg, 2, SCALAR);
    arg1 = convert_to_scalar (arg1);
    arg2 = convert_to_scalar (arg2);
    fwidth = (int) SVALr (e_data (arg1.u.ent));
    fprec = (int) SVALr (e_data (arg2.u.ent));
  }
  else
  {
    /* Reset format to default */
    fwidth = FWIDTH_DEFAULT;
    fprec = FPREC_DEFAULT;
  }
  *return_ptr = (VPTR) scalar_Create (1.0);
}

/* **************************************************************
 * Open a diary file.
 * ************************************************************** */
#include <sys/types.h>

#ifdef __STDC__
#include <time.h>
#endif /* __STDC__ */

#ifdef HAVE_TIME_H
#include <time.h>
#endif

void
Diary (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *string;
  Datum arg;
  FILE *fn;
  time_t r_time;

  if (n_args > 1)
    error_1 ("diary() takes at most a single argument", (char *) 0);

  /* Handle diary() args */
  if (n_args == 1)
  {
    if (write_diary)
      error_1 ("only one diary file may be open at a time", (char *) 0);
    arg = get_bltin_arg ("diary", d_arg, 1, STRING);
    string = string_GetString (e_data (arg.u.ent));
  }
  else
  {				/* No args */
    if (write_diary)
    {
      /*
         * n_args = 0, and a diary file is already open
         * close the existing diary file, and return
       */
      close_file_ds (diary_filenm);
      write_diary = 0;
      diary_file_ptr = 0;
      *return_ptr = (VPTR) scalar_Create ((double) 1.0);
      return;
    }
    else
      /*
         * n_args = 0, and no diary file open, open the
         * default diary file "DIARY"
       */
      string = cpstr ("DIARY");
  }

  if ((fn = get_file_ds (string, "w", 0)) == 0)
  {
    warning_1 (string, "cannot open for write");
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }

  /* Set external variables */
  write_diary = 1;
  diary_file_ptr = fn;
  diary_filenm = cpstr (string);

  /* Write out a header to diary file */
  r_time = time (0);
  fprintf (fn, "RLaB diary file: %s. Opened %s\n",
	   string, ctime (&r_time));

  *return_ptr = (VPTR) scalar_Create ((double) 1.0);
}

void btree_write _PROTO ((Btree * btree, FILE * fn));

/* **************************************************************
 * Write an object to a file.
 * ************************************************************** */
void
Write (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int i;
  char *string;
  Datum arg;
  FILE *fn;

  /* Check n_args */
  if (n_args > 1)
  {
    /* Standard user-specified write() */
  }
  else
  {
    warning_1 ("write() requires at least 2 arg", (char *) 0);
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }

  /* get file name from argument list */
  arg = get_bltin_arg ("write", d_arg, 1, STRING);
  string = string_GetString (e_data (arg.u.ent));

  if ((fn = get_file_ds (string, "w", 0)) == 0)
  {
    warning_1 (string, "cannot open for write");
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }

  /* Now move through arg list writing objects to file */
  for (i = 2; i <= n_args; i++)
  {
    arg = get_bltin_arg ("write", d_arg, i, 0);
    switch (arg.type)
    {
    case CONSTANT:
      scalar_Write (scalar_Create (arg.u.val), fn);
      break;
    case iCONSTANT:
      scalar_Write (scalar_CreateC (0.0, arg.u.val), fn);
      break;
    case ENTITY:
      switch (e_type (arg.u.ent))
      {
      case SCALAR:
	scalar_Write (e_data (arg.u.ent), fn);
	break;
      case MATRIX:
	matrix_Write (e_data (arg.u.ent), fn);
	break;
      case BTREE:
	btree_write (e_data (arg.u.ent), fn);
	break;
      case STRING:
	string_Write (e_data (arg.u.ent), fn);
	break;
      case UNDEF:
	break;
      default:
	warning_1 (e_name (arg.u.ent), "invalid object for write()");
	break;
      }
      break;
    default:
      warning_1 ("invalid object for write()", (char *) 0);
      break;
    }
  }
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

void
btree_write (btree, fn)
     Btree *btree;
     FILE *fn;
{
  btree_Write (btree, fn);
  fflush (fn);
}

/* **************************************************************
 * Read RLaB objects from a file. The default behavior is to read
 * ALL of the objects in the file, and install them on the GLOBAL
 * list. If the file contains a LIST, it is also put on the GLOBAL
 * list.
 * ************************************************************** */
void
Read (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *name, *string, type[20];
  Datum arg, arg2;
  FILE *fn;
  Btree *btree, *symtab;
  ListNode *lnode;
  Matrix *m;
  Scalar *s;
  String *str;

  /* Check n_args */
  if (n_args > 2)
    error_1 ("read: wrong number of input arguments", 0);

  /* get file name from argument list, always the 1st argument */
  arg = get_bltin_arg ("read", d_arg, 1, STRING);
  string = string_GetString (e_data (arg.u.ent));

  if (n_args == 1)
  {
    /* Use the global symbol table */
    symtab = get_symtab_ptr ();
  }
  else
  {
    /* Use a user-supplied list */
    arg2 = d_arg[1];
    
    /* Check to see if variable exists, etc... */
    if (arg.type != ENTITY)
      error_1 ("read: 2nd argument must be a variable", 0);
    if (e_type (arg2.u.ent) != UNDEF)
      listNode_DestroyDataOnly (arg2.u.ent);
    symtab = btree_Create ();
    btree_SetName (symtab, cpstr (e_name (arg2.u.ent)));
    listNode_AttachData (arg2.u.ent, BTREE, symtab, btree_Destroy);
  }
    
  if ((fn = get_file_ds (string, "r", 0)) == 0)
  {
    warning_1 (string, "cannot open for read");
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }

  /* 
   * Save current position in file, we will need to go back to
   * this point when we hand the read task off to the object
   * libraries
   */
  while (fscanf (fn, "# %s :", type) != EOF)
  {
    if (!strcmp ("scalar", type))
    {
      s = scalar_Read (fn);
      name = scalar_GetName (s);
      if ((lnode = btree_FindNode (symtab, name)) != 0)
      {
	listNode_DestroyDataOnly (lnode);
	listNode_AttachData (lnode, SCALAR, s, scalar_Destroy);
      }
      else
	install (symtab, cpstr (name), SCALAR, s);
    }
    else if (!strcmp ("matrix", type))
    {
      m = matrix_Read (fn);
      name = matrix_GetName (m);
      if ((lnode = btree_FindNode (symtab, name)) != 0)
      {
	listNode_DestroyDataOnly (lnode);
	listNode_AttachData (lnode, MATRIX, m, matrix_Destroy);
      }
      else
	install (symtab, cpstr (name), MATRIX, m);
    }
    else if (!strcmp ("string", type))
    {
      str = string_Read (fn);
      name = string_GetName (str);
      if ((lnode = btree_FindNode (symtab, name)) != 0)
      {
	listNode_DestroyDataOnly (lnode);
	listNode_AttachData (lnode, STRING, str, string_Destroy);
      }
      else
	install (symtab, cpstr (name), STRING, str);
    }
    else if (!strcmp ("list", type))
    {
      btree = btree_Read (fn);
      name = btree_GetName (btree);
      if ((lnode = btree_FindNode (symtab, name)) != 0)
      {
	listNode_DestroyDataOnly (lnode);
	listNode_AttachData (lnode, BTREE, btree, btree_Destroy);
      }
      else
	install (symtab, cpstr (name), BTREE, btree);
    }
    else
    {
      close_file_ds (string);
      error_1 (type, "do not know how to read this type");
    }
  }

  close_file_ds (string);

  /* Maybe we should return the list we wrote to ? */
  *return_ptr = (VPTR) scalar_Create (1.0);
  return;
}

/* **************************************************************
 * Reshape a matrix.
 * ************************************************************** */
void
Reshape (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  double drow, dcol;
  Datum arg1, arg2, arg3, targ1;

  if (n_args != 3)
    error_1 ("reshape() requires 3 arguments", (char *) 0);

  arg1 = get_bltin_arg ("reshape", d_arg, 1, NUM);
  arg2 = get_bltin_arg ("reshape", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("reshape", d_arg, 3, NUM);
  targ1 = convert_to_matrix (arg1);
  drow = get_num_scalar_val (arg2);
  dcol = get_num_scalar_val (arg3);

  *return_ptr = (VPTR)
    matrix_Reshape (e_data (targ1.u.ent), (int) drow, (int) dcol);

  TARG_DESTROY (arg1, targ1);
  return;
}

/* **************************************************************
 * Show the attributes of an object.
 * ************************************************************** */
void
Show (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char str[80];
  Datum arg;
  Matrix *m = 0;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to show()", 0);

  /* Get object from argument list */
  arg = get_bltin_arg ("show", d_arg, 1, 0);

  if (arg.type != ENTITY)
    error_1 ("show takes an entity as an argument", 0);

  switch (e_type (arg.u.ent))
  {
  case SCALAR:
    m = matrix_CreateS (5, 1);

    if (e_name (arg.u.ent) == 0)
      sprintf (str, "   name:      %s", "NULL");
    else
      sprintf (str, "   name:      %s", e_name (arg.u.ent));
    MATs (m, 1, 1) = cpstr (str);

    sprintf (str, "   class:     num");
    MATs (m, 2, 1) = cpstr (str);

    if (SVALi (e_data (arg.u.ent)) == 0.0)
      sprintf (str, "   type:      real");
    else
      sprintf (str, "   type:      complex");
    MATs (m, 3, 1) = cpstr (str);

    sprintf (str, "     nr:      %d", 1);
    MATs (m, 4, 1) = cpstr (str);

    sprintf (str, "     nc:      %d", 1);
    MATs (m, 5, 1) = cpstr (str);

    break;

  case MATRIX:
    m = matrix_CreateS (5, 1);

    if (e_name (arg.u.ent) == 0)
      sprintf (str, "   name:      %s", "NULL");
    else
      sprintf (str, "   name:      %s", e_name (arg.u.ent));
    MATs (m, 1, 1) = cpstr (str);

    if (MTYPE (e_data (arg.u.ent)) != STRING)
    {
      sprintf (str, "   class:     num");
      MATs (m, 2, 1) = cpstr (str);

      if (MTYPE (e_data (arg.u.ent)) == REAL)
	sprintf (str, "   type:      real");
      else if (MTYPE (e_data (arg.u.ent)) == COMPLEX)
	sprintf (str, "   type:      complex");
      MATs (m, 3, 1) = cpstr (str);
    }
    else
    {
      sprintf (str, "   class:     string");
      MATs (m, 2, 1) = cpstr (str);

      sprintf (str, "   type:      string");
      MATs (m, 3, 1) = cpstr (str);
    }

    sprintf (str, "     nr:      %d", MNR (e_data (arg.u.ent)));
    MATs (m, 4, 1) = cpstr (str);

    sprintf (str, "     nc:      %d", MNC (e_data (arg.u.ent)));
    MATs (m, 5, 1) = cpstr (str);

    break;

  case STRING:
    m = matrix_CreateS (5, 1);

    if (e_name (arg.u.ent) == 0)
      sprintf (str, "   name:      %s", "NULL");
    else
      sprintf (str, "   name:      %s", e_name (arg.u.ent));
    MATs (m, 1, 1) = cpstr (str);

    sprintf (str, "   class:     string");
    MATs (m, 2, 1) = cpstr (str);

    sprintf (str, "    type:     string");
    MATs (m, 3, 1) = cpstr (str);

    sprintf (str, "     nr:      %d", 1);
    MATs (m, 4, 1) = cpstr (str);

    sprintf (str, "     nc:      %d", 1);
    MATs (m, 5, 1) = cpstr (str);

    break;

  case BTREE:
    m = matrix_CreateS (3, 1);

    if (e_name (arg.u.ent) == 0)
      sprintf (str, "   name:      %s", "NULL");
    else
      sprintf (str, "   name:      %s", e_name (arg.u.ent));
    MATs (m, 1, 1) = cpstr (str);

    sprintf (str, "   class:     list");
    MATs (m, 2, 1) = cpstr (str);

    sprintf (str, "       n:     %d",
	     btree_GetRealNumNodes (e_data (arg.u.ent)));
    MATs (m, 3, 1) = cpstr (str);

    break;

  case BLTIN:
    m = matrix_CreateS (3, 1);

    if (e_name (arg.u.ent) == 0)
      sprintf (str, "   name:      %s", "NULL");
    else
      sprintf (str, "   name:      %s", e_name (arg.u.ent));
    MATs (m, 1, 1) = cpstr (str);

    sprintf (str, "   class:     function");
    MATs (m, 2, 1) = cpstr (str);

    sprintf (str, "    type:     builtin");
    MATs (m, 3, 1) = cpstr (str);

    break;

  case U_FUNCTION:
    m = matrix_CreateS (3, 1);

    if (e_name (arg.u.ent) == 0)
      sprintf (str, "   name:      %s", "NULL");
    else
      sprintf (str, "   name:      %s", e_name (arg.u.ent));
    MATs (m, 1, 1) = cpstr (str);

    sprintf (str, "   class:     function");
    MATs (m, 2, 1) = cpstr (str);

    sprintf (str, "    type:     user");
    MATs (m, 3, 1) = cpstr (str);

    break;

  default:
    error_1 ("invalid type for show()", 0);
    break;
  }
  *return_ptr = (VPTR) m;
}

/* **************************************************************
 * User's error() function. longjmp() back to prompt.
 * ************************************************************** */
void
Error (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;

  if (n_args == 0)
    /* Default error message */
    error_1 ("USER-RAISED-ERROR", (char *) 0);
  else
  {
    arg = get_bltin_arg ("error", d_arg, 1, STRING);
    error_1 (string_GetString (e_data (arg.u.ent)), (char *) 0);
  }
}

/* **************************************************************
 * Read a generic matrix from a file. The format is:
 * All lines:    whitespace separated numbers.
 * ************************************************************** */
void
ReadM (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *string;
  int block_size = 0;
  Datum arg, arg2, targ2;
  FILE *fn;
  Matrix *m;

  /* Check n_args */
  if (n_args != 1 && n_args != 2)
    error_1 ("Wrong number of args to readm()", 0);

  /* Get file name from argument list */
  arg = get_bltin_arg ("readm", d_arg, 1, STRING);
  string = string_GetString (e_data (arg.u.ent));

  if (n_args == 2)
  {
    /* Get block_size from argument list */
    arg2 = get_bltin_arg ("readm", d_arg, 2, NUM);
    targ2 = convert_to_matrix (arg2);
    block_size = (int) MAT (e_data (targ2.u.ent), 1, 1);
    TARG_DESTROY (arg2, targ2);
  }

  /* Try and open() the file */
  if ((fn = get_file_ds (string, "r", 0)) == 0)
  {
    warning_1 (string, "cannot open for read");
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }

  if ((m = matrix_ReadGeneric (fn, block_size)) == 0)
  {
    close_file_ds (string);
    error_1 ("readm: error during read", 0);
  }
  close_file_ds (string);
  *return_ptr = (VPTR) m;
}

void
WriteM (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *string;
  Datum arg1, arg2;
  FILE *fn;
  Matrix *m;

  /* Check n_args */
  if (n_args != 2)
    error_1 ("Wrong number of args to writem()", (char *) 0);

  /* get file name from argument list */
  arg1 = get_bltin_arg ("writem", d_arg, 1, STRING);
  string = string_GetString (e_data (arg1.u.ent));
  arg2 = get_bltin_arg ("writem", d_arg, 2, MATRIX);
  m = (Matrix *) e_data (arg2.u.ent);

  /* Try and open() the file */
  if ((fn = get_file_ds (string, "w", 0)) == 0)
  {
    warning_1 (string, "cannot open for write");
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }

  matrix_WriteGeneric (m, fn);
  *return_ptr = (VPTR) scalar_Create (1.0);
}

/* **************************************************************
 * Return 1 if any value of a matrix is non-zero.
 * Return 0 otherwise.
 * ************************************************************** */
void
Any (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to any()", (char *) 0);

  /* get file name from argument list */
  arg = get_bltin_arg ("any", d_arg, 1, 0);

  switch (arg.type)
  {
  case CONSTANT:
    if (arg.u.val != 0.0)
      *return_ptr = (VPTR) scalar_Create (1.0);
    else
      *return_ptr = (VPTR) scalar_Create (0.0);
    return;

  case iCONSTANT:
    if (SVALr (e_data (arg.u.ent)) != 0.0)
      *return_ptr = (VPTR) scalar_Create (1.0);
    else
      *return_ptr = (VPTR) scalar_Create (0.0);
    return;

  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      if (SVALr (e_data (arg.u.ent)) != 0.0
	  || SVALi (e_data (arg.u.ent)) != 0.0)
	*return_ptr = (VPTR) scalar_Create (1.0);
      else
	*return_ptr = (VPTR) scalar_Create (0.0);
      return;

    case MATRIX:
      *return_ptr = (VPTR) matrix_Any (e_data (arg.u.ent));
      return;
      break;

    default:
      error_1 ("invalid entity type as arg to any()", (char *) 0);
      break;
    }
  }
}

/* **************************************************************
 * Return 1 if all values of a matrix are non-zero
 * Return 0 otherwise.
 * ************************************************************** */
void
All (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to all()", (char *) 0);

  /* get file name from argument list */
  arg = get_bltin_arg ("all", d_arg, 1, 0);

  switch (arg.type)
  {
  case CONSTANT:
    if (arg.u.val != 0.0)
      *return_ptr = (VPTR) scalar_Create (1.0);
    else
      *return_ptr = (VPTR) scalar_Create (0.0);
    return;

  case iCONSTANT:
    if (SVALr (e_data (arg.u.ent)) != 0.0)
      *return_ptr = (VPTR) scalar_Create (1.0);
    else
      *return_ptr = (VPTR) scalar_Create (0.0);
    return;

  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      if (SVALr (e_data (arg.u.ent)) != 0.0
	  || SVALi (e_data (arg.u.ent)) != 0.0)
	*return_ptr = (VPTR) scalar_Create (1.0);
      else
	*return_ptr = (VPTR) scalar_Create (0.0);
      return;

    case MATRIX:
      *return_ptr = (VPTR) matrix_All (e_data (arg.u.ent));
      return;
      break;

    default:
      error_1 ("invalid entity type as arg to all()", (char *) 0);
      break;
    }
  }
}

/* **************************************************************
 * Clear a variable's data.
 * ************************************************************** */
void
Clear (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int i;
  Datum arg;

  /* Check n_args */
  if (n_args == 0)
    error_1 ("clear() requires at least 1 arg", 0);

  for (i = 1; i <= n_args; i++)
  {
    arg = get_bltin_arg ("clear", d_arg, i, 0);

    if ((arg.type == CONSTANT) || (arg.type == iCONSTANT))
      error_1 ("clear() will not operate on CONSTANTS", 0);

    if (e_type (arg.u.ent) == BLTIN)
      error_1 (e_name (arg.u.ent), "cannot clear() a built-in function");

    listNode_DestroyDataOnly (arg.u.ent);
    listNode_SetType (arg.u.ent, UNDEF);
  }
  *return_ptr = (VPTR) scalar_Create ((double) i - 1);
}

/* **************************************************************
 * Get built-in arguments in an automated way for users.
 * The user passes in a pointer to the beginning of the argument
 * array, the number of the arg requested:
 *          bltin(1; 2; 3; 4; 5; ...)
 * and the type of data that SHOULD be returned. If any type of
 * data can be returned, then a 0 is passed for the value of type.
 * The returned value is a Datum.
 * ************************************************************** */
Datum
get_bltin_arg (bltin_name, arg_array, arg_number, type)
     char *bltin_name;
     Datum *arg_array;
     int arg_number, type;
{
  Datum d;
  char *str;
  int e;

  /* get the requested arg */
  d = arg_array[arg_number - 1];

  /*
   * If user did not specify an arg-type, just return it,
   * UNLESS the arg is UNDEFINED, then complain.
   */

  if (type == 0)
  {
    if (d.type == ENTITY && (e_type (d.u.ent) == UNDEF))
      error_1 (e_name (d.u.ent), "UNDEFINED");
    else
      return (d);
  }

  /* Set type-string for error output */
  switch (type)
  {
  case SCALAR:
    str = cpstr ("SCALAR");
    break;
  case MATRIX:
    str = cpstr ("MATRIX");
    break;
  case BTREE:
    str = cpstr ("LIST");
    break;
  case STRING:
    str = cpstr ("STRING");
    break;
  case BLTIN:
    str = cpstr ("BUILT-IN");
    break;
  case U_FUNCTION:
    str = cpstr ("USER-FUNCTION");
    break;
  case UNDEF:
    str = cpstr ("UNDEFINED");
    break;
  case NUM:
    str = cpstr ("NUM");
    break;
  default:
    str = cpstr ("what?");
    break;
  }

  e = 0;

  /* Now check arg against desired type */
  switch (d.type)
  {
  case CONSTANT:
    if (!((type == SCALAR) || (type == CONSTANT) || (type == NUM)))
      error_2 (bltin_name,
	       "wrong type of arg to built-in should be:", str);
    break;
  case iCONSTANT:
    if (!((type == SCALAR) || (type == CONSTANT) || (type == NUM)))
      error_2 (bltin_name,
	       "wrong type of arg to built-in should be:", str);
    break;
  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      if (type != SCALAR && type != NUM)
	e = 1;
      break;
    case MATRIX:
      if (type != MATRIX && type != NUM)
	e = 1;
      break;
    case STRING:
      if (type != STRING)
	e = 1;
      break;
    case BTREE:
      if (type != BTREE)
	e = 1;
      break;
    case BLTIN:
      if (type != BLTIN)
	e = 1;
      break;
    case U_FUNCTION:
      if (type != U_FUNCTION)
	e = 1;
      break;
    case UNDEF:
      if (type != UNDEF)
	e = 1;
      break;
    default:
      error_1 (bltin_name, "wrong type of entity to pass to built-in");
      break;
    }
    break;
  default:
    error_1 (bltin_name, "invalid argument (Datum) passed to built-in");
  }

  if (e == 1)
    error_3 (bltin_name, e_name (d.u.ent),
	     "wrong type of arg to built-in should be:", str);

  FREE (str);
  return (d);
}
