#ifndef lint
static char SCCSid[] = "@(#) ./vectors/dvec.c 07/23/93";
#endif

/*
   This file defines the vector operations in the simplest way possible
   (that still allows for general user contexts):  The first field in the
   user context contains size of the vector, and the actual vector type
   is no more than a pointer to the data 

   These routines are for standard double precision serial vectors.
 */

#include "tools.h"
#include "system/flog.h"
#include "inline/dot.h"
#include "inline/vmult.h"
#include "inline/setval.h"
#include "inline/copy.h"
#include "inline/axpy.h"

#include <math.h>
#include "vectors/vector.h"              /*I "vectors/vector.h" I*/
#include "vectors/dvec.h"                /*I "vectors/dvec.h" I*/

/*@
   DVdot - Dot product for serial double precision vectors.

   Input Parameters:
.  N     - Vector context
.  x,y   - vectors to form dot product of

   Output Parameter:
.  z     - result
 @*/
void DVdot( N, x, y, z )
VEDefaultUsrCntx *N;
register double *x, *y, *z;
{
register int n = N->n;
register double sum = 0.0;
DOT(sum,x,y,n);
*z = sum;
}
/*@
   DVnorm - Two norm for serial double precision vectors.

   Input Parameters:
.  N     - Vector context
.  x     - vector to form 2-norm of

   Output Parameter:
.  z     - result
 @*/
void DVnorm( N, x, z )
VEDefaultUsrCntx *N;
register double *x, *z;
{
register int n = N->n;
register double sum = 0.0;
SQR(sum,x,n);
*z = sqrt( sum );
}
/*@
   DVmax - Max norm for serial double precision vectors.

   Input Parameters:
.  N     - Vector context
.  x     - vector to form max-norm of
.  idx   - index of maximum element.  If there are several elements with the
           maximum value, the index of one (probably but no necessarily the
	   first) is returned.

   Output Parameter:
.  z     - result
 @*/
void DVmax( N, x, idx, z )
VEDefaultUsrCntx *N;
int              *idx;
register double  *x, *z;
{
register int i, j=0, n = N->n;
register double max = 0.0, tmp;
for (i=0; i<n; i++) {
    if ( (tmp = *x++) > 0.0 ) { if (tmp > max) { j = i; max = tmp; } }
    else                      { if (-tmp > max) { j = i; max = -tmp; } }
    }
  *z   = max;
  if (idx)
      *idx = j;
}

/*@
   DVscal - Multiply a serial double precision vector by a scalar.

   Input Parameters:
.  N     - Vector context
.  alpha - scale factor
.  x     - vectors to scale
 @*/
void DVscal( N, alpha, x )
VEDefaultUsrCntx *N;
register double alpha, *x;
{
register int n = N->n;
VSCALIP(x,alpha,n);
}

/*@
   DVcopy - Copy a serial double precision vector.

   Input Parameters:
.  N     - Vector context
.  x     - vector to copy

   Output Parameter:
.  y     - copy of x
 @*/
void DVcopy( N, x, y )
VEDefaultUsrCntx *N;
register double *x, *y;
{
register int n = N->n;
COPY(y,x,n);
}

/*@
   DVswap - Swap two serial double precision vectors.

   Input Parameters:
.  N     - Vector context
.  x,y   - vectors to swap
 @*/
void DVswap( N, x, y )
VEDefaultUsrCntx *N;
register double *x, *y;
{
register int n = N->n, i;
register double t;
for (i=0; i<n; i++) {
    t    = x[i];
    x[i] = y[i];
    y[i] = t;
    }
}

/*@
   DVset - Set the elements in a serial double precision vector
   to a scalar.

   Input Parameters:
.  N     - Vector context
.  alpha - value to set every element in vector to
.  x     - vector to set
 @*/
void DVset( N, alpha, x )
VEDefaultUsrCntx *N;
register double alpha, *x;
{
register int n = N->n;
SET(x,n,alpha);
}

/*@
   DVaxpy -  y = alpha*x + y where x and y are serial double 
   precision vectors. 

   Input Parameters:
.  N     - Vector context
.  alpha - multiplier
.  x,y   - vectors
 @*/
void DVaxpy( N, alpha, x, y )
VEDefaultUsrCntx *N;
register double alpha, *x, *y;
{
register int n = N->n;
/* This should really look at the case alpha = +1 as well */
if (alpha == -1.0) {
    YMX(y,x,n);
    }
else {
    APXY(y,alpha,x,n);
    }
}

/*@
   DVaypx -  y = x + alpha*y where x and y are serial double 
   precision serial vectors. 

   Input Parameters:
.  N     - Vector context
.  alpha - multiplier
.  x,y   - vectors
 @*/
void DVaypx( N, alpha, x, y )
VEDefaultUsrCntx *N;
register double alpha, *x, *y;
{
register int n = N->n;
AYPX(y,alpha,x,n);
}

/*@
   DVwaxpy -  w = alpha*x + y where w, x and y are serial double 
   precision vectors. 

   Input Parameters:
.  N     - Vector context
.  alpha - multiplier
.  w,x,y  - vectors
 @*/
void DVwaxpy( N, alpha, x, y, w )
VEDefaultUsrCntx *N;
register double alpha, *w, *x, *y;
{
register int i, n = N->n;
if (alpha == 1.0) {
    for (i=0; i<n; i++) 
	w[i] = y[i] + x[i];
    }
else {
    for (i=0; i<n; i++) 
	w[i] = y[i] + alpha * x[i];
    }
}

/*@
   DVpmult - Multiply the elements of one vector by another.

   Input Parameters:
.  N     - Vector context
.  x,y   - vectors to multiply together

   Output Parameter:
.  w     - w(i) <- x(i) * y(i)
 @*/
void DVpmult( N, x, y, w )
VEDefaultUsrCntx *N;
register double *x, *y, *w;
{
register int n = N->n, i;
for (i=0; i<n; i++) 
    w[i] = x[i] * y[i];
}

/*@
   DVpdiv - Divide the elements on one vector by another.

   Input Parameters:
.  N     - Vector context
.  x,y   - vectors to divide

   Output Parameter:
.  w     - w(i) <- x(i) / y(i)
 @*/
void DVpdiv( N, x, y, w )
VEDefaultUsrCntx *N;
register double *x, *y, *w;
{
register int n = N->n, i;
for (i=0; i<n; i++) 
    w[i] = x[i] / y[i];
}

#include "inline/spops.h"
/*@
   DVgather - Gather from sparse to dense for serial double precision vectors.

   Input Parameters:
.  N     - Vector context
.  x     - vector to gather from
.  ix    - indices to gather with
.  ni    - number of indices to gather

   Output Parameter:
.  y     - vector to gather to

   Notes:
   y[i] = x[ix[i]], for i=0, ..., ni-1.

 @*/
void DVgather( N, x, ix, ni, y )
VEDefaultUsrCntx *N;
register double  *x, *y;
register int     *ix, ni;
{
GATHER(y,ix,x,ni);
}

/*@
   DVscatter - Scatter from dense to sparse for serial double precision 
               vectors.

   Input Parameters:
.  N     - Vector context
.  x     - vector to scatter from
.  ix    - indices to scatter with
.  ni    - number of indices to scatter

   Output Parameter:
.  y     - vector to gather to

   Notes:
   y[ix[i]] = x[i], for i=0, ..., ni-1.

 @*/
void DVscatter( N, x, ix, ni, y )
VEDefaultUsrCntx *N;
register double  *x, *y;
register int     *ix, ni;
{
SCATTER(x,ix,y,ni);
}

/*@
   DVobtain_vectors - Returns a pointer to a list of pointers
   which point to newly obtained serial double precision vectors. 
   To obtain a single serial double precision vector use 
   DVCreateVector().

   Input Parameters:
.      N  - user context, first element is the vector length
.      m  - number of vectors to obtain
 @*/
void **DVobtain_vectors( N, m )
VEDefaultUsrCntx *N;
int              m;
{
void **v;
int  i, n = N->n;

v = (void **) MALLOC( m * sizeof(void *) );
for (i=0; i<m; i++)
    v[i] = (void *)MALLOC( n * sizeof(double) );
return v;
}
/*@
   DVrelease_vectors - Frees a set of serial double precision vectors
   obtained by DVobtain_vectors. 

   Input Parameters:
.     N  - user context, first element is the vector length
.     v  - pointer obtained from DVobtain_vectors
.     m  - number of vectors which were obtained
 @*/
void DVrelease_vectors( N, v, m )
VEDefaultUsrCntx *N;
int  m;
void **v;
{
int i;
for (i=0; i<m; i++)
    FREE( v[i] );
FREE( v );
}

/*@
   DVCreateVector - Returns a pointer to a newly created
   serial double precision vector. Use DVobtain_vectors() if 
   several vectors are needed.

   Input Parameters:
.     N  - user context, the first element must be the vector length
 @*/
void *DVCreateVector( N )
VEDefaultUsrCntx *N;
{
return (void *)MALLOC( N->n * sizeof(double) );
}

/*@
   DVDestroyVector -  Frees a serial double precision vector obtained
   with DVCreateVector().

  Input Parameters:
.    v  -  pointer to vector
 @*/
void DVDestroyVector( v )
void *v;
{
FREE(v);
}

/*@
  DVSetDefaultFunctions - Set vector operations for serial 
  double precision vectors.
 
  Input Parameters:
.      vopP  - vector context obtained by VECreate()
 @*/
void DVSetDefaultFunctions( vopP )
VECntx *vopP;
{
vopP->dot    = DVdot;
vopP->norm   = DVnorm;
vopP->max    = DVmax;
vopP->tdot   = DVdot;
vopP->scal   = DVscal;
vopP->copy   = DVcopy;
vopP->set    = DVset;
vopP->axpy   = DVaxpy;
vopP->aypx   = DVaypx;
vopP->swap   = DVswap;
vopP->waxpy  = DVwaxpy;
vopP->pmult  = DVpmult;
vopP->pdiv   = DVpdiv;
vopP->gather = DVgather;
vopP->scatter= DVscatter;
vopP->obtain_vectors = DVobtain_vectors;
vopP->release_vectors= DVrelease_vectors;
vopP->create_vector  = DVCreateVector;
vopP->destroy_vector = DVDestroyVector;
}
