#ifndef lint
static char SCCSid[] = "@(#) ./sparse/row/matmult.c 07/23/93";
#endif

#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"
#include "inline/spops.h"

void SpiMapMult(), SpiMapColMult(), SpiMapMultAdd(), SpiMapColMultAdd();

/*
   SpRMult - Routine for matrix-vector product.

   Input parameters:
.  mat - matrix to multiply with
.  v   - vector to multiply

   Output parameters:
.  vout - result vector
 */
void SpRMult( mat, v, vout )
SpMat  *mat;
double *v, *vout;
{
int      n;
SpVec    **rs, *row;
register double *xv, sum;
register int    nz, *xi;
SpRowMat *R;

SPLITTOMAT(mat);
if (mat->map) {
    if (mat->map->colmap) 
	SpiMapColMult( mat, v, vout );
    else 
	SpiMapMult( mat, v, vout );
    return;
    }

R = GETROWMAT(mat);

n      = mat->rows;
rs     = R->rs;
while (n--) {
    row  = *rs++;
    xv   = row->v;
    xi   = row->i;
    nz   = row->nz;
    sum  = 0.0;
    SPARSEDENSEDOT(sum,v,xv,xi,nz);
    *vout++ = sum;
    }
}

/*+
   SpiMapMult - Routine for matrix-vector product (with row-mapping)

   Input Parameters:
.  mat - matrix 
.  v   - input vector 

   Output Parameters:
.  vout - result vector   
 +*/
void SpiMapMult( mat, v, vout )
SpMat    *mat;
double   *v, *vout;
{
int n;
SpVec    **rs, *row;
register double *xv, sum;
register int    nz, *xi, *rowmap = mat->map->rowmap;
SpRowMat *R = GETROWMAT(mat);

n      = mat->rows;
rs     = R->rs;
while (n--) {
    row  = *rs++;
    xv   = row->v;
    xi   = row->i;
    nz   = row->nz;
    sum  = 0.0;
    SPARSEDENSEDOT(sum,v,xv,xi,nz);
    vout[*rowmap++] = sum;
    }
}

/*+
   SpiMapColMult - Routine for matrix-vector product (with column and
                   possibly row-mapping)

   Input Parameters:
.  mat - matrix 
.  v   - input vector 

   Output Parameters:
.  vout - result vector   
 +*/
void SpiMapColMult( mat, v, vout )
SpMat    *mat;
double   *v, *vout;
{
int n;
SpVec    **rs, *row;
register double *xv, sum;
register int    nz, *xi, *rowmap = mat->map->rowmap, *map=mat->map->colmap;
SpRowMat *R = GETROWMAT(mat);

n      = mat->rows;
rs     = R->rs;
while (n--) {
    row  = *rs++;
    xv   = row->v;
    xi   = row->i;
    nz   = row->nz;
    sum  = 0.0;
    SPARSEDENSEMAPDOT(sum,v,xv,xi,map,nz);
    if (rowmap) vout[*rowmap++] = sum;
    else        *vout++         = sum;
    }
}

/*
   SpRMultAdd - Routine for matrix-vector product with addition. 

   Input parameters:
.  mat - matrix to multiply with
.  v   - vector to multiply
.  v1  - vector to add to result

   Output parameters:
.  vout - result vector = mat*v + v1
 */
void SpRMultAdd( mat, v, v1, vout )
SpMat  *mat;
double *v, *v1, *vout;
{
int      n;
SpVec    **rs, *row;
register double *xv, sum;
register int    nz, *xi;
SpRowMat *R;

SPLITTOMAT(mat);
if (mat->map) {
    if (mat->map->colmap) SpiMapColMultAdd( mat, v, v1, vout );
    else SpiMapMultAdd( mat, v, v1, vout );
    return;
    }

R = GETROWMAT(mat);
n      = mat->rows;
rs     = R->rs;
while (n--) {
    row  = *rs++;
    xv   = row->v;
    xi   = row->i;
    nz   = row->nz;
    sum  = 0.0;
    SPARSEDENSEDOT(sum,v,xv,xi,nz);
    *vout++ = sum + *v1++;
    }
}

/*+
   SpiMapMultAdd - Routine for matrix-vector product (with row-mapping)

   Input parameters:
.  mat - matrix to multiply with
.  v   - vector to multiply
.  v1  - vector to add to result

   Output parameters:
.  vout - result vector = mat*v + v1
 +*/
void SpiMapMultAdd( mat, v, v1, vout )
SpMat    *mat;
double   *v, *v1, *vout;
{
int n;
SpVec    **rs, *row;
register double *xv, sum;
register int    nz, *xi, *rowmap = mat->map->rowmap;
SpRowMat *R = GETROWMAT(mat);

n      = mat->rows;
rs     = R->rs;
while (n--) {
    row  = *rs++;
    xv   = row->v;
    xi   = row->i;
    nz   = row->nz;
    sum  = 0.0;
    SPARSEDENSEDOT(sum,v,xv,xi,nz);
    vout[*rowmap] = sum + v1[*rowmap];
    rowmap++;
    }
}

/*+
   SpiMapColMultAdd - Routine for matrix-vector product (with column and
                   possibly row-mapping)

   Input parameters:
.  mat - matrix to multiply with
.  v   - vector to multiply
.  v1  - vector to add to result

   Output parameters:
.  vout - result vector = mat*v + v1
 +*/
void SpiMapColMultAdd( mat, v, v1, vout )
SpMat    *mat;
double   *v, *vout, *v1;
{
int n;
SpVec    **rs, *row;
register double *xv, sum;
register int    nz, *xi, *rowmap = mat->map->rowmap, *map=mat->map->colmap;
SpRowMat *R = GETROWMAT(mat);

n      = mat->rows;
rs     = R->rs;
while (n--) {
    row  = *rs++;
    xv   = row->v;
    xi   = row->i;
    nz   = row->nz;
    sum  = 0.0;
    SPARSEDENSEMAPDOT(sum,v,xv,xi,map,nz);
    if (rowmap) {
	vout[*rowmap] = sum + v1[*rowmap];
	rowmap++;
	}
    else        *vout++         = sum + *v1++;
    }
}

#ifdef FOO
/*
   SpRMultMMR - Sparse matrix * dense matrix in ROW major order.
 */
void SpRMultMMR( mat, nc, v, vout )
SpMat  *mat;
double *v, *vout;
int    nc
{
int      n;
SpVec    **rs, *row;
register double *xv, sum;
register int    nz, *xi;
SpRowMat *R;

SPLITTOMAT(mat);
if (mat->map) {
    nc = nc / 0;
    if (mat->map->colmap) SpiMapColMult( mat, v, vout );
    else SpiMapMult( mat, v, vout );
    return;
    }

R = GETROWMAT(mat);

n      = mat->rows;
rs     = R->rs;
while (n--) {
    row  = *rs++;
    xv   = row->v;
    xi   = row->i;
    nz   = row->nz;
    sum  = 0.0;
    SPARSEDENSEDOT(sum,v,xv,xi,nz);
    *vout++ = sum;
    }
}
#endif

/*
   SpRMultTrans - Routine for matrix-vector product, using the TRANSPOSE
                 of the matrix.

   Input parameters:
.  mat - matrix to multiply with
.  v   - vector to multiply

   Output parameters:
.  vout - result vector = mat*v
 */
void SpRMultTrans( mat, v, vout )
SpMat  *mat;
double *v, *vout;
{
int      n, m;
SpVec    **rs, *row;
register double *xv, alpha;
register int    i, nz, *xi;
SpRowMat *R;

SPLITTOMAT(mat);
if (mat->map) {
    SETERRC(1,"SpRMultTrans does not support mapped matrices");
    return;   /* Broken, need an error return */
    }

R = GETROWMAT(mat);

n      = mat->rows;
m      = mat->cols;
rs     = R->rs;
for (i=0; i<m; i++) vout[i] = 0.0;
while (n--) {
    row  = *rs++;
    xv   = row->v;
    xi   = row->i;
    nz   = row->nz;
    alpha= - *v++;
    SPARSEDENSESMAXPY(vout,alpha,xv,xi,nz);
    }
}

#ifdef FOO
/*
   SpRMultBlk - Routine for matrix-vector product.
   (unmapped for now)

   Input parameters:
.  mat - matrix to multiply with
.  v   - vector to multiply

   Output parameters:
.  vout - result vector = mat*v
 */
void SpRMultBlk( mat, v, vout )
SpMat  *mat;
double *v, *vout;
{
int      n, r, pbi, ninblock, j;
SpVec    **rs, *row;
register double *xv, sum;
register int    nz, *xi;
SpRowMat *R;
/* This is the blocked row structure, if it exists */
SpRowBlock   *blks;

SPLITTOMAT(mat);
if (mat->map) {
    SETERRC(1,"SpRMultBlk does not support mapped matrices");
    return;
    }

R    = GETROWMAT(mat);
blks = R->rblks;

n      = mat->rows;
rs     = R->rs;
for (r=0; r<n; r++) {
    row  = rs[r];
    pbi = row->rblki;
    if (pbi >= 0) {
	/* Do all of these rows at once (at least in blocks) */
	ninblock = blks->blocksize[pbi];
	xi = row->i;
	for (j=0; j<ninblock; j++) {
	    xv  = rs[r+j]->v;
	    sum = 0.0;
	    /* These may be done in blocks by doing:
	       a  = v[xi[k]];
	       s1 += a * xv1[k];
	       s2 += a * xv2[k]; ...
	       This will reduce the number of memory refs per item
	       significantly (since xi AND v[xi] are fetched once for 
	       each block)
	     */
	    /* ??? what is nz */
	    SPARSEDENSEDOT(sum,v,xv,xi,nz);
	    vout[r+j] = sum;
	    }
	r += ninblock - 1;
	}
    else {
	xv      = row->v;
	xi      = row->i;
	nz      = row->nz;
	sum     = 0.0;
	SPARSEDENSEDOT(sum,v,xv,xi,nz);
	vout[r] = sum;
	}
    }
}
#endif
