#ifndef lint
static char SCCSid[] = "@(#) ./solvers/dd/osm2d.c 07/23/93"; 
#endif

/* 
   This file contains routines to setup OSM for 2-d regular meshes.

   These directly exploit the structure of a 2-d mesh; a general approach
   would form the interpolation matrix and then use the transpose of that
   as the restriction operator.

   Later, we want 
   Dot products with:

   (1) bilinear pyramid grid functions on the subdomains cardinal at each
   coarse grid point for "full-weighting"

   (2) subdomain characteristic functions for straight averages

   (3) Kroneckers at the cross-points themselves

   (4) linear ramp grid functions on the edges (not a priority)

   (5) user-defined F(x,y) or F(i,j) (not a priority)

 */

#include "tools.h"
#include "solvers/svctx.h"

void Msh2dGlobalMonitor(), 
     Msh2dSetRegularDomains( ), 
     Msh2dSetRegularOverlap( );

typedef struct {
    int fx, fy; 
    int fgx, fgy; 
    int cx, cy;
    } SVOSM2dGrid;
void SViOSM2dInterpLinear( ctx, lctx, cx, x, ictx ) 
SVctx       *ctx;
SVOSMctx    *lctx;
double      *cx, *x;
SVOSM2dGrid *ictx;
{
Msh2dInterpolateBilinear( x, ictx->fx, ictx->fx, ictx->fy, 
			  cx, ictx->cx-1, ictx->cx, ictx->cy, 
			  ictx->fgx, ictx->fgy, 1 );
}

void SViOSM2dRestrictPC( ctx, lctx, x, cx, ictx )
SVctx       *ctx;
SVOSMctx    *lctx;
double      *cx, *x;
SVOSM2dGrid *ictx;
{
Msh2dRestrictPCopen( x, ictx->fx, ictx->fx, ictx->fy, 
		     cx, ictx->cx-1, ictx->cx, ictx->cy, 
		     ictx->fgx, ictx->fgy );
}

/* @
    SVOSM2dSetup - Setup overlapping Schwarz for a 2-d mesh

    Input Parameters:
.   ctx - Solver context
.   nc  - number of components (must be 1 for now)
.   mx,my - size of mesh
.   cx,cy - number of coarse-grid cells in each direction
.   cmat  - matrix for coarse-grid problem.  May be null
.   ovlp  - overlap amount

    Notes: 
    Call before SVSetup.
@ */
void SVOSM2dSetup( ctx, mx, my, nc, cx, cy, cmat, ovlp )
SVctx *ctx;
int   nc, mx, my, cx, cy, ovlp;
SpMat *cmat;
{
SVOSM2dGrid *ictx;

SVSetOSMDomainsNumber( ctx, cx*cy );
Msh2dSetRegularDomains( ctx, mx, my, cx, cy, nc );
Msh2dSetRegularOverlap( ctx, mx, my, cx, cy, nc, ovlp, ovlp );
if (cmat && cx > 1 && cy > 1) {
    ictx = NEW(SVOSM2dGrid);   CHKPTR(ictx);
    ictx->cx  = cx;
    ictx->cy  = cy;
    ictx->fgx = 1 + (mx + 1) / cx;
    ictx->fgy = 1 + (my + 1) / cy;
    ictx->fx  = mx;
    ictx->fy  = my;
    SVSetOSMGlobalMatrix( ctx, cmat );
    SVSetOSMGlobalInterpolation( ctx, SViOSM2dInterpLinear, ictx );
    SVSetOSMGlobalRestriction( ctx, SViOSM2dRestrictPC, ictx );
    }
EZGopen( "osm.cit" );
SVSetOSMGlobalMonitor( ctx, Msh2dGlobalMonitor, ictx );
/* SVSetOSMLocalMonitor( ctx, Msh2dLocalMonitor, ictx ); */
}


/* Default global monitor */
#include "solvers/svctx.h"
void Msh2dGlobalMonitor( ctx, n1, n2, v1, v2, gctx, phase )
SVctx  *ctx;
int    n1, n2, phase;
double *v1, *v2;
SVOSM2dGrid *gctx;
{
int n1x, n1y, n2x, n2y;    
switch (phase) {
    case 0: n1x = gctx->fx;   n1y = gctx->fy; 
            n2x = gctx->cx-1; n2y = gctx->cy - 1;
            EZGtitle( "After restriction" );
            break;
    case 1: n1x = n2x = gctx->cx - 1;
            n1y = n2y = gctx->cy - 1;
            EZGtitle( "After global solve" );
            break;
    case 2: n1x = gctx->cx - 1; n1y = gctx->cy - 1;
            n2x = gctx->fx;     n2y = gctx->fy;
            EZGtitle( "After interpolation" );
            break;
    }
EZGgrid( n1x, n1y, v1 );
EZGsurface();  EZGeop();
EZGgrid( n2x, n2y, v2 );
EZGsurface();  EZGeop();
}

void Msh2dSetRegularDomains( ctx, mx, my, cx, cy, nc )
SVctx *ctx;
int   mx, my, cx, cy, nc;
{
int i, j, si, ei, sj, ej, *idx, fgx, fgy, cnt;

fgx = (mx + 1) / cx;
fgy = (my + 1) / cy;
/* Test for even? */
/* assume open for the moment */
/* This is WRONG; it needs to allocate enough space if the domain is 
   irregular ( (mx+1) % cx != 0) .  Ditto for overlap */
idx = (int *)MALLOC( (fgx+2)*(fgy+2)*sizeof(int) );  CHKPTR(idx);
for (j=0; j<cy; j++) {
    sj = j * fgy;
    ej = (j+1) * fgy - 2;
    if (sj > 0) sj--;
    if (ej < my-1) ej++;
    if (j == cy-1) ej = my - 1;
    for (i=0; i<cx; i++) {
	si = i * fgx;
	ei = (i+1) * fgx - 2;
	if (si > 0) si--;
	if (ei < mx-1) ei++;
	if (i == cx-1) ei = mx - 1;
	/* printf( "[%d,%d] x [%d,%d]\n", si, sj, ei, ej ); */
	cnt = SViNumber2dDomain( sj, ej, si, ei, nc, mx, idx );
	SViSetOSMDecomp( ctx, i+j*cx, idx, cnt );
	}
    }
FREE( idx );
}

void Msh2dSetRegularOverlap( ctx, mx, my, cx, cy, nc, ovlp )
SVctx *ctx;
int   mx, my, cx, cy, nc, ovlp;
{
int i, j, si, ei, sj, ej, *idx, fgx, fgy, cnt;

fgx = (mx + 1) / cx;
fgy = (my + 1) / cy;
/* Test for even? */
/* assume open for the moment */
idx = (int *)MALLOC( (fgx+2+2*ovlp)*(fgy+2+2*ovlp)*sizeof(int) );  CHKPTR(idx);
for (j=0; j<cy; j++) {
    sj = j * fgy;
    ej = (j+1) * fgy - 2;
    if (sj > 0) sj--;
    if (ej < my-1) ej++;
    sj -= ovlp; 
    if (sj < 0) sj = 0;
    ej += ovlp;
    if (ej >= my) ej = my-1;
    if (j == cy-1) ej = my - 1;
    for (i=0; i<cx; i++) {
	si = i * fgx;
	ei = (i+1) * fgx - 2;
	if (si > 0) si--;
	if (ei < mx-1) ei++;
	si -= ovlp;
	if (si < 0) si = 0;
	ei += ovlp;
	if (ei >= mx-1) ei = mx-1;
	if (i == cx-1) ei = mx - 1;
	/* printf( "[%d,%d] x [%d,%d]\n", si, sj, ei, ej ); */
	cnt = SViNumber2dDomain( sj, ej, si, ei, nc, mx, idx );
	SViSetOSMOverlapDecomp( ctx, i+j*cx, idx, cnt );
	}
    }
FREE( idx );
}

/* 
   This routine dumps out the domains as index sets.
 */
void Msh2dDumpDomains( ctx )
SVctx *ctx;
{
SVOSMctx   *lctx = (SVOSMctx *)(ctx->private);
SVOSMBlock *BB;
int        i, nd = lctx->nd;

for (i=0; i<nd; i++) {
    BB = lctx->bb + i;
    ISiPrint( stdout, BB->ovidx, BB->ovnv );
    }
}
