/*
        @(#) ./iter/itctx.h 07/23/93
	Context variable used by all the Iterative Solvers.
 */

#ifndef _ITCONTEXT
#define _ITCONTEXT
#include "vectors/vector.h"

/* 
   Known methods
 */
typedef enum { ITRICHARDSON, ITCHEBYCHEV, ITCG, ITGMRES, 
               ITCGSPLIT, ITTCQMR, ITBCGS, ITCGS, ITTFQMR, ITLSQR,
	       ITPREONLY } ITMETHOD ;

/*
   Iterative method context.

   The approach used here for the vector operations needs to be modified.
   We should separate the vector context from the user-context, in the
   same manner as the Sparse matrix context.  The vector context can
   then contain all of the data that is needed in one place, rather than
   (as currently) split between the usrP and the context.

   Also, the usrP should probably be removed from the calling sequences
   and stashed within the itP.
 */
typedef struct {
  /*------------------------- User parameters--------------------------*/
  int max_it,                      /* maximum number of iterations */
      right_inv,                   /* flag for right preconditioning */
      guess_zero,                  /* flag for whether initial guess is 0 */
      calc_eigs,                   /* calculate extreme eigenvalues */
      calc_res,                    /* calculate residuals at each iteration*/
      use_pres;                    /* use preconditioned residual */

  double rtol,                     /* relative tolerance */
         atol,                     /* absolute tolerance */
         ttol;                     /* (not set by user)  */
  double rnorm0;                   /* initial residual norm 
				      (used for divergence testing) */

  void *vec_sol, *vec_rhs;         /* pointer to where user has stashed 
                                      the solution and rhs, these are 
                                      never touched by the code, only 
                                      passed back to the user */ 
  double *residual_history;        /* If !0 stores residual at iterations*/
  int    res_hist_size;            /* Size of residual history array */
  int    res_act_size;             /* actual amount of data in residual_history
				      */

  /* --------User (or default) routines (most return -1 on error) --------*/
  void (*usr_monitor)();            /* returns control to user after residual
                                      calculation, allows user to, for 
                                      instance, print residual norm, etc. */
  int (*converged)();
  void *(*BuildSolution)();        /* Returns a pointer to the solution, or
				      calculates the solution in a 
				      user-provided area. */
  void *(*BuildResidual)();        /* Returns a pointer to the residual, or
				      calculates the residual in a 
				      user-provided area.  */
  VECntx *vc;                      /* Vector routines */
  int (*adjust_work_vectors)();    /* should pre-allocate the vectors*/
  void (*amult)(),                 /* application of matrix multiply A*/
       (*binv)(),                  /* application of preconditioner B*/
       (*matop)();                 /* application of AB or BA; if not set,
			                         amult and binv will be used */
  void (*tamult)(),                /* transposes of the same         */
       (*tbinv)(),                  
       (*tmatop)();                

  /*------------ Major routines which act on ITcontext-----------------*/
  ITMETHOD  method;            /* type of solver */
  int  (*solver)();            /* actual solver */
  void (*setup)();
  void (*closedown)();
  void (*adjustwork)();
  void *MethodPrivate;          /* holder for misc stuff associated 
                                   with a particular iterative solver */

  /* ----------------Default work-area management -------------------- */
  int  nwork;
  void **work;

  /* ----------------Keep track of the amount of work----------------- */
  int  nmatop, namult, nbinv, nvectors, nscalar;

  /* ------------Contexts for the user-defined functions-------------- */
  void *matP,       /* Matrix operations */
       *monP,       /* User Monitor */
       *cnvP;       /* Convergence tester */
} ITCntx;

/* --------------Calling sequence of user provided routines---------------    
*  
* void usr_monitor(ITCntx *, void *usrP, int Iteration, double res)
* int  converged(ITCntx *, void *usrP, int Iteration, double res) 
*/

/* -------------Calling sequence of Basic library routines ----------------
* 
* solver(ITCntx *,void *usrP)
* setup(ITCntx *, void *usrP);
* closedown(ITCntx *, void *usrP);
* adjustwork(ITCntx *, void *usrP);
*
*/
#endif


