
/*   Example Program to test Iterative Solver Library */
/*  This is a very simple test and by no means exhaustive */
/*  It tests    Richardson
                Chebychev 
                Conjugate Gradient 
                Gmres
                transpose free QMR
    for the symmetric model Laplacian in one dimension 
    it has two preconditioners, Jacobi and Symmetric Gauss-Seidel
*/

#include <stdio.h>
#include "tools.h"
#include "iter/itall.h"
#include "vectors/vector.h"
#include <ctype.h>
#include <math.h>

/*------------------------------------------------------------*/
/*        This amult is for the discrete laplacian in 1-d     */
/*------------------------------------------------------------*/
void amult( np, v1, v2 )
int     *np;
double  *v1, *v2;
{
int    n = *np;
double *p1, *p2;

p1   = v1;
p2   = v2;
*p2++= 2* *p1 - *(p1+1);
p1++;
n--;
n--;
while (n--) {
    *p2++ = 2.0 * *p1 - *(p1-1) - *(p1+1);
    p1++;
    }
*p2++= 2.0 * *p1 - *(p1-1);
}
/*------------------------------------------------------------*/
/* This routine computes the energy norm of the residual directly from
   amult and the known right-hand-side.  Note that many methods use a
   recursion relation to find the energy norm of the residual; a bug in
   the implementat can produce misleading results.  In addition, depending
   on the form of the preconditioning, the residual reported may not be
   correct for the original problem 

   norm  - energy norm of ?
   rnorm - norm of ax - b
 */
/*------------------------------------------------------------*/
void ActualResiduals( itP, n, norm, rnorm, iflag )
ITCntx *itP;
int       n, iflag;
double    *norm, *rnorm;
{
  double *x, *y, *z, enorm, rrnorm, h, xp, u, *xx;
  int i;

  /* malloc work space */
  y = (double *)MALLOC( 3*n * sizeof(double) );
  z = y + n;
  x = ITGetSolution(itP);
  if (ITGetPreconditionerSide(itP) == 1 && !iflag) {
      xx = (double *)ITGetSolution(itP);
      x  = z + n;
      for (i=0; i<n; i++) x[i] = xx[i];
      ITUnwindPre( (void *)&n, itP, (void *)x, (void *)z );
      }
      
  h = 1.0/(1.0 + (double) n);
  xp = h;
  for ( i=0; i<n; i++ ) {
      u = .5*xp*(1.0 - xp)/(h*h);
      y[i] = x[i] - u;
      xp += h;
  }
  amult( &n, y, z );
  enorm = 0.0; 
  for ( i=0; i<n; i++ ) { enorm += y[i]*z[i]; }
  *norm  = sqrt( enorm );

  /* Exact (unpreconditioned) residual.  RHS is identically one */
  amult( &n, x, z );
  rrnorm = 0.0;
  for ( i=0; i<n; i++) { rrnorm += (z[i] - 1.0) * (z[i] - 1.0); }
  *rnorm = sqrt( rrnorm );
  FREE( y );
}
/*------------------------------------------------------------*/
/* this prints the energy norm of the error at each iteration */
/*------------------------------------------------------------*/
void usr_monitor(itP,np,N,rnorm)
ITCntx *itP;
int       N; /* iteration number */
int       *np; /* size of vector */
double    rnorm;
{
  int    err, n= *np;
  double enorm, rrnorm;
  if (ITMethod(itP) == ITGMRES || ITMethod(itP) == ITTCQMR) {
    printf("%3d %14.6e \n",N,rnorm);
    return;
  }
  ActualResiduals( itP, n, &enorm, &rrnorm, 0 ); 
  printf("%3d %14.7e %14.7e %14.7e\n",N,rnorm,enorm, rrnorm);
}
void PrintHeader()
{
printf( " it   Estimated-R     EnergyNorm     ActualNorm\n" );
}
/*-------------------------------------------------------------*/
/* symmetric Guess-Seidel preconditioning                      */
/* there are MUCH better ways to do this                       */
/*-------------------------------------------------------------*/
void binv(np,v1,v2)  
int    *np;          
double *v1,*v2;
{
  int    i, n = *np;
  double *z;
  z = (double *)MALLOC( n * sizeof(double) );
  z[0] = .5*v1[0];
  for ( i=1; i<n; i++ ) z[i] = .5*(v1[i] + z[i-1]);
  amult(np,z,v2); CHKERR(1);
  for ( i=0; i<n; i++ ) v2[i] = -v2[i] + v1[i];
  v2[n-1] = .5*v2[n-1];
  for ( i=n-2; i>=0; i-- ) v2[i] = .5*(v2[i] + v2[i+1]);
  for ( i=0; i<n; i++ ) v2[i] = v2[i] + z[i];
  FREE( z );
}
/*-------------------------------------------------------------*/
/* SSOR preconditioning with the optimal relaxation factor     */
/* for SOR                                                     */
/*-------------------------------------------------------------*/
void ssor(np,v1,v2)  
int    *np;          
double *v1,*v2;
{
  int    i, n= *np;
  double *z,omega = 1.88401813635332; /* omega is for n = 50 */
  z = (double *)MALLOC( n * sizeof(double) );
  z[0] = .5*v1[0];
  for ( i=1; i<n; i++ ) z[i] = .5*(v1[i] + omega*z[i-1]); 
  for ( i=0; i<n; i++ ) z[i] = 2.0*z[i];
  v2[n-1] = .5*z[n-1];
  for ( i=n-2; i>=0; i-- ) v2[i] = .5*(z[i] + omega*v2[i+1]);
  FREE( z );
}
/* ---------------------------------------------------------------*/
/*             converts string to lower case                      */
/* ---------------------------------------------------------------*/
void  ToLower(name)
char   *name;
{
  while (*name) {
      if (isupper(*name)) *name = tolower(*name);
      name++;
      }
}
/*-------------------------------------------------------------*/
/*  Charactor string names for the accelerators                */
/*-------------------------------------------------------------*/
static char *Names[] = {"Richardson",
                         "Chebychev",
                         "Conjugate Gradient",
                         "GMRES",
                         "Tony Chan's transpose free QMR",
			 "BiCG-Stab",
			 "CGS",
			 "Freund's transpose free QMR" };
/*-------------------------------------------------------------*/
main(Argc,Args)
int  Argc;
char *Args[];
{
  ITCntx *itP;
  double    *x,*b,scale;
  double    emin,emax;
  double    norm, rnorm;
  int       its,i,j,*np, n = 50, runs, defaults, iname;
  ITMETHOD  itmethod;

  /* np will serve as our pointer to the user context, in this simple
     example since the amult and binv routines do not require any
     additional information we do not need to use a full C 
     structure. */
  np = &n;

  if (Argc > 1) {
    runs     = Argc - 1;
    defaults = 0;
  }
  else {
    runs     = 7;
    defaults = 1;
  }
  for ( j=1; j<=runs; j++ ) {

    /* determine which accelerators the user wants to run */
    if (!defaults) {
      ToLower(Args[j]);
      if (!strncmp(Args[j],"richardson",2))     
         { itmethod = ITRICHARDSON; iname = 0;}
      else if (!strncmp(Args[j],"chebychev",2)) 
         {itmethod = ITCHEBYCHEV; iname = 1;}
      else if (!strncmp(Args[j],"cgs",3))
	  { itmethod = ITCGS; iname = 6;}
      else if (!strncmp(Args[j],"cg",2))       
         { itmethod = ITCG; iname = 2;}
      else if (!strncmp(Args[j],"gmres",2))    
         { itmethod = ITGMRES; iname = 3;}
      else if (!strncmp(Args[j],"tcqmr",2))    
         { itmethod = ITTCQMR; iname = 4;}
      else if (!strncmp(Args[j],"bcgs",4))
	  { itmethod = ITBCGS; iname = 5;}
      else if (!strncmp(Args[j],"tfqmr",2))
	  { itmethod = ITTFQMR; iname = 7;}
      else {
        printf("Usage: example solvertype\n");
        printf("Where solvertype is one of \n");
        printf("CGS, BCGS, Richardson, Chebychev, CG, GMRES, TCQMR, TFQMR\n");
        exit(0);
      }
    }
    else { /* default to running all of them */
      if (j == 1)      itmethod = ITRICHARDSON;
      else if (j == 2) itmethod = ITCHEBYCHEV;
      else if (j == 3) itmethod = ITCG;
      else if (j == 4) itmethod = ITGMRES;
      else if (j == 5) itmethod = ITTCQMR; 
      else if (j == 6) itmethod = ITBCGS;
      else if (j == 7) itmethod = ITCGS;
      else if (j == 8) itmethod = ITTFQMR;
      iname = j - 1;  
    }

    /* create the ITCntx variable, this is the place where all the 
       information used in the iterative process is hidden. */

    itP = ITCreate(itmethod); CHKERR(1);

    /* Allocate rhs and solution */
    x = (double *)MALLOC( n * sizeof(double) );
    b = (double *)MALLOC( n * sizeof(double) );

    /* set default vector functions */
    DVSetDefaultFunctions( ITVectorCtx( itP ) );

    /* set parameters for solution */
    ITSetRelativeTolerance(itP,1.e-10);
    ITSetIterations(itP,25);
    ITSetCalculateEigenvalues(itP);
    ITSetAmult(itP,amult);
    /* Set no preconditioner */
    ITSetBinv(itP,ITVectorCtx( itP )->copy);  
    ITSetMonitor(itP,usr_monitor,(void *)0); 

    /* set locations of rhs and solution vectors */
    ITSetSolution(itP,x);
    ITSetRhs(itP,b);

    /* initial rhs to 1 */
    DVset((VEDefaultUsrCntx*)np,1.0,b);

    ITGMRESSetDirections(itP,25);
    ITSetUp(itP,np); CHKERR(2);

    /* Jacobi preconditioner */
    printf("%%Test of %s with Jacobi \n",Names[iname]);

    /* initialize solution to 0 */
    DVset((VEDefaultUsrCntx*)np,0.0,x);
    ITRichardsonSetScale(itP,.4995);
    ITChebychevSetEigenvalues(itP,3.99,.0037);
   
    PrintHeader();
    its = ITSolve(itP,np); 
    if (its == -1) { printf("Error in Solver\n");}
    ActualResiduals( itP, n, &norm, &rnorm, 1 );
    printf("%%Its %d Actual (unpreconditioned) residual %e\n",its,rnorm);

    /* Symmetric Gauss-Seidel preconditioner */
    printf("%%Test of %s with Symmetric Gauss-Seidel \n",Names[iname]);
    ITSetBinv(itP,binv);

    /* initialize solution to 0 */
    DVset((VEDefaultUsrCntx*)np,0.0,x);
    ITRichardsonSetScale(itP,.4995);
    ITChebychevSetEigenvalues(itP,1.001,.0074);
 
    PrintHeader();
    its = ITSolve(itP,np); 
    if ( its == -1 ) {printf("Error in Solver\n");}

#ifndef __MSDOS__
    if ( itmethod == ITCG ) {
      ITCGGetEigenvalues(itP,its-2,&emax,&emin); 
      printf("%%Condition Number for G-S preconditioner %g\n",emax/emin); 
    }
#endif

    ActualResiduals( itP, n, &norm, &rnorm, 1 );
    printf("%%Its %d Actual (unpreconditioned) residual %e\n",its,rnorm);

    /* SSOR preconditioner */
    printf("%%Test of %s with SSOR \n",Names[iname]);
    ITSetBinv(itP,ssor);

    /* initialize solution to 0 */
    DVset((VEDefaultUsrCntx*)np,0.0,x);
    ITRichardsonSetScale(itP,.41);
    ITChebychevSetEigenvalues(itP,4.57,.27);

    PrintHeader();
    its = ITSolve(itP,np); 
    if ( its == -1 ) { printf("Error in Solver\n");}
    ActualResiduals( itP, n, &norm, &rnorm, 1 );
    printf("%%Its %d Actual (unpreconditioned) residual %e\n",its,rnorm );

    /* cleanup after ourselves so as to not waste memory. */
    ITDestroy(itP,np);
  }
}
