#ifndef lint
static char SCCSid[] = "@(#) ./blog/bmerge.c 07/23/93";
#endif

/* #define DODEBUG */

#if defined(LOGCOMMEVENT)
#undef LOGCOMMEVENT
#endif
#if defined(LOGCOMMALL)
#undef LOGCOMMALL
#define LOGCOMMTRACE
#define LOGCOMMCOUNTS
#define LOGCOMMSUMMARY
#endif
/* Eventually turn this into LOGCOMMDISABLE */

#include "tools.h"
#include <stdio.h>
#if !defined(solaris)
#include <strings.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include "comm/comm.h"
#include "blog/blog.h"
#include "system/system.h"

/*
   This file contains routines to merge the log data together into a single
   file.  This code handles the case that the logs may not fit into a single 
   processor by breaking the log into a sequence of parts, and doing 
   an incremental merge of them (basically, the out-of-core merge 
   algorithm).  Since we are using all of the processors in the 
   processor set to manage the merge, we need to mark completion.
   We do this by sending an empty message.

   The actual algorithm (pipelined merge):

   srcs = 1 + has_lchild + has_rchild
   load buffers Left, Right (including time of first entry)
   while (srcs) {
       if (lta <= ltb) {
           if (lta <= ltc) OutputFrom(A)
	   else            OutputFrom(C)
	   }
       else {
           if (ltb <= ltc) OutputFrom(B)
	   else            OutputFrom(C)
	   }
       }
   flushoutput

   For each buffer, we have:
      Current time (lt.)
      Current pointer (lp.)
      Current endpoiner (le.)

   The output routine is
      If data won't fit in buffer, flush buffer
      Add to output buffer
      Update: lt. lp.
      If lp. > le. reload input buffer; check for end (set lt. to INF,
                                        decr srcs

   This is complicated by the fact that the individual records do NOT store
   the processor id, but the merged records need to to allow the processor
   id's to be recovered.  Also, the issue of time-stamps is a delicate one.

   Currently, this code DOES merge the records but does not correctly
   generate the process id fields.
   This is because the individual blog records do NOT contain the 
   process id fields.  The data that is generated by this code
   contains the process id field at the end of the regular BLOG_HEADER
   record.

   Finally, we need to generate the HEADER records first.  These contain:
   title (who, when, etc)
   number of events
   number of processors
   number of event types
   starttime
   endtype

   Notes on the ugliness of this code.
   The desire to provide variable length records causes much of the ugliness
   in this code.

   Not yet done:
   We need to move any negative events to their proper place in the list; 
   that is, we assume that the events are sorted by time; that may not be true
   of user-defined events and states.
 */
#define BLOG_BUF_SIZE 700
typedef struct {
    int    *p, *plast;          /* Pointers to current and last entry ?? */
    int    buf[BLOG_BUF_SIZE];  /* Holds blog buffer */
    double t;                   /* Time of current entry */
    int    (*reload)();         /* routine and context used to reload buf */
    void   *reload_ctx;
    } BLOGMBuf;

/* Here is a structure to hold the original header and the source procid */
typedef struct {
    BLOG_HEADER header;
    int         procid;
    } BLOG_MERGE_HEADER;

/* Compute the time for the current record */
static double timeoffset = 1.0e-6;
#define COMPUTETIME(p) ((p)->time.s1[0] + (p)->time.s1[1] * timeoffset)
#define TIME_INF 1.0e300

/*
   These hold the routines to generate the specific log-files.  The default
   is alog format.
 */
extern void BLOG2AlogHeader(), BLOG2AlogHeaderFull(), BLOG2AlogOutput();
static void (*OutHeader)() = BLOG2AlogHeaderFull;
static void (*OutHeaderShort)() = BLOG2AlogHeader;
static void (*OutBody)()   = BLOG2AlogOutput;
/*
   Routine to set and get the output routines.
 */
BLOGSetOutputRoutines( header, headershort, body )
void (*header)(), (*headershort)(), (*body)();
{
OutHeader      = header;
OutHeaderShort = headershort;
OutBody        = body;
}

BLOGGetOutputRoutines( header, headershort, body )
void (**header)(), (**headershort)(), (**body)();
{
*header        = OutHeader;
*headershort   = OutHeaderShort;
*body          = OutBody;
}

void BLOGOutput();

/* 
  Finally, the flush-output routines.  If there is a parent, send to it.
  Otherwise, write out the data.
 */
static FILE *fp;
static int  procid = -1;

void BLOGMergeSetup( fname )
char *fname;
{
procid = PImytid;
if (procid == 0)
    fp = fopen( fname, "w" );
}

/* 
   There are two routines to reload the buffer.  One gets data from
   another processor (FromChild); the other from the internal buffer 
 */

int BLOGReloadFromChild( b, msgtype, srcs )
int msgtype, *srcs;
BLOGMBuf *b;
{
int ln;
BLOG_HEADER *h;

PIbrecv( msgtype, b->buf, BLOG_BUF_SIZE*sizeof(int), MSG_INT );
ln = PIsize();
if (ln == 0) {
#ifdef DODEBUG
    printf( "[%d] End of data from %d\n", PImytid, PIfrom() ); 
#endif
    b->t  = TIME_INF;
    *srcs = *srcs - 1;
    return 0;
    }
else {
#ifdef DODEBUG
    printf( "[%d] data from %d\n", PImytid, PIfrom() );
#endif
    b->p     = b->buf;
    b->plast = b->p + (ln / sizeof(int));
    h        = (BLOG_HEADER *)(b->p);
    b->t     = COMPUTETIME(h);
    }
return 1;
}
int BLOGReloadFromChildL( b, srcs )
BLOGMBuf *b;
int      *srcs;
{
return BLOGReloadFromChild( b, 100, srcs );
}
int BLOGReloadFromChildR( b, srcs )
BLOGMBuf *b;
int      *srcs;
{
return BLOGReloadFromChild( b, 101, srcs );
}

typedef struct _blog { struct _blog *next; int size; } BLOG_BLOCK;
static BLOG_BLOCK *bblock;

int BLOGReloadFromData( b, srcs )
BLOGMBuf *b;
int      *srcs;
{
BLOG_HEADER       *h;
BLOG_MERGE_HEADER *mh;
int         n, i, *bp, 
            *dp;             /* Destination; repacked as 
				(header)(procid)(fields) */

if (bblock) {
#ifdef DODEBUG
    printf( "[%d] loading from data\n", PImytid );
#endif
    /* We have to insert the procid as an int after the header and before
       any vfields */
    dp   = b->buf;
    n    = bblock->size;
    bp   = (int *)(bblock + 1);
    i    = 0;
    while (i < n) {
	h  = (BLOG_HEADER *)bp;
	mh = (BLOG_MERGE_HEADER *)bp;

	/* Negative events have all times set to zero */
	if (h->event < 0) 
	    BLOGZEROTIME(h);
	MEMCPY( dp, bp, sizeof(BLOG_HEADER) );
	/* Increase len by 1 and insert the procid */
	((BLOG_HEADER *)dp)->len = h->len + 1;
	dp[BLOGHEADERSIZE]       = procid;
	MEMCPY( dp + BLOGHEADERSIZE + 1, bp + BLOGHEADERSIZE, 
	        (h->len - BLOGHEADERSIZE) * sizeof(int) );
	/* Increment the lengths (dp includes the procid) */
	i    += h->len;
	bp   += h->len;
	dp   += h->len + 1;
 	}
    b->p     = b->buf;
    b->plast = dp;
    h        = (BLOG_HEADER *)(b->p);
    b->t     = COMPUTETIME(h);
    bblock   = bblock->next;
    return 1;
    }
else {
#ifdef DODEBUG
    printf( "[%d] no more local data\n", PImytid );
#endif
    b->t = TIME_INF;
    b->p = b->plast = b->buf;
    *srcs = *srcs - 1;
    return 0;
    }
}

static int parent = -1;

extern BLOG_BLOCK *xx_BLOG_get_blog_ptr();

/* This is the parallel merge code */
void BLOGParallelMerge( flushoutput )
void (*flushoutput)();
{
int      srcs, lchild, rchild, tlast, myid, np, mtype, am_left;
BLOGMBuf *ba, *bb, *bc, *bout;
void BLOGFlushOutput();

#ifdef DODEBUG
printf( "[%d] About to get tree nodes\n", PImytid ); 
#endif
myid = PImytid;
np   = PInumtids;
PISetTreeNodes( myid, np, &lchild, &rchild, &parent, &am_left );

/* Use the default flush routine if none provided */
if (!flushoutput)
    flushoutput = BLOGFlushOutput;

/* Generate the header */
BLOGGenerateHeader( );

/* On to the business of generating the logfile */
bblock = xx_BLOG_get_blog_ptr();

ba      = NEW(BLOGMBuf);         CHKPTR(ba);
bb      = NEW(BLOGMBuf);         CHKPTR(bb);
bc      = NEW(BLOGMBuf);         CHKPTR(bc);
bout    = NEW(BLOGMBuf);         CHKPTR(bout);
bout->p     = bout->buf;
bout->plast = bout->buf + BLOG_BUF_SIZE;

srcs = 1;
ba->reload = BLOGReloadFromData;
(*ba->reload)( ba, &srcs );

mtype = (myid & 0x1) ? 100 : 101;

if (lchild >= 0) {
    srcs += 1;
    bb->reload = BLOGReloadFromChildL;
    (*bb->reload)( bb, &srcs );
    }
else
    bb->t = TIME_INF;

if (rchild >= 0) {
    srcs += 1;
    bc->reload = BLOGReloadFromChildR;
    (*bc->reload)( bc, &srcs );
    }
else
    bc->t = TIME_INF;

#ifdef DODEBUG
printf( "[%d] About to start loop\n", PImytid );
#endif
while (srcs > 0) {
#ifdef DODEBUG
    printf( "[%d] %e %e %e\n", PImytid, ba->t, bb->t, bc->t );
#endif
    if (ba->t <= bb->t) {
	if (ba->t <= bc->t) BLOGOutput( ba, bout, flushoutput, mtype, &srcs );
	else                BLOGOutput( bc, bout, flushoutput, mtype, &srcs );
	}
    else {
	if (bb->t <= bc->t) BLOGOutput( bb, bout, flushoutput, mtype, &srcs );
	else                BLOGOutput( bc, bout, flushoutput, mtype, &srcs );
	}
    }
#ifdef DODEBUG
printf( "[%d] About to flush output\n", PImytid );
#endif
if ((int)(bout->p - bout->buf) > 0) {
    (*flushoutput)( bout->buf, (int)(bout->p - bout->buf), parent, mtype );
    }
if (parent >= 0) {
    /* Send end of data message */
    (*flushoutput)( bout->buf, 0, parent, mtype );
    }
FREE(bout);
FREE(ba);
FREE(bb);
FREE(bc);
/* Make sure that everyone has finished before exiting */
PIgsync( PSAllProcs );
}

/* Output from the buffer */    
void BLOGOutput( b, bout, flushoutput, mtype, srcs )
BLOGMBuf *b, *bout;
void     (*flushoutput)();
int      mtype, *srcs;
{
int         ln;
BLOG_HEADER *h;
int         pp[100];

#ifdef DODEBUG
if (PImytid > 0)
    printf( "[%d] about to start BLOGOutput\n", PImytid ); 
#endif
h  = (BLOG_HEADER *)(b->p);
ln = h->len;
#ifdef DODEBUG
if (PImytid > 0)
    printf( "[%d] ln = %d\n", PImytid, ln );
#endif
if (ln <= 0) {
    printf( "[%d] Error in log file; length of entry = %d\n", PImytid, ln );
    *srcs = *srcs - 1;
    return;
    }
if (parent >= 0) {
    if (ln + bout->p > bout->plast) {
#ifdef DODEBUG
	printf( "[%d] about to flush output\n", PImytid );
#endif
	flushoutput( bout->buf, (int)(bout->p - bout->buf), parent, mtype );
	bout->p = bout->buf;
	}
    MEMCPY( bout->p, b->p, ln*sizeof(int) );
    bout->p += ln;
    }
else {
    /* Repack the buffer */
    if (h->event > -100) {
	MEMCPY( pp, b->p, sizeof(BLOG_HEADER) );
	((BLOG_HEADER *)pp)->len--;
	MEMCPY( pp + BLOGHEADERSIZE, b->p + BLOGHEADERSIZE + 1,
	       (h->len - BLOGHEADERSIZE) * sizeof(int) );
	(*OutBody)( fp, b->p[BLOGHEADERSIZE], pp );
	}
    }
b->p    += ln;
if (b->p >= b->plast) {
#ifdef DODEBUG
    printf( "[%d] About to reload data\n", PImytid );
#endif
    if (!(*(b->reload))( b, srcs )) {
	/* Check for end of data */
	b->t = TIME_INF;
	return;
	}
    }
/* compute new t */
h    = (BLOG_HEADER *)(b->p);
b->t = COMPUTETIME(h);
}

/* 
  Finally, the flush-output routines.  If there is a parent, send to it.
 */
void BLOGFlushOutput( p, len, parent, mtype )
int *p, len, parent, mtype;
{
if (parent >= 0) {
#ifdef DODEBUG
    printf( "[%d] Sending %d ints to %d\n", PImytid, len, parent ); 
#endif
    fflush( stdout );
    PIbsend( mtype, p, len * sizeof(int), parent, MSG_INT );
    }
else
    fflush( fp );
}

/* Start times in 0-1, end in 2-3 */
void OrderTime( val, work, n )
int      n;
unsigned long *val, *work;
{
if (work[0] < val[0] || (work[0] == val[0] && work[1] < val[1])) {
    val[0] = work[0];
    val[1] = work[1];
    }
if (work[2] > val[2] || (work[2] == val[2] && work[3] > val[3])) {
    val[2] = work[2];
    val[3] = work[3];
    }
}
    
/* 
   Generate the header 
 */
BLOGGenerateHeader()
{
int           buf[2], work[2];
unsigned long ubuf[4], uwork[4];

/* Get the local info */
BLOGGetStatistics( &buf[0], &buf[1], &ubuf[0], &ubuf[2] );
PIgisum( buf, 2, work, ALLPROCS );
/* Get the min/max time with a special function */
gsetopT( ubuf, 4, uwork, ALLPROCS, sizeof(unsigned long), MSG_INT, 
	 OrderTime );
if (procid == 0) {
    (*OutHeader)( fp, procid, buf[0], buf[1], PInumtids, ubuf, ubuf + 2 );
    }

}

BLOGGetStatistics( nevents, ne_types, starttime, endtime )
int           *nevents, *ne_types;
unsigned long starttime[2], endtime[2];
{
BLOG_BLOCK        *bl;
int               xx_i, n, *bp;
BLOG_HEADER       *ap;
int               ne, net;
double            tstart, tend, ttest;
SYusc_time_t      *t;

ne  = 0;
net = 0;
tstart = TIME_INF;
tend   = -1.0;

bl = xx_BLOG_get_blog_ptr();
while (bl) {
    n      = bl->size;
    bp     = (int *)(bl + 1);
    xx_i   = 0;
    while (xx_i < n) {
	ap    = (BLOG_HEADER *)bp;
	if (ap->event >= 0) {
	    ne++;
	    ttest = SYuscValue( &ap->time );
	    if (ttest < tstart) {
		tstart       = ttest;
		starttime[0] = ap->time.s1[0];
		starttime[1] = ap->time.s1[1];
		}
	    if (ttest > tend) {
		tend         = ttest;
		endtime[0]   = ap->time.s1[0];
		endtime[1]   = ap->time.s1[1];
		}
	    }
	xx_i += ap->len;
	bp   += ap->len;
	}
    bl = bl->next;
    }
*nevents  = ne;
*ne_types = net;
}
