#ifndef lint
static char SCCSid[] = "@(#) ./sparse/chunck.c 07/23/93";
#endif

/*  
   The blocks used by the allocation routines in pool.c are kept in 
   linked list when freed; if they are needed again the one at the end
   of the list is retrived. This can reduce the numbers of calls to 
   malloc and free.

   The first part of each chunk contains a pointer to the previous
   chunck.

   As an enhancement, we use routines to preallocated a number of blocks of
   fixed size.  The user can use this to optimize for the manipulation of large
   matrices.  Experiments with assembling large
   matrices have shown that the incremental allocation is a significant
   source of inefficiency in creating matrices.

 */

static void **LastChunck      = 0;
static int  ChunckCount       = 0;
static int  __ChuncksReused   = 0;
static int  __ChuncksMalloced = 0;

#include <stdio.h>
#include "tools.h"
#include "sparse/pool.h"
#include "sparse/sppriv.h"
#include "system/sbcnst.h"

/* Constant-block allocator control */
static void *sb = 0;
/* Current problem: active chuncks are freed by POOL, not ChunchFree, since
   the pool blocks may be larger than this "BASIC_CHUNCK" in size.
 */   

/* Since the intel has no virtual memory, we trade space-efficiency for time */
#if !defined(intelnx)
#define USE_SBCNST
#endif

#ifdef USE_SBCNST
#define GETSPACE     SBalloc(sb)
#define FREESPACE(p) SBfree(sb,p)
#else
#define GETSPACE     MALLOC(BASIC_CHUNCK_IN_BYTES)
#define FREESPACE(p) FREE(p)
#endif
/*
   SpiChunckMalloc: returns a pointer to a BASIC_CHUNCK size of space.
 */
void *SpiChunckMalloc(m)
int m;
{
  void **ptr;

#ifdef USE_SBCNST
  if (!sb) {
      sb = SBinit( BASIC_CHUNCK_IN_BYTES, 0, 8 );
      CHKPTRN(sb);
      }
#endif  
  if (!ChunckCount) {
    ptr = (void **) GETSPACE; if (!ptr) {SETERRNOMEM; return 0;}
    __ChuncksMalloced++;
  }
  else {
    __ChuncksReused++;
    ptr       = LastChunck;
    LastChunck = *ptr;
    ChunckCount--;
  }
  /* printf( "Chunck alloc returning chunck at %x\n", (char *)ptr ); */
  return (void *) ptr;
}

/*  
  SpiChunckFree - return a BASIC_CHUNCK size of space to a free list
*/
void SpiChunckFree(ptr)
void **ptr;
{
  if (!LastChunck) {
    ChunckCount = 1;
    LastChunck = ptr;
    *ptr = 0;
  }
  else {
    *ptr      = (void *) LastChunck;
    LastChunck = ptr;
    ChunckCount++;
  }
}

/* 
  SpChunckFreeAll - Frees all the free  BASIC_CHUNCK size of space
*/
void SpChunckFreeAll( )
{
  void **ptr;
  while (LastChunck) {
    ptr       = LastChunck;
    LastChunck = *ptr;
    FREESPACE(ptr);
  }
  ChunckCount = 0; 
}

/* Release a chunck back to the block allocator (rather than to the chunck
   pool) */
void SpChunckRelease( p )
void *p;
{
#ifdef USE_SBCNST
SBrelease( sb, p );    
#else
FREE( p );
#endif
}
/* Remove any unused SB chuncks */
void SpChunckFlush()
{
#ifdef USE_SBCNST
if (sb)
    SBFlush( sb );
#endif
}
/* 
  SpChunckPrintUsage - Prints the usage of "chunck" space.
*/
void SpChunckPrintUsage(fp)
FILE *fp;
{
  fprintf(fp,"ChunksMalloced %d ChunksReused %d \n",
             __ChuncksMalloced,__ChuncksReused); 
#ifdef USE_SBCNST
  if (sb) {
      fprintf( fp, "SB Blocks in use\n" );
      SBDump( fp, sb );
      }
#endif
}

/*
   Deallocate any free chuncks
 */
SpChunckClear( )
{
#ifdef USE_SBCNST
  void **ptr;
  while (LastChunck) {
    ptr       = LastChunck;
    LastChunck = *ptr;
    SBrelease( sb, ptr );
  }
  ChunckCount = 0; 
  if (sb) {
      SBReleaseAvail( sb );
      SBFlush( sb );
      }
#endif
}
/*
   SpChunckPrealloc - Preallocate space for nz non-zeros.
 */
SpChunckPrealloc( nz )
int nz;
{
int bytes, blocks;

if (nz < 0) return;

/* Compute the number of blocks needed */

#ifdef USE_SBCNST
bytes  = nz * (sizeof(int) + sizeof(double));
blocks = (bytes + BASIC_CHUNCK_IN_BYTES - 1) / BASIC_CHUNCK_IN_BYTES;
if (!sb) {
    sb = SBinit( BASIC_CHUNCK_IN_BYTES, 0, 8 );
    CHKPTR(sb);
    }
SBPrealloc( sb, blocks );
#endif
}
