#ifndef lint
static char SCCSid[] = "@(#) ./solvers/svdb.c 07/23/93";
#endif

#include <stdio.h>
#include "tools.h"
#include "solvers/svctx.h"
#include "system/nreg.h"
extern NRList *__SVList;

/*
    This file contains routines for a simplified and standardized database
    of results, consisting of a directory file containing a summary of the
    run and a file containing the residual history

    To simplify the output, each entry is a multi-line block.  An attempt 
    was made to use a single line, but this was really too restrictive, 
    particularly in generating enough information to identify a particular 
    run (time, system, problem choice) and details about the run (number of
    matrix-vector products) etc.
 */
/* @
    SVDBHeader - Writes out the header containing problem information.

    Input parameters:
.   sd  - svdb structure (from SVDBOpen)
.   svctx - solver context
.   str   - user string (may be null)

    Notes:
    The output is... (should be a summary listing)
@ */
void SVDBHeader( sd, svctx, str )
SVDB  *sd;
SVctx *svctx;
char  *str;
{
char   *svname;
char   *itname = "";
int    flops;
double srate, svrate;
int    sf  = 0;     /* setup flops -- later */
char   name[25];

if (!sd || !sd->dir) return;

svname = NRFindName( __SVList, svctx->type );
strncpy( name, svname, 24 );
if (svctx->itctx) {
    itname = ITGetMethodName( svctx->method );
    if (itname) {
	strncat( name, " ", 24 );
        strncat( name, itname, 24 );
	}
    }
SVGetFlops( svctx, flops );

srate  = 0.0;
svrate = 0.0;
if (svctx->t_setup > 0)
    srate = ((double) sf) / (1.e6 * svctx->t_setup);
if (svctx->t_solve > 0)
    svrate = ((double) flops) / (1.e6 * svctx->t_solve);

if (svctx->its >= 0) 
    fprintf( sd->dir, "*%24s %11.3e %11.3e %4.1fMf %4.1fMf %4d",
	   name, svctx->t_setup, svctx->t_solve, srate, svrate, svctx->its );
else
    fprintf(sd->dir,  "%24s %11.3e %11.3e %4.1fMf %4.1fMf   -1",
	   name, svctx->t_setup, svctx->t_solve, srate, svrate );
if (sd->hname) 
    fprintf( sd->dir, " %s\n", sd->hname );
else 
    fprintf( sd->dir, " -No-History-\n" );

/* Add details about the system on the second line */
{
char hname[128], arch[12], date[100];
SYGetHostName( hname, 128 );
SYGetArchType( arch, 12 );
SYGetDate( date );
fprintf( sd->dir, " %s %s %s\n", hname, arch, date );
}
}

/* @
   SVDBHistory - Outputs the residual history, if any

   Input Parmeters:
.  sd  - svdb structure (from SVDBOpen)
.  svctx - solver context
.  fmat  - format (see below).  May be null.

   Notes:
   The fmat string gives the format to use; the iteration count always comes
   before the residual value.  The default fmat is "%d %e\n".  

   The residual values are the ESTIMATES provided by the iterative accelerator.
   For some methods, these are at best approximate.
@ */
void SVDBHistory( sd, svctx, fmat )
SVDB  *sd;
SVctx *svctx;
char  *fmat;
{
int i, na;

if (!sd || !sd->history) return;

if (svctx->type != SVLU) {
    if (svctx->itctx->residual_history) {
    	/* Output the residual history */
	na = svctx->itctx->res_act_size;
	if (!fmat) fmat = "%d %e\n";
	for (i=0; i<na; i++) {
	    fprintf( sd->history, fmat, i, svctx->itctx->residual_history[i] );
	    }
        }
    }
}

#ifndef MAXFILELEN
#define MAXFILELEN 1024
#endif
#ifndef MAXTITLELEN 
#define MAXTITLELEN 1024
#endif

/* @
   SVDBOpen - Opens a simplified solvers database.

   Input Parameters:
.  Argc - pointer to argc
.  argv - argument vector

   Returns:
   Pointer to database structure.  If no database selected, it returns null.

   Notes:
   This routine takes command line arguments and opens a simplified 
   solvers database entry.  This package is provided to give a common
   method for running tests and saving their results.  Eventually, a GUI that
   understands these files will be provided to give users of SLES a simple
   way to compare methods for solving linear systems.

   The arguments recognized are
$    -fdb name        - for the directory file
$    -fdbres name     - for the residual file.  If the name is a directory, 
                        then a filename that is unique within the directory
			will be generated (the file name is contained in the
			directory file specified by -fdb).
$    -fdbtitle string - specifies name string to add to entry

   A good approach is the following

   ... -fdb mytest/summary -fdbres mytest

   This puts the residual history into a file in the directory mytest; the
   name is chosen by SVDBOpen.
   
   See also:
   SVDBHeader, SVDBHistory, SVDBClose
@ */
SVDB *SVDBOpen( Argc, argv )
int  *Argc;
char **argv;
{
FILE *fp;
SVDB *sd;
char dirname[MAXFILELEN];
char histname[MAXFILELEN];
char title[MAXTITLELEN];

sd   = NEW(SVDB);    CHKPTRN(sd);
sd->dir     = 0;
sd->history = 0;
sd->hname   = 0;
if (SYArgGetString( Argc, argv, 1, "-fdb", dirname, MAXFILELEN )) {
    if (SYIsDirectory( dirname ))
	strcat( dirname, "/summary" );
    sd->dir     = fopen( dirname, "a+" );	CHKPTRN(sd->dir);
    }
if (SYArgGetString( Argc, argv, 1, "-fdbres", histname, MAXFILELEN )) {
    if (SYIsDirectory( histname )) {
	/* Convert to file name in the directory */
	if (!sd->dir) {
	    strcat( histname, "/history" );
	    }
	else {
	    int ncnt, c;
	    char num[10];
	    /* Use the number of lines in the directory as the file name */
	    ncnt = 0;
	    fseek( sd->dir, 0L, 0 );
	    while ((c = getc( sd->dir)) != EOF) if (c == '\n') ncnt++;
	    sprintf( num, "/%d", ncnt );
	    strcat( histname, num );
	    }
	}
    sd->history = fopen( histname, "w" );	  CHKPTRN(sd->history);
    sd->hname   = MALLOC( strlen(histname) + 1 ); CHKPTRN(sd->hname);
    strcpy( sd->hname, histname );
    }
if (SYArgGetString( Argc, argv, 1, "-fdbtitle", title, MAXTITLELEN )) {
    sd->title   = MALLOC( strlen(title) + 1 );  CHKPTRN(sd->title);
    strcpy( sd->title, title );
    }
else
    sd->title = 0;

if (!sd->dir && !sd->history) {
    FREE( sd );
    sd = 0;
    }
return sd;
}

void SVDBClose( sd )
SVDB *sd;
{
if (!sd) return;

if (sd->hname)
    FREE( sd->hname );
if (sd->title) 
    FREE( sd->title );
if (sd->dir) 
    fclose( sd->dir );
if (sd->history)
    fclose( sd->history );
FREE( sd );
}

