#ifndef LINT
static char SCCSid[] = "@(#) ./solvers/dd/rgmsh2d.c 07/23/93";
#endif

/*
   This file contains (relatively) simple routines for interpolation and 
   restriction on a 2-d regular mesh.

   These routines handle the following kinds of interpolations:

   Piecewise constant

   Piecewise linear

   Piecewise linear with the edges handled separately.

   Note that there are two (major) cases to handle.  In the first, the 
   "boundary" is not part of the problem.  The grid then looks like this

   + - - - - - - - +
   | . . . . . . . |
   | . C . C . C . |
   | . . . . . . . |
   | . C . C . C . |
   | . . . . . . . |
   | . C . C . C . |
   | . . . . . . . |
   + - - - - - - - +
   
   Here, the "coarse grid" is every second point in each direction.  Note
   that the "missing" boundary points require special handling.  The big 
   problem with this case is that the "missing" elements can cause a lot of
   trouble in the code.  There are several obvious ways to handle this.
   One is to copy the mesh to one that includes the zeros and work with that.
   Another is to use "if" tests at every point.  Yet another is to effectively
   unroll the loops and handle the various special cases.

   Another is to effectively ignore the regular structure of the meshes and
   form interpolation matrices directly; the restriction matrices are then
   just the transposes of the interpolation matrices.

   Finite difference problems involving non-Dirichlet conditions often 
   include the boundaries.  In this case, the grid (complete with coarse
   grid problem) looks like

   C . C . C . C . C
   . . . . . . . . .
   C . C . C . C . C
   . . . . . . . . .
   C . C . C . C . C
   . . . . . . . . .
   C . C . C . C . C
   . . . . . . . . .
   C . C . C . C . C

   This APPEARS more regular but in fact still requires special care at
   the boundaries because the boundary conditions need to be 
   interpolated/restricted separately from the interiors.

   Because of all of this and also because we'd like to use these routines
   in a distributed-memory sense, we define a general interface that
   handles a general "interior"

   Finally, for many operations, the appropriate restriction operator
   is the transpose of the interpolation operator.  Because of this, 
   we define this operators together.

 */

/* @
   Msh2dInterpolateCell - Interpolates a 2-d cell with linear interpolation

   Input Parameters:
.  fg    - fine grid.  Must be first point to be set (see below)
.  fgx,fgy - size of cell
.  fxl   - "declared" size in x of fine grid
.  cg    - coarse grid
.  cgl   - "declared" size in x of coarse grid
.  top   - 0 if top should be interpolated
.  right - 0 if right should be interpolated
.  bottom - 0 if bottom should be interpolated
.  left   - 0 if left should be interpolated

   In the case where the bottom or left is not to be interpolated, it is
   necessary to adjust fg to correspond to the first point to be set.  This
   is required since the mesh fg may not even exist at the "corner" points.
@ */
void Msh2dInterpolateCellLinear( fg, fxl, fgx, fgy, cg, cgl, 
				 top, bottom, left, right )
double *fg, *cg;
int    fxl, fgx, fgy, cgl, top, bottom, left, right;
{
double le, re;          /* left and right edge values */
double leftDf, rightDf; /* Left and right differences */
double edgeDf;          /* Difference along an edge */
int    i, j, k;
int    sj = bottom, ej = fgy - top;
int    si = left,   ei = fgx - right;

leftDf  = (cg[cgl]   - cg[0]) / (fgy - 1.0);
rightDf = (cg[cgl+1] - cg[1]) / (fgy - 1.0);

for (j=sj; j<ej; j++) {
    le     = cg[0] + j * leftDf;
    re     = cg[1] + j * rightDf;
    edgeDf = (re - le) / (fgx - 1.0);
    k = 0;
    for (i=si; i<ei; i++) {
	fg[k++] = le + i * edgeDf;
	}
    fg += fxl;
    }
}

/* @
    Msh@dInterpolateBilinear - Interpolate from a coarse grid to a fine grid

    For now, isopen gives the choice of the above grids.  We'll probably 
    want to expand this into the top, bottom, left, right.  Note that if
    isopen is 1, then along the edges we need to pass a faked-up coarse
    grid that contains 0 values.

    cx,cy - number of CELLS
    fgx,fgy - number of points along a cell side (inclusive)
@ */
void Msh2dInterpolateBilinear( fg, fxl, fx, fy, cg, cxl, cx, cy, 
			       fgx, fgy, isopen )
double *fg, *cg;
int    fxl, fx, fy, cxl, fgx, fgy, isopen;
{
int    i, j, top, bottom, left, right;
double dcg[4]; 

if (cx <= 0 || cy <= 0) return;

if (isopen) {
    double *fgUpperleft = fg + (fgy-2)*fxl + (cy-2)*(fgy-1)*fxl;
    double *cgUpperleft = cg + (cy-2) * cxl;

    /* This requires more care since the edges are tricky.  For now, we will
       cull out each of the cases.  Eventually, this should be unified 
       (particularly when we plan to do more than 2-d)
     */
    /* First, get the interior out of the way */
    Msh2dInterpolateBilinear( fg + (fgx-2) + (fgy-2)*fxl, fxl, fx, fy, 
			      cg, cxl, cx - 2, cy - 2, fgx, fgy, 0 );
		   
    /* The corners */
    /* Bottom Left */
    dcg[0] = 0.0; dcg[1] = 0.0; dcg[2] = 0.0; dcg[3] = cg[0];
    Msh2dInterpolateCellLinear( fg, fxl, fgx, fgy, dcg, 2, 
			        cy == 1, 1, 1, cx == 1 );

    /* Bottom Right */
    if (cx > 1) {
	dcg[0] = 0.0; dcg[1] = 0.0; dcg[2] = cg[cx-2]; dcg[3] = 0.0;
	Msh2dInterpolateCellLinear( fg + (fgx-2) + (cx-2)*(fgx-1), 
				    fxl, fgx, fgy, 
				    dcg, 2, cy == 1, 1, 0, 1 );
	}
    if (cy > 1) {
	/* Top Left */
	dcg[0] = 0.0; dcg[1] = cg[(cy-2)*cxl]; dcg[2] = 0.0; dcg[3] = 0.0;
	Msh2dInterpolateCellLinear( fgUpperleft, fxl, fgx, fgy, dcg, 2, 
			            1, 0, 1, cx == 1 );
	
	/* Top Right */
	dcg[0] = cg[(cy-2)*cxl+cx-2]; dcg[1] = 0.0; dcg[2] = 0.0; dcg[3] = 0.0;
	Msh2dInterpolateCellLinear( fgUpperleft+(fgx-2)+(cx-2)*(fgx-1)+1, 
				    fxl, fgx, fgy, 
				    dcg, 2, 1, 0, 1, 1 );
	}
    /* The edges */
    /* Bottom */
    for (i=1; i<cx-1; i++) {
	dcg[0] = 0.0; dcg[1] = 0.0; dcg[2] = cg[i-1]; dcg[3] = cg[i];
	Msh2dInterpolateCellLinear( fg + (fgx-2) + (i-1)*(fgx-1) + 1, 
				    fxl, fgx, fgy, 
				    dcg, 2, cy == 1, 1, 1, 0 );
	}
    /* Top */
    for (i=1; i<cx-1; i++) {
	dcg[0] = cgUpperleft[i-1]; dcg[1] = cgUpperleft[i]; 
	dcg[2] = 0.0; dcg[3] = 0.0;
	Msh2dInterpolateCellLinear( fgUpperleft + (fgx-2) + (i-1)*(fgx-1) + 1, 
				    fxl, fgx, fgy, 
				    dcg, 2, 1, 0, 1, 0 );

	}
    /* Left */
    for (j=1; j<cy-1; j++) {
	dcg[0] = 0.0; dcg[1] = cg[(j-1)*cxl]; 
	dcg[2] = 0.0; dcg[3] = cg[j*cxl];
	Msh2dInterpolateCellLinear( fg + (fgy-2)*fxl + (j-1)*(fgy-1)*fxl,
				    fxl, fgx, fgy, 
				    dcg, 2, 1, 0, 1, 1 );
	}
    /* Right */
    for (j=1; j<cy-1; j++) {
	dcg[0] = cg[cx-2+(j-1)*cxl]; dcg[1] = 0.0;
	dcg[2] = cg[cx-2+j*cxl];     dcg[3] = 0.0;
	Msh2dInterpolateCellLinear( fg + (cx-1)*(fgx-1)+(fgy-2)*fxl + 
				    (j-1)*(fgy-1)*fxl,
				    fxl, fgx, fgy, 
				    dcg, 2, 1, 0, 1, 1 );
	}
    }
else {
    for (j=0; j<cy; j++) {
	for (i=0; i<cx; i++) {
	    Msh2dInterpolateCellLinear( fg + i * (fgx-1), fxl, fgx, fgy, 
				       cg + i, cxl, j!=cy-1, 0, 0, i!=cx-1 );
	    }
	cg += cxl;
	fg += (fgy - 1) * fxl;
	}
    }
}

/*
   Restrition/Aggregation routines.
 */

/*
   I'll start with a simple piecewise-constant aggregation routine
  
   This is the idea.  We want to simulate the effect of integrating over
   the fine mesh, using some weighting function.

   Piecewise constant -
   For each fine-grid cell, take the average of the surrounding points, divide
   by the number of cells, and add 1/4 to each surrounding coarse-grid point.
   This has the property that a uniform value is preserved.  Again, the 
   case of open boundaries makes things a little more complicated.

   Also, since the 
 */
void Msh2dRestrictPCopen( fg, fxl, fx, fy, cg, cxl, cx, cy, 
			  fgx, fgy )
double *fg, *cg;
int    fxl, fx, fy, cxl, fgx, fgy;
{
int    i, j, i1, j1;
double sum, *fgp;

/* For each coarse-grid point */
for (j=1; j<cy; j++) {
    for (i=1; i<cx; i++) {
	/* Do the interior of the surrounding cells.  Note that we 
	   end up accessing the values on the mesh roughly 4 times,
	   but in compensation, we get a simpler code */
	sum = 0.0;
	fgp = fg + (j-1)*(fgy-1)*fxl + (i-1)*(fgx-1);
	for (j1=0; j1<2*(fgy-1)-1; j1++) {
	    for (i1=0; i1<2*(fgx-1)-1; i1++) {
		sum += fgp[i1];
		}
	    fgp += fxl;
	    }
	sum = sum / ( (2.0*(fgx-1)-1)*(2.0*(fgy-1)-1) ); 
	cg[i-1+(j-1)*cxl] = sum;
	}
    }
}

/* This is the version for the closed mesh */
void Msh2dRestrictPCclosed( fg, fxl, fx, fy, cg, cxl, cx, cy, 
			    fgx, fgy )
double *fg, *cg;
int    fxl, fx, fy, cxl, fgx, fgy;
{
int    i, j, i1, j1;
double sum, *fgp, *fgg;

for (j=0; j<cy; j++) 
    for (i=0; i<cx; i++) 
	cg[i+j*cxl] = 0.0;

/* For each cell */
for (j=0; j<cy; j++) {
    for (i=0; i<cx; i++) {
	/* Do each cell element */
	sum = 0.0;
	fgp = fg + j*(fgy-1)*fxl + fxl + i*(fgx-1) + 1;
	for (j1=0; j1<fgy-1; j1++) {
	    for (i1=1; i1<fgx-1; i1++) 
		sum += (fgp[i1] + fgp[i1+1] + 
			fgp[i1+fxl] + fgp[i1+1+fxl]) / 4.0;
	    fgp += fxl;
	    }
	sum = sum / ((fgx-1)*(fgy-1));
	sum = sum / 4.0;
	cg[i+j*cxl]       += sum;
	cg[i+1+j*cxl]     += sum;
	cg[i+(j+1)*cxl]   += sum;
	cg[i+1+(j+1)*cxl] += sum;
	}
    }
}
