#ifndef lint
static char SCCSid[] = "@(#) ./nonlin/min2/neq_trust.c 07/23/93";
#endif

#include "tools.h"
#include "nonlin/nlall.h"
#include "nonlin/min2/ntctx.h"
#include <math.h>

#define MIN(a,b) ( ((a)<(b)) ? a : b )

/*
     This subroutine solves the system of equations f(x) = 0
     by a trust region verion of a truncated Newton method.
 */
int NLntrustSolve( nlP, usrP )
NLCntx *nlP;
void   *usrP;
{
double  fnorm, delta, fnormt, xnorm;
double  actred, prered, dirder, decrease, gpnorm, pnorm, flnorm;
void    *x, *f, *p, *xtrial, *ftrial, *flinear, **work, *tmp;
int     info, i, N = nlP->max_it;
NT2Cntx *ntctx;
FILE    *fp = nlP->fp;
double  *res = nlP->residual_history;

ntctx = (NT2Cntx *)nlP->MethodPrivate;

x       = nlP->vec_sol;
work    = VGETVECS(nlP->vc,usrP,5); CHKPTRV(work,0);
f       = work[0]; p = work[1]; xtrial = work[2]; ftrial = work[3]; 
flinear = work[4];

if (nlP->initial_guess) (*nlP->initial_guess)(nlP,usrP,x);  /* x <- x_0  */
else VSET(nlP->vc,usrP,0.0,x);

/* unpack the data */
/* Evaluate the function at the starting point and calculate its norm. */
(*nlP->fun)( nlP, usrP, x, f );
nlP->nfunc++;

VNORM(nlP->vc,usrP,f,&fnorm);  nlP->nvectors++;
if (res) *res++ = fnorm;

/* Initialize trust region parameter delta */
delta = 0.1 * fnorm;
info  = 0;
if (nlP->usr_monitor) 
    (*nlP->usr_monitor)(nlP,usrP,x,f,fnorm);
for ( i=0; i<N && !info; i++ ) {
    /* Setup the step code */
    if (nlP->stepSetup)
	(*nlP->stepSetup)( nlP, usrP, x );
    do {
	(*nlP->stepCompute)( nlP, usrP, x, f, p, fnorm, delta, nlP->trunctol,
			     &gpnorm, &pnorm );
	nlP->nsteps++;
	/* Store the direction p and x + p. Calculate the norm of p. */
	VWAXPY(nlP->vc,usrP,1.0,x,p,xtrial);  /* xtrial <- x + p */
	nlP->nvectors ++;

	/* Evaluate the function at x + p and calculate its norm. */
	(*nlP->fun)( nlP, usrP, xtrial, ftrial );  nlP->nfunc++;

	VNORM(nlP->vc,usrP,ftrial,&fnormt);   nlP->nvectors++;

	if (fp) fprintf(fp,"%d f %g ftrial %g pnorm %g ",i,fnorm,fnormt,pnorm);
	if (fp) fprintf(fp,"gpred %g delta %g\n",gpnorm,delta);

	/* Compute the scaled actual reduction. */
	if (0.1*fnormt <= fnorm) 
	    actred = 1.0 - pow(fnormt/fnorm,2.0);
        else
	    actred = - 1.0;

	/* Compute the scaled predicted reduction */
	(*ntctx->Jv)( ntctx->Jcntx, p, flinear );
	VAXPY( nlP->vc, usrP, 1.0, f, flinear );   /* flinear = f + fjac*p */
	
	VNORM( nlP->vc, usrP, flinear, &flnorm );    nlP->nvectors += 2;

	if (fp) fprintf( fp, "flinear %f\n", flnorm );
	prered = 1 - pow( flnorm/fnorm, 2.0 );

	/* Compute the scaled directional derivative. */
	VDOT( nlP->vc, usrP, f, flinear, &dirder );  nlP->nvectors++;
	dirder = ( dirder / fnorm ) / fnorm - 1.0;

	/* Update the step bound. */
	if (actred <= 0.25 * prered) {
	    if (actred >= 0.0)
		decrease = 0.5;
	    else
		decrease = 0.5*dirder/(dirder + 0.5*actred);
	    if (0.1*fnormt >= fnorm || decrease < 0.1) 
		decrease = 0.1;
	    delta = decrease*MIN(delta,pnorm/0.1);
	    }
	else {
	    /* if (flnorm == 0.0)
		delta *= 2.0; */
	    if (actred >= 0.75 * prered) 
		delta = pnorm / 0.5;
	    }
	
	/* Test for successful step. */
	if (actred >= 1.0e-4 * prered) {
	    /* Successful iteration. Update x, f, and their norms. */
	    if (res) *res++ = fnormt;
	    tmp = xtrial;  xtrial = x; x = tmp;
	    tmp = ftrial;  ftrial = f; f = tmp;
	    VNORM(nlP->vc,usrP,x,&xnorm);    nlP->nvectors++;
	    fnorm = fnormt;
	    if (fp) fprintf( fp, "Accepting step\n" );
	    }
	else {
	    if (fp) fprintf( fp, "Rejecting step\n" );
	    }

	ntctx->delta  = delta;
	ntctx->actred = actred;
	ntctx->prered = prered;
	if (nlP->usr_monitor) 
	    (*nlP->usr_monitor)(nlP,usrP,x,f,fnorm);

	/* Tests for convergence. */
	info = (*nlP->converged)( nlP, usrP, xnorm, pnorm, fnorm );
	
	} while (actred < 1.0e-4 * prered && !info);
    if (nlP->stepDestroy)
	(*nlP->stepDestroy)( nlP, usrP );
    }
    
VFREEVECS(nlP->vc,usrP,work,5);
if (x != nlP->vec_sol)
    VCOPY( nlP->vc, usrP, x, nlP->vec_sol );
return i + 1;
}

/*ARGSUSED*/
int NLTR2DefaultConverged( nlP, usrP, xnorm, pnorm, fnorm )
NLCntx *nlP;
void   *usrP;
double  xnorm, pnorm, fnorm;
{
int    info;
double ftol = nlP->ftol;
double xtol = nlP->xtol;
double actred, prered, delta;
double epsmch = 1.0e-14;    /* need to fix this */
NT2Cntx *ntctx;

ntctx = (NT2Cntx *)nlP->MethodPrivate;
actred = ntctx->actred;
prered = ntctx->prered;
delta  = ntctx->delta;

info = 0;
/* Test for successful convergence */
if (fabs(actred) <= ftol && prered <= ftol && 0.5*actred <= prered) info = 1;
if (delta <= xtol*xnorm) info = 2;
if (fabs(actred) <= ftol && prered <= ftol &&
    0.5*actred <= prered && info == 2) info = 3;
if (info != 0) return info;

/* Tests for termination and stringent tolerances. (failure and stop) */
if (nlP->nfunc >= nlP->max_func) info = 5;
if (fabs(actred) <= epsmch && prered <= epsmch && 0.5*actred <= prered) 
    info = 6;
if (delta <= epsmch*xnorm) info = 7;
return info;
}

/*------------------------------------------------------------*/
void NLNewtonTR2SetUp(nlP,usrP)
NLCntx *nlP;
void   *usrP;
{
NT2Cntx   *neP = (NT2Cntx *)nlP->MethodPrivate;

  if (!nlP->stepCompute) {
    fprintf(stderr,"NLNewtonTR2SetUp needs stepCompute!\n");
    SETERR(1);
  }
  if (!nlP->fun) {
    fprintf(stderr,"NLNewtonTR2SetUp needs fun!\n");
    SETERR(1);
  }
  if (!neP->Jv) {
    fprintf(stderr,"NLNewtonTR2SetUp needs Jv!\n");
    SETERR(1);
  }
  if (!nlP->vc) {
    fprintf(stderr,"NLNewtonTR2SetUp needs vector ops!\n");
    SETERR(1);
  }
}
/*------------------------------------------------------------*/
void NLNewtonTR2Destroy(nlP)
NLCntx *nlP;
{
  FREE(nlP->MethodPrivate);
  FREE(nlP);
}


NLCntx *NLNewtonTR2Create()
{
  NLCntx    *nlP;
  NT2Cntx   *neP;

  nlP                    = NEW(NLCntx); CHKPTRN(nlP);
  NLSetDefaults( nlP );

  nlP->method            = NLNEWTONTR2;

  nlP->converged         = NLTR2DefaultConverged;
  nlP->setup             = NLNewtonTR2SetUp;
  nlP->solver            = NLntrustSolve;
  nlP->destroy           = NLNewtonTR2Destroy;

  nlP->trunctol          = 1.e-10;

  neP                    = NEW(NT2Cntx); CHKPTRN(neP);
  neP->delta             = .2;
  nlP->MethodPrivate     = (void *) neP;

  return nlP;
}

void NLSetJv( nlP, Jv, Jctx )
NLCntx    *nlP;
void      (*Jv)(), *Jctx;
{
NT2Cntx   *neP = (NT2Cntx *)nlP->MethodPrivate;

neP->Jv    = Jv;
neP->Jcntx = Jctx;
}
