#ifndef lint
static char SCCSid[] = "@(#) ./iter/rich/rich.c 07/23/93";
#endif
/*          
            This implements Richardson Iteration.       
*/
#include <stdio.h>
#include <math.h>
#include "tools.h"
#include "iter/itctx.h"
#include "iter/itfunc.h"
#include "iter/richctx.h"
#include "iter/itpriv.h"

/*+
    ITRichardsonCreate - Creates an ITCntx variable for the Richardson
    method. The preferred calling sequence is ITCreate(ITRICHARDSON).
+*/
ITCntx *ITRichardsonCreate()
{
  ITCntx           *itP;
  ITRichardsonCntx *richardsonP;

  itP = NEW(ITCntx); CHKPTRV(itP,0);
  ITSetDefaults( itP );
  richardsonP = NEW(ITRichardsonCntx); CHKPTRV(richardsonP,0);
  itP->MethodPrivate = (void *) richardsonP;

  itP->method               = ITRICHARDSON;
  itP->max_it               = 50;
  itP->right_inv            = 0;
  itP->calc_res             = 1;
  itP->guess_zero           = 0;
  itP->use_pres             = 0;
  itP->rtol                 = 1.e-5;
  itP->atol                 = 1.e-50;
  richardsonP->scale        = 1.0;

  itP->setup      = ITRichardsonSetUp;
  itP->solver     = ITRichardsonSolve;
  itP->adjustwork = ITDefaultAdjustWork;
  itP->closedown  = ITRichardsonDestroy;
  return(itP);
}

/*+
    ITRichardsonSetUp - Called after a call to ITRichardsonCreate() or
    ITCreate(ITRICHARDSON), allocates space needed in the Richardson 
    solution. Preferred call sequence is ITSetUP(itP,usrP).

    Input Parameters:
.   itP - the iterative context
.   usrP - the user context
+*/
void ITRichardsonSetUp(itP,usrP)
ITCntx *itP;
void   *usrP;
{
  if (itP->method != ITRICHARDSON) {
      SETERRC(1,"Attempt to use Richardson Setup on wrong context"); return;}
 
  /* check user parameters and functions */
  if (itP->right_inv) {
      SETERRC(2,"Right-inverse preconditioning not supported for Richardson"); 
      return; }
  if (ITCheckDef( itP )) return;
 
  /* get work vectors from user code */
  ITDefaultGetWork( itP, usrP, 2 );
}

/*@
    ITRichardsonSetScale - Called after a call to ITRichardsonCreate() or 
    ITCREATE(ITRICHARDSON), sets the "relaxation" factor for 
    Richardson, if this routine is not called the relaxation 
    factor defaults to 1.0

    Input Parameters:
.   itP - the iterative context
.   scale - the relaxation factor
@*/
void ITRichardsonSetScale(itP,scale)
ITCntx *itP;
double scale;
{
  ITRichardsonCntx *richardsonP;
  if (itP->method != ITRICHARDSON) return;
  richardsonP = (ITRichardsonCntx *) itP->MethodPrivate;
  richardsonP->scale = scale;
}

/*+
    ITRichardsonSolve - Called after a call to ITRichardsonCreate() or 
    ITCreate(ITRICHARDSON) and the call to ITRichardsonSetUp() or ITSetUp().
    Actually solves the linear system using Richardson's method.
    Preferred calling sequence ITSolve(itP,usrP).

    Input Parameters:
.   itP - the iterative context
.   usrP - the user context

    Returns:
    the number of iterations required or -1 on error.
+*/
int  ITRichardsonSolve(itP,usrP)
void   *usrP;
ITCntx *itP;
{
  int               i = 0,maxit,pres, brokeout = 0, hist_len, cerr;
  double            rnorm,scale,*history;
  void              *x,*b,*r,*z;
  ITRichardsonCntx  *richardsonP;
  richardsonP = (ITRichardsonCntx *) itP->MethodPrivate;

  x       = itP->vec_sol;
  b       = itP->vec_rhs;
  r       = itP->work[0];
  z       = itP->work[1];
  history = itP->residual_history;
  hist_len= itP->res_hist_size;
  scale   = richardsonP->scale;
  maxit   = itP->max_it;
  pres    = itP->use_pres;

  if (!itP->guess_zero) {                       /*   r <- b - A x     */
    MM(x,r);
    DAYPX(-1.0,b,r);
  }
  else COPY(b,r);

  for ( i=0; i<maxit; i++ ) {
     PRE(r,z);                               /*   z <- B r         */
     if (itP->calc_res) {
	if (!pres) NORM(r,&rnorm);           /*   rnorm <- r'*r    */
	else       NORM(z,&rnorm);           /*   rnorm <- z'*z    */
        if (history && hist_len > i) history[i] = rnorm;
        MONITOR(rnorm,i);
        if (CONVERGED(rnorm,i)) {brokeout = 1; break;}
     }
   
     DAXPY(scale,z,x);                       /*   x  <- x + scale z */
     MM(x,r);                                /*   r  <- b - Ax      */
     DAYPX(-1.0,b,r);
  }
  if (itP->calc_res && !brokeout) {
    if (!pres) NORM(r,&rnorm);                /*   rnorm <- r'*r    */
    else {
      PRE(r,z);                               /*   z <- B r         */
      NORM(z,&rnorm);                         /*   rnorm <- z'*z    */
    }
    if (history && hist_len > i) history[i] = rnorm;
    MONITOR(rnorm,i);
  }
  if (history) itP->res_act_size = (hist_len < i) ? hist_len : i;

  itP->namult += (i+1);
  itP->nbinv  += (i+1);
  itP->nvectors += 4*(i+1);

  return RCONV(i+1);
}

/*+
    ITRichardsonDestroy - Destroys a iterative context variable obtained
    by a call to ITRichardsonCreate() or ITCreate(ITRICHARDSON).
    Prefered calling sequence ITDestroy().

    Input Parameters:
.   itP          the iterative context obtained from ITRichardsonCreate
.                or ITCreate(ITRICHARDSON);
.   usrP         the user context
+*/
void ITRichardsonDestroy(itP,usrP)
ITCntx *itP;
void   *usrP;
{
  ITRichardsonCntx *richardsonP;
  richardsonP = (ITRichardsonCntx *) itP->MethodPrivate;
 
  ITDefaultFreeWork( itP, usrP );
  
  /* free the context variables */
  FREE(richardsonP); FREE(itP);
}
