#ifndef lint
static char SCCSid[] = "@(#) ./ilu/iflow.c 07/23/93";
#endif

#include <math.h>
#include "tools.h"

/* Private structure for the flow-ordering routines.  This is not a
   compact structure; if desired, lots could be done to compress it.
   The fields are:
   marked = -1 if unmarked, index of node if marked.
   next,prev = index of next and previous.  Either marked or unmarked,
               depending on value of marked
 */               
typedef struct {
    int marked;
    int next, prev;
    } ILUFLOWW;

/*@
    IFFlowOrder - Finds a "flow" ordering, given an array of "next" nodes.

    Input Parameters:
.   n     - number of nodes
.   nxn   - for each of those, the "next node".  -1 if there is no next node
.   start - starting nodes (see below)
.   ns    - number of starting nodes

    Output Parameters:
.   order - ordering of nodes

    Notes:
    This routine is provided to generate orderings in cases where it is
    relatively easy to generate a next node but not necessarily easy to
    generate an ordering (which may not, for example, duplicate nodes).

    Generating start:
    This should be a list of starting nodes (if none are provided, some
    will be picked by the routine).  For problems arising from PDEs, good
    choices of starting nodes include the inflow boundaries.
    
    Generating nxn:
    Since it will rarely be the case that there is an obvious unique next
    node for each node, several approaches can be used to generate nxn.
    One is to take the local "best", breaking ties in any way.  Another
    is to choose amoung the possibilities at random, weighting the choice
    appropriately.  Another approach that could have been used here is
    to call a user-function for each node and give that function a context
    value that it could use in deciding on the next node.  This would allow
    Bresenham-like line following algorithms to be used.

    Nodes start from 0 and go to n-1.

    Algorithm:
    Two lists, one of marked and one of unmarked nodes, are maintained.
    Elements are marked either by being selected as starting nodes or by
    following (through the nxn array) another node.

    Use:
    This routine should be used with approximate factorizations to provide
    better performance.  Use the order parameter as the row and column
    mapping (or as the column mapping only if an unsymmetric ordering is
    appropriate) in SpSetMappings for the factored
    matrix before computing the factor with SpFindILUFactor.
@*/    
void IFFlowOrder( n, nxn, start, ns, order )
int n, *nxn, *start, ns, *order;
{
ILUFLOWW *flow;
int      unmarked = 0;     /* List of unmarked nodes starts with first */
int      marked   = -1;    /* List of marked nodes starts empty */
int      mtail    = -1;
int      cur, next, prev, i;
	
/* Allocate initial storage areas */
flow = (ILUFLOWW *)MALLOC( n * sizeof(ILUFLOWW) );   CHKPTR(flow);
/* Set initial list of nodes as all unmarked */
for (i=0; i<n; i++) {
    flow[i].marked = -1;
    flow[i].next   = i + 1;
    flow[i].prev   = i - 1;
    }
flow[0].prev   = -1;
flow[n-1].next = -1;    

while (unmarked >= 0) {
    /* Look for a starting node (careful of starting nodes that have already
       been used) */
    do {
        cur = IFiStartingNode( &ns, start, unmarked );
    } while (flow[cur].marked >= 0);
    do { 
    	/* We can use this node */
    	flow[cur].marked = cur;
    	prev = flow[cur].prev;
    	next = flow[cur].next;
    	/* Remove node cur from unmarked list */
    	if (prev >= 0) flow[prev].next = next;
    	else           unmarked = next;
    	if (next >= 0) flow[next].prev = prev;
    	
    	/* Add to the tail of the marked nodes */
	if (mtail < 0) {
	    /* First in marked list */
	    marked         = cur;
	    mtail          = cur;
	    flow[cur].prev = -1;
	    }
	else {
	    /* Add to end of list */
	    flow[mtail].next = cur;
	    flow[cur].prev   = mtail;
	    mtail            = cur;
	    }
	/* cur is at the end of the marked list */
	flow[cur].next = -1;

	/* For more general routines, replace this reference with
	   a routine call */
    	cur = nxn[cur];
        } while (cur >= 0 && flow[cur].marked < 0);
    }
/* Now, we order by following the marked list through and getting the
   values.  The values are just those stored in the marked field */
cur = marked;   
for (i=0; i<n; i++) {
    order[i] = flow[cur].marked;   /* Should == cur */
    cur      = flow[cur].next;
    }
FREE( flow );    
}	

/*
    Internal routine to find the next candidate starting node.
    This routine could be changed to not accept stagnation points
    until all non-stagnation points have been used.
 */
int IFiStartingNode( ns, starts, unmarked )
int *ns, *starts, unmarked;
{
if (*ns > 0) {
    *ns = *ns - 1;
    return starts[*ns];
    }
/* Else no remaining starting nodes, so return the head of the unmarked
   list */    
return unmarked;
}

/* @
    IFFlow2d - Creates a flow ordering from a velocity field

    Input parameters:
.   vx.vy - velocity field
.   mx    - declared size of velocity field
.   nx,ny - size of mesh
.   flag  - specifies type of formult to use
$            flag & 0x1 = 0 for 4 direction
$        		 1 for 8 direction
$ 	    flag & 0x2 = 0 for deterministic
$			 1 for random    

    ? Starting nodes.

    Notes:
    This routine provides an example of generating a "next node" array
    for use by IFFlowOrder.  It is useful for regular 2-dimensional meshes.

    ? Do we add the Rndm Ctx?

    We use the following connection star to choose directions:

$   6  2  3
$   4  .  1
$  12  8  9
$
   This uses bits for the various directions (e.g., 6 = 4 | 2).
@ */
void IFFlow2d( vx, vy, mx, nx, ny, nxn, flag )
double *vx, *vy;
int    mx, nx, ny, *nxn, flag;
{
int    i, j;
double Vx, Vy, AVx, AVy, r, V1, V2;
int    dir1, dir2, dir;       /* axis of dominant flow */
int    cnt, d1prob;

cnt = 0;
for (j=0; j<ny; j++) {
    for (i=0; i<nx; i++) {
	nxn[cnt] = -1;
    	Vx       = vx[i];
    	Vy       = vy[i];
    	AVx      = fabs( Vx );
    	AVy      = fabs( Vy );
    	/* Determine the direction */
    	if (AVx >= AVy) {
	    dir = (Vx > 0.0) ? 1 : 4;
	    V1  = AVx;
	    V2  = AVy;
    	    }
    	else {
	    dir = (Vy > 0.0) ? 2 : 8;
	    V1  = AVy;
	    V2  = AVx;
    	    }

    	if (flag & 0x1) {
    	    /* 8 point */
  	    if (flag & 0x2) {
	        r = SYDRndm( r );
		dir = (r < d1prob) ? dir1 : dir2;
	        if (r < (V1 - V2) / V1) {
		    /* take 1 else 2;*/
		    }
	        }
	    else {
  	    	/* take 1 else 2; */
	        }
	    }
	else {
	    /* Four point */
  	    if (flag & 0x2) {
	        r = SYDRndm( r );
  	    	if (r < V1 / (V1 + V2)) {
		    /* take 1 else 2; */
		    }
	        }
	    else {
	    	/* take 1 else 2; */
	        }
	    }
	nxn[cnt++] = dir;
        }
    vx += mx;
    vy += mx;
    }
}	
