#ifndef LINT
static char SCCSid[] = "@(#) ./c2fort/header.c 07/23/93";
#endif

#include <ctype.h>
#include "tools.h"
#include "doc/doc.h"

/* 
   This is designed to work with comments in C programs.  
   It uses the standardized documentation to issue dummy routine
   definitions to allow the creation of a Fortran to C library.
 */
extern char GetSubClass();

static int NoFortMsgs = 1;
/* This says to convert char **a to int*a, and cast to (char **)*a */
static int MultipleIndirectAreInts    = 1;
static int MultipleIndirectsAreNative = 0;

/*D
  bfort - program to extract short definitions for a Fortran to C interface

  Input:
. filenames - Names the files from which lint definitions are to be extracted
. -nomsgs   - Do not generate messages for routines that can not be converted
              to Fortran.
  Author: 
  Bill Gropp
D*/
main( argc, argv )
int  argc;
char **argv;
{
char routine[MAX_ROUTINE_NAME];
char *infilename;
char outfilename[1024];
char dirname[1024];
char fname[1024], *p;
FILE *fd, *fout, *incfd;
int  nread;
char kind;
char incfile[MAX_FILE_SIZE];
char incbuffer[1024];

/* process all of the files */
strcpy( dirname, "." );
incfile[0] = 0;
SYArgGetString( &argc, argv, 1, "-dir", dirname, 1024 );
SYArgGetString( &argc, argv, 1, "-I",   incfile, 1024 );
NoFortMsgs = SYArgHasName( &argc, argv, 1, "-nomsgs" );
if (SYArgHasName( &argc, argv, 1, "-help" )) {
    fprintf( stderr, "%s - write a Fortran interface to C routines with\n", 
	     argv[0] );
    fprintf( stderr, "routines documented in the `doctext' format\n" );
    fprintf( stderr, "Optional arguments:\n" );
    fprintf( stderr, " -dir <name> - directory write output files in\n" );
    fprintf( stderr, 
	   " -nomsgs     - don't generate messages for routines that can not\n\
be converted to Fortran\n" );
    exit( 1 );
    }
/* Open up the file of public includes */    
if (incfile[0])
    incfd = fopen( incfile, "r" );
else
    incfd = 0;

argc--; argv++;
while (argc--) {
    /* Input filename */
    infilename = *argv++;
    fd = fopen( infilename, "r" );
    if (!fd) {
    	fprintf( stderr, "Could not open file %s\n", infilename );
    	continue;
        }

    /* Set the output filename */
    SYGetRelativePath( infilename, fname, 1024 );
    /* Strip the trailer */
    p = fname + strlen(fname) - 1;
    while (p > fname && *p != '.') p--;
    *p = 0;
    /* Add an extra h to include files */
    if (p[1] == 'h') {
	p[0] = 'h';
	p[1] = 0;
	}
    sprintf( outfilename, "%s/%sf.c", dirname, fname );
    /* Don't open the filename yet (wait until we know that we'll have
       some output for it) */
    fout = NULL;

    while (FoundLeader( fd, routine, &kind )) {
	/* We need this test first to avoid creating an empty file, 
	   particularly for initf.c */
	if ((kind == ROUTINE || kind == MACRO) && GetSubClass() == 'C') {
	    if (!NoFortMsgs) {
		fprintf( stderr, "%s %s can not be translated into Fortran\n",
			(kind == ROUTINE) ? "Routine" : "Macro", routine );
		}
	    SkipText( fd, routine, infilename, kind );
	    continue;
	    }
	if ((kind == ROUTINE || kind == MACRO) && fout == NULL) {
	    OutputBuf( &fout, infilename, outfilename, incfd, (char *)0 );
	    }
	if (kind == ROUTINE) {
	    OutputRoutine( fd, fout, routine, infilename, kind );
	    }
	else if (kind == MACRO) {
	    /* Eventually we can handle this by using the Synopsis to
	       construct an equivalent definition */
	    OutputMacro( fd, fout, routine, infilename );
	    }
	else if (kind == INCLUDE) {
	    strcpy( incbuffer, "#include " );
	    /* Grumble.  We'll have to fix this eventually */
	    if (routine[0] != '"' && routine[0] != '<') {
		p = routine + strlen(routine) - 1;
		if (*p == '>')
		    strcat( incbuffer, "<" );
		else if (*p == '"') 
		    strcat( incbuffer, p );
		}
	    strcat( incbuffer, routine );
	    CopyIncludeName( fd, incbuffer + strlen(incbuffer) );
	    strcat( incbuffer, "\n" );
	    OutputBuf( &fout, infilename, outfilename, incfd, incbuffer );
	    }
	}
    fclose( fd );
    if (fout) fclose( fout );
    }
return 0;
}

/* We also need to make some edits to the types occasionally.  First, note
   that double indirections are often bugs */
typedef struct {
    char *name;
    int  has_star, is_char, is_native, type, is_FILE;
    } ARG_LIST;
typedef struct {
    char type[60];
    } TYPE_LIST;

OutputRoutine( fin, fout, name, filename, kind )
FILE *fin, *fout;
char *name, *filename, kind;
{
int       is_function;
ARG_LIST  args[512];
TYPE_LIST types[60];
int       nargs, nstrings;
int       ntypes;

/* Check to see if this is a C-only routine */
if (GetSubClass() == 'C') {
    if (!NoFortMsgs) {
	fprintf( stderr, "Routine %s can not be translated into Fortran\n",
		name );
	}
    SkipText( fin, name, filename, kind );
    return;
    }

/* Skip to trailer */
SkipText( fin, name, filename, kind );

/* Get the call to the routine, including finding the argument names */
SkipWhite( fin );
ProcessArgList( fin, fout, filename, &is_function, name, args, &nargs, 0 );

SkipWhite( fin );
ProcessArgDefs( fin, fout, args, nargs, types, &ntypes, &nstrings, 0, name );
PrintBody( fout, is_function, name, nstrings, nargs, args, types );
}

/*
    This routine skips the text part of a text page.
 */        
SkipText( fin, name, filename, kind )
FILE *fin;
char *name, *filename;
char kind;
{
int  c;
char lineBuffer[MAX_LINE], *lp;
	
lineBuffer[0] = '+';   /* Sentinal on lineBuffer */
while (1) {
    lp = lineBuffer + 1;
    c  = getc( fin );
    if (c == EOF) break;
    if (c == ARGUMENT || c == VERBATIM)
	SkipLine( fin );
    else if (c == '\n')
	;
    else {
	if (isspace(c) && c != '\n')
	    SkipWhite( fin );
	else 
	    *lp++ = c;
    	/* Copy to end of line; do NOT include the EOL */
    	while ((c = getc( fin )) != EOF && c != '\n') 
    	    *lp++ = c;
    	lp--;
    	while (isspace(*lp)) lp--;
    	lp[1] = '\0';    /* Add the trailing null */
    	if (lineBuffer[1] == kind && strcmp(lineBuffer+2,"*/") == 0)
    	    break;
        }
    }
}

int SkipToSynopsis( fin, kind )
FILE *fin;
char kind;
{
int  c;
char lineBuffer[MAX_LINE], *lp;
	
lineBuffer[0] = '+';   /* Sentinal on lineBuffer */
while (1) {
    lp = lineBuffer + 1;
    c  = getc( fin );
    if (c == EOF) break;
    if (c == ARGUMENT || c == VERBATIM)
	SkipLine( fin );
    else if (c == '\n')
	;
    else {
	if (isspace(c) && c != '\n')
	    SkipWhite( fin );
	else 
	    *lp++ = c;
    	/* Copy to end of line; do NOT include the EOL */
    	while ((c = getc( fin )) != EOF && c != '\n') 
    	    *lp++ = c;
    	lp--;
    	while (isspace(*lp)) lp--;
    	lp[1] = '\0';    /* Add the trailing null */
    	if (lineBuffer[1] == kind && strcmp(lineBuffer+2,"*/") == 0)
    	    break;
	if (lp[0] == ':') {
	    lp = lineBuffer + 1;
	    while (isspace(*lp)) lp++;
	    LowerCase( lp );
	    if (strcmp( lp, "synopsis:" ) == 0) 
		return 1;
	    }
        }
    }
return 0;
}

/* Convert a string to lower case, in place */
LowerCase( s )
char *s;
{
char c;
while (*s) {
    c = *s;
    if (isascii(c) && isupper(c)) *s = tolower(c);
    s++;
    }
}

#ifdef FOO
/* Convert a string to upper case, in place */
/* Defined in tools.core/doc/doc.c */
UpperCase( s )
char *s;
{
char c;
while (*s) {
    c = *s;
    if (isascii(c) && islower(c)) *s = toupper(c);
    s++;
    }
}
#endif

/* Find the next space delimited token; put the text into token.
   The number of leading spaces is kept in nsp.
   Alpha-numeric tokens are terminated by a non-alphanumeric character
   (_ is allowed in alpha-numeric tokens) */
int FindNextANToken( fd, token, nsp )
FILE *fd;
char *token;
int  *nsp;
{
int fc, c, Nsp;

Nsp = SkipWhite( fd );

fc = c = getc( fd );
if (fc == EOF) return fc;
*token++ = c;
if (isalnum(c) || c == '_') {
    while ((c = getc( fd )) != EOF) {
	if (c != '\n' && (isalnum(c) || c == '_')) *token++ = c;
	else break;
	}
    ungetc( (char)c, fd );
    }
*token++ = '\0';
*nsp     = Nsp;
return fc;
}

OutputBuf( fout, infilename, outfilename, incfd, buffer )
FILE **fout, *incfd;
char *infilename, *outfilename, *buffer;
{
char arch[20];
if (!*fout) {
    *fout = fopen( outfilename, "w" );
    if (!*fout) {
	fprintf( stderr, "Could not open file %s\n", outfilename );
	return;
	}
    fprintf( *fout, "/* %s */\n", infilename );
    SYGetArchType( arch, 20 );
    fprintf( *fout, "/* Fortran interface file for %s */\n", arch );
    /* Turn on the base debugging */
    fprintf( *fout, "#ifndef DEBUG_ALL\n#define DEBUG_ALL\n#endif\n" );

    if (incfd) {
	int c;
	fseek( incfd, 0L, 0 );
	while ((c = getc( incfd )) != EOF) 
	    putc( (char)c, *fout );
	}
    }
if (buffer) 
    fputs( buffer, *fout );
}


/* 
   There are a number of things to watch for.  One is that leading blanks are
   considered significant; since the text is being formated, we usually dont
   agree with that. 
 */
OutputMacro( fin, fout, name, filename )
FILE *fin, *fout;
char *name, *filename;
{
int       is_function;
ARG_LIST  args[512];
TYPE_LIST types[60];
int       nargs, nstrings;
int       ntypes;
int       has_synopsis;
int       c;
int       done;

/* Check to see if this is a C-only macro */
if (GetSubClass() == 'C') {
    if (!NoFortMsgs) {
	fprintf( stderr, "Macro %s can not be translated into Fortran\n",
		name );
	}
    SkipText( fin, name, filename, MACRO );
    return;
    }

/* Skip to the synopsis in the body */
has_synopsis = SkipToSynopsis( fin, MACRO );

done = 0;
if (has_synopsis) {
    /* Get the call to the routine, including finding the argument names */
    SkipWhite( fin );
    ProcessArgList( fin, fout, filename, &is_function, name, args, &nargs, 1 );
    
    SkipWhite( fin );
    done = 
	ProcessArgDefs( fin, fout, args, nargs, types, &ntypes, &nstrings, 1, 
		        name );
    PrintBody( fout, is_function, name, nstrings, nargs, args, types );
    }
else {
    fprintf( stderr, "%s has no synopsis section\n", name );
    }
/* finish up the section */
if (!done)
    SkipText( fin, name, filename, MACRO );
}

/* Read the arg list */
ProcessArgList( fin, fout, filename, is_function, name, args, Nargs, flag )
FILE *fin, *fout;
char *filename, *name;
ARG_LIST args[512];
int      *Nargs;
int      *is_function, flag;
{
int             c;
char            token[1024], *p;
int             i, nsp, bl, leadingm;
static char     rcall[1024];
int             nargs, ln, in_args;
int             reading_function_type = 0;
int             found_name;

SkipWhite( fin );
nargs       = 0;
in_args     = 0;
p           = rcall;
c           = FindNextANToken( fin, p, &nsp );
/* 
   We check for routines that return (functions) versus ones that don't
   by looking for "void".  A special case is functions that return 
   pointers to void; we check for these by looking at the first character
   of the first token after the void 
 */
*is_function          = strcmp( p, "void" );
reading_function_type = !*is_function;

for (i=0; i<nsp; i++) putc( ' ', fout );
fputs( p, fout );
p += strlen( p );
*p++ = ' ';
leadingm = 0;    /* If a newline is encountered before this is one, AND
                    this is a macro, insert one and exit */
found_name = 0;
while (1) {
    c = FindNextANToken( fin, p, &nsp );
    if (c == EOF) {
	fprintf( stderr, "Unexpected EOF in %s\n", filename );
	return;
	}
    if (reading_function_type) {
	reading_function_type = 0;
	if (c == '*') *is_function = 1;
	}
    if (flag && c == '\n' && leadingm == 0) {
	fputs( "()", fout );
	break;
	}
    if (c == '\n') leadingm = 1;
    if (c == '(') {
	in_args += 1;
	}
    if (c == ')') {
	in_args -= 1;
	if (in_args == 0) {
	    fputs( ")", fout ); 
	    break;
	    }
	}
    if (in_args == 0) {
	if (strcmp( p, name ) == 0) {
	    /* Convert to Fortran name.  For now, this just does the
	       lowercase_ version */
	    found_name = 1;
#if defined(FORTRANCAPS)
	    UpperCase( p );
#elif defined(FORTRANUNDERSCORE)	    
	    LowerCase( p );
	    ln = strlen( p );
	    p[ln] = '_';
	    p[ln+1] = 0;
#else
	    LowerCase( p );
#endif
	    }
	else {
	    if (p[0] != '*') 
		fprintf( stderr, "%s:Did not find matching name: %s != %s\n", 
			 filename, p, name );
	    }
	}
    if (in_args == 1) {
	if (c != ',' && c != '(' && c != '\n') {
	    /* Assume that it is a name and remember it */
	    args[nargs].name     = p;
	    args[nargs].has_star = 0;
	    args[nargs].is_char  = 0;
	    args[nargs].is_FILE  = 0;
	    args[nargs].is_native= 1;   /* Unspecified args are ints */
	    args[nargs].type     = 0;
	    nargs++;
	    }
	}
    for (i=0; i<nsp; i++) putc( ' ', fout );
    fputs( p, fout );
    if (in_args) {
	p += strlen( p );
	*p++ = 0;
	}
    }

if (!found_name) {
    fprintf( stderr, "%s:Did not find routine name (may be untyped): %s \n", 
 	     filename, name );
    }

/* Handle definitions of the form "type (*Name( args, ... ))()" (this is
   function returns pointer to function returning type). */
SkipWhite( fin );
c = getc( fin );
if (c == '(') {
    SkipWhite( fin );
    c = getc(fin);
    if (c == ')') 
	fputs( "()", fout );
    else 
	ungetc( (char)c, fin );
    }
else 
    ungetc( (char)c, fin );

*Nargs = nargs;
}

/* if flag == 1, stop on empty line rather than { */
/* This needs to distinguish between pointers and values, since all
   parameters are passed by reference in Fortran.  Just to keep things
   lively, there are two ways to indicate a pointer in C:
     type *foo;
     type foo[];
 */
int ProcessArgDefs( fin, fout, args, nargs, types, Ntypes, Nstrings, flag, 
		    name )
FILE      *fin, *fout;
ARG_LIST  *args;
int       nargs;
TYPE_LIST *types;
int       *Ntypes, *Nstrings, flag;
char      *name;
{
int      c;
char     token[1024], *p;
int      i, nsp, bl, newline;
char     rcall[1024];
int      is_function;
int      ln, in_args, has_star, n_strings, is_char, nstrings,
         is_native, has_array, is_FILE;
int      ntypes;
int      done = 0;         /* set to 1 if ate end-of-definition */

newline  = 1;
if (flag) newline = 0;
has_star = 0;
has_array= 0;
is_char  = 0;
is_FILE  = 0;
nstrings = 0;
ntypes   = 0;
while (1) {
    c = FindNextANToken( fin, token, &nsp );
    if (c == EOF || token[0] == '{') break;
    /* Check for empty line; if found, exit.  Otherwise, 
       check for M * / (Macro definition) and handle that case */
    if (flag) {
	if (newline && c == '\n') break;
	if (c == MACRO) {
	    c = getc( fin );
	    if (c == '*') {
		c = getc( fin );
		if (c == '/') {
		    done = 1;
		    break;
		    }
		else { 
		    /* This won't work on all systems. */
		    ungetc( '*', fin );
		    ungetc( (char)c, fin );
		    }
		}
	    else 
		ungetc( (char)c, fin );
	    }
	}

    /* Don't output register */
    if (strcmp( token, "register" ) == 0) continue;
    /* Handle various argument features */
    if (c == '*')                  has_star++;
    else if (c == ',' || c == ';') {has_star = 0; has_array = 0;}
    else if (c == '\n') {
	newline  = 1;
	is_char  = 0;
	is_FILE  = 0;
	has_star = 0;
	has_array= 0;
	is_native= 0;
	}
    else if (newline) {
	if (strcmp( token, "char" ) == 0) is_char = 1;
	if (strcmp( token, "FILE" ) == 0) is_FILE = 1;
	is_native = 0;
	if (strcmp( token, "double" ) == 0 ||
	    strcmp( token, "int"    ) == 0 ||
	    strcmp( token, "float"  ) == 0 ||
	    strcmp( token, "char"   ) == 0 ||
	    strcmp( token, "complex") == 0 ||
	    strcmp( token, "dcomplex")== 0 ||
	    strcmp( token, "void"   ) == 0 ||
	    strcmp( token, "BCArrayPart") == 0)
	    is_native = 1;
	newline = 0;
	strcpy( types[ntypes].type, token );
	if (strcmp( token, "struct" ) == 0 || 
	    strcmp( token, "unsigned") == 0) {
	    /* Flush struct to the output */
	    for (i=0; i<nsp; i++) putc( ' ', fout );
	    fputs( token, fout );
	    c = FindNextANToken( fin, token, &nsp );
	    strcat( types[ntypes].type, " " );
	    strcat( types[ntypes].type, token );
	    }
	ntypes++;
	}

    /* Check for "[]" */
    c = getc( fin );
    if (c == '[') {
	has_star++;
	while ((c = getc(fin)) != EOF && c != ']') ;
	has_array = 1;
	}
    else
	ungetc( c, fin );

    /* Look up name */
    for (i=0; i<nargs; i++) {
	if (strcmp( token, args[i].name ) == 0) {
	    args[i].has_star = has_star;
	    args[i].is_char  = is_char;
	    args[i].is_FILE  = is_FILE;
	    args[i].type     = ntypes-1;
	    args[i].is_native = is_native;
	    if (is_char) nstrings++;
	    if (!has_star) {
		/* This makes it look nicer */
		for (i=0; i<nsp; i++) putc( ' ', fout );
		nsp = 0;
		fputs( "*", fout );
		}
	    else if (has_star > 1) {
		if (!NoFortMsgs) {
		    fprintf( stderr, "%s has multiple indirection for %s\n",
			     name, args[i].name );
		    }
		if (!MultipleIndirectsAreNative) args[i].is_native = 0;
		}
	    break;
	    }
	}
    for (i=0; i<nsp; i++) putc( ' ', fout );
    fputs( token, fout );
    if (has_array) fputs( "[]", fout );
    }
*Ntypes   = ntypes;
*Nstrings = nstrings;
return done;
}

/*
   A major question is whether "void *" should be considered the actual
   pointer or an address containing the value of the pointer (the usual "int"
   trick). 

   Since "void *" is used heavily in the communications routines to refer
   to any one of the type double*, int*, ..., I'm going to add void * to
   the list of types that are not translated
 */
PrintBody( fout, is_function, name, nstrings, nargs, args, types )
FILE      *fout;
char      *name;
int       is_function, nstrings, nargs;
ARG_LIST  *args;
TYPE_LIST *types;
{
int  i;

fputs( "{\n", fout );
if (nstrings) {
    fprintf( fout, "int d0" );
    for (i=1; i<nstrings; i++) fprintf( fout, ",d%d", i );
    fputs( ";\n", fout );
    }
/* Look for special-case translations (currently, "FILE") */
for (i=0; i<nargs; i++) {
    if (args[i].is_FILE) {
	fprintf( fout, "FILE *_fp%d = stdout;\n", i );
	}
    }

/* Generate the routine call with the return */
if (is_function) fputs( "return ", fout );
fputs( name, fout );
fputs( "(", fout );
for (i=0; i<nargs; i++) {
    if (args[i].is_FILE) 
	fprintf( fout, "_fp%d", i );
    else if (!args[i].is_native && args[i].has_star) {
	if (args[i].has_star == 1 || !MultipleIndirectAreInts) 
	    fprintf( fout, "(%s *)*((int*)%s)", 
		    types[args[i].type].type, args[i].name );
	else {
	    int j;
	    fprintf( fout, "(%s ", types[args[i].type].type );
	    for (j = 0; j<args[i].has_star; j++) fputs( "*", fout );
	    fprintf( fout, ")*((int *)%s)", args[i].name );
	    }
	}
    else {
	if (!args[i].has_star) 
	    fputs( "*", fout );
	fputs( args[i].name, fout );
	}
    if (i < nargs-1) fputs( ",", fout );
    }
for (i=0; i<nstrings; i++) fprintf( fout, ",d%d", i );
/* fputs( rcall, fout ); */
fputs( ");\n}\n", fout );
}
